#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import Tac
import Tracing
from CliPlugin import ( EthIntfCli, FabricIntfCli, IntfCli )
from CliPlugin.QosCliIntfTypes import ethOrLagOrFabricIntfPrefixes
from CliPlugin.QosCliModel import ( EcnCountersModel, EcnDelayModel,
                                   EcnDelayParametersModel,
                                   EcnIntfQueueCountersModel, EcnParametersModel,
                                   EcnQueueCounterModel, GlobalAllEcnModel,
                                   GlobalEcnModel, IntfEcnCollectionModel,
                                   IntfAllEcnCollectionModel, IntfEcnModel,
                                   IntfNonEctModel, TxQueueEcnModel,
                                   TxQueueNonEctModel, IntfAllBufferBasedDecnModel,
                                   IntfBufferBasedDecnModel, DecnParametersModel )
from CliPlugin.QosCliCommon import ( getIntfListFromMode, setIntfConfig,
                                     cliBlockingFail, QosProfileMode,
                                     defaultWeight, setQosProfileConfig )
from CliPlugin.QosCliEcn import ( EcnCounterKey, ecnCounterTable, ecnSnapshotTable,
                                  qosConfig, qosHwCounter, qosHwStatus,
                                  qosInputConfig, qosInputProfileConfig,
                                  qosSliceHwStatus, qosStatus, setTxQueueNonEct,
                                  ecnMaxDroprate, setEcnQueueCounterConfig,
                                  qosHwConfig )
from CliPlugin.QosCliServicePolicy import pmapHasDropPrecedenceAction
from QosLib import ( fabricIntfName, invalidGuaranteedBw, invalidShapeRate,
                     tacFeatureName, isLagPort )
from QosTypes import ( tacEcnDelayThreshold, tacEcnDelayThresholdUnit, tacPercent,
                      tacQueueThresholdUnit, tacTxQueuePriority )

__defaultTraceHandle__ = Tracing.Handle( 'QosCliEcnHandler' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

def setEcnDelayThresholdUs( mode, config, noOrDefaultKw=None, thresholdUs=None ):
   if noOrDefaultKw or not thresholdUs:
      config.ecnDelayThreshold = tacEcnDelayThreshold
   else:
      # Currently we store as ns internally while displaying microseconds in CLI.
      config.ecnDelayThreshold = \
          Tac.Value( 'Qos::EcnDelayThreshold',
                     tacEcnDelayThresholdUnit.ecnDelayThresholdNs,
                     thresholdUs * 1000 )

def setTxQueueEcnDelayThresholdUs( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   threshold = args.get( 'THRESHOLD' )
   if qosInputConfig.ecnDelayThreshold == tacEcnDelayThreshold:
      print( "WARNING: Delay based ECN is disabled globally." )
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         if noOrDefaultKw:
            return
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      txQueueConfig.delayEcnEnabled = not noOrDefaultKw
      setEcnDelayThresholdUs( mode, txQueueConfig, noOrDefaultKw, threshold )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if noOrDefaultKw:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if noOrDefaultKw:
                  continue
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )

         prevEcnDelayEnabled = txQueueConfig.delayEcnEnabled
         prevEcnDelayThreshold = txQueueConfig.ecnDelayThreshold
         txQueueConfig.delayEcnEnabled = not noOrDefaultKw
         setEcnDelayThresholdUs( mode, txQueueConfig, noOrDefaultKw, threshold )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.delayEcn, "Delay ECN",
                             intf ):
            intfConfig = qosInputConfig.intfConfig.get( intf )
            if intfConfig is None:
               intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            txQueueConfig.delayEcnEnabled = prevEcnDelayEnabled
            txQueueConfig.ecnDelayThreshold = prevEcnDelayThreshold
            setIntfConfig( intf )

def wredApplied( ):
   for intf in qosConfig.intfConfig:
      intfConfig = qosConfig.intfConfig[ intf ]
      for txQueue in intfConfig.txQueueConfig:
         txQueueConfig = intfConfig.txQueueConfig[ txQueue ]
         if ( txQueueConfig.wredConfig or
            ( txQueueConfig.dpConfig and
              len( txQueueConfig.dpConfig.dpWredConfig ) != 0 ) ):
            return True
   return False

def setGlobalEcn( mode, args ):
   weight = args.get( 'WEIGHT', defaultWeight )
   minThd, maxThd, unit = args[ 'THD_RANGE' ]
   timestamp = Tac.now()
   errMsg = "Cannot apply ECN configuration. WRED configuration is already present."
   if wredApplied():
      mode.addError( errMsg )
      return
   prevEcnConfig = qosInputConfig.globalEcnConfig
   if ( prevEcnConfig and prevEcnConfig.minThd == minThd and
        prevEcnConfig.maxThd == maxThd and prevEcnConfig.unit == unit and
        prevEcnConfig.weight == weight ):
      return
   qosInputConfig.globalEcnConfig = ( minThd, maxThd, unit,
                                      ecnMaxDroprate, weight )
   if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN" ):
      if prevEcnConfig:
         qosInputConfig.globalEcnConfig = ( prevEcnConfig.minThd,
                                            prevEcnConfig.maxThd, prevEcnConfig.unit,
                                            ecnMaxDroprate, prevEcnConfig.weight )
      else:
         qosInputConfig.globalEcnConfig = None

def setNoOrDefaultGlobalEcn( mode, args ):
   timestamp = Tac.now()
   prevEcnConfig = qosInputConfig.globalEcnConfig
   if not prevEcnConfig:
      return
   qosInputConfig.globalEcnConfig = None
   if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN" ):
      if prevEcnConfig:
         qosInputConfig.globalEcnConfig = ( prevEcnConfig.minThd,
                                            prevEcnConfig.maxThd, prevEcnConfig.unit,
                                            ecnMaxDroprate, prevEcnConfig.weight )
      else:
         qosInputConfig.globalEcnConfig = None

# -----------------------------------------------------------------------------------
# The "[ mls ] qos random-detect ecn allow non-ect" command, in "global config" mode.
# -----------------------------------------------------------------------------------
def configureAllowNonEct( mode, args ):
   timestamp = Tac.now()
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.ecnAllowNonEct = False
   elif qosStatus.ecnAllowNonEctWithDropPrecedence:
      qosInputConfig.ecnAllowNonEct = True
   else:
      spConfigs = [ qosInputConfig.servicePolicyConfig,
                    qosInputProfileConfig.servicePolicyConfig ]
      for spConfig in spConfigs:
         for key in spConfig:
            if pmapHasDropPrecedenceAction( key.pmapName ):
               mode.addError( "Policy-map action drop-precedence and allow "
                              "non-ect cannot be configured together." )
               return
      qosInputConfig.ecnAllowNonEct = True

   if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN" ):
      qosInputConfig.ecnAllowNonEct = not qosInputConfig.ecnAllowNonEct

# -----------------------------------------------------------------------------------
# The "[ mls ] qos random-detect ecn allow non-ect chip-based" command,
# in "global config" mode.
# -----------------------------------------------------------------------------------
def configureAllowNonEctChipBased( mode, args ):
   timestamp = Tac.now()
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.ecnAllowNonEctWithDropPrecedence = False
   else:
      qosInputConfig.ecnAllowNonEctWithDropPrecedence = True

   if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN" ):
      qosInputConfig.ecnAllowNonEctWithDropPrecedence = not \
                                    qosInputConfig.ecnAllowNonEctWithDropPrecedence

def setGlobalEcnDelayThresholdUs( mode, args ):
   threshold = args.get( 'ECNDELAYTHRESHOLDUSVALUE' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   timestamp = Tac.now()
   prevEcnDelayThreshold = qosInputConfig.ecnDelayThreshold
   setEcnDelayThresholdUs( mode, qosInputConfig, noOrDefaultKw, threshold )
   if cliBlockingFail( mode, timestamp, tacFeatureName.delayEcn, "Delay ECN" ):
      qosInputConfig.ecnDelayThreshold = prevEcnDelayThreshold

def setEcnQueueCount( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   txQueueId = mode.txQueue.id
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      ecnTxQueueCounterCfg = profile.ecnTxQueueCounterConfig.newMember( txQueueId )
      setEcnQueueCounterConfig( mode, None, txQueueId, ecnTxQueueCounterCfg, not no )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfCounterCfg = qosInputConfig.ecnIntfCounterConfig.newMember( intf )
         ecnTxQueueCounterCfg = intfCounterCfg.ecnTxQueueCounterConfig.newMember(
            txQueueId )
         setEcnQueueCounterConfig( mode, intf, txQueueId, ecnTxQueueCounterCfg,
            not no )

# -----------------------------------------------------------------------------------
# The "[ mls ] qos random-detect ecn  minimum-threshold <min_threashold>
# maximum-threshold <max_threshold>" command, in "uc-tx-queue" mode.
# -----------------------------------------------------------------------------------
def setTxQueueEcn( mode, txQueueConfig, noOrDefaultKw=True, minThd=None, maxThd=None,
                   ecnUnit=None, maxMarkRate=tacPercent.max, weight=defaultWeight ):
   prevEcnConfig = txQueueConfig.ecnConfig
   if noOrDefaultKw:
      if prevEcnConfig:
         txQueueConfig.ecnConfig = None
   else:
      if not ( prevEcnConfig and
               prevEcnConfig.minThd == minThd and
               prevEcnConfig.maxThd == maxThd and
               prevEcnConfig.unit == ecnUnit and
               prevEcnConfig.maxDroprate == maxMarkRate and
               prevEcnConfig.weight == weight ):
         txQueueConfig.ecnConfig = ( minThd, maxThd, ecnUnit,
                                     maxMarkRate, weight )
   return txQueueConfig

def setNoOrDefaultEcn( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return

      setTxQueueEcn( mode, txQueueConfig )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            continue
         txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
         if txQueueConfig is None:
            continue

         prevEcnConfig = txQueueConfig.ecnConfig

         setTxQueueEcn( mode, txQueueConfig )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevEcnConfig:
               txQueueConfig.ecnConfig = ( prevEcnConfig.minThd,
                                           prevEcnConfig.maxThd,
                                           prevEcnConfig.unit,
                                           prevEcnConfig.maxDroprate,
                                           prevEcnConfig.weight )
            else:
               txQueueConfig.ecnConfig = None
            setIntfConfig( intf )

# -----------------------------------------------------------------------------------
# The "[ mls ] qos random-detect non-ect minimum-threshold <min_threashold>
# maximum-threshold <max_threshold>" command, in "uc-tx-queue" mode.
# -----------------------------------------------------------------------------------
def setNoOrDefaultNonEctParams( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return

      setTxQueueNonEct( mode, txQueueConfig )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            continue
         txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
         if txQueueConfig is None:
            continue

         prevNonEctConfig = txQueueConfig.nonEctConfig

         setTxQueueNonEct( mode, txQueueConfig )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn,
                             "NON-ECT parameters", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevNonEctConfig:
               txQueueConfig.nonEctConfig = ( prevNonEctConfig.minThd,
                                              prevNonEctConfig.maxThd,
                                              prevNonEctConfig.unit,
                                              ecnMaxDroprate,
                                              prevNonEctConfig.weight )
            else:
               txQueueConfig.nonEctConfig = None
            setIntfConfig( intf )

def showEcnCounters( mode, args ):
   ecnCounter = EcnCountersModel()
   for sliceCounter in qosHwCounter.sliceHwCounter.values():
      for chipName, chipCount in sliceCounter.perChipCount.items():
         ecnCounter.insert( chipName, chipCount.ecnCount )
   return ecnCounter

def setEcn( mode, args ):
   maxMarkRate = args.get( 'MAX_DROPRATE', tacPercent.max )
   weight = args.get( 'WEIGHT', defaultWeight )
   minThd, maxThd, unit = args[ 'THD_RANGE' ]

   msg = "Cannot apply ECN configuration. WRED configuration is already present."
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      # ECN and WRED ( or dpWred ) configs are mutually exclusive only when hw
      # capability wredEcnMutuallyExclusive is true
      prevWredConfig = txQueueConfig.wredConfig
      prevDpWredConfig = ( txQueueConfig.dpConfig and
                           len( txQueueConfig.dpConfig.dpWredConfig ) != 0 )
      # Prior to qosHwStatus update, we don't know whether wred ecn are mutually
      # exclusiveMX or not, to be on safer side storing both config into qos
      # wred input config
      if qosHwStatus.hwInitialized and qosHwStatus.wredEcnMutuallyExclusive and \
         ( prevWredConfig or prevDpWredConfig ):
         mode.addError( msg )
         return

      setTxQueueEcn( mode, txQueueConfig, False, minThd,
                     maxThd, unit, maxMarkRate, weight )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )

         prevEcnConfig = txQueueConfig.ecnConfig

         # ECN and WRED ( or dpWred ) configs are mutually exclusive only when hw
         # wredEcnMutuallyExclusive is true
         prevWredConfig = txQueueConfig.wredConfig
         prevDpWredConfig = ( txQueueConfig.dpConfig and
                              len( txQueueConfig.dpConfig.dpWredConfig ) != 0 )
         # Prior to qosHwStatus update, we don't know whether wred ecn are mutually
         # exclusive or not, to be on safer side storing both config into qos
         # wred input config
         if qosHwStatus.hwInitialized and qosHwStatus.wredEcnMutuallyExclusive and \
            ( prevWredConfig or prevDpWredConfig ):
            mode.addError( msg )
            continue
         setTxQueueEcn( mode, txQueueConfig, False, minThd,
                        maxThd, unit, maxMarkRate, weight )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevEcnConfig:
               txQueueConfig.ecnConfig = ( prevEcnConfig.minThd,
                                           prevEcnConfig.maxThd,
                                           prevEcnConfig.unit,
                                           prevEcnConfig.maxDroprate,
                                           prevEcnConfig.weight )
            else:
               txQueueConfig.ecnConfig = None
            setIntfConfig( intf )

def setNoOrDefaultEcnDelayTxQ( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return
      txQueueConfig.ecnDelayConfig = None
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            continue
         txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
         if txQueueConfig is None:
            continue
         txQueueConfig.ecnDelayConfig = None
         setIntfConfig( intf )

def setDelayEcnTxQ( mode, args ):
   thd, unit = args[ 'THRESHOLD' ]
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
      txQueueConfig.ecnDelayConfig = ( 0, thd, unit, tacPercent.max, 0 )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
         txQueueConfig.ecnDelayConfig = ( 0, thd, unit, tacPercent.max, 0 )
         setIntfConfig( intf )

# -----------------------------------------------------------------------------------
# The "show qos interfaces [ INTF ] ecn counters queue" command
# -----------------------------------------------------------------------------------
def ecnCounterFromSmash( intfId, queueId ):
   key = EcnCounterKey( intfId, queueId )
   counter = ecnCounterTable.ecnCounter.get( key )
   snapshot = ecnSnapshotTable.ecnCounter.get( key )
   if counter is None:
      packetCount = 0
   elif snapshot is None:
      packetCount = counter.pkts
   else:
      packetCount = counter.pkts - snapshot.pkts
   return packetCount

def showEcnIntfQueueCounters( mode, args ):
   intf = args.get( 'INTF' )
   ecnIntfQueueCounterModel = EcnIntfQueueCountersModel()
   intfs = IntfCli.Intf.getAll( mode, intf, None, intfType=EthIntfCli.EthIntf )
   intfNames = [ intf_.name for intf_ in intfs ]
   for sliceHwStatus in qosSliceHwStatus.values():
      for intfId, ecnCounterStatus in \
          sliceHwStatus.ecnIntfCounterHwStatus.items():
         if intfId not in intfNames:
            continue
         queueCounterModel = EcnQueueCounterModel()
         numTxQueueIds = qosHwStatus.numTxQueueSupported
         if qosHwStatus.numMulticastQueueSupported > 0:
            numTxQueueIds = qosHwStatus.numUnicastQueueSupported
         for queueId in range( numTxQueueIds ):
            if ecnCounterStatus.counterEnabled.get( queueId ):
               if qosHwStatus.useEcnCountersSmash:
                  packetCount = ecnCounterFromSmash( intfId, queueId )
               else:
                  txCount = ecnCounterStatus.txQueuePktCount.get( queueId, 0 )
                  txCountClear = ecnCounterStatus.txQueuePktCountLastClear.get(
                     queueId, 0 )
                  packetCount = txCount - txCountClear
               queueCounterModel.insert( queueId, str( packetCount ) )
            else:
               queueCounterModel.insert( queueId, '-' )
         ecnIntfQueueCounterModel.insert( intfId, queueCounterModel )
   return ecnIntfQueueCounterModel

# -----------------------------------------------------------------------------------
# The "show [ mls ] qos random-detect ecn" command, in "enable" mode.
# The "show [ mls ] qos ecn" command, in "enable" mode.
# -----------------------------------------------------------------------------------
def showGlobalEcn( mode, args ):
   allEcn = 'random-detect' not in args
   globalEcn = GlobalAllEcnModel() if allEcn else GlobalEcnModel()
   globalEcn.globalEcnSupported = qosHwStatus.globalEcnSupported
   globalEcn.globalEcnWeightSupported = qosHwStatus.globalEcnWeightSupported
   globalEcn.delayEcnSupported = qosHwStatus.ecnDelaySupported
   globalEcnStatus = qosStatus.globalEcnStatus
   if globalEcnStatus:
      globalEcn.globalEcnParameters = EcnParametersModel()
      globalEcn.globalEcnParameters.minThreshold = globalEcnStatus.minThd
      globalEcn.globalEcnParameters.maxThreshold = globalEcnStatus.maxThd
      globalEcn.globalEcnParameters.unit = globalEcnStatus.unit
      globalEcn.globalEcnParameters.weight = globalEcnStatus.weight
      if globalEcnStatus.unit == tacQueueThresholdUnit.segments:
         globalEcn.globalEcnParameters.segmentSizeInBytes = \
             qosHwStatus.ecnSegmentSizeInBytes
   if allEcn and \
      qosStatus.ecnDelayThreshold != tacEcnDelayThreshold:
      globalEcn.globalEcnDelayParameters = EcnDelayParametersModel()
      globalEcn.globalEcnDelayParameters.configuredThreshold = \
            qosStatus.ecnDelayThreshold.threshold / 1000.0
      actualThresh = \
            ( qosStatus.ecnDelayThreshold.threshold //
              qosHwStatus.ecnDelayGranularity.threshold ) * \
            qosHwStatus.ecnDelayGranularity.threshold
      globalEcn.globalEcnDelayParameters.threshold = actualThresh / 1000.0
   return globalEcn

# -----------------------------------------------------------------------------------
# The "show [ mls ] qos ecn [delay|counters] " command, in "enable" mode.
# -----------------------------------------------------------------------------------
def showEcnDelay( mode, args ):
   ecn = EcnDelayModel()
   if qosStatus.ecnDelayThreshold != tacEcnDelayThreshold:
      ecn.ecnDelayParameters = EcnDelayParametersModel()
      ecn.ecnDelayParameters.configuredThreshold = \
            qosStatus.ecnDelayThreshold.threshold / 1000.0
      actualThresh = \
            ( qosStatus.ecnDelayThreshold.threshold //
              qosHwStatus.ecnDelayGranularity.threshold ) * \
            qosHwStatus.ecnDelayGranularity.threshold
      ecn.ecnDelayParameters.threshold = actualThresh / 1000.0
      if qosStatus.ecnDelayThreshold.unit != \
         tacEcnDelayThresholdUnit.ecnDelayThresholdNs:
         assert False, "Unknown unit"
   return ecn

def showEcnInterface( intf, allEcn=True ):
   intfStatus = qosStatus.intfStatus.get( intf.name )
   if intfStatus:
      intfStatus = qosStatus.intfStatus[ intf.name ]

   intfEcnModel = IntfEcnModel()

   if 0 == qosHwStatus.numTxQueueSupported:
      return intfEcnModel

   for hwtxqid in range( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         prefix = clitxq.type[ : 2 ].upper()
         txQName = prefix + str( clitxq.id )
         if prefix != 'UC':
            continue
      else:
         txQName = str( clitxq.id )

      txQueueEcn = TxQueueEcnModel()
      txQueueEcn.txQueue = txQName
      txQueueEcn.ecnMarkProbSupported = qosHwStatus.ecnMarkProbSupported
      txQueueEcn.ecnWeightSupported = qosHwStatus.ecnWeightSupported or \
                                      qosHwStatus.queueWeightSupported
      ecnSetting = None
      queueWeight = defaultWeight
      delayThresh = tacEcnDelayThreshold
      delayThreshSrc = 'none'
      if isLagPort( intf.name ): # For Lag read from input/config
         if intf.name in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intf.name ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
            if txQueueConfig:
               if txQueueConfig.weightConfig:
                  queueWeight = txQueueConfig.weightConfig
               if txQueueConfig.delayEcnEnabled and \
                  qosStatus.ecnDelayThreshold != tacEcnDelayThreshold:
                  delayThresh = txQueueConfig.ecnDelayThreshold \
                     if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold \
                     else qosStatus.ecnDelayThreshold
                  delayThreshSrc = 'local' \
                        if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold \
                        else 'global'
      elif intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )
         txQueueConfig = None
         if intf.name in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intf.name ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
         if txQueueStatus:
            if txQueueStatus.weightStatus:
               queueWeight = txQueueStatus.weightStatus
            if txQueueStatus.delayEcnEnabled:
               delayThresh = txQueueStatus.ecnDelayThreshold
               if txQueueConfig:
                  delayThreshSrc = 'local' \
                     if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold \
                     else 'global'

      qosShowCommandHelper = Tac.Type( "Qos::QosShowCommandHelper" )
      ecnSetting = qosShowCommandHelper.getEcnStatus(
         qosConfig, qosStatus, intf.name, clitxq )
      if ecnSetting:
         txQueueEcn.txQueueEcnParameters = EcnParametersModel()
         txQueueEcn.txQueueEcnParameters.minThreshold = ecnSetting.minThd
         txQueueEcn.txQueueEcnParameters.maxThreshold = ecnSetting.maxThd
         txQueueEcn.txQueueEcnParameters.unit = ecnSetting.unit
         txQueueEcn.txQueueEcnParameters.maxDroprate = ecnSetting.maxDroprate
         txQueueEcn.txQueueEcnParameters.operationalMaxDroprate = \
               ecnSetting.maxDroprate
         if not isLagPort( intf.name ):
            for sliceHwStatus in qosSliceHwStatus.values():
               intfEcnMaxDropProb = sliceHwStatus.ecnMaxDropProb.get( intf.name )
               if intfEcnMaxDropProb is not None:
                  break
            if intfEcnMaxDropProb and clitxq in intfEcnMaxDropProb.txQEcnMaxDropProb:
               maxDropRate = intfEcnMaxDropProb.txQEcnMaxDropProb[ clitxq ]
               if maxDropRate != tacPercent.invalid:
                  txQueueEcn.txQueueEcnParameters.operationalMaxDroprate = \
                           intfEcnMaxDropProb.txQEcnMaxDropProb[ clitxq ].dropProb
         txQueueEcn.txQueueEcnParameters.weight = ecnSetting.weight
         if queueWeight != defaultWeight:
            txQueueEcn.txQueueEcnParameters.weight = queueWeight
         if ecnSetting.unit == tacQueueThresholdUnit.segments:
            txQueueEcn.txQueueEcnParameters.segmentSizeInBytes = \
                qosHwStatus.ecnSegmentSizeInBytes

      txQueueEcn.txQueueEcnDelaySupported = qosHwStatus.ecnDelaySupported or \
                               qosHwStatus.ecnDelayPerTxQSupported
      ecnDelayPerTxQConfig = qosShowCommandHelper.getEcnDelayPerTxQConfig(
                                qosInputConfig, intf.name, clitxq )
      ecnDelayPerTxQStatus = qosShowCommandHelper.getEcnDelayPerTxQStatus(
                                qosStatus, intf.name, clitxq )
      if allEcn and delayThresh != tacEcnDelayThreshold:
         txQueueEcn.txQueueEcnDelayParameters = EcnDelayParametersModel()
         txQueueEcn.txQueueEcnDelayParameters.threshold = \
               ( delayThresh.threshold //
                 qosHwStatus.ecnDelayGranularity.threshold ) * \
               qosHwStatus.ecnDelayGranularity.threshold / 1000.0
         txQueueEcn.txQueueEcnDelayParameters.configuredThreshold = \
               delayThresh.threshold / 1000.0
         txQueueEcn.txQueueEcnDelayParameters.source = delayThreshSrc
      if ecnDelayPerTxQConfig or ecnDelayPerTxQStatus:
         txQueueEcn.txQueueEcnDelayParameters = EcnDelayParametersModel()
         txQueueEcn.txQueueEcnDelayParameters.source = 'local'
         txQueueEcn.txQueueEcnDelayParameters.threshold = 0.0
         txQueueEcn.txQueueEcnDelayParameters.configuredThreshold = 0.0
      if ecnDelayPerTxQStatus:
         # EcnDelayParametersModel has only unit microseconds in helpstr, so
         # convert milliseconds into microseconds
         thd = float( ecnDelayPerTxQStatus.maxThd ) \
            if ecnDelayPerTxQStatus.unit == tacQueueThresholdUnit.microseconds \
            else ecnDelayPerTxQStatus.maxThd * 1000.0 # Convert to microseconds
         txQueueEcn.txQueueEcnDelayParameters.threshold = thd
      if ecnDelayPerTxQConfig:
         # EcnDelayParametersModel has only unit microseconds in helpstr, so
         # convert milliseconds into microseconds
         thd = float( ecnDelayPerTxQConfig.maxThd ) \
            if ecnDelayPerTxQConfig.unit == tacQueueThresholdUnit.microseconds \
            else ecnDelayPerTxQConfig.maxThd * 1000.0 # Convert to microseconds
         txQueueEcn.txQueueEcnDelayParameters.configuredThreshold = thd

      intfEcnModel.txQueueEcnList.append( txQueueEcn )

   return intfEcnModel

def showNonEctInterface( intf ):
   intfStatus = qosStatus.intfStatus.get( intf.name )
   if intfStatus:
      intfStatus = qosStatus.intfStatus[ intf.name ]

   intfNonEctModel = IntfNonEctModel()

   if not qosHwStatus.numTxQueueSupported:
      return intfNonEctModel

   for hwtxqid in range( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         prefix = clitxq.type[ : 2 ].upper()
         txQName = prefix + str( clitxq.id )
         if prefix != 'UC':
            continue
      else:
         txQName = str( clitxq.id )

      txQueueNonEct = TxQueueNonEctModel()
      txQueueNonEct.txQueue = txQName
      txQueueNonEct.nonEctWeightSupported = qosHwStatus.nonEctWeightSupported
      nonEctSetting = None
      if isLagPort( intf.name ): # For Lag read from input/config
         if intf.name in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intf.name ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
            if txQueueConfig:
               if txQueueConfig.nonEctConfig:
                  nonEctSetting = txQueueConfig.nonEctConfig
      elif intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )
         txQueueConfig = None
         if intf.name in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intf.name ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
         if txQueueStatus:
            if txQueueStatus.nonEctStatus:
               nonEctSetting = txQueueStatus.nonEctStatus
      if nonEctSetting:
         txQueueNonEct.txQueueNonEctParameters = EcnParametersModel()
         txQueueNonEct.txQueueNonEctParameters.minThreshold = nonEctSetting.minThd
         txQueueNonEct.txQueueNonEctParameters.maxThreshold = nonEctSetting.maxThd
         txQueueNonEct.txQueueNonEctParameters.unit = nonEctSetting.unit
         txQueueNonEct.txQueueNonEctParameters.maxDroprate = \
               nonEctSetting.maxDroprate
         txQueueNonEct.txQueueNonEctParameters.operationalMaxDroprate = \
               nonEctSetting.maxDroprate
         if not isLagPort( intf.name ):
            for sliceHwStatus in qosSliceHwStatus.values():
               intfNonEctMaxDropProb = \
                     sliceHwStatus.nonEctMaxDropProb.get( intf.name )
               if intfNonEctMaxDropProb is not None:
                  break
            if intfNonEctMaxDropProb and clitxq \
                  in intfNonEctMaxDropProb.txQEcnMaxDropProb:
               maxDropRate = intfNonEctMaxDropProb.txQEcnMaxDropProb[ clitxq ]
               if maxDropRate != tacPercent.invalid:
                  txQueueNonEct.txQueueNonEctParameters.operationalMaxDroprate = \
                     intfNonEctMaxDropProb.txQEcnMaxDropProb[ clitxq ].dropProb
         txQueueNonEct.txQueueNonEctParameters.weight = nonEctSetting.weight
         if nonEctSetting.unit == tacQueueThresholdUnit.segments:
            txQueueNonEct.txQueueNonEctParameters.segmentSizeInBytes = \
               qosHwStatus.ecnSegmentSizeInBytes

      intfNonEctModel.txQueueNonEctList.append( txQueueNonEct )

   return intfNonEctModel

# -----------------------------------------------------------------------------------
# The "show [ mls ] qos interfaces [ INTF | fabric ] random-detect ecn" command, in
# "enable" mode.
# The "show [ mls ] qos interfaces [ INTF | fabric ] ecn" command, in "enable" mode.
# -----------------------------------------------------------------------------------
def showInterfacesEcn( mode, args ):
   allEcn = 'random-detect' not in args
   intfEcnCollection = IntfAllEcnCollectionModel() if allEcn else \
                       IntfEcnCollectionModel()

   if 'fabric' in args:
      intfs = [ FabricIntfCli.FabricIntf( fabricIntfName, mode )  ]
   else:
      intfs = IntfCli.Intf.getAll(
         mode, args.get( 'INTF' ), None, intfType=EthIntfCli.EthIntf )

   if not intfs:
      return intfEcnCollection

   for intf in intfs:
      if not intf.name.startswith( ethOrLagOrFabricIntfPrefixes ):
         continue

      intfEcn = showEcnInterface( intf, allEcn )

      if allEcn and qosHwStatus.nonEctThdSupported:
         intfNonEct = showNonEctInterface( intf )
         intfEcn.txQueueNonEctList = intfNonEct.txQueueNonEctList
      intfEcnCollection.intfEcnCollection[ intf.name ] = intfEcn

   return intfEcnCollection

def populateIntfBufferBasedDecnModel( intf ):
   model = IntfBufferBasedDecnModel()
   intfStatus = qosStatus.intfStatus.get( intf.name )

   for txQId in range( qosHwStatus.numUnicastQueueSupported - 1, -1, -1 ):
      txQ = Tac.newInstance( "Qos::TxQueue" )
      txQ.id = txQId
      txQ.type = "ucq"

      decnParams = None
      if isLagPort( intf.name ):
         # LAG ports must be read from input/config instead of status (doesn't exist)
         intfConfig = qosConfig.intfConfig.get( intf.name, None )
         txQConfig = ( intfConfig.txQueueConfig.get( txQ, None )
                       if intfConfig else None )
         decnParams = txQConfig.bufferBasedDecnConfig if txQConfig else None
      else:
         txQStatus = ( intfStatus.txQueueStatus.get( txQ, None )
                       if intfStatus else None )
         decnParams = txQStatus.bufferBasedDecnStatus if txQStatus else None

      txQModel = None
      if decnParams:
         txQModel = DecnParametersModel()
         txQModel.floor = decnParams.floor
         txQModel.offset = decnParams.offset
         txQModel.unit = decnParams.unit
      model.txQueues[ f"UC{txQ.id}" ] = txQModel

   return model

def showInterfacesBufferBasedDecn( mode, args ):
   model = IntfAllBufferBasedDecnModel()
   intfs = IntfCli.Intf.getAll(
      mode, args.get( "INTF" ), None, intfType=EthIntfCli.EthIntf )

   for intf in intfs:
      if not intf.name.startswith( ethOrLagOrFabricIntfPrefixes ):
         continue
      model.intfs[ intf.name ] = populateIntfBufferBasedDecnModel( intf )

   return model

def maybeGetIntfTxQueueConfig( inputConfig, intf, txQueue, createIfNeeded=False ):
   intfConfig = inputConfig.intfConfig.get( intf )
   if intfConfig is None:
      if not createIfNeeded:
         return None, None
      intfConfig = inputConfig.intfConfig.newMember( intf )

   txQueueConfig = intfConfig.txQueueConfig.get( txQueue )
   if txQueueConfig is None:
      if not createIfNeeded:
         return intfConfig, None
      txQueueConfig = intfConfig.txQueueConfig.newMember(
         txQueue, tacTxQueuePriority.priorityInvalid,
         tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )

   return intfConfig, txQueueConfig

def setBufferBasedDecn( mode, args, isNoOrDefault=False ):
   decnConfigTuple = args.get( 'OFFSET_FLOOR_EXPR', None )

   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
            mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
            invalidShapeRate, invalidGuaranteedBw )
      txQueueConfig.bufferBasedDecnConfig = decnConfigTuple
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         timestamp = Tac.now()
         _, txQueueConfig = maybeGetIntfTxQueueConfig(
            qosInputConfig, intf, mode.txQueue, createIfNeeded=not isNoOrDefault )
         if not txQueueConfig:
            # In the case of "no", if the queue config doesn't exist, DECN is
            # implicitly disabled, so there is nothing to do for this intf
            continue

         prevDecnConfig = txQueueConfig.bufferBasedDecnConfig
         if ( ( not prevDecnConfig and
                not decnConfigTuple ) or
              ( prevDecnConfig and
                decnConfigTuple and
                prevDecnConfig.offset == decnConfigTuple[ 0 ] and
                prevDecnConfig.floor == decnConfigTuple[ 1 ] and
                prevDecnConfig.unit == decnConfigTuple[ 2 ] ) ):
            # Applying the same config that already exists; nothing to do
            return

         # DECN waits for feature-level config update time in cliBlockingFail, so
         # we need to manually set the version we are waiting for here.
         qosHwConfig.configVersionUpdateTime = Tac.now()
         txQueueConfig.bufferBasedDecnConfig = decnConfigTuple
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.bufferBasedDecn,
                             "buffer-based dynamic ECN", intf ):
            # The failed to apply the configuration; revert
            _, txQueueConfig = maybeGetIntfTxQueueConfig(
               qosInputConfig, intf, mode.txQueue, createIfNeeded=True )
            if prevDecnConfig:
               txQueueConfig.bufferBasedDecnConfig = (
                  prevDecnConfig.offset, prevDecnConfig.floor, prevDecnConfig.unit )
            else:
               txQueueConfig.bufferBasedDecnConfig = None
            setIntfConfig( intf )

def setNoOrDefaultBufferBasedDecn( mode, args ):
   setBufferBasedDecn( mode, args, isNoOrDefault=True )
