# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import utcTimestampToStr
import Arnet
from CliGlobal import CliGlobal
from CliModel import Dict, Enum, Float, Int, List, Model
from IntfModels import Interface
import LazyMount
import Smash
import SmashLazyMount
from TableOutput import Headings, TableFormatter
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'CliDynamicQosCliModel' )
t0 = traceHandle.trace0  # report errors
t8 = traceHandle.trace8  # trace functions
t9 = traceHandle.trace9  # trace optional details

PfcBuffersCounterKey = TacLazyType( "Qos::PfcBuffersCounterKey" )
PfcBuffersCounter = TacLazyType( "Qos::PfcBuffersCounter" )
PfcProfileState = TacLazyType( "Pfc::ProfileState" )
QosCos = TacLazyType( "Qos::Cos" )

# --------------------------------------------------------------------------------
# Mount path holders (Define all mount path holders here)
# --------------------------------------------------------------------------------
gv = CliGlobal(
   dcbPfcBuffersStatus=None,
   packetBufferProfileStatus=None,
   pfcBuffersStatus=None,
   pfcConfig=None,
)

# --------------------------------------------------------------------------------
# Utility Functions
# --------------------------------------------------------------------------------

profileStateToModelEnum = { PfcProfileState.profilePending: "pending",
                            PfcProfileState.profileDefault: "active",
                            PfcProfileState.profileModifiedDefault: "modified",
                            PfcProfileState.profileInvalidDefault: "invalid",
                            PfcProfileState.profileAssigned: "active",
                            PfcProfileState.profileInvalid: "invalid",
                            PfcProfileState.profileOutOfResources: "invalid", }

# --------------------------------------------------------------------------------
# Model definitions
# --------------------------------------------------------------------------------

#
# Model for PFC buffers counters values (per interface and traffic class).
#
# Usage: sub-model
#
class PfcBuffersCountersModel( Model ):
   droppedPackets = Int( "Number of dropped packets", optional=True )
   droppedBytes = Int( "Number of dropped bytes", optional=True )
   wordsMaxTimestamp = Float( "UTC timestamp when max watermark was set" )
   wordsMaxWatermarkBytes = Int( "Max watermark in bytes since clear" )
   wordsPolledWatermarkBytes = Int( "Max watermark in bytes since poll" )

   @classmethod
   def newModel( cls, intfId, priority, switchTimeUtcDelta ):
      t8( "PfcBuffersCountersModel.newModel(", intfId, ",", priority, ",",
          switchTimeUtcDelta, ")" )
      counterKey = PfcBuffersCounterKey( intfId, priority )
      counter = gv.pfcBuffersStatus.counter.get( counterKey, PfcBuffersCounter() )
      dropsCounter = gv.pfcBuffersStatus.dropsCounter.get( counterKey )
      if dropsCounter is None:
         droppedPackets = None
         droppedBytes = None
      else:
         droppedPackets = dropsCounter.packets - dropsCounter.snapshotPackets
         droppedBytes = dropsCounter.bytes - dropsCounter.snapshotBytes

      if counter.wordsMaxTimestamp == 0.0 and droppedPackets is None:
         return None

      return cls( droppedPackets=droppedPackets,
                  droppedBytes=droppedBytes,
                  wordsMaxTimestamp=counter.wordsMaxTimestamp and
                     counter.wordsMaxTimestamp + switchTimeUtcDelta,
                  wordsMaxWatermarkBytes=counter.wordsMaxWatermarkBytes,
                  wordsPolledWatermarkBytes=counter.wordsPolledWatermarkBytes )

#
# Model for PFC buffer counters values (per interface ).
#
# Usage: sub-model
#
class PfcIntfBufferCountersModel( Model ):
   priorities = Dict( help="A mapping of priority to priority flow control "
                           "buffers counters",
                      valueType=PfcBuffersCountersModel,
                      keyType=int )

   @classmethod
   def newModel( cls, intfId, switchTimeUtcDelta ):
      t8( "PfcIntfBufferCountersModel.newModel(", intfId, ",",
          switchTimeUtcDelta, ")" )
      model = cls()
      for priority in range( QosCos.min, QosCos.max + 1 ):
         priorityModel = PfcBuffersCountersModel.newModel(
               intfId, priority, switchTimeUtcDelta )
         if priorityModel is None:
            continue
         model.priorities[ priority ] = priorityModel

      return model

#
# Model for PFC buffer counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show priority-flow-control [ interfaces <range> ] buffer counters
#  - show priority-flow-control [ interfaces <range> ] buffer counters
#
class PfcIntfRangeBufferCountersModel( Model ):
   lastCleared = Float( help="UTC timestamp when clear command was last issued",
                        optional=True )
   lastUpdated = Float( help="UTC timestamp when counters were last polled",
                        optional=True )
   interfaces = Dict( help="A mapping of interface name to priority flow control "
                           "buffers counters",
                      keyType=Interface,
                      valueType=PfcIntfBufferCountersModel )

   @classmethod
   def newModel( cls, intfs, switchTimeUtcDelta ):
      t8( "PfcIntfRangeBufferCountersModel.newModel(", intfs, ",",
          switchTimeUtcDelta, ")" )

      lastCleared = gv.dcbPfcBuffersStatus.lastCleared
      if lastCleared:
         lastCleared += switchTimeUtcDelta

      lastUpdated = gv.dcbPfcBuffersStatus.lastUpdated
      if lastUpdated:
         lastUpdated += switchTimeUtcDelta

      model = cls( lastCleared=lastCleared or None, lastUpdated=lastUpdated or None )
      for intf in intfs:
         intfId = intf.status().intfId
         model.interfaces[ intfId ] = PfcIntfBufferCountersModel.newModel(
               intfId, switchTimeUtcDelta )

      return model

   @staticmethod
   def newTable():
      """Return TableFormatter object with formatted columns."""

      # Tuple is required for heading to be properly justified.
      columns = ( ( "Source Interface", "l" ),
                  ( "Priority", "r" ),
                  ( "Max Polled Bytes", "r" ),
                  ( "Max Latched Bytes", "r" ),
                  ( "Max Latched Time", "r" ),
                  ( "Dropped Packets", "r" ),
                  ( "Dropped Bytes", "r" ), )
      headings = Headings( columns )
      # Adjust formatting parameters inaccessible via initialization.
      for fmt in headings.formats:
         fmt.padLimitIs( True )  # No extra padding around columns.
      headings.formats[ 0 ].noPadLeftIs( True )  # No left padding for 1st column.

      table = TableFormatter()
      headings.doApplyHeaders( table )
      # Return column names because they can't be readily obtained from table.
      return table, tuple( c[ 0 ] for c in columns )

   def render( self ):
      """Render command output to stdout.

         Design document: AID11570
      """
      t8( "PfcIntfRangeBufferCountersModel.render" )
      # Trivial case: Nothing to do.
      if not self.interfaces:
         t9( "PfcIntfRangeBufferCountersModel.render: No interfaces to output." )
         return

      # Converters from raw to display formats per column.
      def identity( x ):
         return x  # Default converter.

      def intfIdShortName( intfId ):
         return Arnet.IntfId( intfId ).shortName

      def numToCommaStr( x ):
         return f'{x:,}'

      def funcOrNa( func, x ):
         return 'n/a' if x is None else func( x )

      # Conversion table for columns that need it.
      convert = { "Source Interface": intfIdShortName,
                  "Priority": lambda p: f'PFC{p}',
                  "Max Polled Bytes": numToCommaStr,
                  "Max Latched Bytes": numToCommaStr,
                  "Max Latched Time": utcTimestampToStr,
                  "Dropped Packets": lambda x: funcOrNa( numToCommaStr, x ),
                  "Dropped Bytes": lambda x: funcOrNa( numToCommaStr, x ) }

      # Table of data & ordered column names.
      table, columns = self.newTable()

      # Make sure all keys in convert map to columns.
      assert all( col in columns for col in convert )

      pfcBuffersCounterDefault = PfcBuffersCounter()
      for intfId in Arnet.sortIntf( self.interfaces ):
         interface = self.interfaces[ intfId ]
         for priority in range( QosCos.min, QosCos.max + 1 ):
            model = interface.priorities.get( priority, pfcBuffersCounterDefault )

            # rowArgs follows order of columns.
            rowArgs = ( intfId, priority,
                        model.wordsPolledWatermarkBytes,
                        model.wordsMaxWatermarkBytes,
                        model.wordsMaxTimestamp,
                        getattr( model, "droppedPackets", None ),
                        getattr( model, "droppedBytes", None ), )

            # Transform rowArgs with convert function, or identity, for each column.
            table.newRow( *( convert.get( col, identity )( arg )
                          for col, arg in zip( columns, rowArgs ) ) )

      # Output preamble.
      print( 'Last polled:', utcTimestampToStr( self.lastUpdated ) )
      print( 'Last cleared:', utcTimestampToStr( self.lastCleared ) )
      # Defer trailing whitespace to after postamble.
      print( table.output().rstrip() )
      # Output postamble.
      print( 'Max Polled Bytes: The max watermark over the polling interval',
             'that ended at the "Last polled" time.' )
      print( 'Max Latched Bytes: The max watermark since the "Last cleared" time.' )

#
# Model for individual PFC packet-buffer profile.
#
# Usage: Model associated with these commands:
#  - show priority-flow-control packet-buffer profile { PROFILE }
#
class PfcPacketBufferProfileModel( Model ):
   pauseMin = Int( help="Minimum buffered bytes before PFC pause" )
   pauseMax = Int( help="Maximum buffered bytes before PFC pause" )
   pauseAlpha = Int( help="Dynamic PFC pause burst-absorption-factor" )
   resumeOffset = Int( help="Bytes offset from PFC pause threshold to unpause" )
   headroom = Int( help="Reserved bytes to buffer packets in-flight after PFC " )
   rejectMin = Int( help="Minimum buffered bytes before packets dropped" )
   rejectMax = Int( help="Maximum buffered bytes before packets dropped" )
   rejectAlpha = Int( help="Dynamic PFC drop threshold burst-absorption-factor" )
   state = Enum( set( profileStateToModelEnum.values() ),
                 help="Profile state after profile validation" )
   _index = Int( help="Show command sorting index" )

   @classmethod
   def newModel( cls, profileStatus ):
      return cls( pauseMin=profileStatus.pauseParams.min,
                  pauseMax=profileStatus.pauseParams.max,
                  pauseAlpha=profileStatus.pauseParams.alpha,
                  resumeOffset=profileStatus.resumeOffset,
                  headroom=profileStatus.headroomBytes,
                  rejectMin=profileStatus.rejectParams.min,
                  rejectMax=profileStatus.rejectParams.max,
                  rejectAlpha=profileStatus.rejectParams.alpha,
                  state=profileStateToModelEnum[ profileStatus.state ],
                  _index=profileStatus.index )

   def addToTable( self, name, table ):
      table.newRow( name,
                    self.pauseMin,
                    self.pauseMax,
                    self.pauseAlpha,
                    self.resumeOffset,
                    self.headroom,
                    self.state.capitalize() )

#
# Model for PFC packet-buffer profile top-level show command.
#
# Usage: Model associated with these commands:
#  - show priority-flow-control packet-buffer profile { PROFILE }
#
class PfcPacketBufferProfileRangeModel( Model ):
   headroomPercent = Int(
         help="Percent of global resources allocated to headroom pool" )
   losslessPercent = Int(
         help="Percent of global resources allocated to lossless pool" )
   lossyPercent = Int(
         help="Percent of global resources allocated to lossy pool" )
   losslessPriorities = List( valueType=int,
         help="Priorities assigned to lossless profiles" )
   profiles = Dict(
         help="A mapping of priority-flow-control packet buffer profile name to "
              "profile model",
         valueType=PfcPacketBufferProfileModel )

   @classmethod
   def newModel( cls, profiles ):
      t8( "PfcIntfRangeBufferCountersModel.newModel(", profiles, ")" )

      config = gv.pfcConfig
      profileStatus = gv.packetBufferProfileStatus
      model = cls( headroomPercent=profileStatus.poolAllocationPercent.headroom,
                   losslessPercent=profileStatus.poolAllocationPercent.lossless,
                   lossyPercent=profileStatus.poolAllocationPercent.lossy(),
                   losslessPriorities=list( config.losslessPriorities ) )

      if not profiles:
         profiles = profileStatus.fadtProfileStatus.keys()

      for name in profiles:
         profile = profileStatus.fadtProfileStatus.get( name )
         if profile is None:
            continue
         model.profiles[ name ] = PfcPacketBufferProfileModel.newModel( profile )

      return model

   @staticmethod
   def newTable():
      """Return TableFormatter object with formatted columns."""

      # Column names and alignment.
      columns = ( ( "Profile", "l" ),
                  ( "Pause FADT Min (Bytes)", "r" ),
                  ( "Pause FADT Max (Bytes)", "r" ),
                  ( "Pause FADT Alpha", "r" ),
                  ( "Resume Offset (Bytes)", "r" ),
                  ( "Headroom (Bytes)", "r" ),
                  ( "State", "l" ), )
      headings = Headings( columns )
      # Adjust formatting parameters inaccessible via initialization.
      for fmt in headings.formats:
         fmt.padLimitIs( True ) # No extra padding around columns.
      headings.formats[ 0 ].noPadLeftIs( True ) # No left padding for 1st column.

      table = TableFormatter()
      headings.doApplyHeaders( table )

      return table

   def render( self ):
      """Render command output to stdout.

         Design document: AID13291.
      """
      t8( 'PfcPacketBufferProfileModel.render' )

      # Table of data & ordered column names.
      table = self.newTable()
      # pylint: disable=protected-access
      sortedProfiles = sorted( self.profiles.items(),
                               # Sort by index, then profile name.
                               key=lambda item: ( item[ 1 ]._index, item[ 0 ] ) )
      for name, profile in sortedProfiles:
         profile.addToTable( name, table )

      # Output preamble.
      print( f'Lossy pool {self.lossyPercent}%' )
      print( f'Lossless pool {self.losslessPercent}%' )
      print( f'Total headroom {self.headroomPercent}%' )
      print( 'Lossless priorities:', end=' ' if self.losslessPriorities else '' )
      print( ', '.join( f'PFC{priority}' for priority in
                        sorted( self.losslessPriorities ) ) )
      print( 'Flow control thresholds:' )
      print( table.output() )

# --------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# --------------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.dcbPfcBuffersStatus = LazyMount.mount( entityManager, "dcb/pfc/buffersStatus",
                                             "Pfc::BuffersStatus", "r" )

   gv.packetBufferProfileStatus = LazyMount.mount(
         entityManager, "dcb/pfc/packetBufferProfile/status",
         "Pfc::PacketBufferProfileStatus", "r" )

   smashReaderInfo = Smash.mountInfo( 'reader' )
   gv.pfcBuffersStatus = SmashLazyMount.mount(
         entityManager,
         "interface/status/counter/pfcBuffers",
         "Qos::PfcBuffersStatus",
         smashReaderInfo,
         autoUnmount=True )

   gv.pfcConfig = LazyMount.mount( entityManager,
                                   "dcb/pfc/config", "Pfc::Config", "r" )
