# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import switchTimeToUtc
from CliDynamicSymbol import CliDynamicPlugin
from CliGlobal import CliGlobal
from CliPlugin.QosCliCommon import counterSupportedEthPhyIntfs
import ConfigMount
import LazyMount
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'CliDynamicQosCliPfc' )
t9 = traceHandle.trace9  # debugging traces

PercentAllowedZero = TacLazyType( "Pfc::PercentAllowedZero" )
PoolAllocationPercent = TacLazyType( "Pfc::PoolAllocationPercent" )

# --------------------------------------------------------------------------------
# Lazy-loaded model modules
# --------------------------------------------------------------------------------

QosCliModel = CliDynamicPlugin( "QosCliModel" )

# --------------------------------------------------------------------------------
# Mount path holders (Define all mount path holders here)
# --------------------------------------------------------------------------------
gv = CliGlobal(
   packetBufferProfileConfig=None,
   qosInputConfig=None,
)

# --------------------------------------------------------------------------------
# Configure Cli handlers
# --------------------------------------------------------------------------------

def configPacketBufferAllocation( mode, args ):
   """Configure handler for CliPlugins QosCliPfc.PfCPacketBufferAllocationCmd"""
   config = gv.packetBufferProfileConfig
   headroom = PercentAllowedZero( args.get( 'HEADROOM' ) )
   lossless = PercentAllowedZero( args.get( 'LOSSLESS' ) )
   if headroom + lossless == PercentAllowedZero.invalid:
      mode.addError( f'Buffer allocation total exceeds {PercentAllowedZero.max} '
                     'percent: '
                     f'{headroom.value} (headroom) + '
                     f'{lossless.value} (lossless) = '
                     f'{headroom.value + lossless.value}' )
      return
   config.poolAllocationPercent = PoolAllocationPercent( headroom, lossless )

def noOrDefaultPacketBufferAllocation( mode, args ):
   """No or default handler for CliPlugins QosCliPfc.PfCPacketBufferAllocationCmd"""
   config = gv.packetBufferProfileConfig
   config.poolAllocationPercent = PoolAllocationPercent()

# --------------------------------------------------------------------------------
# Show Cli handlers
# --------------------------------------------------------------------------------
def showPfcInterfaceBufferCounters( mode, args ):
   """Return the CLI model for the command:

      - show priority-flow-control [ interfaces <range> ] buffer counters
      - show [ interfaces <range> ] priority-flow-control buffer counters

      Model := PfcIntfRangeBufferCountersModel"""

   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )
   # Successive calls to switchTimeToUtc on the same input show some variation over
   # time. Call the function once to fix the time delta across all times in the model
   # to ensure the relative times are maintained (particularly times that should be
   # precisely equal).
   now = Tac.now()
   switchTimeUtcDelta = switchTimeToUtc( now ) - now
   t9( "showPfcInterfaceBufferCounters", "switchTimeUtcDelta:", switchTimeUtcDelta )
   return QosCliModel.PfcIntfRangeBufferCountersModel.newModel( intfs,
                                                                switchTimeUtcDelta )

def showPfcPacketBufferProfileRange( mode, args ):
   """Return the CLI model for the command:

   - show priority-flow-control packet-buffer profile [ { PROFILES } ]

   Model := PfcPacketBufferProfileModel"""

   profiles = args.get( 'PROFILES', [] )
   return QosCliModel.PfcPacketBufferProfileRangeModel.newModel( profiles )

# --------------------------------------------------------------------------------
# Other Cli handlers
# --------------------------------------------------------------------------------

#
# clear priority-flow-control buffer counters
#
def clearPfcBufferCounters( mode, args ):
   """Set the buffer counters clear time attribute to now"""
   gv.qosInputConfig.clearBufferCounters = Tac.now()

# --------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# --------------------------------------------------------------------------------

def Plugin( entityManager ):
   gv.packetBufferProfileConfig = ConfigMount.mount(
         entityManager, "dcb/pfc/packetBufferProfile/config",
         "Pfc::PacketBufferProfileConfig", "w" )
   gv.qosInputConfig = LazyMount.mount( entityManager, "qos/input/config/cli",
                                        "Qos::Input::Config", "w" )
