#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclCliLib
import AclLib
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import LazyMount
import MultiRangeRule
import QosLib
import ShowCommand
import Tac
import Tracing

from CliMode.Qos import ClassMapModeBase
from CliPlugin import AclCli
from CliPlugin.QosCliModel import ClassMapAllModel
from CliPlugin.QosCliServicePolicy import (
   getL2ParamsMatchCombo, allowMatchOptionInClassMap, gv, nodeMpls, nodeTrafficClass,
   nodeDscpMatch, matcherEct, matcherEctCe, matcherCe, matcherNonEct )
from CliPlugin.QosCliCommon import (
   matcherMatch, nodeDscpEcn, nodeMac, matcherAccessGroup, guardIpAcls,
   guardIp6Acls )
from CliPlugin.QosCliBasicQos import (
   VlanMatchExpression, InnerVlanMatchExpression, CosMatchExpression,
   InnerCosMatchExpression, DeiMatchExpression, cosKeywordMatcher, deiKeywordMatcher,
   innerCosKeywordMatcher, vlanKeywordMatcher, guardVlanMatch, guardInnerVlanMatch,
   innerVlanKeywordMatcher, guardCosMatch, guardInnerCosMatch, guardDeiMatch )
from CliPlugin.QosCliCopp import AbortCmd
from QosLib import ( matchOptionToEnum, matchOptionOtherToEnum, rangeToMinMaxValue,
                     mapTypeToEnum )
from QosCliLib import CMapModelContainer
from QosTypes import tacMatchOption

__defaultTraceHandle__ = Tracing.Handle( 'QosCliServicePolicyMode' )
t0 = Tracing.trace0

# -----------------------------------------------------------------------------------
# Variables for Qos Service Policy associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosAclHwStatus = None
qosHwStatus = None
qosSliceHwStatus = None

ecnDontCare = Tac.Type( "Acl::Ecn" ).dontCare
invalidDscp = Tac.Value( "Qos::DscpVal" ).invalid

class ClassMapMode( ClassMapModeBase, BasicCli.ConfigModeBase ):
   name = "Class Map Configuration"
   showActiveCmdRegistered_ = True

   # Each mode object has a session object. We associate the classmap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      self.cmapContext = context
      self.cmapName_ = context.cmapName()
      self.mapType_ = context.mapType()
      self.matchType_ = context.matchType()
      self.subConfig = None
      param = ( self.mapType_, self.cmapName_ )
      ClassMapModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'ClassMapMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.subConfig = None
      self.cmapContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.cmapContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.cmapContext
      self.cmapContext = None
      context.commit()

   @staticmethod
   def matchOptionExists( entry, matchOption ):
      return matchOption in entry.match

   @staticmethod
   def isCollSame( rangeColl, values, ecnCheck=True ):
      ids = set()
      for key in rangeColl:
         ids.update( list( range( key.min, key.max + 1 ) ) )
      return ids == set( values ) and ecnCheck

   @staticmethod
   def setMatchVlanValue( entry, value, mask, matchOption, vlanRangeString ):
      vlanValue = value
      isMaskValid = True
      if vlanRangeString:
         vlanValue = Tac.Value( "Bridging::VlanId" ).invalid
         mask = 0
         isMaskValid = False
      entry.match[ matchOption ].vlanValue = ( vlanValue, mask )
      entry.match[ matchOption ].vlanValue.vlanColl.clear()
      if vlanRangeString:
         valueString = MultiRangeRule.multiRangeToCanonicalString(
            vlanRangeString )
         valueString = valueString.split( ',' )
         for rangeStr in valueString:
            minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
            minVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 0 ] )
            maxVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 1 ] )
            key = Tac.Value( "Qos::VlanRange" )
            key.min = minVal
            key.max = maxVal
            entry.match[ matchOption ].vlanValue.vlanColl[ key ] = True
      entry.match[ matchOption ].vlanValue.maskValid = isMaskValid

   @staticmethod
   def setMatchInnerVlanValue( entry, value, mask, innerVlanRangeString ):
      matchOption = matchOptionToEnum( 'l2Params' )
      innerVlanValue = value
      isMaskValid = True
      if innerVlanRangeString:
         innerVlanValue = Tac.Value( "Bridging::VlanId" ).invalid
         mask = 0
         isMaskValid = False
      entry.match[ matchOption ].innerVlanValue = ( innerVlanValue, mask )
      entry.match[ matchOption ].innerVlanValue.innerVlanColl.clear()
      if innerVlanRangeString:
         valueString = MultiRangeRule.multiRangeToCanonicalString(
            innerVlanRangeString )
         valueString = valueString.split( ',' )
         for rangeStr in valueString:
            minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
            minVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 0 ] )
            maxVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 1 ] )
            key = Tac.Value( "Qos::VlanRange" )
            key.min = minVal
            key.max = maxVal
            entry.match[ matchOption ].innerVlanValue.innerVlanColl[ key ] = True
      entry.match[ matchOption ].innerVlanValue.maskValid = isMaskValid

   @staticmethod
   def setMatchCosValue( entry, value, matchOption ):
      entry.match[ matchOption ].cosValue = ( '', )
      entry.match[ matchOption ].cosValue.cosColl.clear()

      valueString = None
      rangeStr = str( value ) if isinstance( value, int ) else value
      valueString = MultiRangeRule.multiRangeToCanonicalString( rangeStr )
      cosString = valueString.split( ',' )
      for rangeStr in cosString:
         minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
         minVal = Tac.Value( "Qos::Cos", minMaxValue[ 0 ] )
         maxVal = Tac.Value( "Qos::Cos", minMaxValue[ 1 ] )
         key = Tac.Value( "Qos::CosRange" )
         key.min = minVal
         key.max = maxVal
         entry.match[ matchOption ].cosValue.cosColl[ key ] = True

   @staticmethod
   def setMatchInnerCosValue( entry, value, matchOption ):
      entry.match[ matchOption ].innerCosValue = ( '', )
      entry.match[ matchOption ].innerCosValue.innerCosColl.clear()

      valueString = None
      rangeStr = str( value ) if isinstance( value, int ) else value
      valueString = MultiRangeRule.multiRangeToCanonicalString( rangeStr )
      innerCosString = valueString.split( ',' )
      for rangeStr in innerCosString:
         minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
         minVal = Tac.Value( "Qos::InnerCos", minMaxValue[ 0 ] )
         maxVal = Tac.Value( "Qos::InnerCos", minMaxValue[ 1 ] )
         key = Tac.Value( "Qos::InnerCosRange" )
         key.min = minVal
         key.max = maxVal
         entry.match[ matchOption ].innerCosValue.innerCosColl[ key ] = True

   @staticmethod
   def setMatchDeiValue( entry, value, matchOption ):
      val = Tac.Value( "Qos::DeiValue", value[ 0 ] )
      entry.match[ matchOption ].deiValue = ( val, )

   def setMatchDscpEcnValue( self, entry, value, ecnValue, matchRange, matchOption,
                                valueString, dscpNameValid ):
      dscpValue = value
      if not ecnValue:
         ecnValue = ecnDontCare
      else:
         ecnValue = AclCliLib.ecnValueFromCli( self, ecnValue )
      # dscp = 0 is a valid value, hence 'not dscpValue' will be wrong here
      if matchRange or dscpValue is None:
         # 'match ecn' case
         dscpValue = invalidDscp
      entry.match[ matchOption ].dscpEcnValue = ( dscpValue, ecnValue )
      entry.match[ matchOption ].dscpEcnValue.dscpColl.clear()
      entry.match[ matchOption ].dscpEcnValue.dscpNameValid = dscpNameValid
      if matchRange:
         # dscp range configured
         dscpString = valueString.split( ',' )
         for rangeStr in dscpString:
            minMaxValue = rangeToMinMaxValue( rangeStr )
            minVal = Tac.Value( "Qos::DscpVal", minMaxValue[ 0 ] )
            maxVal = Tac.Value( "Qos::DscpVal", minMaxValue[ 1 ] )
            key = Tac.Value( "Qos::DscpRange" )
            key.min = minVal
            key.max = maxVal
            entry.match[ matchOption ].dscpEcnValue.dscpColl[ key ] = True

   @staticmethod
   def setMatchMplsTrafficClassValue( entry, value, matchOption ):
      entry.match[ matchOption ].mplsTrafficClassVal = ( '', )
      for key in entry.match[ matchOption ].mplsTrafficClassVal.\
          mplsTrafficClassColl:
         del entry.match[ matchOption ].mplsTrafficClassVal.mplsTrafficClassColl[
            key ]

      rangeStr = str( value ) if isinstance( value, int ) else value
      mplsTrafficClassString = MultiRangeRule.multiRangeToCanonicalString( rangeStr )

      mplsTrafficClassString = mplsTrafficClassString.split( ',' )
      for rangeStr in mplsTrafficClassString:
         minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
         minVal = Tac.Value( 'Qos::MplsTrafficClassVal', minMaxValue[ 0 ] )
         maxVal = Tac.Value( 'Qos::MplsTrafficClassVal', minMaxValue[ 1 ] )
         key = Tac.Value( "Qos::MplsTrafficClassRange" )
         key.min = minVal
         key.max = maxVal
         entry.match[ matchOption ].mplsTrafficClassVal.mplsTrafficClassColl[
            key ] = True
         entry.match[ matchOption ].mplsTrafficClassVal.isValid = True

   def setL2ParamsMatchValue( self, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      matchCombo = args.get( 'matchCombo' )

      vlanValue = args.get( 'VLAN_VALUE' )
      vlanMask = args.get( 'VLAN_MASK' )
      vlanRangeString = args.get( 'VLAN_SET' )

      innerVlanValue = args.get( 'INNER_VLAN_VALUE' )
      innerVlanMask = args.get( 'INNER_VLAN_MASK' )
      innerVlanRangeString = args.get( 'INNER_VLAN_SET' )

      cosValue = args.get( 'COS_RANGE' )
      innerCosValue = args.get( 'INNER_COS_RANGE' )

      deiValue = args.get( 'DEI_VALUE' )

      context = self.cmapContext
      entry = context.currentEntry()
      matchType = 'l2Params'
      matchOption = matchOptionToEnum( matchType )

      removeUnconditionally = False
      vlanPresent = 'vlan' in matchCombo
      cosPresent = 'cos' in matchCombo
      deiPresent = 'dei' in matchCombo
      innerCosPresent = 'cos inner' in matchCombo
      innerVlanPresent = 'vlan inner' in matchCombo

      if self.matchOptionExists( entry, matchOption ):
         entryMatch = entry.match[ matchOption ]
         # Check matchCombo of the existing ones
         existingMatchCombo = getL2ParamsMatchCombo( entryMatch )
      else:
         existingMatchCombo = []

      if noOrDefault:
         # If there is no match l2Params configured and no command is executed
         if not existingMatchCombo:
            return
         # Removing unconditionally is only applicable if none of the values are
         # entered and the matchCombo matches with the existing matchCombo
         if ( vlanValue is None and cosValue is None and deiValue is None and
              innerCosValue is None and vlanRangeString is None and
              innerVlanValue is None and innerVlanRangeString is None ):
            matchComboSet = set( matchCombo )
            existingMatchComboSet = set( existingMatchCombo )
            if matchComboSet.issubset( existingMatchComboSet ):
               removeUnconditionally = True
            elif len( matchComboSet.intersection( existingMatchComboSet ) ) == 0:
               return

      if vlanMask is None:
         vlanMask = 0xFFF
      if innerVlanMask is None:
         innerVlanMask = 0xFFF

      if not noOrDefault:
         allow, pmapNames, _macAclClassCounts, _ipv6ClassCounts, vxlanIntfPresent = \
                     allowMatchOptionInClassMap( entry, matchType, vlanPresent,
                                                 cosPresent,
                                                 innerVlanPresent,
                                                 innerCosPresent,
                                                 dei=deiPresent )
         if not allow:
            errorMsg = f'The match type {matchType} is not allowed in the ' \
               f'class-map. '
            errorMsg += f'\nThe class-map is present in the following policy-maps:' \
               f' {pmapNames} '
            if qosAclHwStatus.vlanMatchInClassMapSupported and vlanPresent:
               errorMsg += 'which are applied on VLAN interfaces'
            elif qosAclHwStatus.vxlanPolicyQosSupported and vxlanIntfPresent and \
                 not vlanPresent and cosPresent and innerVlanPresent:
               errorMsg += 'which are applied on the VTI'
            else:
               errorMsg += ( 'which have a policer configured in the class-default. '
                             'When class-default has policer, the policy-map is not '
                             'allowed to have both ipv4 class-maps and ipv6 '
                             'class-maps' )
            self.addError( errorMsg )
            return

         if not self.matchOptionExists( entry, matchOption ):
            entry.match.newMember( matchOption )
         else:
            # Option exists, now check if the matchCombos are the same
            if set( existingMatchCombo ) != set( matchCombo ):
               del entry.match[ matchOption ]
               entry.match.newMember( matchOption )

         if vlanPresent:
            self.setMatchVlanValue( entry, vlanValue, vlanMask, matchOption,
                                    vlanRangeString )
         if innerVlanPresent:
            self.setMatchInnerVlanValue( entry, innerVlanValue, innerVlanMask,
                                         innerVlanRangeString )
         if cosPresent:
            self.setMatchCosValue( entry, cosValue, matchOption )

         if innerCosPresent:
            self.setMatchInnerCosValue( entry, innerCosValue, matchOption )

         if deiPresent:
            self.setMatchDeiValue( entry, deiValue, matchOption )
         entry.match[ matchOption ].strValue = matchType

         # Delete other matchOption if any
         matchOptionsOtherToEnum = matchOptionOtherToEnum( matchOption )
         for matchOption_ in matchOptionsOtherToEnum:
            if self.matchOptionExists( entry, matchOption_ ):
               del entry.match[ matchOption_ ]
      else:
         if removeUnconditionally:
            # To do this because we want 'no match vlan' to remove
            # 'match vlan 5 0xfff cos 3'
            # When we do 'no match vlan'
            # CLI passes matchVlan as the option type
            # So explicitly checking if CLI passed matchVlan and only if matchVlanCos
            # is configured we delete the entry
            del entry.match[ matchOption ]
         elif self.matchOptionExists( entry, matchOption ):
            # If incoming commands have more parameters than the existing one
            # reject it
            if len( matchCombo ) != len( existingMatchCombo ):
               return

            if vlanValue is not None or vlanRangeString is not None:
               if not entryMatch.vlanValue:
                  return
               if vlanRangeString is None:
                  if entryMatch.vlanValue.vlan != vlanValue or \
                     entryMatch.vlanValue.vlanMask != vlanMask:
                     return
               else:
                  rangeColl = entryMatch.vlanValue.vlanColl
                  if not self.isCollSame( rangeColl, vlanRangeString ):
                     return

            if innerVlanValue is not None or innerVlanRangeString is not None:
               if not entryMatch.innerVlanValue:
                  return
               if innerVlanRangeString is None:
                  if ( entryMatch.innerVlanValue.innerVlan != innerVlanValue or
                       entryMatch.innerVlanValue.innerVlanMask != innerVlanMask ):
                     return
               else:
                  rangeColl = entryMatch.innerVlanValue.innerVlanColl
                  if not self.isCollSame( rangeColl, innerVlanRangeString ):
                     return

            if cosValue is not None:
               if entryMatch.cosValue is None:
                  return
               if isinstance( cosValue, int ):
                  cosValue = [ cosValue ]
               if not self.isCollSame( entryMatch.cosValue.cosColl, cosValue ):
                  return

            if innerCosValue is not None:
               if entryMatch.innerCosValue is None:
                  return
               if isinstance( innerCosValue, int ):
                  innerCosValue = [ innerCosValue ]
               if not self.isCollSame( entryMatch.innerCosValue.innerCosColl,
                       innerCosValue ):
                  return

            if deiValue is not None:
               if entryMatch.deiValue is None:
                  return
               if entryMatch.deiValue.deiVal.value != deiValue[ 0 ]:
                  return

            del entry.match[ matchOption ]

   def setMatchValue( self, no, matchType, value, mask=None,
                      ecnValue=None ):

      context = self.cmapContext
      entry = context.currentEntry()

      removeUnconditionally = False
      if no and value is None and not mask and not ecnValue:
         removeUnconditionally = True

      matchRange = False
      matchOption = matchOptionToEnum( matchType )
      if matchOption == tacMatchOption.matchDscpEcn and \
          value and isinstance( value, list ):
         matchRange = True

      dscpNameValid = False
      if matchOption == tacMatchOption.matchDscpEcn and value and \
         isinstance( value, str ):
         value = AclCliLib.dscpValueFromCli( self, value )
         value = value[ 0 ]
         dscpNameValid = True

      valueString = None
      if matchRange:
         valueString = MultiRangeRule.multiRangeToCanonicalString( value )

      if not no:
         allow, pmapNames, macAclClassCounts, ipv6ClassCounts, vxlanIntfPresent = \
                     allowMatchOptionInClassMap( entry, matchType )
         if not allow:
            errorMsg = f'The match type {matchType} is not allowed in the ' \
               f'class-map. '
            errorMsg += f'\nThe class-map is present in the following policy-maps:' \
               f' {pmapNames} '
            if qosAclHwStatus.matchMacInClassMapSupported:
               if( matchType in ( 'mac', 'ipv6' ) and not
                  qosAclHwStatus.matchMacInClassMapV6Supported ):
                  if matchType == 'mac' and ipv6ClassCounts or \
                        matchType == 'ipv6' and macAclClassCounts:
                     errorMsg = 'Invalid class map match types are configured in '
                     errorMsg += f'the following policy-maps {pmapNames}: '
                     errorMsg += 'The policy-map is not allowed to have both ipv6 '
                     errorMsg += 'class-maps and mac class-maps'
                  elif matchType == 'mac':
                     errorMsg = 'Invalid action configured in the following '
                     errorMsg += f'policy-maps {pmapNames}: '
                     errorMsg += 'Reconfigure the policy-map to use police action'
               elif matchType == 'mac':
                  errorMsg = 'Invalid action configured in the following policy-maps'
                  errorMsg += f' {pmapNames}: '
                  errorMsg += 'Reconfigure the policy-map to use police action'
            elif qosAclHwStatus.vxlanPolicyQosSupported and vxlanIntfPresent and \
                 matchType != 'l2Params':
               errorMsg += 'which are applied on the VTI'
            else:
               errorMsg += 'which have a policer configured in the class-default. '
               errorMsg += ' When class-default has policer, the policy-map is not '
               errorMsg += 'allowed to have both ipv4 class-maps and ipv6 class-maps'
            self.addError( errorMsg )
            return
         if not self.matchOptionExists( entry, matchOption ):
            entry.match.newMember( matchOption )
         # access-group ip/ipv6/mac
         if matchOption not in ( tacMatchOption.matchDscpEcn,
            tacMatchOption.matchMplsTrafficClass ):
            entry.match[ matchOption ].strValue = value
         elif matchOption == tacMatchOption.matchDscpEcn:
            self.setMatchDscpEcnValue( entry, value, ecnValue, matchRange,
                                       matchOption, valueString, dscpNameValid )
            entry.match[ matchOption ].strValue = 'dscpEcn'  # 'dscpEcn'
         elif matchOption == tacMatchOption.matchMplsTrafficClass:
            self.setMatchMplsTrafficClassValue( entry, value, matchOption )
            entry.match[ matchOption ].strValue = matchType
         # Delete other matchOption if any
         matchOptionsOtherToEnum = matchOptionOtherToEnum( matchOption )
         for matchOption_ in matchOptionsOtherToEnum:
            if self.matchOptionExists( entry, matchOption_ ):
               del entry.match[ matchOption_ ]
      else:
         if removeUnconditionally:
            del entry.match[ matchOption ]
         elif self.matchOptionExists( entry, matchOption ):
            entryMatch = entry.match[ matchOption ]
            if entryMatch.option == matchOption:
               if ( matchOption != tacMatchOption.matchDscpEcn and
                    value == entryMatch.strValue ) or ( matchOption
                    == tacMatchOption.matchMacAccessGroup and
                    ( value == entryMatch.strValue or removeUnconditionally ) ):
                  del entry.match[ matchOption ]
               elif not matchRange and matchOption == tacMatchOption.matchDscpEcn:
                  # match dscp when no range is configured
                  onlyEcnCfg = entryMatch.dscpEcnValue.dscp == invalidDscp and \
                               len( entryMatch.dscpEcnValue.dscpColl ) == 0 and \
                               entryMatch.dscpEcnValue.ecn != ecnDontCare
                  if removeUnconditionally and matchType == 'ecn' and onlyEcnCfg:
                     del entry.match[ matchOption ]
                  elif removeUnconditionally and not onlyEcnCfg:
                     del entry.match[ matchOption ]
                  if removeUnconditionally:
                     return
                  if not ecnValue:
                     ecnValue = ecnDontCare
                  else:
                     ecnValue = AclCliLib.ecnValueFromCli( self, ecnValue )
                  if value is None:
                     value = invalidDscp
                  if entryMatch.dscpEcnValue.dscpNameValid == dscpNameValid and \
                     value == entryMatch.dscpEcnValue.dscp and \
                     ecnValue == entryMatch.dscpEcnValue.ecn:
                     del entry.match[ matchOption ]
               elif matchOption == tacMatchOption.matchMplsTrafficClass:
                  if value is None:
                     del entry.match[ matchOption ]
                     return
                  if isinstance( value, int ):
                     value = [ value ]
                  if self.isCollSame( entryMatch.mplsTrafficClassVal.
                                      mplsTrafficClassColl, value ):
                     del entry.match[ matchOption ]
               elif matchRange:
                  # match vlan/dscp when range is configured
                  if removeUnconditionally:
                     del entry.match[ matchOption ]
                     return
                  ecnCheck = True
                  if matchOption == tacMatchOption.matchDscpEcn:
                     rangeColl = entryMatch.dscpEcnValue.dscpColl
                  if matchOption == tacMatchOption.matchDscpEcn and not ecnValue:
                     ecnValue = ecnDontCare
                  elif matchOption == tacMatchOption.matchDscpEcn:
                     ecnValue = AclCliLib.ecnValueFromCli( self, ecnValue )
                  if matchOption == tacMatchOption.matchDscpEcn and \
                     ecnValue != entryMatch.dscpEcnValue.ecn:
                     ecnCheck = False
                  if self.isCollSame( rangeColl, value, ecnCheck ):
                     del entry.match[ matchOption ]

   # print one or all class maps
   @staticmethod
   def showClassMap( mode, args ):
      mapType = args.get( 'control-plane', 'qos' )
      cmapName = args.get( 'CMAP' )
      classMapsAllModel = ClassMapAllModel()
      classMapsContainer = CMapModelContainer( gv.qosAclConfig, qosHwStatus,
                                               qosAclHwStatus,
                                               qosSliceHwStatus,
                                               mapTypeToEnum( mapType ),
                                               classMapsAllModel )
      if cmapName is None:
         classMapsContainer.populateAll()
      else:
         classMapsContainer.populateClassMap( cmapName, classPrio=1 )
      return classMapsAllModel

class ClassMapModeQos( ClassMapMode ):
   name = "Class Map Qos Configuration"

# -----------------------------------------------------------------
# The show command in 'config-cmap' mode
#
#              show active|pending|diff
# -----------------------------------------------------------------
class CmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new class-map configuration to be applied',
      'active': 'Display the class-map configuration in the current running-config',
      'current': ( 'Display the class-map configuration '
         'in the current running-config' ),
      'diff': ( 'Display the diff between class-map configuration '
          'current running-config and to be applied' ),
   }

   handler = 'QosCliServicePolicyHandler.cmapModeShowCmdHandler'

ClassMapModeQos.addShowCommandClass( CmapModeShowCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match mpls traffic-class ( { TC } | TC_RANGE )
# --------------------------------------------------------------------------------
class MatchMplsTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'match mpls traffic-class ( { TC } | TC_RANGE )'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'mpls': nodeMpls,
      'traffic-class': nodeTrafficClass,
      'TC': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Match packets by Mpls Traffic Class value' ),
      'TC_RANGE': MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 0, 7 ), noSingletons=True,
         helpdesc=( 'Mpls traffic-class value(s) or range(s) '
            'of Mpls traffic-class values' ), priority=CliParser.PRIO_HIGH ),
   }

   handler = "QosCliServicePolicyHandler.matchMplsTrafficClassCmdHandler"
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchMplsTrafficClassCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match dscp ( DSCPVALUE | DSCP_NAME | DSCPRANGE )
#       [ ecn ( ect-ce | non-ect | ce | ect ) ]
# --------------------------------------------------------------------------------
class MatchDscpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'match dscp ( DSCPVALUE | DSCP_NAME | DSCPRANGE ) '
              '[ ecn ( ect-ce | non-ect | ce | ect ) ]' )
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'dscp': nodeDscpMatch,
      'DSCPVALUE': CliMatcher.IntegerMatcher( 0, AclLib.MAX_DSCP,
         helpdesc='DSCP Value' ),
      'DSCP_NAME': CliMatcher.EnumMatcher(
         { k: v[ 1 ] for k, v in AclCliLib.dscpAclNames.items() } ),
      'DSCPRANGE': MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 0, AclLib.MAX_DSCP ), noSingletons=True,
         helpdesc='DSCP value(s) or range(s) of DSCP values',
         priority=CliParser.PRIO_HIGH ),
      'ecn': nodeDscpEcn,
      'ect': matcherEct,
      'ect-ce': matcherEctCe,
      'ce': matcherCe,
      'non-ect': matcherNonEct,
   }

   handler = 'QosCliServicePolicyHandler.matchDscpCmdHandler'
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchDscpCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match ecn ( ect-ce | non-ect | ce | ect )
# --------------------------------------------------------------------------------
class MatchEcnCmd( CliCommand.CliCommandClass ):
   syntax = 'match ecn ( ect-ce | non-ect | ce | ect )'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'ecn': nodeDscpEcn,
      'ect': matcherEct,
      'ect-ce': matcherEctCe,
      'ce': matcherCe,
      'non-ect': matcherNonEct,
   }

   handler = 'QosCliServicePolicyHandler.matchEcnCmdHandler'
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchEcnCmd )

# --------------------------------------------------------------------------------
# ( no | default ) match ( dscp | ecn | ( mpls traffic-class ) |
#       ( mac access-group ) )
# --------------------------------------------------------------------------------
class NoOrDefaultClassMapModeMatchCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = ( 'match ( dscp | ecn | ( mpls traffic-class ) | '
                         '( mac access-group ) )' )
   data = {
      'match': matcherMatch,
      'dscp': nodeDscpMatch,
      'ecn': nodeDscpEcn,
      'mpls': nodeMpls,
      'traffic-class': nodeTrafficClass,
      'mac': nodeMac,
      'access-group': matcherAccessGroup,
   }

   noOrDefaultHandler = 'QosCliServicePolicyHandler.'\
      'noOrDefaultClassMapModeMatchCmdNoOrDefaultHandler'

ClassMapModeQos.addCommandClass( NoOrDefaultClassMapModeMatchCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match mac access-group GROUP
# --------------------------------------------------------------------------------
class MatchMacAccessGroupAccessCmd( CliCommand.CliCommandClass ):
   syntax = 'match mac access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'mac': nodeMac,
      'access-group': matcherAccessGroup,
      'GROUP': AclCli.userMacAclNameMatcher,
   }

   handler = "QosCliPolicyHandler.matchMacAccessGroupAccessCmdHandler"
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchMacAccessGroupAccessCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match ip access-group GROUP
# --------------------------------------------------------------------------------
class MatchIpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'match ip access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'ip': CliCommand.guardedKeyword( 'ip',
         helpdesc='Specify Ip Access-groups', guard=guardIpAcls ),
      'access-group': matcherAccessGroup,
      'GROUP': AclCli.ipAclNameMatcher,
   }

   handler = "QosCliPolicyHandler.matchIpAccessGroupCmdHandler"
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchIpAccessGroupCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match ipv6 access-group GROUP
# --------------------------------------------------------------------------------
class MatchIpv6AccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'match ipv6 access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'ipv6': CliCommand.guardedKeyword( 'ipv6',
         helpdesc='Specify Ipv6 Access-groups', guard=guardIp6Acls ),
      'access-group': matcherAccessGroup,
      'GROUP': AclCli.ip6AclNameMatcher,
   }

   handler = "QosCliPolicyHandler.matchIpv6AccessGroupCmdHandler"
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchIpv6AccessGroupCmd )

class L2ParamsMatchCmd( CliCommand.CliCommandClass ):
   syntax = "match { VLAN | INNER_VLAN | COS | INNER_COS | DEI }"
   noOrDefaultSyntax = syntax
   data = {
      "match": "Match the access rule specified",
      "VLAN": VlanMatchExpression,
      "INNER_VLAN": InnerVlanMatchExpression,
      "COS": CosMatchExpression,
      "INNER_COS": InnerCosMatchExpression,
      "DEI": DeiMatchExpression
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      matchCombo = []
      if 'vlan' in args:
         matchCombo.append( 'vlan' )
         args.pop( 'vlan' )
      if 'inner' in args:
         matchCombo.append( 'vlan inner' )
         args.pop( 'vlan_' )
         args.pop( 'inner' )
      if 'cos' in args:
         matchCombo.append( 'cos' )
         args.pop( 'cos' )
      if 'inner_' in args:
         matchCombo.append( 'cos inner' )
         args.pop( 'cos_' )
         args.pop( 'inner_' )
      if 'dei' in args:
         matchCombo.append( 'dei' )
         args.pop( 'dei' )
      args[ 'matchCombo' ] = matchCombo

   handler = ClassMapModeQos.setL2ParamsMatchValue
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( L2ParamsMatchCmd )

# This command is to remove matches on L2Params when the input
# command is 'no match vlan' to remove 'match vlan 5 cos 2'
# The command works when match options in the command exists in
# class-map match.
class NoL2ParamsMatchCmd( CliCommand.CliCommandClass ):
   syntax = None
   noOrDefaultSyntax = \
            "match ( vlan | ( vlan_ inner ) | cos | ( cos_ inner_ ) | dei )"

   data = {
      "match": "Match the access rule specified",
      "vlan": CliCommand.Node( matcher=vlanKeywordMatcher,
                                guard=guardVlanMatch ),
      "vlan_": CliCommand.Node( matcher=vlanKeywordMatcher,
                                 guard=guardInnerVlanMatch ),
      "inner": CliCommand.Node( matcher=innerVlanKeywordMatcher,
                                 guard=guardInnerVlanMatch ),
      "cos": CliCommand.Node( matcher=cosKeywordMatcher,
                               guard=guardCosMatch ),
      "cos_": CliCommand.Node( matcher=cosKeywordMatcher,
                               guard=guardInnerCosMatch ),
      "inner_": CliCommand.Node( matcher=innerCosKeywordMatcher,
                               guard=guardInnerCosMatch ),
      "dei": CliCommand.Node( matcher=deiKeywordMatcher,
                               guard=guardDeiMatch )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      matchCombo = []
      if 'vlan' in args:
         matchCombo.append( 'vlan' )
         args.pop( 'vlan' )
      if 'inner' in args:
         matchCombo.append( 'vlan inner' )
         args.pop( 'vlan_' )
         args.pop( 'inner' )
      if 'cos' in args:
         matchCombo.append( 'cos' )
         args.pop( 'cos' )
      if 'inner_' in args:
         matchCombo.append( 'cos inner' )
         args.pop( 'cos_' )
         args.pop( 'inner_' )
      if 'dei' in args:
         matchCombo.append( 'dei' )
         args.pop( 'dei' )
      args[ 'matchCombo' ] = matchCombo

   handler = None
   noOrDefaultHandler = ClassMapModeQos.setL2ParamsMatchValue

ClassMapModeQos.addCommandClass( NoL2ParamsMatchCmd )

ClassMapModeQos.addCommandClass( AbortCmd )

def Plugin( entityManager ):
   global qosAclHwStatus, qosHwStatus, qosSliceHwStatus
   qosAclHwStatus = LazyMount.mount(
      entityManager, "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = \
      "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
