#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import ConfigMount
import LazyMount
import Tac
import Tracing
from CliPlugin.QosCliCommon import ( QosProfileMode,
                                     getIntfListFromMode,
                                     setIntfConfig,
                                     setQosProfileConfig,
                                     invertIdToName )
from QosLib import ( invalidShapeRate, invalidGuaranteedBw )
from QosTypes import ( tacTxQueueSchedulerProfileName,
                       tacTxQueuePriority,
                       tacPercent )

__defaultTraceHandle__ = Tracing.Handle( 'QosGlobalConfigModeHandler' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

# -----------------------------------------------------------------------------------
# Variables for Qos GlobalConfigMode associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosConfig = None
qosGlobalConfig = None
qosInputConfig = None
qosHwStatus = None

#--------------------------------------------------------------------------------
# Handler of QosRandomDetectWredAllowEctCmd
# --------------------------------------------------------------------------------
def configureGlobalWredAllowEct( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.wredAllowEct = False
   else:
      qosInputConfig.wredAllowEct = True

# --------------------------------------------------------------------------------
# Handler of QosTxQueueNameCmd
# --------------------------------------------------------------------------------
def handleTxQueueName( mode, args ):
   txQueue = args.get( 'TXQ_VALUE' )
   txQueueName = args.get( 'TXQ_NAME' )
   queueNameToIdMap = qosInputConfig.queueNameToTxQueueIdMap.queueNameToTxQueueId
   invertIdToName( txQueue, queueNameToIdMap, txQueueName )

#--------------------------------------------------------------------------------
# Handler of QosTxQueueSchedulerProfileCmd
#--------------------------------------------------------------------------------
def setGlobalTxQueueSchedulerProfile( mode, args ):
   txQueueId = args[ 'TXQ_VALUE' ]
   profileName = tacTxQueueSchedulerProfileName.fixedResponsive
   qosInputConfig.globalTxQueueToSchedulerProfile[ txQueueId ] = profileName

#--------------------------------------------------------------------------------
# No handler of QosTxQueueSchedulerProfileCmd
#--------------------------------------------------------------------------------
def noGlobalTxQueueSchedulerProfile( mode, args ):
   txQueueId = args[ 'TXQ_VALUE' ]
   del qosInputConfig.globalTxQueueToSchedulerProfile[ txQueueId ]

#--------------------------------------------------------------------------------
# Handler of QosTxQueuePerPortSchedulerProfileCmd
#--------------------------------------------------------------------------------
def setPerPortTxQueueSchedulerProfile( mode, args ):
   schProfileName = tacTxQueueSchedulerProfileName.fixedResponsive

   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if txQueueConfig is None:
         txQueueConfig = profile.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
      txQueueConfig.schedulerProfile = schProfileName
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
         txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
         if txQueueConfig is None:
            txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
         txQueueConfig.schedulerProfile = schProfileName
         setIntfConfig( intf )

#--------------------------------------------------------------------------------
# No handler of QosTxQueuePerPortSchedulerProfileCmd
#--------------------------------------------------------------------------------
def noPerPortTxQueueSchedulerProfile( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if txQueueConfig:
         txQueueConfig.schedulerProfile = \
               tacTxQueueSchedulerProfileName.defaultProfile
         setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig:
               txQueueConfig.schedulerProfile = \
                     tacTxQueueSchedulerProfileName.defaultProfile
               setIntfConfig( intf )

#--------------------------------------------------------------------------------
# Handler of LagSubIntfRebalanceCmd
# --------------------------------------------------------------------------------
def enableLagSubIntfRebalance( mode, args ):
   qosGlobalConfig.lagRebalanceDisabled = False

# --------------------------------------------------------------------------------
# No handler of LagSubIntfRebalanceCmd
# --------------------------------------------------------------------------------
def disableLagSubIntfRebalance( mode, args ):
   qosGlobalConfig.lagRebalanceDisabled = True

# --------------------------------------------------------------------------------
# Handler of TxQueueShapeRateAdaptiveCmd
# --------------------------------------------------------------------------------
def enableTxQueueShapeRateAdaptive( mode, args ):
   qosInputConfig.txQueueShapeRateAdaptive = True
   mode.addWarning( "Change will take effect only after switch reboot." )

# --------------------------------------------------------------------------------
# No or default handler of TxQueueShapeRateAdaptiveCmd
# --------------------------------------------------------------------------------
def disableTxQueueShapeRateAdaptive( mode, args ):
   qosInputConfig.txQueueShapeRateAdaptive = False
   mode.addWarning( "Change will take effect only after switch reboot." )

# --------------------------------------------------------------------------------
# Handler of LagSubIntfLoadbalanceCmd
# --------------------------------------------------------------------------------
def enableLagSubIntfLoadbalance( mode, args ):
   if qosGlobalConfig.lagSubIntfLoadbalance is False:
      qosGlobalConfig.lagSubIntfLoadbalance = True
      mode.addWarning( "Change will take effect only after switch reboot." )

# --------------------------------------------------------------------------------
# No or default handler of LagSubIntfLoadbalanceCmd
# --------------------------------------------------------------------------------
def disableLagSubIntfLoadbalance( mode, args ):
   if qosGlobalConfig.lagSubIntfLoadbalance is True:
      qosGlobalConfig.lagSubIntfLoadbalance = False
      mode.addWarning( "Change will take effect only after switch reboot." )

# --------------------------------------------------------------------------------
# Handler of QosParentFairSchedCmd
# --------------------------------------------------------------------------------
def enableParentFairSched( mode, args ):
   if not qosGlobalConfig.parentFairSched:
      qosGlobalConfig.parentFairSched = True
      mode.addWarning( "Change will take effect only after switch reboot." )

# --------------------------------------------------------------------------------
# No or default handler of QosParentFairSchedCmd
# --------------------------------------------------------------------------------
def disableParentFairSched( mode, args ):
   if qosGlobalConfig.parentFairSched:
      qosGlobalConfig.parentFairSched = False
      mode.addWarning( "Change will take effect only after switch reboot." )

def Plugin( entityManager ):
   global qosConfig, qosGlobalConfig, qosInputConfig, qosHwStatus
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
