# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
from ArnetModel import IpGenericAddrAndMask, IpGenericAddress
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import List
from CliModel import Submodel
import Tac

class ClientInfo( Model ):
   hostname = Str( help='IP-address, subnet or hostname of the client' )
   authPort = Int( help='Specifies authorization port number' )
   acctPort = Int( help='Specifies accounting port number' )
   vrf = Str( help='The VRF over which this client is reachable' )
   resolvedIpAddr = IpGenericAddrAndMask( help='Resolved IP-address of the client',
                                          optional=True )

class ClientStats( Model ):
   clientInfo = Submodel(
      valueType=ClientInfo, help='Configured client information' )
   messagesReceived = Int( help='Number of authentication and accounting request'
                                ' messages received from the client' )
   messagesSent = Int( help='Number of authentication and accounting response'
                            ' messages sent to the client' )
   acceptRespSent = Int( help='Number access-accept and password-ack messages sent'
                              ' to this client' )
   rejectRespSent = Int( help='Number of access-reject and password-reject messages'
                              ' sent to this client' )
   acctStartsReceived = Int( help='Number of accounting start messages received'
                                  ' from this client' )
   acctInterimUpdatesReceived = Int( help='Number of accounting interim update'
                                          ' messages received from this client' )
   acctStopsReceived = Int( help='Number of accounting stop messages received'
                                 ' from this client' )
   badRequests = Int( help='Number of bad authentication and accounting requests'
                           ' received from this client' )
   coaRequestsSent = Int( help='Number of CoA requests sent to this client' )
   dmRequestsSent = Int( help='Number of DM requests sent to this client' )
   coaAcksReceived = Int( help='Number of CoA Ack responses received from this'
                               ' client' )
   dmAcksReceived = Int( help='Number of DM Ack responses received from this'
                              ' client' )
   coaNaksReceived = Int( help='Number of CoA Nak responses received from this'
                               ' client' )
   dmNaksReceived = Int( help='Number of DM Nak responses received from this'
                              ' client' )
   badResponses = Int( help='Number of bad CaA and DM responses received from'
                            ' this client' )
   connectionErrors = Int( help='Number of connection errors to the client' )

class ClientGroupInfo( Model ):
   members = List( valueType=ClientStats, help='List of clients in a client group' )
   lastCounterClearTime = Float( help='UTC time when the counters were '
                                      'last cleared' )

class showClientGroupCmd( Model ):
   groups = Dict( keyType=str, valueType=ClientGroupInfo,
                  help='Mapping of client group name to client group info' )

   def render( self ):
      template = '%35s:%11d'
      for gName, group in self.groups.items():
         print( f'Radius Client Group                :'
                f' { gName }' )
         print( 'Last time counters were cleared    :',
                Ark.utcTimeRelativeToNowStr( group.lastCounterClearTime ) )
         for stats in group.members:
            print()
            print( f'   Client                          :'
                   f' { stats.clientInfo.hostname },'
                   f' authentication port { stats.clientInfo.authPort },'
                   f' accounting port { stats.clientInfo.acctPort },'
                   f' vrf { stats.clientInfo.vrf }' )
            if stats.clientInfo.resolvedIpAddr is None:
               print( '   Resolved IP address             : Not Resolved' )
            else:
               print( f'   Resolved IP address             :'
                      f' { stats.clientInfo.resolvedIpAddr }' )
            print( template % ( 'Messages received', stats.messagesReceived ) )
            print( template % ( 'Messages sent', stats.messagesSent ) )
            print( template % ( 'Accept response sent', stats.acceptRespSent ) )
            print( template % ( 'Reject response sent', stats.rejectRespSent ) )
            print( template % ( 'Accounting start received',
                                stats.acctStartsReceived ) )
            print( template % ( 'Interim updates received',
                                stats.acctInterimUpdatesReceived ) )
            print( template % ( 'Accounting stop received',
                                stats.acctStopsReceived ) )
            print( template % ( 'Bad requests', stats.badRequests ) )
            print( template % ( 'CoA requests sent', stats.coaRequestsSent ) )
            print( template % ( 'DM requests sent', stats.dmRequestsSent ) )
            print( template % ( 'CoA ACKs received', stats.coaAcksReceived ) )
            print( template % ( 'DM ACKs received', stats.dmAcksReceived ) )
            print( template % ( 'CoA NAKs received', stats.coaNaksReceived ) )
            print( template % ( 'DM NAKs received', stats.dmNaksReceived ) )
            print( template % ( 'Bad responses', stats.badResponses ) )
            print( template % ( 'Connection errors', stats.connectionErrors ) )
         print()

class ServerInfo( Model ):
   serverId = Int( help='ID of the RADIUS server' )
   hostName = Str( help='IP address or hostname of the RADIUS server' )
   authPort = Int( help='Specifies authorization port number', optional=True )
   acctPort = Int( help='Specifies accounting port number', optional=True )
   vrf = Str( help='The VRF over which this server is reachable' )
   resolvedIpAddr = IpGenericAddress(
      help='Resolved IP-address of the RADIUS server', optional=True )
   tlsPort = Int( help='RADIUS server TLS port number', optional=True )
   status = Str( help='Status of the RADIUS server' )
   lastDeclaredDead = Float( help='UTC time when the server was last declared dead',
      optional=True )

class ServerGroupInfo( Model ):
   members = List( valueType=ServerInfo, help='List of servers in a server group' )

class showServerGroupCmd( Model ):
   groups = Dict( keyType=str, valueType=ServerGroupInfo,
                  help='Mapping of server group name to server group info' )
   def render( self ):
      for gName, group in sorted ( self.groups.items() ):
         print( 'RADIUS Server Group:'
                f' { gName }' )
         for serverInfo in group.members:
            if serverInfo.tlsPort:
               print( f'   RADIUS Server { serverInfo.serverId }: '
                      f'{ serverInfo.hostName }, '
                      f'TLS port { serverInfo.tlsPort }, '
                      f'VRF { serverInfo.vrf }' )
            else:
               print( f'   RADIUS Server { serverInfo.serverId }: '
                      f'{ serverInfo.hostName }, '
                      f'authentication port { serverInfo.authPort }, '
                      f'accounting port { serverInfo.acctPort }, '
                      f'VRF { serverInfo.vrf }' )
            if serverInfo.resolvedIpAddr is None:
               print( '   Resolved IP address: not resolved' )
            else:
               print( f'   Resolved IP address:'
                      f' { serverInfo.resolvedIpAddr }' )
            if serverInfo.status == 'active':
               print( '   Status: active' )
            else:
               failedForTime = float( Tac.now() ) - serverInfo.lastDeclaredDead
               print( f'   Status: '
                      f'failed for {( failedForTime ):.0f} seconds'  )
            print()
         print()

