# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import difflib
import os
import sys
import tempfile

import Cell
import BasicCli
import BasicCliUtil
import CliCommand
import CliGlobal
import ConfigMount
import LazyMount
import CliMatcher
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin import ConfigTagCommon
from CliPlugin import RcfCliLib
from CliPlugin import RcfCliHelpers
from CliPlugin.RcfCliHelpers import addCompilationStatusToMode
from CliPlugin.RcfCliSessionHelpers import (
   isStartupConfig,
   isInteractive,
   isEapi,
)
from CliPlugin import RcfCli
from CliPlugin import RcfDebugLib
from CliPlugin.RcfCli import unitNameMatcher
from CliPlugin.RouterGeneralCli import routerGeneralCleanupHook
from CliPlugin.RcfScratchpad import (
   getOrInitScratchpad,
   getScratchpad,
   initializeScratchpad,
   removeScratchpad,
)
from CliPlugin.IraServiceCli import getEffectiveProtocolModel

from CliToken import RcfCliTokens
from IpLibTypes import ProtocolAgentModelType

import ShowCommand

import Tracing
import Tac
import Url
from CliMode.Rcf import ControlFunctionsBaseMode
from CodeUnitMapping import (
   CodeUnitKey,
   CodeUnitMapping,
   generateCodeUnitsForDomain,
)
from RcfHelperTypes import RcfExternalConfig
from RcfLinter import ( RcfLinter, RcfLintRequest )
import RcfLabeling
from RcfOpenConfigFunctionTextGen import generateOpenConfigFunctionTexts
from RcfOpenConfigFunctionValidator import (
      RcfOpenConfigFunctionValidationRequest,
      validateOpenConfigFunctions,
)
import RcfTypeFuture as Rcf
from Toggles import ConfigTagToggleLib, RcfLibToggleLib

traceHandle = Tracing.Handle( 'RcfCli' )
t0 = traceHandle.trace0

gv = CliGlobal.CliGlobal( dict(
      rcfConfig=RcfCli.gv.rcfConfig,
      codeUnitScratchpadHelper=RcfCli.gv.codeUnitScratchpadHelper,
      openConfigScratchpadHelper=RcfCli.gv.openConfigScratchpadHelper,
      rcfCompiler=None,
      rcfLinter=None,
      rcfStatus=None,
      debugSymbols=RcfDebugLib.gv.debugSymbols,
      redundancyStatus=None,
      redundancyStatusReactor=None,
      startupConfigIndentLen=6,
      rcfExternalConfig=None,
      configTagConfig=None,
      configTagIdState=None,
      configTagInput=None,
      configTagStatus=None,
   )
)

RcfCliModels = CliDynamicPlugin( 'RcfCliModels' )

configTagExpr = ConfigTagCommon.ConfigTagExpr
TagState = Tac.Type( "ConfigTag::ConfigTagState" )

def rcfCleanup( mode ):
   if gv.rcfConfig and gv.rcfConfig.isNonDefault():
      prompt = 'Remove all routing control functions? [Y/n] '
      if BasicCliUtil.confirm( mode, prompt ):
         RcfCliHelpers.commitRcfCode( gv, None )

routerGeneralCleanupHook.addExtension( rcfCleanup )

#----------------------------------------------------------------------------------
#                                  M O D E S
#----------------------------------------------------------------------------------

class ControlFunctionsMode( ControlFunctionsBaseMode, BasicCli.ConfigModeBase ):
   """CLI mode for interacting with routing control functions."""
   name = "control-functions"

   def __init__( self, parent, session ):
      ControlFunctionsBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if getScratchpad( self ):
         # Do automatic compile+commit in EAPI case
         if isInteractive( self ) and not isEapi( self ):
            self.addWarning( 'There are pending routing control function changes' )
            self.addWarning( 'These can be accessed by going back into '
                             'control-functions mode' )
            return

         if not isStartupConfig( self ):
            # Don't block startup on linting RCF config.
            # The results would be ignored anyway.
            RcfCliHelpers.lintRcfHelper( self, gv )

         RcfCliHelpers.commitRcfHelper( self, gv )

#----------------------------------------------------------------------------------
#                               C O M M A N D S
#----------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# "control-functions" in router general mode
#--------------------------------------------------------------------------------
def handlerControlFunctionsCmd( mode, args ):
   childMode = mode.childMode( ControlFunctionsMode )
   mode.session_.gotoChildMode( childMode )
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                    "configured for routing control functions configuration" )
   if getScratchpad( mode ):
      mode.addWarning( "There are pending routing control function changes" )

def noOrDefaultHandlerControlFunctionsCmd( mode, args ):
   rcfCleanup( mode )

#--------------------------------------------------------------------------------
# "code" in control-functions mode
#--------------------------------------------------------------------------------
class CodeCmd( CliCommand.CliCommandClass ):
   syntax = 'code [ unit UNIT_NAME ]'
   noOrDefaultSyntax = syntax
   data = {
      'code': RcfCliTokens.codeKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      unitName = RcfCliHelpers.getUnitName( gv, args )
      multiLineRcfInput = BasicCliUtil.getMultiLineInput( mode, cmd="code",
                                                          prompt="Enter RCF code" )
      if not isInteractive( mode ):
         rcfText = RcfCliHelpers.removeStartupConfigIndentation(
               multiLineRcfInput,
               gv.startupConfigIndentLen )
      else:
         rcfText = multiLineRcfInput
      # save RCF text to scratchpad before compilation
      scratchpad = getOrInitScratchpad( mode )
      scratchpad.setCodeUnit( unitName=unitName,
                              unitText=rcfText,
                              editSincePull=True,
                              url=None )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unitName = RcfCliHelpers.getUnitName( gv, args )
      scratchpad = getOrInitScratchpad( mode )
      scratchpad.setCodeUnit( unitName=unitName,
                              unitText=None,
                              editSincePull=True,
                              url=None )
      if ( unitName in gv.rcfConfig.codeUnitToConfigTag or
           unitName in scratchpad.rcfCodeUnitConfigTags ):
         scratchpad.rcfCodeUnitConfigTags[ unitName ] = None

ControlFunctionsMode.addCommandClass( CodeCmd )

ocFunctionNameMatcher = CliMatcher.DynamicNameMatcher(
   # Wrap with a lambda so we don't access openConfigScratchpadHelper before it's
   # initialized.
   # pylint: disable=unnecessary-lambda
   lambda mode: gv.openConfigScratchpadHelper.getEffectiveNames( mode ),
   # pylint: enable=unnecessary-lambda
   helpdesc='Unique name to identify an RCF function',
   pattern=RcfCliLib.rcfUnscopedFunctionPattern,
   helpname='WORD' )

# -------------------------------------------------------------------------------
# "function openconfig" in control-functions mode
# -------------------------------------------------------------------------------
class FunctionOpenConfig( CliCommand.CliCommandClass ):
   syntax = 'function openconfig FUNCTION_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'function': RcfCliTokens.functionKw,
      'openconfig': RcfCliTokens.openConfigKw,
      'FUNCTION_NAME': ocFunctionNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      functionName = args[ 'FUNCTION_NAME' ]
      multiLineRcfInput = BasicCliUtil.getMultiLineInput(
            mode, cmd="function openconfig",
            prompt="Enter OpenConfig formatted RCF function" )
      if not isInteractive( mode ):
         openConfigFunction = RcfCliHelpers.removeStartupConfigIndentation(
               multiLineRcfInput,
               gv.startupConfigIndentLen )
      else:
         openConfigFunction = multiLineRcfInput
      # save RCF text to scratchpad before compilation
      scratchpad = getOrInitScratchpad( mode )
      scratchpad.setOpenConfigFunction( functionName, openConfigFunction )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      functionName = args[ 'FUNCTION_NAME' ]
      scratchpad = getOrInitScratchpad( mode )
      scratchpad.setOpenConfigFunction( functionName, None )

if RcfLibToggleLib.toggleRcfPolicyDefinitionsEnabled():
   ControlFunctionsMode.addCommandClass( FunctionOpenConfig )

#--------------------------------------------------------------------------------
# "pull" in control-functions mode
#--------------------------------------------------------------------------------
class PullCmd( CliCommand.CliCommandClass ):
   syntax = 'pull [ unit UNIT_NAME ] replace URL'
   data = {
      'pull': RcfCliTokens.pullKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
      'replace': RcfCliTokens.replaceKw,
      'URL': RcfCliTokens.urlMatcherSource,
   }

   @staticmethod
   def handler( mode, args ):
      unitName = RcfCliHelpers.getUnitName( gv, args )
      url = args[ 'URL' ]
      # populate scratchpad with contents from file
      filename = tempfile.mktemp( prefix="rcfText-" )
      rcfInputText = None
      pullError = False
      with open( filename, 'w' ):
         try:
            url.get( filename )
            with open( filename ) as rcfFile:
               rcfInputText = rcfFile.read()
         except OSError as e:
            pullError = True
            if e.filename:
               e.filename = Url.filenameToUrl( e.filename )
            mode.addError( e )
      os.remove( filename )
      # If the pull was successful and we got some RCF text from the input file,
      # go ahead and populate the scratchpad.
      if not pullError and rcfInputText:
         scratchpad = getOrInitScratchpad( mode )
         scratchpad.setCodeUnit( unitName=unitName,
                                 unitText=rcfInputText,
                                 editSincePull=False,
                                 url=url.url )
         # For backwards compatability, auto-lint the unnamed code unit
         if unitName == gv.rcfConfig.unnamedCodeUnitName:
            RcfCliHelpers.lintRcfHelper( mode, gv )

ControlFunctionsMode.addCommandClass( PullCmd )

#--------------------------------------------------------------------------------
# "push" in control-functions mode
#--------------------------------------------------------------------------------
class PushCmd( CliCommand.CliCommandClass ):
   syntax = 'push [ unit UNIT_NAME ] [ pending | running-config ] URL'
   data = {
      'push': RcfCliTokens.pushKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
      'pending': RcfCliTokens.pendingKw,
      'running-config': RcfCliTokens.runningConfigKw,
      'URL': RcfCliTokens.urlMatcherPush,
   }

   @staticmethod
   def handler( mode, args ):
      rcfTextToWrite = ""
      scratchpad = getScratchpad( mode )
      unitName = RcfCliHelpers.getUnitName( gv, args )
      # If the user does not explicitly specify 'pending', there are three cases
      # where we fetch the RCF text from Sysdb instead of the scratchpad:
      #    1. User asked for 'running-config'
      #    2. Scratchpad does not exist (no pending changes from the user)
      #    3. The unit specified is not in the scratchpad
      if ( ( 'pending' not in args ) and
           ( 'running-config' in args or
             not scratchpad or
             unitName not in scratchpad.rcfCodeUnitTexts ) ):
         rcfTextToWrite = gv.rcfConfig.rcfCodeUnitText.get( unitName, None )
         if rcfTextToWrite is None:
            # Specified unit does not exist in Sysdb
            codeUnitStr = RcfCliHelpers.getCodeUnitStr( gv, unitName,
                                                        capitalize=True,
                                                        sentenceFormat=True )
            mode.addError( "%s is not configured" % codeUnitStr )
            return
      else:
         if ( ( not scratchpad ) or
              ( unitName not in scratchpad.rcfCodeUnitTexts ) ):
            # Specified unit does not exist in scratchpad
            codeUnitStr = RcfCliHelpers.getCodeUnitStr( gv, unitName,
                                                        capitalize=True,
                                                        sentenceFormat=True )
            mode.addError( "%s is not configured" % codeUnitStr )
            return
         # The rcfCodeUnitText dict might have "unitName" with a value of None (unit
         # is being deleted) or "unitName" might not exist in the dict. In either
         # case,treat rcfTextToWrite as "".
         rcfTextToWrite = scratchpad.rcfCodeUnitTexts.get( unitName )
         if rcfTextToWrite is None:
            rcfTextToWrite = ""
      url = args[ 'URL' ]
      filename = None
      # place contents from either scratchpad or config to file
      filename = tempfile.mktemp( prefix="rcfText-" )
      with open( filename, 'w' ) as rcfFile:
         rcfFile.write( rcfTextToWrite )
      try:
         url.put( filename )
      except OSError as e:
         if e.filename:
            e.filename = Url.filenameToUrl( e.filename )
         mode.addError( e )
      os.remove( filename )

ControlFunctionsMode.addCommandClass( PushCmd )

# -------------------------------------------------------------------------------
# "edit" in control-functions mode
# -------------------------------------------------------------------------------
class EditCmd( CliCommand.CliCommandClass ):
   syntax = 'edit [ unit UNIT_NAME ]'
   data = {
      'edit': RcfCliTokens.editKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
   }

   @staticmethod
   @BasicCliUtil.EapiIncompatible()
   def handler( mode, args ):
      scratchpad = getScratchpad( mode )
      unitName = RcfCliHelpers.getUnitName( gv, args )

      if ( ( not scratchpad ) or
           ( unitName not in scratchpad.rcfCodeUnitTexts ) ):
         # Specified unit does not exist in scratchpad, so get it from Sysdb.
         # If it's 'None' in Sysdb as well, we treat it as "" below and create unit.
         rcfTextToWrite = gv.rcfConfig.rcfCodeUnitText.get( unitName, None )
      else:
         rcfTextToWrite = scratchpad.rcfCodeUnitTexts.get( unitName )

      # The rcfCodeUnitText dict might have "unitName" with a value of None (unit
      # is being deleted) or "unitName" might not exist in the dict. In both
      # cases, treat rcfTextToWrite as "".
      if rcfTextToWrite is None:
         rcfTextToWrite = ""
      try:
         with tempfile.NamedTemporaryFile( suffix=".rcf", mode="w+" ) as f:
            # Push
            f.write( rcfTextToWrite )
            f.flush()
            # Launch editor
            # default editor is nano in restricted mode '-R'
            Tac.run( [ 'nano', '-R', f.name ],
                  stdout=sys.stdout, stderr=sys.stderr, stdin=sys.stdin )
            # Pull
            f.seek( 0 )
            rcfInputText = f.read()
      except ( Tac.SystemCommandError, OSError ) as e:
         mode.addError( "Editor failed: " + str( e ) )
         return
      # if the user didn't make any changes, don't set code unit and don't initialize
      if rcfTextToWrite == rcfInputText:
         return
      if not scratchpad:
         scratchpad = initializeScratchpad( mode )
      scratchpad.setCodeUnit( unitName=unitName,
                              unitText=rcfInputText,
                              editSincePull=True,
                              url=None )

ControlFunctionsMode.addCommandClass( EditCmd )

#--------------------------------------------------------------------------------
# "discard" in control-functions mode
#--------------------------------------------------------------------------------
class DiscardCmd( CliCommand.CliCommandClass ):
   syntax = 'discard [ ( unit UNIT_NAME ) | all ]'
   data = {
      'discard': RcfCliTokens.discardKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
      'all': RcfCliTokens.allKw,
   }

   @staticmethod
   def handler( mode, args ):
      if 'all' in args:
         removeScratchpad( mode )
      else:
         unitName = RcfCliHelpers.getUnitName( gv, args )
         scratchpad = getScratchpad( mode )
         if scratchpad:
            if unitName in scratchpad.rcfCodeUnitTexts:
               del scratchpad.rcfCodeUnitTexts[ unitName ]
            if unitName in scratchpad.rcfCodeUnitConfigTags:
               del scratchpad.rcfCodeUnitConfigTags[ unitName ]
            if scratchpad.rcfCodeUnitTexts:
               # If other code units still exist in scratchpad, clear compile result
               scratchpad.rcfLintResult = None
            elif not scratchpad.rcfCodeUnitConfigTags:
               # No more code units remain in scratchpad, clean it up
               removeScratchpad( mode )

ControlFunctionsMode.addCommandClass( DiscardCmd )

#--------------------------------------------------------------------------------
# "compile" in control-functions mode
#--------------------------------------------------------------------------------
class CompileCmd( CliCommand.CliCommandClass ):
   syntax = 'compile'
   data = {
      'compile': RcfCliTokens.compileKw,
   }

   @staticmethod
   def handler( mode, args ):
      # An explicit "compile" command should be ignored in startup-config
      if isStartupConfig( mode ):
         return

      RcfCliHelpers.lintRcfHelper( mode, gv )

ControlFunctionsMode.addCommandClass( CompileCmd )

#--------------------------------------------------------------------------------
# "commit" in control-functions mode
#--------------------------------------------------------------------------------
class CommitCmd( CliCommand.CliCommandClass ):
   syntax = 'commit'
   data = {
      'commit': RcfCliTokens.commitKw,
   }

   @staticmethod
   def handler( mode, args ):
      if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
         mode.addWarning( "Routing protocols model multi-agent must be "
                       "configured for routing control functions configuration" )

      # An explicit "commit" command should be ignored in startup-config
      if isStartupConfig( mode ):
         return

      RcfCliHelpers.commitRcfHelper( mode, gv )

ControlFunctionsMode.addCommandClass( CommitCmd )

#--------------------------------------------------------------------------------
# "code source" in control-functions mode
#--------------------------------------------------------------------------------
class CodeSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'code [ unit UNIT_NAME ] source pulled-from URL [ edited ]'
   noOrDefaultSyntax = 'code [ unit UNIT_NAME ] source pulled-from ...'
   data = {
      'code': RcfCliTokens.codeKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
      'source': RcfCliTokens.sourceKw,
      'pulled-from': RcfCliTokens.pulledFromKw,
      'URL': RcfCliTokens.urlMatcherSource,
      'edited': RcfCliTokens.editedKw,
   }

   @staticmethod
   def handler( mode, args ):
      if isInteractive( mode ):
         mode.addError( "This command is not supported in interactive mode" )
      else:
         url = args[ 'URL' ]
         unitName = RcfCliHelpers.getUnitName( gv, args )
         scratchpad = getOrInitScratchpad( mode )
         codeUnitUrlInfo = scratchpad.getOrInitCodeUnitUrlInfo( unitName )
         codeUnitUrlInfo.lastPulledUrl = url.url
         codeUnitUrlInfo.editSincePull = 'edited' in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Removing a code source URL takes immediate effect (no "commit" needed).
      # There is no need to bump up rcfConfig.rcfCodeVersion since the code source
      # URL has no functional impact on RCF compilation or AETs.
      unitName = RcfCliHelpers.getUnitName( gv, args )
      del gv.rcfConfig.rcfCodeUnitUrlInfo[ unitName ]

ControlFunctionsMode.addCommandClass( CodeSourceCmd )

#--------------------------------------------------------------------------------
# "code ( replace | delete | insert ) block ..." in control-functions mode
#
# "code [ unit UNIT_NAME ] replace block function FOO @LABEL"
# "code [ unit UNIT_NAME ] delete block function FOO @LABEL"
# "code [ unit UNIT_NAME ] insert block function FOO {before,after} @LABEL"
#
#--------------------------------------------------------------------------------
class CodeLabeledEditingCmd( CliCommand.CliCommandClass ):
   # Use subsyntaxes to enforce "before" and "after" only applying to "insert" action
   _deleteOrReplaceSubSyntax = '( ( replace | delete ) block function FUNC_NAME )'
   _insertSubSyntax = '( insert block function FUNC_NAME ( before | after ) )'
   syntax = ( 'code [ unit UNIT_NAME ] ( %s | %s ) LABEL' %
              ( _deleteOrReplaceSubSyntax, _insertSubSyntax ) )
   data = {
      'code': RcfCliTokens.codeKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
      'replace': RcfCliTokens.replaceKw,
      'delete': RcfCliTokens.deleteKw,
      'insert': RcfCliTokens.insertKw,
      'block': RcfCliTokens.blockKw,
      'function': RcfCliTokens.functionKw,
      'FUNC_NAME': RcfCliTokens.funcNameMatcher,
      'before': RcfCliTokens.beforeKw,
      'after': RcfCliTokens.afterKw,
      'LABEL': RcfCliTokens.labelMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      unitName = RcfCliHelpers.getUnitName( gv, args )
      unitKey = CodeUnitKey( Rcf.Metadata.FunctionDomain.USER_DEFINED, unitName )
      funcName = args[ 'FUNC_NAME' ]
      label = args[ 'LABEL' ]

      # Fetch code unit text for the specified code unit
      unitText = gv.codeUnitScratchpadHelper.getEffectiveText( mode, unitName )
      if unitText is None:
         codeUnitStr = RcfCliHelpers.getCodeUnitStr( gv, unitName,
                                                     capitalize=True,
                                                     sentenceFormat=True )
         mode.addError( "%s is not configured" % codeUnitStr )
         return

      # Get LabelLocPerFunction for the specified code unit
      llocRequest = RcfLabeling.LabelLocRequest( unitName, unitText )
      llocResult = RcfLabeling.buildLabelLoc( llocRequest )
      if llocResult.failed():
         codeUnitMapping = CodeUnitMapping( { unitKey: unitText } )
         for error in llocResult.fatalErrorList:
            mode.addError( error.render( codeUnitMapping ) )
         return

      # Check that the specified function exists in the specified code unit
      if funcName not in llocResult.llocPerFunction:
         codeUnitStr = RcfCliHelpers.getCodeUnitStr( gv, unitName,
                                                     sentenceFormat=True )
         mode.addError( f"Function {funcName} not found in {codeUnitStr}" )
         return

      # Get LabelLoc for the specified label
      labelLoc = llocResult.llocPerFunction[ funcName ].get( label )
      if labelLoc is None:
         mode.addError( "Label {} does not exist in function {}".format( label,
                                                                         funcName ) )
         return

      # Now that the CLI command has been sanity checked, prompt user for input
      if 'replace' in args:
         action = RcfLabeling.EditTextRequest.Action.REPLACE_BLOCK
         userText = BasicCliUtil.getMultiLineInput( mode, cmd="code replace",
                                                          prompt="Enter RCF code" )
      elif 'delete' in args:
         action = RcfLabeling.EditTextRequest.Action.DELETE_BLOCK
         userText = None
      elif 'insert' in args:
         if 'before' in args:
            action = RcfLabeling.EditTextRequest.Action.BLOCK_INSERT_BEFORE
         elif 'after' in args:
            action = RcfLabeling.EditTextRequest.Action.BLOCK_INSERT_AFTER
         else:
            assert False # Unexpected relative location type for "code insert" cmd
         userText = BasicCliUtil.getMultiLineInput( mode, cmd="code insert",
                                                          prompt="Enter RCF code" )
      else:
         assert False # Unexpected action type for "code" labeled editing cmd

      # Invoke RcfLabeling edit text API
      editTextReq = RcfLabeling.EditTextRequest(
            unitName,
            unitText,
            labelLoc,
            llocResult.llocPerFunction[ funcName ],
            action,
            userText )
      editTextRes = RcfLabeling.editRcfText( editTextReq )

      if editTextRes.failed():
         # Four possible reasons for failure

         # 1. Parse errors in the existing code unit's text
         if editTextRes.codeUnitTextErrors:
            codeUnitMapping = CodeUnitMapping( { unitKey: unitText } )
            for error in editTextRes.codeUnitTextErrors:
               mode.addError( error.render( codeUnitMapping ) )

         # 2. Parse errors in the user's multi line input
         if editTextRes.userTextErrors:
            userTextCodeUnitMapping = CodeUnitMapping( { unitKey: userText } )
            for error in editTextRes.userTextErrors:
               mode.addError( error.render( userTextCodeUnitMapping ) )

         # 3. The provided label is bad (exact semantics depend on insert/delete)
         if editTextRes.badLabelName:
            if 'replace' in args:
               err = ( "The new label %s must match the label (%s) being replaced" %
                       ( editTextRes.badLabelName, label ) )
            else: # 'insert'
               err = ( "The new label %s must not already exist in function %s" %
                       ( editTextRes.badLabelName, funcName ) )
            mode.addError( err )

         # 4. Surrounding text found in the user's multi line input
         if editTextRes.surroundingTextFound:
            mode.addError( "Must specify exactly one labeled block with no " +
                           "surrounding comments" )

         return
      else:
         scratchpad = getOrInitScratchpad( mode )
         scratchpad.setCodeUnit( unitName=unitName,
                                 unitText=editTextRes.resultCodeUnitText,
                                 editSincePull=True,
                                 url=None )

ControlFunctionsMode.addCommandClass( CodeLabeledEditingCmd )

# --------------------------------------------------------------------------------
# "code command-tag" in control-functions mode
# --------------------------------------------------------------------------------
class CodeConfigTagCmd( CliCommand.CliCommandClass ):
   syntax = 'code [ unit UNIT_NAME ] CONFIG_TAG_EXPR'
   noOrDefaultSyntax = 'code [ unit UNIT_NAME ] command-tag ...'
   data = {
      'code': RcfCliTokens.codeKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
      'CONFIG_TAG_EXPR': configTagExpr,
   }

   @staticmethod
   def handler( mode, args ):
      unitName = RcfCliHelpers.getUnitName( gv, args )
      configTag = args.get( 'CONFIG_TAG', None )

      ConfigTagCommon.commandTagConfigCheck( mode, configTag )
      ConfigTagCommon.checkTagInRemovedOrDisassociatedState( mode, configTag,
                                                             featureCheck=True )
      scratchpad = getOrInitScratchpad( mode )
      scratchpad.rcfCodeUnitConfigTags[ unitName ] = configTag

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unitName = RcfCliHelpers.getUnitName( gv, args )

      scratchpad = getOrInitScratchpad( mode )
      scratchpad.rcfCodeUnitConfigTags[ unitName ] = None

if ConfigTagToggleLib.toggleRcfConfigTagSupportEnabled():
   ControlFunctionsMode.addCommandClass( CodeConfigTagCmd )

#--------------------------------------------------------------------------------
# "show router rcf pending"
#--------------------------------------------------------------------------------
def handlerShowRcfPending( mode, args ):
   rcfPendingModel = RcfCliModels.ShowRcfPendingModel()
   scratchpad = getScratchpad( mode )

   if not scratchpad:
      # No pending changes
      rcfPendingModel.pendingChanges = False
      return rcfPendingModel
   else:
      rcfPendingModel.pendingChanges = True

   # Handle unit texts
   pendingCodeUnitTexts = dict( gv.rcfConfig.rcfCodeUnitText )
   pendingCodeUnitTexts.update( scratchpad.rcfCodeUnitTexts )

   # Handle unit config tag associations
   pendingCodeUnitConfigTags = {}
   for rcUnitName, rcConfigTagId in gv.rcfConfig.codeUnitToConfigTag.items():
      # Convert ConfigTagIds to their original tag strings
      tagStr = gv.configTagIdState.tagIdToTagStr.get( rcConfigTagId, None )
      if tagStr is not None:
         pendingCodeUnitConfigTags[ rcUnitName ] = tagStr
   pendingCodeUnitConfigTags.update( scratchpad.rcfCodeUnitConfigTags )

   # Handle unit urls
   pendingCodeUnitUrls = {}
   for rcUnitName, rcUnitUrlInfo in gv.rcfConfig.rcfCodeUnitUrlInfo.items():
      pendingCodeUnitUrls[ rcUnitName ] = rcUnitUrlInfo.lastPulledUrl
   scratchpadUrls = {}
   for unitName, unitUrlInfo in scratchpad.rcfCodeUnitUrlInfos.items():
      if unitUrlInfo.lastPulledUrl:
         scratchpadUrls[ unitName ] = unitUrlInfo.lastPulledUrl
   pendingCodeUnitUrls.update( scratchpadUrls )

   # Populate CAPI model with Sysdb+Scratchpad effective content
   # NOTE: we are intentionally skipping code units where the URL info has changed
   # but the RCF text of the unit has not been modified
   for unitName, pendingCodeUnitText in pendingCodeUnitTexts.items():
      rcfPendingCodeUnitModel = RcfCliModels.RcfPendingCodeUnitModel()
      rcfPendingCodeUnitModel.rcfTextModified = bool(
            scratchpad and unitName in scratchpad.rcfCodeUnitTexts )
      if pendingCodeUnitText is not None:
         rcfPendingCodeUnitModel.rcfText = pendingCodeUnitText
      pendingCodeUnitUrl = pendingCodeUnitUrls.get( unitName )
      if pendingCodeUnitUrl:
         rcfPendingCodeUnitModel.url = pendingCodeUnitUrl

      if unitName == gv.rcfConfig.unnamedCodeUnitName:
         rcfPendingModel.rcfCode = rcfPendingCodeUnitModel
      else:
         rcfPendingModel.rcfCodeUnits[ unitName ] = rcfPendingCodeUnitModel

   # Populate CAPI model with config tag associations. New pending code unit entries
   # are created for code units which have not yet been created but are associated
   # with a config tag
   for unitName, configTagStr in pendingCodeUnitConfigTags.items():
      # Get the model for the unit
      if unitName == gv.rcfConfig.unnamedCodeUnitName:
         model = rcfPendingModel.rcfCode
      else:
         model = rcfPendingModel.rcfCodeUnits.get( unitName )
      if model is None:
         model = RcfCliModels.RcfPendingCodeUnitModel()

      # Set command-tag info
      model.commandTag = configTagStr
      model.commandTagModified = bool( scratchpad and
                                       unitName in scratchpad.rcfCodeUnitConfigTags )

      # Set the model back
      if unitName == gv.rcfConfig.unnamedCodeUnitName:
         rcfPendingModel.rcfCode = model
      else:
         rcfPendingModel.rcfCodeUnits[ unitName ] = model

   return rcfPendingModel

#--------------------------------------------------------------------------------
# "show router rcf errors"
#--------------------------------------------------------------------------------
def handlerShowRcfErrorsCmd( mode, args ):
   rcfErrorsModel = RcfCliModels.ShowRcfErrorsModel()
   rcfErrorsModel.compilationErrors = \
         gv.rcfStatus.lastCompilationError.split( "\n" )
   rcfErrorsModel.active = gv.rcfStatus.lastCompilationError == ""
   return rcfErrorsModel

#--------------------------------------------------------------------------------
# "show router rcf code [ unit UNIT_NAME ]"
#--------------------------------------------------------------------------------
def handlerShowRcfCodeUnitsCmd( mode, args ):
   codeUnitsModel = RcfCliModels.ShowRcfCodeUnitModel()
   unitName = RcfCliHelpers.getUnitName( gv, args )
   unitText = gv.rcfConfig.rcfCodeUnitText.get( unitName, None )
   if unitText is None:
      mode.addWarning( RcfCliHelpers.getCodeUnitStr( gv, unitName,
                                                     capitalize=True )
                       + " not found" )
   else:
      codeUnitsModel.rcfCodeUnits[ unitName ] = unitText
   return codeUnitsModel

#--------------------------------------------------------------------------------
# "show pending"
#--------------------------------------------------------------------------------
class ShowRcfPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': RcfCliTokens.pendingKw,
   }

   cliModel = RcfCliModels.ShowRcfPendingModel
   handler = handlerShowRcfPending

ControlFunctionsMode.addShowCommandClass( ShowRcfPendingCmd )

#--------------------------------------------------------------------------------
# "show diff"
#--------------------------------------------------------------------------------
def handlerShowRcfDiffCmd( mode, args ):
   rcfDiffModel = RcfCliModels.ShowRcfDiffModel()
   scratchpad = getScratchpad( mode )
   if scratchpad:
      rcfCodeUnitDiffs = []
      for unitName in sorted( scratchpad.rcfCodeUnitTexts ):

         # Get rcf text for this unit in Sysdb
         rcRcfText = gv.rcfConfig.rcfCodeUnitText.get( unitName, "" )

         # Get rcf text for this unit in the scratchpad
         spRcfText = scratchpad.rcfCodeUnitTexts.get( unitName )
         if spRcfText is None:
            spRcfText = ""

         # Generate diff str for this unit
         codeCmd = RcfCliHelpers.getCodeCmd( gv, unitName )
         rcfCodeUnitDiffs.append( "".join( difflib.unified_diff(
            rcRcfText.splitlines( True ),
            spRcfText.splitlines( True ),
            fromfile='running-config: %s' % codeCmd,
            tofile='pending: %s' % codeCmd ) ) )

      rcfDiffModel.rcfDiff = '\n'.join( rcfCodeUnitDiffs )
   else:
      rcfDiffModel.rcfDiff = ""
   return rcfDiffModel

class ShowRcfDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': RcfCliTokens.diffKw
   }

   cliModel = RcfCliModels.ShowRcfDiffModel
   handler = handlerShowRcfDiffCmd

ControlFunctionsMode.addShowCommandClass( ShowRcfDiffCmd )

# -------------------------------------------------------------------------------
# "show router rcf function location [ ( function FUNC_NAME ) | ( unit UNIT_NAME ) ]"
# ------------------------------------------------------------------------------
def handlerShowRcfFunctionLocationCmd( mode, args ):
   functionLocationModel = RcfCliModels.ShowRcfFunctionLocationModel()

   functionSymbols = gv.debugSymbols.functionSymbol

   def insertFunctionLocationToModel( funcName ):
      funcSymbol = functionSymbols[ funcName ]
      if not funcSymbol:
         return

      if funcSymbol.functionDomain != Rcf.Metadata.FunctionDomain.USER_DEFINED:
         return

      funcDef = funcSymbol.functionDefinition
      func = RcfCliModels.FunctionLocationModel()
      func.codeUnit = funcDef.codeUnitName
      func.lineNum = funcDef.definitionStartLine
      functionLocationModel.functions[ funcName ] = func

   if "unit" in args:
      unitName = args[ 'UNIT_NAME' ]
      if unitName in gv.rcfConfig.rcfCodeUnitText:
         for funcName, funcSymbol in functionSymbols.items():
            functionDef = funcSymbol.functionDefinition

            if functionDef and unitName == functionDef.codeUnitName:
               insertFunctionLocationToModel( funcName )
      else:
         mode.addErrorAndStop( "Code unit " + unitName + " not found" )
      return functionLocationModel

   if "FUNC_KW" in args:
      funcName = args[ 'FUNC_NAME' ]
      funcSymbol = functionSymbols.get( funcName )

      if funcSymbol:
         if funcSymbol.functionDomain == Rcf.Metadata.FunctionDomain.USER_DEFINED:
            insertFunctionLocationToModel( funcName )
         else:
            # POA wrapper functions will return a KeyError. POA wrapper names dont't
            # parse on the command line and aren't meant to be visible, so this is a
            # reasonable "assert" to make.
            fnTypeStr = {
                  Rcf.Metadata.FunctionDomain.BUILTIN: "built-in",
                  Rcf.Metadata.FunctionDomain.OPEN_CONFIG: "OpenConfig",
            }[ funcSymbol.functionDomain ]
            mode.addErrorAndStop( f"Function {funcName} is a {fnTypeStr} function" )
      else:
         mode.addErrorAndStop( f"Function {funcName} not found" )
      return functionLocationModel

   # Gather all functions from all code units
   for funcName in functionSymbols:
      insertFunctionLocationToModel( funcName )

   return functionLocationModel

class RcfCommandTagHelper(
      ConfigTagCommon.ConfigTagDependentBase ):

   def processAndValidateConfig( self, mode, commandTagInfo ):
      # This is used by every command-tag command to validate RCF config post
      # modification, before making any modifications.

      # Since this is used for RCF validation of command-tag CLI commands,
      # we ignore the code units present in scratchpad and only validate
      # RCF config in sysdb/running-config.
      tag = commandTagInfo.commandTag
      tagId = commandTagInfo.commandTagId
      operState = commandTagInfo.operState

      # Check if the tag exists in commandTag CLI input.
      if not gv.configTagInput.configTagEntry.get( tag, None ):
         return
      codeUnitList = gv.rcfConfig.configTagToCodeUnit.get( tagId )
      if not codeUnitList:
         return

      # Create local copies of the sysdb config to filter out input that will be
      # removed from compilation after this CLI command is completed.

      codeUnitsForLinter = {}
      validationResults = []

      # Handle user-defined code units
      userDefinedCodeUnitTexts = dict( gv.rcfConfig.rcfCodeUnitText )
      rcfCodeVersion = gv.rcfConfig.rcfCodeVersion
      removedUnits = []
      if operState in ( TagState.removed, TagState.disabled ):
         # If this command-tag is associated with code unit texts, they will be
         # removed from sysdb in case of 'removed' operState and removed from
         # compilation by the rcfAgent in case of 'disabled' operState. Removing
         # these unit texts for validation.
         for unitName in codeUnitList.codeUnit:
            if unitName in userDefinedCodeUnitTexts:
               del userDefinedCodeUnitTexts[ unitName ]
            removedUnits.append( unitName )
      elif operState in ( TagState.disassociated, TagState.enabled ):
         pass
      else:
         assert False, "Unhandled action"
      codeUnitsForLinter.update( generateCodeUnitsForDomain(
         Rcf.Metadata.FunctionDomain.USER_DEFINED, userDefinedCodeUnitTexts ) )

      # Generate and validate OpenConfig functions
      openConfigFunctions = dict( gv.rcfConfig.openConfigFunction )
      if RcfLibToggleLib.toggleRcfPolicyDefinitionsEnabled():
         openConfigValidationRequest = RcfOpenConfigFunctionValidationRequest(
            openConfigFunctions, rcfCodeVersion )
         ocValidationResult = validateOpenConfigFunctions(
               openConfigValidationRequest )
         validationResults.append( ocValidationResult )
         if ocValidationResult.success:
            openConfigFunctionTexts = generateOpenConfigFunctionTexts(
                  ocValidationResult.openConfigFunctions,
                  gv.rcfExternalConfig )
            # Add generated text to the linter input
            codeUnitsForLinter.update( generateCodeUnitsForDomain(
               "OPEN_CONFIG", openConfigFunctionTexts ) )

      # Validate RCF text
      codeUnitMapping = CodeUnitMapping( codeUnitsForLinter )
      lintRequest = RcfLintRequest( codeUnitMapping,
                                 rcfCodeVersion = rcfCodeVersion,
                                 strictMode=isInteractive( mode ) )
      # TODO: BUG870823: Pass a custom external config object to the linter that
      # includes/excludes references associated with the commandTag.
      rcfExternalConfig = None
      if lintRequest.strictMode:
         rcfExternalConfig = gv.rcfExternalConfig
      rcfLintResult = gv.rcfLinter.lint( lintRequest, rcfExternalConfig )
      validationResults.append( rcfLintResult )

      # Only printing the compilation errors in this case and skipping the
      # warnings. Any errors are emitted by a config tag command, so directly
      # reference RCF in the error message to suggest the origin of the error.
      errorMsg = "RCF compilation failed"
      addCompilationStatusToMode( mode, validationResults, codeUnitMapping,
                                  addWarnings=False, errorMsg=errorMsg )

      # Validation is successful at this point. Handle both 'removed' and
      # 'disassociated' cases which require sysdb modification. 'enabled' and
      # 'disabled' states are handled by the RcfAgent.
      if operState in ( TagState.removed, TagState.disassociated ):
         gv.rcfConfig.rcfCodeVersionPending += 1
         # Copy the local copy of rcfCodeUnitText to sysDb, overwriting
         # other config that may have been added to ensure we have a validated
         # config in sysdb.
         if operState == TagState.removed:
            gv.rcfConfig.rcfCodeUnitText.clear()
            for unitName, unitText in userDefinedCodeUnitTexts.items():
               gv.rcfConfig.rcfCodeUnitText[ unitName ] = unitText
            # Remove existing url infos of removed units.
            for unitName in removedUnits:
               del gv.rcfConfig.rcfCodeUnitUrlInfo[ unitName ]
         # Remove command-tag associations from sysdb for both operStates.
         for unitName in codeUnitList.codeUnit:
            del gv.rcfConfig.codeUnitToConfigTag[ unitName ]
         del gv.rcfConfig.configTagToCodeUnit[ tagId ]
         # Increment the rcfCodeVersion to start another compilation of the rcf
         # config by the rcfAgent.
         gv.rcfConfig.rcfCodeVersion = gv.rcfConfig.rcfCodeVersionPending
      elif operState in ( TagState.enabled, TagState.disabled ):
         pass
      else:
         assert False, "Unhandled action"

   # 'removeTaggedConfig' and 'disassociateConfigFromTag' operations are handled by
   # the 'processAndValidateConfig' method.
   def removeTaggedConfig( self, mode, tag ):
      pass

   def disassociateConfigFromTag( self, mode, tag ):
      pass

ConfigTagCommon.ConfigTagState.registerConfigTagSupportingClass(
   RcfCommandTagHelper )

def Plugin( entMan ):
   # aclConfig is config mounted so that we can lint inside a config-session and
   # lookup the external reference config within the session. It is mounted as
   # writable because ConfigMounts must be mounted writable. Since this is all
   # happening in ConfigAgent, there is no multi writer issue.
   aclConfig = ConfigMount.mount( entMan, 'routing/acl/config', 'Acl::AclListConfig',
                                  'wS' )
   # BUG795911 - ROA table status is not config-session aware
   roaTableStatusDir = LazyMount.mount( entMan,
                                        Cell.path( 'routing/rpki/roaTable/status' ),
                                        'Rpki::RoaTableStatusDir', 'r' )
   dynPfxListConfigDir = ConfigMount.mount( entMan,
                                             "routing/dynPfxList/config",
                                             "Routing::DynamicPrefixList::Config",
                                             "wS" )
   tunnelRibNameIdMap = None
   if RcfLibToggleLib.toggleRcfResolutionRibsEnabled():
      tunnelRibNameIdMap = LazyMount.mount( entMan,
                                           "tunnel/tunnelRibs/tunnelRibNameIdMap",
                                           "Tunnel::TunnelTable::TunnelRibNameIdMap",
                                           'r' )
   gv.rcfStatus = LazyMount.mount( entMan, Cell.path( 'routing/rcf/status' ),
                                   'Rcf::Status', 'r' )
   gv.rcfExternalConfig = RcfExternalConfig( aclConfig, roaTableStatusDir,
                                             dynPfxListConfigDir,
                                             tunnelRibNameIdMap )
   gv.rcfLinter = RcfLinter()
   gv.configTagConfig = LazyMount.mount( entMan, 'configTag/config',
                                         'ConfigTag::ConfigTagConfig', 'r' )
   gv.configTagIdState = LazyMount.mount( entMan, 'configTag/configTagIdState',
                                          'ConfigTag::ConfigTagIdState', 'r' )
   gv.configTagInput = LazyMount.mount( entMan, 'configTag/input/cli',
                                        'ConfigTag::ConfigTagInput', 'w' )
   gv.configTagStatus = LazyMount.mount( entMan, 'configTag/status',
                                         'ConfigTag::ConfigTagStatus', 'r' )
