# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
      Bool,
      Dict,
      List,
      Model,
      Str,
      Submodel,
      Int
)

import TableOutput
from CliToken import RcfCliTokens

class RcfRouteMapConvertedTextModel( Model ):
   """
   Models the converted RCF text and whether it has converted successfully
   """
   rcfText = Str( help='Generated RCF text converted from route maps' )
   functionName = Str( help='Generated RCF function name converted from route maps' )
   conversionFailed = Bool( help='This route map has failed to convert to RCF' )

class RcfRouteMapConversionModel( Model ):
   """
   Models the RCF conversion of route maps
   """
   routeMaps = Dict(
         keyType=str,
         valueType=RcfRouteMapConvertedTextModel,
         help='Routing control functions converted from route maps, keyed by route '
         'map name' )

   def render( self ):
      print( "\n".join( func.rcfText for func in self.routeMaps.values() ) )

class RcfPendingCodeUnitModel( Model ):
   """
   Models the RCF pending code unit contents
   """
   rcfTextModified = Bool( help='This pending code unit text has been modified',
                           default=False )
   rcfText = Str( optional=True,
                  help='Pending routing control function code unit text' )
   url = Str( optional=True, help='Source file URL' )
   commandTagModified = Bool( help='The command tag association for this code unit'
                              'has been modified', default=False )
   commandTag = Str( optional=True,
                    help='Command tag associated with this code unit' )

   def renderPendingCodeUnitConfigTag( self, unitName ):
      if self.commandTag is None:
         # Only print 'no code ... command-tag' if the command-tag has been modified,
         # meaning it is pending removal
         if self.commandTagModified:
            if unitName is None:
               print( "no code command-tag" )
            else:
               print( f"no code unit {unitName} command-tag" )
      else:
         if unitName is None:
            print( f"code command-tag {self.commandTag}" )
         else:
            print( f"code unit {unitName} command-tag {self.commandTag}" )

   def renderPendingCodeUnitText( self, unitName ):
      if self.rcfText is None:
         # Only print 'no code ...' if the code unit has been modified, meaning it is
         # pending removal
         if self.rcfTextModified:
            if unitName is None:
               print( "no code" )
            else:
               print( "no code unit " + unitName )
      else:
         if unitName is None:
            print( "code" )
         else:
            print( "code unit " + unitName )
         if self.url:
            print( "! Source: " + self.url )
         print( self.rcfText + "EOF" )

   def renderPendingCodeUnit( self, unitName ):
      self.renderPendingCodeUnitConfigTag( unitName )
      self.renderPendingCodeUnitText( unitName )

class ShowRcfPendingModel( Model ):
   """
   Models the RCF pending contents (scratchpad)
   """
   __revision__ = 2
   pendingChanges = Bool( help='There are pending routing control function changes' )
   rcfCode = Submodel( valueType=RcfPendingCodeUnitModel,
                       optional=True,
                       help='Pending routing control function code' )
   rcfCodeUnits = Dict(
         keyType=str,
         valueType=RcfPendingCodeUnitModel,
         help="Pending routing control function code units keyed by unit name" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # Degrade to pre-code-units model
         del dictRepr[ 'rcfCodeUnits' ]
         # Unpack the RcfPendingCodeUnitModel for unnamed code unit into flattened
         # revision 1 model which contains optional url and rcfCode
         if 'rcfCode' in dictRepr:
            unnamedCodeUnitModel = dictRepr[ 'rcfCode' ]
            if 'url' in unnamedCodeUnitModel:
               dictRepr[ 'url' ] = unnamedCodeUnitModel[ 'url' ]
            if 'rcfText' in unnamedCodeUnitModel:
               dictRepr[ 'rcfCode' ] = unnamedCodeUnitModel[ 'rcfText' ]
            else:
               dictRepr[ 'rcfCode' ] = ""
            if 'commandTag' in unnamedCodeUnitModel:
               dictRepr[ 'commandTag' ] = unnamedCodeUnitModel[ 'commandTag' ]

      return dictRepr

   def render( self ):
      if not self.pendingChanges:
         print( "No pending routing control function changes" )
      if self.rcfCode:
         self.rcfCode.renderPendingCodeUnit( None )
      # Code Units are always rendered in alphabetical order
      for unitName, codeUnit in sorted( self.rcfCodeUnits.items() ):
         codeUnit.renderPendingCodeUnit( unitName )

class ShowRcfDiffModel( Model ):
   """
   Models the contents of RCF diff between scratchpad and commited code
   """
   rcfDiff = Str( help='Diff between commited and pending routing control '
                  'function code' )

   def render( self ):
      if self.rcfDiff:
         print( self.rcfDiff )

class ShowRcfErrorsModel( Model ):
   """
   Models compilation errors
   """
   active = Bool( help='Routing control functions are active' )
   compilationErrors = \
         List( valueType=str, help='Routing control function running-config errors' )

   def render( self ):
      if self.active:
         # no error to print
         print( 'no running-config routing control function errors' )
      else:
         for error in self.compilationErrors:
            print( error )

class ShowRcfCodeUnitModel( Model ):
   """
   Model for RCF code units
   """
   rcfCodeUnits = Dict( keyType=str, valueType=str,
                        help="RCF code unit text keyed by code unit name" )

   def render( self ):
      assert len( self.rcfCodeUnits ) <= 1, "the 'all' keyword is not supported yet"
      for codeUnitText in self.rcfCodeUnits.values():
         print( codeUnitText )

class FunctionLocationModel( Model ):
   """
   Model for RCF function
   """
   codeUnit = Str( help="Code unit that the function is stored in" )
   lineNum = Int(
      help="Line number of the start of the function within the code unit" )

class ShowRcfFunctionLocationModel( Model ):
   """
   Model for RCF function and code unit mapping
   """
   functions = Dict( keyType=str, valueType=FunctionLocationModel,
                  help="RCF function location information keyed by function name" )

   def render( self ):
      table = TableOutput.createTable(
         ( "Function Name", "Code Unit", "Line Number" ) )
      fl = TableOutput.Format( justify='left' )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )

      fr = TableOutput.Format( justify='right' )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )

      table.formatColumns( fl, fl, fr )
      sortedFunctionNames = sorted( self.functions, key=str.lower )

      for funcName in sortedFunctionNames:
         func = self.functions[ funcName ]
         codeUnit = func.codeUnit
         if not codeUnit:
            codeUnit = RcfCliTokens.unnamedCodeUnitString
         lineNum = func.lineNum
         table.newRow( funcName, codeUnit, lineNum )

      print( table.output() )
