# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CallbackRegistry import CallbackRegistry
from CliPlugin.RcfRouteMapConversion import gv
from CliDynamicSymbol import CliDynamicPlugin
from RouteMapLib import printRouteMapEntryAttributes
import re
from RcfFunctionTextGen import (
   Directive,
   fragmentsToFunctionText,
   RcfFragmentsMixin,
   TextBlock,
)
RcfCliModels = CliDynamicPlugin( 'RcfCliModels' )

class MatchStatementConverter( CallbackRegistry, RcfFragmentsMixin ):
   """
   Handles conversion of a Routing::RouteMap::MatchRule to
   RCF text.

   To handle a match rule of type "matchAbc", the class must
   provide a function with name "_matchAbcHandler" that accepts
   rule.

   The entry point here is "convert()"
   """

   @classmethod
   def convert( cls, rule ):
      """
      Translates a match statement into RCF string representation. If
      the translation fails (matcher not supported for translation
      yet) then returns None.

      Arguments
      ---------
      rule : Routing::RouteMap::MatchRule

      Returns
      -------
      str on success; None on failure;
      """

      maybeCallbacks = cls.getCallbacks( str( rule.option ) )
      if maybeCallbacks:
         callback, = maybeCallbacks
         return callback( cls, rule )
      return None

@MatchStatementConverter.registerCallback( "matchLocalPref" )
def matchLocalPrefCb( cls, rule ):
   return cls.matchLocalPref( rule.intValue )

@MatchStatementConverter.registerCallback( "matchTag" )
def matchTagCb( cls, rule ):
   return cls.matchTag( rule.intValue )

@MatchStatementConverter.registerCallback( "matchMetric" )
def matchMedCb( cls, rule ):
   return cls.matchMed( rule.intValue )

class SetStatementConverter( CallbackRegistry, RcfFragmentsMixin ):
   """
   Handles conversion of a setter in a Routing::RouteMap::MapEntry
   to RCF text identified by flag.

   To handle a setter with setterFlag "has<Attr>" the class must provide
   a function with name "_has<Attr>Handler" that accepts mapEntry.

   Entry point is convert()
   """

   @classmethod
   def convert( cls, setterFlag, mapEntry ):
      """
      Translates a set statement into RCF string representation. If
      the translation fails (setter not supported for translation
      yet) then returns None.

      Arguments
      ---------
      setterFlag : Routing::RouteMap::MapEntry::SetFlag
      mapEntry : Routing::RouteMap::MapEntry

      Returns
      -------
      str on success; None on failure;
      """
      maybeCallbacks = cls.getCallbacks( str( setterFlag ) )
      if maybeCallbacks:
         callback, = maybeCallbacks
         return callback( cls, mapEntry )
      return None

@SetStatementConverter.registerCallback( "hasLocalPref" )
def setLocalPrefCb( cls, mapEntry ):
   return cls.setLocalPref( mapEntry.localPref )

@SetStatementConverter.registerCallback( "hasWeight" )
def setWeightCb( cls, mapEntry ):
   return cls.setWeight( mapEntry.weight )

@SetStatementConverter.registerCallback( "hasMetric" )
def setMedCb( cls, mapEntry ):
   return cls.setMed( mapEntry.metric.medType, mapEntry.metric.metricValue )

@SetStatementConverter.registerCallback( "hasOrigin" )
def setOriginCb( cls, mapEntry ):
   originValue = mapEntry.origin.lstrip( "bgp" ).upper()
   return cls.setOrigin( originValue )

class GenerateRcfFunctionName:

   def __init__( self ):
      self.genRcfFnName = []

   def getRcfFunctionName( self, rmapName ):
      """
      Provides an RCF compliant function name for route map
      names. In case two route maps map to the same RCF name,
      a number is appended at the end of the RCF function name
      to identify them uniquely
      """
      # Substitute unsupported characters with "_"
      newRcfName = re.sub( "[^a-zA-Z0-9_]+", "_", rmapName )

      # If the route map name starts with a number, we prepend a "_" to it
      if newRcfName[ 0 ].isdigit():
         newRcfName = f"_{newRcfName}"
      newRcfNameBase = newRcfName

      idx = 1
      while newRcfName in self.genRcfFnName:
         newRcfName = f"{newRcfNameBase}{idx}"
         idx += 1

      self.genRcfFnName.append( newRcfName )
      return newRcfName

def handleUnsupportedRouteMapStmt( rmap, seqs, seqNo ):
   """
   Arguments
   ---------
   rmap : Route map config
   seqs : Sorted route map seqs
   seqNo : sequence number conversion failed at

   Returns
   -------
   A TextBlock instance with a comment containing the failed route-map and a
   FAIL_COMPILE directive.
   """
   rmOut = []
   for seq in seqs:
      mapEntry = rmap.mapEntry[ seq ]
      permit = "permit" if mapEntry.permit == "permitMatch" else "deny"
      rmOut.append( f"route-map {rmap.name} {permit} {seq}" )
      printRouteMapEntryAttributes( mapEntry, output=rmOut )

   textBlock = TextBlock.initEmpty()
   textBlock.commentLines.append( f"Stopped at sequence {seqNo} due to unsupported "
                                  + "route map configuration" )

   for line in rmOut:
      if not line.startswith( 'route-map' ):
         # Extra indent for rules
         line = f"   {line}"
      textBlock.commentLines.append( line )

   textBlock.directive = Directive.FAIL_COMPILE

   return textBlock

def addRmapSeqs( rmap ):
   """
   Arguments
   ---------
   rmap : Route map config

   Returns
   -------
   Tuple of:
      A list of TextBlockFragments to translate into an RCF function.
      A boolean to indicate whether conversion failed.
   """

   # Store each successfully converted sequence. Fully converted sequences will be
   # printed out if we encounter an unsupported statement in a later sequence.
   seqs = sorted( rmap.mapEntry )
   textBlocks = []
   for seq in seqs:
      mapEntry = rmap.mapEntry[ seq ]
      textBlock = TextBlock.initEmpty()
      textBlocks.append( textBlock )

      # Add description in generated RCF as a comment
      if mapEntry.description:
         textBlock.commentLines += list( mapEntry.description.values() )

      matchRules = mapEntry.matchRule
      rcfMatchers = []
      if matchRules:
         for rule in matchRules.values():
            rcfMatcher = MatchStatementConverter.convert( rule )
            if rcfMatcher is None:
               # BUG717185 rcfMatcher support
               textBlock.updateWith(
                     handleUnsupportedRouteMapStmt( rmap, seqs, seq ) )
               return textBlocks, True
            rcfMatchers.append( rcfMatcher )

      if mapEntry.subRouteMap:
         # BUG717203 sub route map support
         textBlock.updateWith(
               handleUnsupportedRouteMapStmt( rmap, seqs, seq ) )
         return textBlocks, True

      rcfSetters = []
      if mapEntry.setFlags:
         for flag in [ flag for flag in dir( mapEntry.setFlags )
                       if getattr( mapEntry.setFlags, flag ) is True ]:
            rcfSetter = SetStatementConverter.convert( flag, mapEntry )
            if rcfSetter is None:
               # BUG717195 setter support
               # BUG717202 handle continue statement
               textBlock.updateWith(
                     handleUnsupportedRouteMapStmt( rmap, seqs, seq ) )
               return textBlocks, True
            rcfSetters += rcfSetter

      # Check route-map permit or deny and return accordingly
      textBlock.returnValue = RcfFragmentsMixin.returnValue(
            mapEntry.permit == "permitMatch" )

      textBlock.conditionFragments += rcfMatchers
      textBlock.modificationFragments += rcfSetters

      if not rcfMatchers:
         # No match/sub-route-map in this sequence so any sequence after this
         # one is unreachable
         return ( textBlocks, False )

   return textBlocks, False

def generateRcfFunction( rmap, rcfName, outputByFn ):
   """
   Return value: bool
   ------------------
   False when conversion fails; True otherwise.

   outputByFn gets updated in place. outputByFn is
   keyed by RCF function name.
   """

   textBlocks, convFailed = addRmapSeqs( rmap )
   output = fragmentsToFunctionText( rcfName, textBlocks )

   outputByFn[ rmap.name ] = RcfCliModels.RcfRouteMapConvertedTextModel()
   outputByFn[ rmap.name ].rcfText = output
   outputByFn[ rmap.name ].functionName = rcfName
   outputByFn[ rmap.name ].conversionFailed = convFailed

# -----------------------------------------------------------------------------------
#                               C O M M A N D S
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# "route-map [ MAP ] convert rcf" in router exec mode
# -----------------------------------------------------------------------------------
def handlerRcfRouteMapCmd( mode, args ):
   name = args.get( 'MAP' )

   # outputByFn is keyed by RCF function name
   rmConversionModel = RcfCliModels.RcfRouteMapConversionModel()
   outputByRm = rmConversionModel.routeMaps

   rcfNameGenerator = GenerateRcfFunctionName()

   if name:
      rmap = gv.mapConfig.routeMap.get( name )
      if rmap:
         rcfName = rcfNameGenerator.getRcfFunctionName( rmap.name )
         generateRcfFunction( rmap, rcfName, outputByRm )
      else:
         mode.addErrorAndStop( f"Route map {name} not found" )
   else:
      for rmap in gv.mapConfig.routeMap.values():
         rcfName = rcfNameGenerator.getRcfFunctionName( rmap.name )
         generateRcfFunction( rmap, rcfName, outputByRm )

   return rmConversionModel
