# Copyright (c) 2018, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------------------------------------
# This module provides "show rib route ip[v6]" functions
#------------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function
from CliPlugin.IraCommonCli import ribRouteModel
from IpLibConsts import DEFAULT_VRF
from RibCapiLib import (
   showRibCapiCommand,
   EmptyResponseException,
)
from socket import AF_INET, AF_INET6
import json
import sys

def showRibdRoute( af, mode, vrfName, protocol, prefix, fibExcluded=None,
      shamlink=False ):
   assert 'IpRib' in ribRouteModel

   args = {}
   args[ 'af' ] = AF_INET if af == 'ipv4' else AF_INET6

   # routingProtocols means show all supported protocols
   if protocol != ribRouteModel[ 'IpRib' ].routingProtocol.routingProtocols:
      for i, p in enumerate( ribRouteModel[ 'IpRib' ].ribdSupportProtos ):
         if p == protocol:
            args[ 'protocol' ] = i
            break
      else:
         return

   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if fibExcluded:
      args[ 'show_fib_denied' ] = 1

   if prefix:
      # even if mlen was not specified, prefix was built as IpGenPrefix which
      # defaults to host mlen, so we always have '/<mlen>' in stringValue
      addr, mlen = prefix.stringValue.split( '/' )

      if prefix.af == 'ipv4':
         args[ 'addrv4' ] = addr
         args[ 'masklenv4' ] = int( mlen )
      else:
         args[ 'addrv6' ] = addr
         args[ 'masklenv6' ] = int( mlen )

   if shamlink:
      args[ 'sham_link' ] = 1

   cmd = "MIO_DGET_RIB_ROUTE"

   try:
      if shamlink:
         model = showRibCapiCommand( mode,
                                     ribRouteModel[ 'IpRib' ].RibRoutesByProtocol,
                                     cmd, args=args, highPerf=True,
                                     clientName='OSPF' if af == 'ipv4' else 'OSPF3' )
      else:
         model = showRibCapiCommand( mode,
                                     ribRouteModel[ 'IpRib' ].RibRoutesByProtocol,
                                     cmd, args=args, highPerf=True )
   except EmptyResponseException:
      model = ribRouteModel[ 'IpRib' ].RibRoutesByProtocol()

   if not mode.session_.shouldPrint():
      revision = mode.session_.requestedModelRevision()
      encoder = json.JSONEncoder( indent=4, sort_keys=False )
      for chunk in encoder.iterencode( model.toDict( streaming=True,
                                                     revision=revision ) ):
         sys.stdout.write( chunk )
      # Output is always newline terminated
      sys.stdout.write( '\n' )
   elif prefix:
      model.renderProtocols( vrfName=vrfName, prefixProto=protocol )
   else:
      model.renderProtocols( vrfName=vrfName )

