# Copyright (c) 2010, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#------------------------------------------------------------------------------------
# This module provides an interface to retrieve Rib-internal state
# for all Rib agents (across default and non-default VRFs)
#------------------------------------------------------------------------------------
from __future__ import absolute_import, division, print_function
from Arnet import NsLib
import Cell
from CliPlugin.VrfCli import (
   VrfExecCmdDec,
   getVrfNames,
)
from IpLibConsts import DEFAULT_VRF
import LazyMount
import RibCliLib
import sys
import Tac

# pylint: disable-msg=W0108

allVrfStatusLocal = None
l3Config = None

@VrfExecCmdDec( getVrfsFunc=getVrfNames )
def showAgentDump( mode, **kwargs ):
   vrfName = kwargs.get( "vrfName" )
   agentName = kwargs.get( "AGENT" )
   debug = kwargs.get( "debug" )
   ribout = kwargs.get( "ribout" )
   verbose = kwargs.get( "verbose" )

   # filter out any vrfs that have a VRF status not 'active'
   if vrfName != DEFAULT_VRF:
      vsl = allVrfStatusLocal.vrf.get( vrfName )
      if not vsl or vsl.state != 'active':
         mode.addError( "VRF %s is not active" % vrfName )
         return
      doShowAgentDump( mode, agentName, vrfName=vrfName, ns=vsl.networkNamespace,
                      debug=debug, ribout=ribout, verbose=verbose )
   else:
      doShowAgentDump( mode, agentName, debug=debug, ribout=ribout,
                       verbose=verbose )

def doShowAgentDump( mode, agentName,
                     vrfName=DEFAULT_VRF, ns=NsLib.DEFAULT_NS, debug=None,
                     ribout=None, verbose=None ):
   multiagent = l3Config.protocolAgentModel == "multi-agent"
   if agentName == 'ribd':
      if multiagent:
         mode.addError( "Rib agent is not active" )
         return
      cmds = [ '/usr/bin/RibdShowTech' ]
      if debug:
         cmds.append( 'debug' )
      if ribout:
         cmds.append( 'ribout' )
      if verbose:
         cmds.append( 'verbose' )
   elif agentName == 'ospf':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/OspfShowTech' ]
   elif agentName == 'isis':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/IsisShowTech' ]
   elif agentName == 'ospfv3':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/Ospf3ShowTech' ]
   elif agentName == 'rip':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/RipShowTech' ]

   try:
      NsLib.runMaybeInNetNs( ns, cmds,
                             stdout=sys.stdout,
                             stderr=Tac.DISCARD,
                             asRoot=True )
   except Tac.SystemCommandError as e:
      vrfErrStr = ''
      if vrfName != DEFAULT_VRF:
         vrfErrStr += ' for VRF %s' % vrfName
      if e.error in ( 1, 2 ):
         mode.addError( 'Failed to connect to %s%s' % ( agentName, vrfErrStr ) )
      elif e.error == 3:
         mode.addError( 'Timeout encountered while processing '
                        'show tech-support %s%s' % ( agentName, vrfErrStr ) )

def handlerShowTechSupportRibd( mode, args ):
   args.setdefault( "vrfName", None )
   showAgentDump( mode, **args )

@VrfExecCmdDec( getVrfsFunc=getVrfNames )
def doShowRibdRunningConfig( mode, **args ):
   vrfName = args.get( "vrfName", DEFAULT_VRF )
   RibCliLib.cliRibdShowCommand( mode, 'show running-config', cport=None,
                                 vrf=vrfName )

def handlerShowTechSupportRibdRunningConfig( mode, args ):
   args.setdefault( "vrfName", None )
   doShowRibdRunningConfig( mode, **args )

def handlerShowTechSupportExtendedAgents( mode, args ):
   args.setdefault( "vrfName", DEFAULT_VRF )
   showAgentDump( mode, **args )

def Plugin( entityManager ):
   global allVrfStatusLocal
   global l3Config
   # This is a little silly.  We use interface/status/all as a fake
   # CliSave anchor in our CliSavePlugin, so it has to get mounted here,
   # even though we never use it.
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
