#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from RouteTrack import RouteTrack
from TableOutput import Format, createTable
from CliPlugin.RouterGeneralCli import generalConfig
from CliDynamicSymbol import CliDynamicPlugin
from CliModel import cliPrinted
import json

RouteTrackModel = CliDynamicPlugin( 'RouteTrackModel' )

# This method exists so that doShowRouteTrack can be stubbed in the
# RouteTrackShowTest.
def showRouteTrackCommandHandler( mode, args ):
   doShowRouteTrack( mode, args )
   return cliPrinted( RouteTrackModel.RouteTrackLog )

def doShowRouteTrack( mode, args ):
   detail = 'detail' in args

   config = generalConfig.routeTrackConfig

   outputFormat = mode.session_.outputFormat_
   if outputFormat == 'text':
      showRouteTrackText( config, detail )
   else:
      assert outputFormat == 'json'
      showRouteTrackJson( config, detail )

def showRouteTrackJson( config, detail ):
   if config.vrfName:
      log = getMergedLogForEpoch( config.epoch )
   else:
      log = { 'events' : [] }
   print( json.dumps( log ) )

def showRouteTrackText( config, detail ):
   if not config.vrfName:
      print( 'Route tracking is not configured\n' )
      return

   log = getMergedLogForEpoch( config.epoch )

   print( f'Tracked events for VRF: {config.vrfName}, prefix: {config.prefix}' )

   formatLeft = Format( justify="left" )
   formatLeft.noPadLeftIs( True )
   formatLeft.padLimitIs( True )
   formatRight = Format( justify="right" )
   formatRight.noPadLeftIs( True )
   formatRight.padLimitIs( True )
   if detail:
      print( f'epoch: {config.epoch}' )
      table = createTable( ( 'Time',
                             'Offset (us)',
                             'Agent',
                             'Id',
                             'Context',
                             'Event' ),
                           tableWidth=200 )
      table.formatColumns( formatLeft, formatRight, formatLeft, formatRight,
                           formatLeft, formatLeft )
   else:
      table = createTable( ( 'Time',
                             'Offset (us)',
                             'Agent',
                             'Event' ),
                           tableWidth=200 )
      table.formatColumns( formatLeft, formatRight, formatLeft, formatLeft )

   wrappedLogs = log.get( 'wrappedLogs' )
   if wrappedLogs:
      print( f'Log has wrapped for: { ", ".join( wrappedLogs ) }' )
   print()

   events = log[ 'events' ]
   if not events:
      print( 'No events to display\n' )
      return

   firstEventNs = events[ 0 ][ 'nsSinceBoot' ]
   for event in events:
      offsetUs = f'+{ int( ( event[ "nsSinceBoot" ] - firstEventNs ) / 1000 ) }'
      if detail:
         table.newRow( event[ 'timestamp' ],
                       offsetUs,
                       event[ 'agent' ],
                       event[ 'eventId' ],
                       event[ 'context' ],
                       event[ 'description' ] )
      else:
         table.newRow( event[ 'timestamp' ],
                       offsetUs,
                       event[ 'agent' ],
                       event[ 'description' ] )
   print( table.output() )

# This indirection esists so that it can be stubbed in the
# RouteTrackShowHandlerTest.
def getMergedLogForEpoch( epoch ):
   return RouteTrack().getMergedLogForEpoch( epoch )
