#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
from Arnet import (
   IntfId,
   IpGenAddr,
)
import BgpLib
from BgpLib import (
   addressFamilies,
   NoOrDefault,
   PeerConfigKey,
   PeerConfigKeyType,
   vpnAfTypeCliKeywords,
)
import CliCommand
import CliCommon
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin import ConfigTagCommon
from CliPlugin import IntfCli
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RoutingBgpCli import (
   addPathSendConfig,
   attrIsPresent,
   BgpCmdBaseClass,
   BgpNEDCmdBaseClass,
   bgpNeighborConfig,
   configForVrf,
   configModeCmdForAf,
   emitWarningIfOrrNotSupportedInRibdMode,
   emitWarningIfRfdNotSupportedInRibdMode,
   getConflictingAttrsForAf,
   getTimeInSeconds,
   haveConflictingOutDelay,
   haveConflictingRouteMaps,
   haveConflictingWeight,
   MissingPolicyConfig,
)
import CliPlugin.RoutingBgpNeighborCli as Globals
from CliPlugin.RoutingBgpNeighborCli import (
   neighborAfActivateHelper,
   validatePeer,
)
import CliToken.RoutingBgp as bgpTokens
from CliToken.RoutingBgpConstants import evpnNexthopSelfSupportNlris
import ConfigMount
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from RouteMapLib import getLbwCommValue
import Tac
from TypeFuture import TacLazyType

bgpVrfConfigDir = None
configTagConfig = None
configTagInput = None

RoutingBgpCliHandler = CliDynamicPlugin( "RoutingBgpCliHandler" )
delNeighborConfigIfDefault = RoutingBgpCliHandler.delNeighborConfigIfDefault
ExtendedNextHopCapabilityEnum = TacLazyType(
   "Routing::Bgp::ExtendedNextHopCapability" )
GrHelperRestartTimeClassEnum = TacLazyType(
   "Routing::Bgp::GrHelperRestartTimeClass" )
LinkBandwidthGenerationStateEnum = TacLazyType(
   'Routing::Bgp::LinkBandwidthGenerationState' )
OutDelayApplyEnum = TacLazyType( "Routing::Bgp::OutDelayApplyType" )
RRClientEnum = TacLazyType( "Routing::Bgp::ReflectorClient" )
SendCommunityLinkBandwidthStateEnum = TacLazyType(
   'Routing::Bgp::SendCommunityLinkBandwidthState' )
SoftReconfigInboundStateEnum = TacLazyType(
   "Routing::Bgp::SoftReconfigInboundState" )
RemoveOrDisassociate = TacLazyType( "ConfigTag::RemoveOrDisassociate" )

def isPrivateAsn( asn ):
   """
   Checks whether the AS number is private per RFC 6996.
   There are two ranges for this:
      64512 - 65534 inclusive
      4200000000 - 4294967294 inclusive
   """
   return ( ( 64512 <= asn <= 65534 ) or
            ( 4200000000 <= asn <= 4294967294 ) )

def isDynamicPeerGroupByName( pgName ):
   if pgName in Globals.bgpConfig.listenRange:
      return True
   for v in bgpVrfConfigDir.vrfConfig:
      config = bgpVrfConfigDir.vrfConfig[ v ]
      if pgName in config.listenRange:
         return True
   return False

def isNeighIntfConfigPeerGroup( pgName ):
   """
   Checks whether the given peer-group is used as part of a
   'neighbor interface' config command.
   """
   for intf in Globals.bgpConfig.neighIntfConfig:
      peerConfig = Globals.bgpConfig.neighIntfConfig[ intf ]
      if peerConfig.peerGroup.stringValue == pgName:
         return True

   for v in bgpVrfConfigDir.vrfConfig:
      config = bgpVrfConfigDir.vrfConfig[ v ]
      for intf in config.neighIntfConfig:
         peerConfig = config.neighIntfConfig[ intf ]
         if peerConfig.peerGroup.stringValue == pgName:
            return True
   return False

def getModuleFunction( functionName ):
   # Returns a function by name from the current module (file).
   # Throws if function does not exist.
   function = globals()[ functionName ]
   if not callable( function ):
      raise NotImplementedError
   return function

# RoutingBgpCli.BgpCmdBaseClass
def BgpCmdBaseClass_handler( commandName, mode, args ):
   BgpCmdBaseClass.callHandler(
      getModuleFunction( commandName + "_handleNormal" ),
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

def BgpCmdBaseClass_noOrDefaultHandler( commandName, mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

# RoutingBgpCli.BgpNEDCmdBaseClass
def BgpNEDCmdBaseClass_handler( commandName, mode, args ):
   BgpNEDCmdBaseClass.callHandler(
      getModuleFunction( commandName + "_handleNormal" ),
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

def BgpNEDCmdBaseClass_noOrDefaultHandler( commandName, mode, args ):
   BgpNEDCmdBaseClass.callNoOrDefaultHandler(
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

# class SetNeighborMaxAcceptedRoutesCmd( BgpNEDCmdBaseClass ):
def SetNeighborMaxAcceptedRoutesCmd_setNeighborMaxAcceptedRoutes(
      mode, peer, numAcceptedRoutes, threshold ):
   addrFamily = mode.addrFamily
   if ( getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent and
         addrFamily != 'all' ):
      msg = ( 'Address family maximum accepted routes is only supported in '
               'multi-agent mode.' )
      if mode.session.isInteractive():
         mode.addError( msg )
         return

      mode.addWarning( msg )

   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'maxAcceptedRoutesConfig' ].get(
      addrFamily )
   if not threshold:
      defaultAttr = attr + 'Default' if attr else 'maxAcceptedRoutesConfigDefault'
      threshold = getattr( config, defaultAttr ).maxAcceptedRoutesThreshold
   # As MIO commits are batched and asynchronous, the order of these assignments
   # is critical. That is, we want threshold to be set before or with
   # numAcceptedRoutes (not after), and so the configuration must be mutated in
   # this order.
   maxAcceptedRoutesConfig = Tac.Value( 'Routing::Bgp::MaxAcceptedRoutesConfig',
                                          numAcceptedRoutes, threshold )
   if attr is not None:
      setattr( config, attr, maxAcceptedRoutesConfig )
      setattr( config, attr + 'Present', True )
   else:
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                           *BgpLib.addressFamilies.get( addrFamily ) )
      config.maxAcceptedRoutesAfiSafiConfig[ afiSafi ] = maxAcceptedRoutesConfig
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMaxAcceptedRoutesCmd_noNeighborMaxAcceptedRoutes(
      mode, peer, noOrDefault ):
   addrFamily = mode.addrFamily
   if ( getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent and
         addrFamily != 'all' ):
      msg = ( 'Address family maximum accepted routes is only supported in '
               'multi-agent mode.' )
      if mode.session.isInteractive():
         mode.addError( msg )
         return

      mode.addWarning( msg )

   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      attr = BgpLib.peerConfigAttrsAfMap[ 'maxAcceptedRoutesConfig' ].get(
         addrFamily )
      if attr is not None:
         setattr( config, attr + 'Present', noOrDefault == NoOrDefault.NO and
                  config.isPeerGroupPeer )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         if addrFamily == 'all':
            config.maxAcceptedRoutesAfiSafiConfig.clear()
      else:
         afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                              *BgpLib.addressFamilies.get( addrFamily ) )
         del config.maxAcceptedRoutesAfiSafiConfig[ afiSafi ]
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMaxAcceptedRoutesCmd_handleNormal( mode, args ):
   SetNeighborMaxAcceptedRoutesCmd_setNeighborMaxAcceptedRoutes(
      mode,
      args[ 'PEER' ],
      args[ 'NUM_ACCEPTED_ROUTES' ],
      args.get( 'THRESHOLD' ) )

def SetNeighborMaxAcceptedRoutesCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborMaxAcceptedRoutesCmd_noNeighborMaxAcceptedRoutes(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborMaxAcceptedRoutesCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborMaxAcceptedRoutesCmd", mode, args )

def SetNeighborMaxAcceptedRoutesCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborMaxAcceptedRoutesCmd", mode, args )

# class SetNeighborMaxAdvRoutesCmd( BgpCmdBaseClass ):
def SetNeighborMaxAdvRoutesCmd_setNeighborMaxAdvRoutes(
      mode, peer, numAdvRoutes, threshold, percent ):
   addrFamily = mode.addrFamily
   if ( getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent and
         addrFamily != 'all' ):
      msg = ( 'Address family maximum advertised routes is only supported in'
               ' multi-agent mode.' )
      if mode.session.isInteractive():
         mode.addError( msg )
         return

      mode.addWarning( msg )

   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = 'maxAdvRoutesConfig'
   if not threshold:
      threshold = getattr( config, attr + 'Default' ).maxAdvRoutesThreshold
   percentagePresent = percent is not None
   if percent:
      percentage = threshold
      threshold = int( ( numAdvRoutes * threshold ) / 100 )
   else:
      percentage = getattr(
         config, attr + 'Default' ).maxAdvRoutesThresholdPercentage
   maxAdvRoutesConfig = Tac.Value( 'Routing::Bgp::MaxAdvRoutesConfig',
                                    numAdvRoutes, threshold,
                                    percentage, percentagePresent )
   if addrFamily == 'all':
      setattr( config, attr, maxAdvRoutesConfig )
      setattr( config, attr + 'Present', True )
   else:
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                           *BgpLib.addressFamilies.get( addrFamily ) )
      config.maxAdvRoutesAfiSafiConfig[ afiSafi ] = maxAdvRoutesConfig
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMaxAdvRoutesCmd_noNeighborMaxAdvRoutes( mode, peer, noOrDefault ):
   addrFamily = mode.addrFamily
   if ( getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent and
         addrFamily != 'all' ):
      msg = ( 'Address family maximum advertised routes is only supported in'
               ' multi-agent mode.' )
      if mode.session.isInteractive():
         mode.addError( msg )
         return

      mode.addWarning( msg )

   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      if addrFamily == 'all':
         attr = 'maxAdvRoutesConfig'
         setattr( config, attr + 'Present', False )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
      else:
         afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                              *BgpLib.addressFamilies.get( addrFamily ) )
         del config.maxAdvRoutesAfiSafiConfig[ afiSafi ]
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMaxAdvRoutesCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborMaxAdvRoutesCmd_noNeighborMaxAdvRoutes(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborMaxAdvRoutesCmd_handleNormal( mode, args ):
   SetNeighborMaxAdvRoutesCmd_setNeighborMaxAdvRoutes(
      mode,
      args[ 'PEER' ],
      args[ 'NUM_ADV_ROUTES' ],
      args.get( 'THRESHOLD' ),
      args.get( 'percent' ) )

def SetNeighborMaxAdvRoutesCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborMaxAdvRoutesCmd", mode, args )

def SetNeighborMaxAdvRoutesCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborMaxAdvRoutesCmd", mode, args )

# class SetNeighborMissingPolicyCmd( BgpCmdBaseClass ):
def SetNeighborMissingPolicyCmd_handleNormal( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                        "configured for missing-policy neighbor configuration" )
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   mpCfg = MissingPolicyConfig( args )
   actionType = mpCfg.actionType
   setattr( config, mpCfg.actionAttr, actionType )
   setattr( config, mpCfg.addrFamilyAllAttr,
            'isTrue' if mpCfg.addressFamilyAll else 'isInvalid' )
   setattr( config, mpCfg.includeSubAttr,
            'isTrue' if mpCfg.includeSubRm else 'isInvalid' )
   setattr( config, mpCfg.includePfxListAttr,
            'isTrue' if mpCfg.includePfxList else 'isInvalid' )
   setattr( config, mpCfg.includeCommListAttr,
            'isTrue' if mpCfg.includeCommList else 'isInvalid' )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMissingPolicyCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      mpCfg = MissingPolicyConfig( args )
      for attr in [
            mpCfg.actionAttr, mpCfg.addrFamilyAllAttr, mpCfg.includeSubAttr,
            mpCfg.includePfxListAttr, mpCfg.includeCommListAttr ]:
         setattr( config, attr, ( config.missingPolicyActionInvalid
                                    if attr == mpCfg.actionAttr else 'isInvalid' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMissingPolicyCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborMissingPolicyCmd", mode, args )

def SetNeighborMissingPolicyCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborMissingPolicyCmd", mode, args )

# class SetNeighborMissingPolicyAfCmd( BgpCmdBaseClass ):
def SetNeighborMissingPolicyAfCmd_handleNormal( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                        "configured for missing-policy neighbor configuration" )
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   mpCfg = MissingPolicyConfig( args, af=True )
   actionAfConfig = getattr( config, mpCfg.actionAttr )
   includeSubAfConfig = getattr( config, mpCfg.includeSubAttr )
   includePfxAfConfig = getattr( config, mpCfg.includePfxListAttr )
   includeCommAfConfig = getattr( config, mpCfg.includeCommListAttr )
   af = Tac.Value( "Routing::Bgp::AfiSafi", *addressFamilies[ mode.addrFamily ] )
   actionAfConfig[ af ] = mpCfg.actionType
   includeSubAfConfig[ af ] = 'isTrue' if mpCfg.includeSubRm else 'isFalse'
   includePfxAfConfig[ af ] = 'isTrue' if mpCfg.includePfxList else 'isFalse'
   includeCommAfConfig[ af ] = 'isTrue' if mpCfg.includeCommList else 'isFalse'
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMissingPolicyAfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if config:
      mpCfg = MissingPolicyConfig( args, af=True )
      actionAfConfig = getattr( config, mpCfg.actionAttr )
      includeSubAfConfig = getattr( config, mpCfg.includeSubAttr )
      includePfxAfConfig = getattr( config, mpCfg.includePfxListAttr )
      includeCommAfConfig = getattr( config, mpCfg.includeCommListAttr )
      af = Tac.Value( "Routing::Bgp::AfiSafi",
                        *addressFamilies[ mode.addrFamily ] )
      del actionAfConfig[ af ]
      del includeSubAfConfig[ af ]
      del includePfxAfConfig[ af ]
      del includeCommAfConfig[ af ]
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMissingPolicyAfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborMissingPolicyAfCmd", mode, args )

def SetNeighborMissingPolicyAfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborMissingPolicyAfCmd", mode, args )

# class SetNeighborLocalAsCmd( BgpNEDCmdBaseClass ):
def SetNeighborLocalAsCmd_setLocalAs( mode, peer, asNumber, fallbackAs=False ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.localAs = Tac.Value( 'Routing::Bgp::LocalAs', asNumber,
                                 fallbackAs )
   config.localAsPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLocalAsCmd_noLocalAs( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.localAsPresent = ( noOrDefault == NoOrDefault.NO and
                                 config.isPeerGroupPeer )
      config.localAs = config.localAsDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLocalAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborLocalAsCmd_noLocalAs(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborLocalAsCmd_handleNormal( mode, args ):
   SetNeighborLocalAsCmd_setLocalAs(
      mode,
      args[ 'PEER' ],
      args[ 'AS_NUM' ],
      'fallback' in args )

def SetNeighborLocalAsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLocalAsCmd", mode, args )

def SetNeighborLocalAsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLocalAsCmd", mode, args )

# class RouterBgpAfActivateCommand( CliCommand.CliCommandClass ):
def RouterBgpAfActivateCommand_activateNeighbor( mode, peer ):
   validatePeer( mode, peer )
   neighborAfActivateHelper( mode, peer, Globals.PeerAfStateEnum.afActive )

def RouterBgpAfActivateCommand_noActivateNeighbor( mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   if noOrDefault != NoOrDefault.DEFAULT:
      neighborAfActivateHelper( mode, peer, Globals.PeerAfStateEnum.afInactive )
   else:
      neighborAfActivateHelper( mode, peer, Globals.PeerAfStateEnum.afDefault )

def RouterBgpAfActivateCommand_handler( mode, args ):
   peer = args.get( 'PEER' )
   if 'deactivate' in args:
      RouterBgpAfActivateCommand_noActivateNeighbor(
         mode,
         peer,
         NoOrDefault.NO )
   else:
      RouterBgpAfActivateCommand_activateNeighbor( mode, peer )

def RouterBgpAfActivateCommand_noHandler( mode, args ):
   noOrDefault = NoOrDefault.DEFAULT if Globals.bgpConfig.noEqualsDefaultMode \
      else NoOrDefault.NO
   RouterBgpAfActivateCommand_noActivateNeighbor(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def RouterBgpAfActivateCommand_defaultHandler( mode, args ):
   RouterBgpAfActivateCommand_noActivateNeighbor(
      mode,
      args[ 'PEER' ],
      NoOrDefault.DEFAULT )

# class NeighborReflectorClientOrrPositionCmd( BgpNEDCmdBaseClass ):
def NeighborReflectorClientOrrPositionCmd_orrHelper(
      mode, peer, config, present, value ):
   validatePeer( mode, peer )
   attr = BgpLib.peerConfigAttrsAfMap[ 'orr' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, value )
      setattr( config, attr + 'Present', present )
   else:
      mode.addError( "Invalid address family %s" % peer )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborReflectorClientOrrPositionCmd_handleNormal( mode, args ):
   emitWarningIfOrrNotSupportedInRibdMode( mode )
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if 'peer-address' in args:
      value = Tac.Value( 'Routing::Bgp::PeerOrrState', '', True )
   else:
      positionName = args.get( 'POSITION_NAME' )
      if not positionName:
         mode.addError( 'Invalid position name' )
         return
      value = Tac.Value( 'Routing::Bgp::PeerOrrState', positionName, False )
   assert value
   # Sanity check that this CLI is only enabled for AfiSafis that supports ORR.
   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                        *BgpLib.addressFamilies.get( mode.addrFamily ) )
   assert afiSafi.orrSupported()
   NeighborReflectorClientOrrPositionCmd_orrHelper( mode, peer, config, True, value )

def NeighborReflectorClientOrrPositionCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if not config:
      return
   present = ( noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
   NeighborReflectorClientOrrPositionCmd_orrHelper(
      mode, peer, config, present, config.orrCommonDefault )

def NeighborReflectorClientOrrPositionCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborReflectorClientOrrPositionCmd", mode, args )

def NeighborReflectorClientOrrPositionCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "NeighborReflectorClientOrrPositionCmd", mode, args )

# class NeighborFwdFailoverTriggerSessionCmd( BgpCmdBaseClass ):
def NeighborFwdFailoverTriggerSessionCmd_handleNormal( mode, args ):
   addrFamily = mode.addrFamily
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "forwarding failover is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'fwdFailoverTriggerSession' ].\
      get( addrFamily )
   setattr( config, attr + "Present", True )
   setattr( config, attr, True )

def NeighborFwdFailoverTriggerSessionCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   addrFamily = mode.addrFamily
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if noOrDefault == NoOrDefault.DEFAULT:
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         attr = BgpLib.peerConfigAttrsAfMap[ 'fwdFailoverTriggerSession' ].\
            get( addrFamily )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         setattr( config, attr + 'Present', False )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )
   else:
      # Explicitly "disabled"
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ 'fwdFailoverTriggerSession' ].\
         get( addrFamily )
      setattr( config, attr + "Present", True )
      setattr( config, attr, False )

def NeighborFwdFailoverTriggerSessionCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborFwdFailoverTriggerSessionCmd", mode, args )

def NeighborFwdFailoverTriggerSessionCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "NeighborFwdFailoverTriggerSessionCmd", mode, args )

# class RouterBgpNeighborActivate6peCommand( CliCommand.CliCommandClass ):
def RouterBgpNeighborActivate6peCommand_enableV6Unicast( mode, peer ):
   neighborAfActivateHelper( mode, peer, Globals.PeerAfStateEnum.afActive )

def RouterBgpNeighborActivate6peCommand_disableV6Unicast( mode, peer, noOrDefault ):
   if noOrDefault != NoOrDefault.DEFAULT:
      neighborAfActivateHelper( mode, peer, Globals.PeerAfStateEnum.afInactive )
   else:
      neighborAfActivateHelper( mode, peer, Globals.PeerAfStateEnum.afDefault )

def RouterBgpNeighborActivate6peCommand_enableSixPe( mode, peer ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.sixPe = True
   config.sixPePresent = True

def RouterBgpNeighborActivate6peCommand_disableSixPe( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.sixPePresent = ( noOrDefault != NoOrDefault.DEFAULT )
   config.sixPe = config.sixPeDefault

def RouterBgpNeighborActivate6peCommand_activateNeighbor(
      mode, peer, sixPe=False, ipv6Uni=False ):
   validatePeer( mode, peer )

   if ipv6Uni:
      # enables 6pe and v6 unicast
      RouterBgpNeighborActivate6peCommand_enableSixPe( mode, peer )
      RouterBgpNeighborActivate6peCommand_enableV6Unicast( mode, peer )
   elif sixPe:
      # enables only 6pe and disables v6 unicast
      RouterBgpNeighborActivate6peCommand_enableSixPe( mode, peer )
      RouterBgpNeighborActivate6peCommand_disableV6Unicast(
         mode,
         peer,
         NoOrDefault.NO )
   else:
      # enables only v6 unicast and disables 6pe
      RouterBgpNeighborActivate6peCommand_disableSixPe( mode, peer, NoOrDefault.NO )
      RouterBgpNeighborActivate6peCommand_enableV6Unicast( mode, peer )

def RouterBgpNeighborActivate6peCommand_noActivateNeighbor(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   RouterBgpNeighborActivate6peCommand_disableSixPe( mode, peer, noOrDefault )
   RouterBgpNeighborActivate6peCommand_disableV6Unicast( mode, peer, noOrDefault )

def RouterBgpNeighborActivate6peCommand_handler( mode, args ):
   peer = args.get( 'PEER' )
   if 'deactivate' in args:
      RouterBgpNeighborActivate6peCommand_noActivateNeighbor(
         mode,
         peer,
         NoOrDefault.NO )
   else:
      RouterBgpNeighborActivate6peCommand_activateNeighbor(
         mode,
         peer,
         sixPe=( '6pe' in args ),
         ipv6Uni=( 'ipv6-unicast' in args ) )

def RouterBgpNeighborActivate6peCommand_noHandler( mode, args ):
   noOrDefault = NoOrDefault.DEFAULT if Globals.bgpConfig.noEqualsDefaultMode \
      else NoOrDefault.NO
   RouterBgpNeighborActivate6peCommand_noActivateNeighbor(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def RouterBgpNeighborActivate6peCommand_defaultHandler( mode, args ):
   RouterBgpNeighborActivate6peCommand_noActivateNeighbor(
      mode,
      args[ 'PEER' ],
      NoOrDefault.DEFAULT )

# class RouterBgpNeighborActivateEvpnCommand( CliCommand.CliCommandClass ):
def RouterBgpNeighborActivateEvpnCommand_setActivateRouteType(
      mode, peer, routeType ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if routeType == 'layer-2':
      config.afEvpnActivateMacIpRouteType = True
      config.afEvpnActivateMacIpRouteTypePresent = True
   elif routeType == 'layer-3':
      config.afEvpnActivateIpPrefixRouteType = True
      config.afEvpnActivateIpPrefixRouteTypePresent = True
   else:
      mode.addError( "Invalid route type %s" % routeType )

def RouterBgpNeighborActivateEvpnCommand_noActivateRouteType(
      mode, peer, noOrDefault, routeType ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if config:
      if routeType == 'layer-2':
         if noOrDefault == NoOrDefault.DEFAULT:
            config.afEvpnActivateMacIpRouteType = (
                  config.afEvpnActivateMacIpRouteTypeDefault )
            config.afEvpnActivateMacIpRouteTypePresent = False
         else:
            config.afEvpnActivateMacIpRouteType = False
            config.afEvpnActivateMacIpRouteTypePresent = True
      elif routeType == 'layer-3':
         if noOrDefault == NoOrDefault.DEFAULT:
            config.afEvpnActivateIpPrefixRouteType = (
                  config.afEvpnActivateIpPrefixRouteTypeDefault )
            config.afEvpnActivateIpPrefixRouteTypePresent = False
         else:
            config.afEvpnActivateIpPrefixRouteType = False
            config.afEvpnActivateIpPrefixRouteTypePresent = True
      else:
         mode.addError( "Invalid route type %s" % routeType )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def RouterBgpNeighborActivateEvpnCommand_handler( mode, args ):
   RouterBgpNeighborActivateEvpnCommand_setActivateRouteType(
         mode,
         args.get( 'PEER' ),
         args.get( 'routeType' ) )

def RouterBgpNeighborActivateEvpnCommand_defaultHandler( mode, args ):
   RouterBgpNeighborActivateEvpnCommand_noActivateRouteType(
         mode,
         args.get( 'PEER' ),
         NoOrDefault.DEFAULT,
         args.get( 'routeType' ) )

def RouterBgpNeighborActivateEvpnCommand_noHandler( mode, args ):
   RouterBgpNeighborActivateEvpnCommand_noActivateRouteType(
         mode,
         args.get( 'PEER' ),
         NoOrDefault.NO,
         args.get( 'routeType' ) )

# class RouterBgpNeighborDomainRemoteEvpnCommand( BgpCmdBaseClass ):
def RouterBgpNeighborDomainRemoteEvpnCommand_setDomainRemote( mode, peer ):
   peerConfig = bgpNeighborConfig( peer, mode.vrfName )
   peerConfig.afEvpnRemoteDomainPresent = True
   peerConfig.afEvpnRemoteDomain = True

def RouterBgpNeighborDomainRemoteEvpnCommand_noDomainRemote( mode, peer ):
   peerConfig = bgpNeighborConfig( peer, mode.vrfName, create=False )
   if peerConfig:
      peerConfig.afEvpnRemoteDomain = False
      peerConfig.afEvpnRemoteDomainPresent = False

def RouterBgpNeighborDomainRemoteEvpnCommand_handleNormal( mode, args ):
   RouterBgpNeighborDomainRemoteEvpnCommand_setDomainRemote(
      mode,
      peer=args[ 'PEER' ] )

def RouterBgpNeighborDomainRemoteEvpnCommand_handleNoOrDefault(
      mode, args, noOrDefault ):
   RouterBgpNeighborDomainRemoteEvpnCommand_noDomainRemote(
      mode,
      peer=args[ 'PEER' ] )

def RouterBgpNeighborDomainRemoteEvpnCommand_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpNeighborDomainRemoteEvpnCommand", mode, args )

def RouterBgpNeighborDomainRemoteEvpnCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RouterBgpNeighborDomainRemoteEvpnCommand", mode, args )

# class RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd( BgpCmdBaseClass ):
def RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_setNeighborNexthopSelfEvpn(
      mode, peer, routeTypes, interDomain=False ):
   peerConfig = bgpNeighborConfig( peer, mode.vrfName )
   for rtype in routeTypes:
      if rtype in evpnNexthopSelfSupportNlris:
         attr = 'afEvpnNexthopSelfType%d' % rtype
         if interDomain:
            attr += 'InterDomain'
         setattr( peerConfig, attr + 'Present', True )
         setattr( peerConfig, attr, True )

def RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_noNeighborNexthopSelfEvpn(
      mode, peer, routeTypes, noOrDefault, interDomain=False ):
   peerConfig = bgpNeighborConfig( peer, mode.vrfName )
   if peerConfig:
      for rtype in routeTypes:
         if rtype in evpnNexthopSelfSupportNlris:
            attr = 'afEvpnNexthopSelfType%d' % rtype
            if interDomain:
               attr += 'InterDomain'
            val = getattr( peerConfig, attr + 'Default' )
            setattr( peerConfig, attr, val )
            setattr( peerConfig, attr + 'Present', False )

def RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_noNeighborNexthopSelfEvpn(
      mode,
      peer=args[ 'PEER' ],
      routeTypes=args[ 'ROUTE_TYPES' ],
      noOrDefault=noOrDefault,
      interDomain=args.get( 'inter-domain' ) )

def RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_handleNormal( mode, args ):
   RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_setNeighborNexthopSelfEvpn(
      mode,
      peer=args[ 'PEER' ],
      routeTypes=args[ 'ROUTE_TYPES' ],
      interDomain=args.get( 'inter-domain' ) )

def RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd", mode, args )

def RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd", mode, args )

# class SetNeighborPeerTimersCmd( BgpNEDCmdBaseClass ):
def SetNeighborPeerTimersCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   hold = args.get( 'HOLD_TIME', 0 )
   keepalive = args.get( 'KEEPALIVE' )

   if keepalive is not None:
      if hold < keepalive:
         mode.addError( 'keepalive time cannot be greater than hold time' )
         return
      if hold != 0 and keepalive == 0:
         mode.addError(
            'keepalive time cannot be zero when hold time is not zero' )
         return

   minHoldTime = args.get( 'MIN_HOLD_TIME' )
   if keepalive is not None and hold is not None and hold != 0 and \
      minHoldTime is not None and hold < minHoldTime:
      mode.addError( 'hold time cannot be less than minimum hold time' )
      return

   sendFailureHoldTime = args.get( 'SEND_FAILURE_HOLD_TIME' )

   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )

   if keepalive is not None:
      config.keepaliveTime = keepalive
      config.holdTime = hold
      config.timersPresent = True
   else:
      config.keepaliveTime = config.keepaliveTimeDefault
      config.holdTime = config.holdTimeDefault
      config.timersPresent = False

   if minHoldTime is not None:
      config.minHoldTime = minHoldTime
      config.minHoldTimePresent = True
   else:
      config.minHoldTime = config.minHoldTimeDefault
      config.minHoldTimePresent = False

   if sendFailureHoldTime is not None:
      config.sendFailureHoldTime = sendFailureHoldTime
      config.sendFailureHoldTimePresent = True
   else:
      config.sendFailureHoldTime = config.sendFailureHoldTimeDefault
      config.sendFailureHoldTimePresent = False

   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborPeerTimersCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.timersPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                 config.isPeerGroupPeer )

      config.minHoldTimePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                    config.isPeerGroupPeer )
      config.sendFailureHoldTimePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                             config.isPeerGroupPeer )
      config.minHoldTime = config.minHoldTimeDefault
      config.sendFailureHoldTime = config.sendFailureHoldTimeDefault
      config.keepaliveTime = config.keepaliveTimeDefault
      config.holdTime = config.holdTimeDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborPeerTimersCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborPeerTimersCmd", mode, args )

def SetNeighborPeerTimersCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborPeerTimersCmd", mode, args )

# class SetIdleRestartTimerCmd( BgpNEDCmdBaseClass ):
def SetIdleRestartTimerCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   irtime = args[ 'IDLE_RESTART_TIMER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.idleHoldTime = irtime
   config.idleHoldTimePresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetIdleRestartTimerCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.idleHoldTimePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                       config.isPeerGroupPeer )
      config.idleHoldTime = config.idleHoldTimeDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetIdleRestartTimerCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetIdleRestartTimerCmd", mode, args )

def SetIdleRestartTimerCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetIdleRestartTimerCmd", mode, args )

# class SetNeighborAsCmd( BgpNEDCmdBaseClass ):
def SetNeighborAsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   asNumber = args[ 'AS_NUM' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.asNumber = asNumber
   config.asNumberPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.asNumberPresent = ( noOrDefault == NoOrDefault.NO and
                                 config.isPeerGroupPeer )
      config.asNumber = config.asNumberDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborAsCmd", mode, args )

def SetNeighborAsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborAsCmd", mode, args )

# class SetNeighborRouteMapBaseCmd( BgpNEDCmdBaseClass ):
# class SetNeighborRouteMapCmd( SetNeighborRouteMapBaseCmd ):
# class SetNeighborRouteMapSrTeCmd( SetNeighborRouteMapBaseCmd ):
def SetNeighborRouteMapBaseCmd_setNeighborRouteMap( mode, peer, mapName, inOut ):
   if ( mode.addrFamily == 'ipv6 labeled-unicast' and
         getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd ):
      mode.addError( "IPv6 neighbor route-maps is only supported in "
                        "multi-agent labeled-unicast mode" )
   elif ( inOut == 'out' and mode.addrFamily in [ 'ipv4 sr-te', 'ipv6 sr-te' ] and
            getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent ):
      mode.addError( "SR-TE neighbor outbound route-map is only supported in "
                     "multi-agent mode" )
   else:
      attr = 'routeMapIn' if inOut == 'in' else 'routeMapOut'
      config = bgpNeighborConfig( peer, mode.vrfName )
      if haveConflictingRouteMaps( mode, config, attr, mapName, mode.addrFamily ):
         return
      af = mode.addrFamily
      attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
      setattr( config, attr, str( mapName ) )
      setattr( config, attr + 'Present', True )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborRouteMapBaseCmd_noNeighborRouteMap( mode, peer, inOut, noOrDefault ):
   attr = 'routeMapIn' if inOut == 'in' else 'routeMapOut'
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
   if config:
      af = mode.addrFamily
      attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
      if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
         setattr( config, attr + 'Present', True )
      else:
         setattr( config, attr + 'Present', False )
      setattr( config, attr, getattr( config, attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborRouteMapBaseCmd_handleNormal( mode, args ):
   SetNeighborRouteMapBaseCmd_setNeighborRouteMap(
      mode,
      args[ 'PEER' ],
      args[ 'MAP_NAME' ],
      args[ 'inout' ] )

def SetNeighborRouteMapBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborRouteMapBaseCmd_noNeighborRouteMap(
      mode,
      args[ 'PEER' ],
      args[ 'inout' ],
      noOrDefault )

def SetNeighborRouteMapBaseCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborRouteMapBaseCmd", mode, args )

def SetNeighborRouteMapBaseCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborRouteMapBaseCmd", mode, args )

# class SetNeighborPrefixListCmd( BgpNEDCmdBaseClass ):
def SetNeighborPrefixListCmd_setPrefixList( mode, peer, plName, attrName ):
   addrFamily = 'ipv4' if mode.addrFamily == 'all' else mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ attrName ].get( addrFamily )
   setattr( config, attr, str( plName ) )
   setattr( config, attr + 'Present', True )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborPrefixListCmd_noPrefixList( mode, peer, attrName, noOrDefault ):
   addrFamily = 'ipv4' if mode.addrFamily == 'all' else mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
   if config:
      attr = BgpLib.peerConfigAttrsAfMap[ attrName ].get( addrFamily )
      attrVal = noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
      setattr( config, attr + 'Present', attrVal )
      setattr( config, attr, getattr( config,
                                       attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborPrefixListCmd_handleNormal( mode, args ):
   SetNeighborPrefixListCmd_setPrefixList(
         mode,
         args.get( 'PEER' ),
         args.get( 'PREFIX_LIST_NAME' ),
         args.get( 'inout' ) )

def SetNeighborPrefixListCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborPrefixListCmd_noPrefixList(
         mode,
         args.get( 'PEER' ),
         args.get( 'inout' ),
         noOrDefault )

def SetNeighborPrefixListCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborPrefixListCmd", mode, args )

def SetNeighborPrefixListCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborPrefixListCmd", mode, args )

# class SetUnsupportedNeighborPrefixListCmd( BgpNEDCmdBaseClass ):
def SetUnsupportedNeighborPrefixListCmd_setPrefixList(
      mode, peer, plName, attrName ):
   if mode.session.isInteractive():
      mode.addWarning( 'neighbor prefix-list unsupported in ' +
                        mode.addrFamily + ' address family' )
      return
   else:
      mode.addWarning( 'neighbor prefix-list unsupported in ' + mode.addrFamily
                        + ' address family, applying it to ipv4 address family' )
      addrFamily = 'ipv4'

   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ attrName ].get( addrFamily )
   setattr( config, attr, str( plName ) )
   setattr( config, attr + 'Present', True )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetUnsupportedNeighborPrefixListCmd_noPrefixList(
      mode, peer, attrName, noOrDefault ):
   if mode.session.isInteractive():
      mode.addWarning( 'neighbor prefix-list unsupported in ' +
                        mode.addrFamily + ' address family' )
      return
   else:
      mode.addWarning( 'neighbor prefix-list unsupported in ' + mode.addrFamily
                        + ' address family, applying it to ipv4 address family' )
      addrFamily = 'ipv4'

   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
   if config:
      attr = BgpLib.peerConfigAttrsAfMap[ attrName ].get( addrFamily )
      attrVal = noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
      setattr( config, attr + 'Present', attrVal )
      setattr( config, attr, getattr( config,
                                       attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetUnsupportedNeighborPrefixListCmd_handleNormal( mode, args ):
   SetUnsupportedNeighborPrefixListCmd_setPrefixList(
      mode,
      args.get( 'PEER' ),
      args.get( 'PREFIX_LIST_NAME' ),
      args.get( 'inout' ) )

def SetUnsupportedNeighborPrefixListCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetUnsupportedNeighborPrefixListCmd_noPrefixList(
      mode,
      args.get( 'PEER' ),
      args.get( 'inout' ),
      noOrDefault )

def SetUnsupportedNeighborPrefixListCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetUnsupportedNeighborPrefixListCmd", mode, args )

def SetUnsupportedNeighborPrefixListCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetUnsupportedNeighborPrefixListCmd", mode, args )

# class SetNeighborImportLocalPrefCmd( BgpNEDCmdBaseClass ):
def SetNeighborImportLocalPrefCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   localpref = args[ 'LOCAL_PREF' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.importLocalPref = localpref
   config.importLocalPrefPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborImportLocalPrefCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.importLocalPrefPresent = ( noOrDefault == NoOrDefault.NO and
                                        config.isPeerGroupPeer )
      config.importLocalPref = config.importLocalPrefDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborImportLocalPrefCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborImportLocalPrefCmd", mode, args )

def SetNeighborImportLocalPrefCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborImportLocalPrefCmd", mode, args )

# class SetNeighborExportLocalPrefCmd( BgpNEDCmdBaseClass ):
def SetNeighborExportLocalPrefCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   localpref = args[ 'LOCAL_PREF' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.exportLocalPref = localpref
   config.exportLocalPrefPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborExportLocalPrefCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.exportLocalPrefPresent = ( noOrDefault == NoOrDefault.NO and
                                        config.isPeerGroupPeer )
      config.exportLocalPref = config.exportLocalPrefDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborExportLocalPrefCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborExportLocalPrefCmd", mode, args )

def SetNeighborExportLocalPrefCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborExportLocalPrefCmd", mode, args )

# class SetNeighborGracefulRestartCmd( BgpNEDCmdBaseClass ):
def SetNeighborGracefulRestartCmd_configureNeighborGracefulRestart(
      mode, config, enable, noOrDefault, addrFamily ):
   if addrFamily == 'ipv4 multicast':
      mode.addError( "Graceful restart is currently not supported "
                     "for IPv4 multicast mode" )
      return
   if addrFamily == 'ipv6 multicast':
      mode.addError( "Graceful restart is currently not supported "
                     "for IPv6 multicast mode" )
      return

   gr = False
   grPresent = False
   if enable:
      gr = grPresent = True
   else:
      assert noOrDefault
      grPresent = noOrDefault != NoOrDefault.DEFAULT

   attr = BgpLib.bgpConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
   setattr( config, attr, gr )
   setattr( config, attr + 'Present', grPresent )

def SetNeighborGracefulRestartCmd_setNeighborGracefulRestart( mode, peer ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   SetNeighborGracefulRestartCmd_configureNeighborGracefulRestart(
      mode, config, True, None, mode.addrFamily )

def SetNeighborGracefulRestartCmd_noNeighborGracefulRestart(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                               vrfName=mode.vrfName )
   if config:
      SetNeighborGracefulRestartCmd_configureNeighborGracefulRestart(
         mode, config, False, noOrDefault, mode.addrFamily )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborGracefulRestartCmd_handleNormal( mode, args ):
   SetNeighborGracefulRestartCmd_setNeighborGracefulRestart(
      mode,
      args[ 'PEER' ] )

def SetNeighborGracefulRestartCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborGracefulRestartCmd_noNeighborGracefulRestart(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborGracefulRestartCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborGracefulRestartCmd", mode, args )

def SetNeighborGracefulRestartCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborGracefulRestartCmd", mode, args )

# class SetNeighborGrHelperCmd( BgpNEDCmdBaseClass ):
def SetNeighborGrHelperCmd_setNeighborGrHelper(
      mode, peer, extRestartTime, timeClass, llgrHelper, routeMap, grOptional ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = 'grHelperRouteMap'
   if ( routeMap and
         haveConflictingRouteMaps( mode, config, attr, routeMap,
                                   mode.addrFamily ) ):
      return
   config.grHelper = True
   config.grHelperPresent = True
   if extRestartTime:
      extRestartTime = getTimeInSeconds( extRestartTime, timeClass )
      config.grHelperRestartTime = extRestartTime
      config.grHelperRestartTimePresent = True
      config.grHelperRestartTimeClass = timeClass
   else:
      config.grHelperRestartTime = config.grHelperRestartTimeDefault
      config.grHelperRestartTimePresent = False
      config.grHelperRestartTimeClass = config.grHelperRestartTimeClassDefault
   if llgrHelper:
      config.llgrHelper = True
      config.llgrHelperPresent = True
   else:
      config.llgrHelper = config.llgrHelperDefault
      config.llgrHelperPresent = False
   if routeMap:
      config.grHelperRouteMap = routeMap
      config.grHelperRouteMapPresent = True
   else:
      config.grHelperRouteMap = config.grHelperRouteMapDefault
      config.grHelperRouteMapPresent = False
   if grOptional:
      config.grHelperRmGrOptional = True
      config.grHelperRmGrOptionalPresent = True
   else:
      config.grHelperRmGrOptional = config.grHelperRmGrOptionalDefault
      config.grHelperRmGrOptionalPresent = False

def SetNeighborGrHelperCmd_noNeighborGrHelper( mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                 vrfName=mode.vrfName )
   if config:
      if noOrDefault == NoOrDefault.DEFAULT:
         # GR-Helper is enabled by default
         config.grHelper = True
         config.grHelperPresent = False
         config.grHelperRestartTimePresent = False
         config.llgrHelperPresent = False
         config.grHelperRouteMapPresent = False
         config.grHelperRmGrOptionalPresent = False
      else:
         config.grHelper = False
         config.grHelperPresent = True
         config.grHelperRestartTimePresent = True
         config.llgrHelperPresent = True
         config.grHelperRouteMapPresent = True
         config.grHelperRmGrOptionalPresent = True
      config.grHelperRestartTime = config.grHelperRestartTimeDefault
      config.grHelperRestartTimeClass = config.grHelperRestartTimeClassDefault
      config.llgrHelper = config.llgrHelperDefault
      config.grHelperRouteMap = config.grHelperRouteMapDefault
      config.grHelperRmGrOptional = config.grHelperRmGrOptionalDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborGrHelperCmd_handleNormal( mode, args ):
   timeClass = getattr( GrHelperRestartTimeClassEnum,
         args.get( 'UNITS', 'seconds' ) )
   SetNeighborGrHelperCmd_setNeighborGrHelper( mode, args[ 'PEER' ],
                                               args.get( 'RESTART_TIME' ),
                                               timeClass,
                                               ( 'long-lived' in args ),
                                               args.get( 'MAP_NAME' ),
                                               ( 'optional' in args ) )

def SetNeighborGrHelperCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborGrHelperCmd_noNeighborGrHelper( mode, args[ 'PEER' ], noOrDefault )

def SetNeighborGrHelperCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborGrHelperCmd", mode, args )

def SetNeighborGrHelperCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborGrHelperCmd", mode, args )

# class SetNeighborGrHelperStaleRmAfCmd( BgpNEDCmdBaseClass ):
def SetNeighborGrHelperStaleRmAfCmd_setNeighborGrHelperStaleRm(
      mode, peer, routeMap, grOptional ):
   validatePeer( mode, peer )
   attr = 'grHelperRouteMap'
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   globalConfig = configForVrf( mode.vrfName )
   if haveConflictingRouteMaps( mode, config, attr, routeMap, mode.addrFamily ):
      return
   if globalConfig.grHelper == 'isFalse':
      mode.addWarning( "Graceful restart helper disabled globally for this vrf" )
   if not config.grHelper:
      mode.addWarning( "Graceful restart helper disabled for this peer" )
   af = mode.addrFamily
   attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
   setattr( config, attr, str( routeMap ) )
   setattr( config, attr + 'Present', True )
   attr = BgpLib.peerConfigAttrsAfMap[ 'grHelperRmGrOptional' ].get( af )
   setattr( config, attr, grOptional )
   setattr( config, attr + 'Present', grOptional )

def SetNeighborGrHelperStaleRmAfCmd_noNeighborGrHelperStaleRm(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      af = mode.addrFamily
      attr = BgpLib.peerConfigAttrsAfMap[ 'grHelperRouteMap' ].get( af )
      setattr( config, attr + 'Present', False )
      setattr( config, attr, getattr( config, attr + 'Default' ) )
      attr = BgpLib.peerConfigAttrsAfMap[ 'grHelperRmGrOptional' ].get( af )
      setattr( config, attr + 'Present', False )
      setattr( config, attr, getattr( config, attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborGrHelperStaleRmAfCmd_handleNormal( mode, args ):
   SetNeighborGrHelperStaleRmAfCmd_setNeighborGrHelperStaleRm(
      mode, args[ 'PEER' ], args.get( 'MAP_NAME' ), ( 'optional' in args ) )

def SetNeighborGrHelperStaleRmAfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborGrHelperStaleRmAfCmd_noNeighborGrHelperStaleRm(
      mode, args[ 'PEER' ], noOrDefault )

def SetNeighborGrHelperStaleRmAfCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborGrHelperStaleRmAfCmd", mode, args )

def SetNeighborGrHelperStaleRmAfCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborGrHelperStaleRmAfCmd", mode, args )

# class SetNeighborMetricOutCmd( BgpNEDCmdBaseClass ):
def SetNeighborMetricOutCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   metric = args[ 'METRIC' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.metricOut = metric
   config.metricOutState = Globals.MetricOutStateEnum.metricOutSet
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMetricOutCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
         config.metricOutState = Globals.MetricOutStateEnum.metricOutNone
      else:
         config.metricOutState = Globals.MetricOutStateEnum.metricNotConfigured
      config.metricOut = config.metricOutDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborMetricOutCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborMetricOutCmd", mode, args )

def SetNeighborMetricOutCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborMetricOutCmd", mode, args )

# class NeighborPasswordCommand( BgpNEDCmdBaseClass ):
def NeighborPasswordCommand_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   bgpTcpAuthKey = args.get( 'AUTH_KEY' )
   if bgpTcpAuthKey is None:
      # If the secret failed to decode, there is nothing to save
      return
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.password = bgpTcpAuthKey
   config.passwordPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborPasswordCommand_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.passwordPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                 config.isPeerGroupPeer )
      config.password = config.passwordDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborPasswordCommand_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborPasswordCommand", mode, args )

def NeighborPasswordCommand_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborPasswordCommand", mode, args )

# class SetNeighborMaxRoutesCmd( BgpNEDCmdBaseClass ):
def SetNeighborMaxRoutesCmd_setNeighborMaxRoutes(
      mode, peer, numRoutes, threshold=None, percent=None, action=False ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   if ( getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent and
        addrFamily != 'all' ):
      msg = ( 'Address family maximum routes is only supported in '
               'multi-agent mode.' )
      if mode.session.isInteractive():
         mode.addError( msg )
         return

      mode.addWarning( msg )

   config = bgpNeighborConfig( peer, vrfName=vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ].get( addrFamily )
   # Newer address-family support is via the maxRoutesAfiSafiConfig collection
   if attr is not None:
      maxRoutesConfig = getattr( config, attr )
      maxRestart = maxRoutesConfig.maxRoutesRestart
   else:
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                           *BgpLib.addressFamilies.get( addrFamily ) )
      prevConfig = config.maxRoutesAfiSafiConfig.get( afiSafi )
      maxRestart = prevConfig.maxRoutesRestart if prevConfig else 0

   if not threshold:
      threshold = 0
   percentagePresent = percent is not None
   if percent:
      percentage = threshold
      # The Tac model expects a integer, and Python division now returns a float.
      threshold = ( numRoutes * threshold ) // 100
   else:
      percentage = 100
   maxRoutesConfig = Tac.Value( 'Routing::Bgp::MaxRoutesConfig',
      numRoutes, threshold, maxRestart, action, percentage, percentagePresent )
   if attr is not None:
      setattr( config, attr, maxRoutesConfig )
      setattr( config, attr + 'Present', True )
   else:
      config.maxRoutesAfiSafiConfig[ afiSafi ] = maxRoutesConfig
   delNeighborConfigIfDefault( peer, vrfName=vrfName )

def SetNeighborMaxRoutesCmd_noNeighborMaxRoutes(
      mode, peer, noOrDefault, numRoutes=0,
      threshold=None, action=False, restart=None ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   if ( getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent and
         addrFamily != 'all' ):
      msg = ( 'Address family maximum routes is only supported in '
               'multi-agent mode.' )
      if mode.session.isInteractive():
         mode.addError( msg )
         return

      mode.addWarning( msg )

   config = bgpNeighborConfig( peer, vrfName=vrfName, create=False )
   if config:
      attr = BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ].get( addrFamily )
      if attr is not None:
         setattr( config, attr + 'Present',
                  noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         if addrFamily == 'all':
            config.maxRoutesAfiSafiConfig.clear()
      else:
         afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                              *BgpLib.addressFamilies.get( addrFamily ) )
         del config.maxRoutesAfiSafiConfig[ afiSafi ]
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

def SetNeighborMaxRoutesCmd_handleNormal( mode, args ):
   validatePeer( mode, args[ 'PEER' ] )
   SetNeighborMaxRoutesCmd_setNeighborMaxRoutes(
      mode,
      args[ 'PEER' ],
      args[ 'NUM_ROUTES' ],
      threshold=args.get( 'THRESHOLD' ),
      percent=args.get( 'percent' ),
      action=( 'warning-only' in args ) )

def SetNeighborMaxRoutesCmd_handleNoOrDefault( mode, args, noOrDefault ):
   if noOrDefault == NoOrDefault.DEFAULT:
      SetNeighborMaxRoutesCmd_noNeighborMaxRoutes(
         mode,
         args[ 'PEER' ],
         NoOrDefault.DEFAULT )
   else:
      SetNeighborMaxRoutesCmd_noNeighborMaxRoutes(
         mode,
         args[ 'PEER' ],
         NoOrDefault.NO )

def SetNeighborMaxRoutesCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborMaxRoutesCmd", mode, args )

def SetNeighborMaxRoutesCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborMaxRoutesCmd", mode, args )

# class SetNeighborRcfCmd( BgpCmdBaseClass ):
def SetNeighborRcfCmd_setRcf( mode, peer, inout, function ):
   validatePeer( mode, peer )
   # construct rcf field to be modified from af and inbound vs outbound
   direction = inout.title()
   af = mode.addrFamily
   config = bgpNeighborConfig( peer, mode.vrfName )
   attr = 'rcf' + direction
   attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
   setattr( config, attr, function )
   setattr( config, attr + 'Present', True )

def SetNeighborRcfCmd_noRcf( mode, peer, inout, noOrDefault ):
   validatePeer( mode, peer )
   # construct rcf field to be modified from af and inbound vs outbound
   af = mode.addrFamily
   config = bgpNeighborConfig( peer, mode.vrfName, create=False )
   direction = inout.title()
   attr = 'rcf' + direction
   attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
   if config:
      setattr( config, attr, getattr( config, attr + 'Default' ) )
      setTrue = noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
      setattr( config, attr + 'Present', setTrue )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborRcfCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   function = args[ 'FUNCTION' ]
   inout = args[ 'DIRECTION' ]
   SetNeighborRcfCmd_setRcf( mode, peer, inout, re.sub( r'\(\)', '', function ) )

def SetNeighborRcfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   inout = args[ 'DIRECTION' ]
   SetNeighborRcfCmd_noRcf( mode, peer, inout, noOrDefault )

def SetNeighborRcfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborRcfCmd", mode, args )

def SetNeighborRcfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborRcfCmd", mode, args )

# class SetRouteToPeerCmd( BgpNEDCmdBaseClass ):
def SetRouteToPeerCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.routeToPeerPresent = config.isPeerGroupPeer
   config.routeToPeer = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetRouteToPeerCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer,
                               create=( noOrDefault != NoOrDefault.DEFAULT ),
                               vrfName=mode.vrfName )
   if config:
      if noOrDefault == NoOrDefault.DEFAULT:
         # default route-to-peer is true
         config.routeToPeer = config.routeToPeerDefault
         config.routeToPeerPresent = False
      else:
         config.routeToPeer = False
         config.routeToPeerPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetRouteToPeerCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetRouteToPeerCmd", mode, args )

def SetRouteToPeerCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetRouteToPeerCmd", mode, args )

# class NeighborRemovePrivateAsCmd( BgpNEDCmdBaseClass ):
def NeighborRemovePrivateAsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   always = 'all' in args
   replace = 'replace-as' in args
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   asNumber = configForVrf( mode.vrfName ).asNumber
   if mode.vrfName != DEFAULT_VRF and asNumber == 0:
      asNumber = configForVrf( DEFAULT_VRF ).asNumber
   if config.asNumber == asNumber:
      mode.addWarning( "This command is a no-op - "
                        "neighbor's AS number is same as "
                        "device's AS number" )
   elif isPrivateAsn( config.asNumber ):
      mode.addWarning( 'eBGP neighbor has a private AS number' )
   config.removePrivateAs = Tac.Value( 'Routing::Bgp::RemovePrivateAs', True,
                                       always, replace )
   config.removePrivateAsPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborRemovePrivateAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.removePrivateAsPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
      config.removePrivateAs = config.removePrivateAsDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborRemovePrivateAsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborRemovePrivateAsCmd", mode, args )

def NeighborRemovePrivateAsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborRemovePrivateAsCmd", mode, args )

# class NeighborRemovePrivateAsIngressCmd( BgpNEDCmdBaseClass ):
def NeighborRemovePrivateAsIngressCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   replace = 'replace-as' in args
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   asNumber = configForVrf( mode.vrfName ).asNumber
   if mode.vrfName != DEFAULT_VRF and asNumber == 0:
      asNumber = configForVrf( DEFAULT_VRF ).asNumber
   if config.asNumber == asNumber:
      mode.addWarning( "This command is a no-op - "
                       "neighbor's AS number is same as "
                       "device's AS number" )
   elif isPrivateAsn( config.asNumber ):
      mode.addWarning( 'eBGP neighbor has a private AS number' )
   config.removePrivateAsIngress = \
      Tac.Value( 'Routing::Bgp::RemovePrivateAs', True, True, replace )
   config.removePrivateAsIngressPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborRemovePrivateAsIngressCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.removePrivateAsIngressPresent = \
         ( noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
      config.removePrivateAsIngress = config.removePrivateAsIngressDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborRemovePrivateAsIngressCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborRemovePrivateAsIngressCmd", mode, args )

def NeighborRemovePrivateAsIngressCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "NeighborRemovePrivateAsIngressCmd", mode, args )

# class NeighborPrependOwnAsDisabledCmd( BgpNEDCmdBaseClass ):
def NeighborPrependOwnAsDisabledCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.prependOwnDisabled = True
   config.prependOwnDisabledPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborPrependOwnAsDisabledCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.prependOwnDisabledPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                           config.isPeerGroupPeer )
      config.prependOwnDisabled = config.prependOwnDisabledDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborPrependOwnAsDisabledCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborPrependOwnAsDisabledCmd", mode, args )

def NeighborPrependOwnAsDisabledCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "NeighborPrependOwnAsDisabledCmd", mode, args )

# class NeighborReplaceRemoveAsCmd( BgpNEDCmdBaseClass ):
def NeighborReplaceRemoveAsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   asNumber = configForVrf( mode.vrfName ).asNumber
   if mode.vrfName != DEFAULT_VRF and asNumber == 0:
      asNumber = configForVrf( DEFAULT_VRF ).asNumber
   if config.asNumber == asNumber:
      mode.addWarning( "This command is a no-op - " +
                        "neighbor's AS number is same as " +
                        "device's AS number" )
   config.replaceRemoteAs = True
   config.replaceRemoteAsPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborReplaceRemoveAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.replaceRemoteAsPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
      config.replaceRemoteAs = config.replaceRemoteAsDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborReplaceRemoveAsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborReplaceRemoveAsCmd", mode, args )

def NeighborReplaceRemoveAsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborReplaceRemoveAsCmd", mode, args )

# class SetNeighborNextHopSelfCmd( BgpNEDCmdBaseClass ):
def SetNeighborNextHopSelfCmd_setNeighborNexthopSelf( mode, peer ):
   addrFamily = mode.addrFamily

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd and\
         addrFamily != 'all':
      msg = 'Address family next-hop-self only supported in multi-agent mode'
      mode.addWarning( msg )
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'nextHopSelf' ].get( addrFamily )
   setattr( config, attr, True )
   setattr( config, attr + 'Present', True )

def SetNeighborNextHopSelfCmd_noNeighborNexthopSelf( mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   addrFamily = mode.addrFamily
   attr = BgpLib.peerConfigAttrsAfMap[ 'nextHopSelf' ].get( addrFamily )

   if addrFamily == 'all':
      # in global mode, 'no' same as 'default' (sets Present to False),
      # unless peer in peer group
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      isPeerGroupPeer = False
      if config:
         isPeerGroupPeer = config.isPeerGroupPeer
      # if peer in peer group, 'no' sets Present to True
      setPresent = noOrDefault != NoOrDefault.DEFAULT and isPeerGroupPeer
   else:
      # in address family mode, 'no' always sets Present to True
      setPresent = noOrDefault != NoOrDefault.DEFAULT
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName,
                                  create=( noOrDefault != NoOrDefault.DEFAULT ) )

   if config:
      if noOrDefault != NoOrDefault.DEFAULT:
         setattr( config, attr + 'Present', setPresent )
         setattr( config, attr, False )
      else:
         setattr( config, attr + 'Present', False )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborNextHopSelfCmd_handleNormal( mode, args ):
   SetNeighborNextHopSelfCmd_setNeighborNexthopSelf( mode, args[ 'PEER' ] )

def SetNeighborNextHopSelfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborNextHopSelfCmd_noNeighborNexthopSelf( mode, args[ 'PEER' ],
                                                    noOrDefault )

def SetNeighborNextHopSelfCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborNextHopSelfCmd", mode, args )

def SetNeighborNextHopSelfCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborNextHopSelfCmd", mode, args )

# class NeighborNextHopPeerCmd( BgpNEDCmdBaseClass ):
def NeighborNextHopPeerCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.nextHopPeer = True
   config.nextHopPeerPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborNextHopPeerCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.nextHopPeerPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                    config.isPeerGroupPeer )
      config.nextHopPeer = config.nextHopPeerDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborNextHopPeerCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborNextHopPeerCmd", mode, args )

def NeighborNextHopPeerCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborNextHopPeerCmd", mode, args )

# class NeighborDescriptionCmd( BgpNEDCmdBaseClass ):
def NeighborDescriptionCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   desc = args[ 'DESC' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.description = desc
   config.descriptionPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborDescriptionCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.descriptionPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                    config.isPeerGroupPeer )
      config.description = config.descriptionDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborDescriptionCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborDescriptionCmd", mode, args )

def NeighborDescriptionCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborDescriptionCmd", mode, args )

def setNeighborSendCommunity( mode, peer, standard=None, extended=None,
                              large=None, lbw=None, add=None, remove=None ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )

   # For each send-community command, there are 4 command variants for each
   # community-type (standard, extended, large, link-bandwidth):
   #   (1) The community type is present with add
   #   (2) The community type is present with remove
   #   (3) The community type is present without add or remove
   #   (4) The community type is not present without add or remove
   isReplace = ( not add and not remove ) # case (3) and (4)

   preExistingSendCommunity = ( config.sendStandardCommunity or
                                config.sendExtendedCommunity or
                                config.sendLargeCommunity or
                                config.sendCommunity )

   if ( not remove and standard ):
      config.sendStandardCommunity = True
   elif ( ( remove and standard ) or
          ( isReplace and not standard ) ):
      config.sendStandardCommunity = config.sendStandardCommunityDefault

   if ( not remove and extended ):
      config.sendExtendedCommunity = True
   elif ( ( remove and extended ) or
          ( isReplace and not extended ) ):
      config.sendExtendedCommunity = config.sendExtendedCommunityDefault

   if ( not remove and large ):
      config.sendLargeCommunity = True
   elif ( ( remove and large ) or
          ( isReplace and not large ) ):
      config.sendLargeCommunity = config.sendLargeCommunityDefault

   # If neither send standard/extended/large community are
   # explicitly enabled, fallback to the default case and set the send
   # community attribute to True to enable send community for all (except
   # in the case that this is a remove command and there were no
   # pre-existing comm-type configured - in which case, ignore the cmd).
   # Otherwise, if one or two are explicitly enabled, then
   # set the send community attribute to False.
   if ( not config.sendStandardCommunity and not config.sendExtendedCommunity
        and not config.sendLargeCommunity ):
      config.sendCommunity = preExistingSendCommunity if remove else True
   else:
      config.sendCommunity = config.sendCommunityDefault

   # check if all or any individual send-communities values configured
   if ( config.sendCommunity or config.sendStandardCommunity or
        config.sendExtendedCommunity or config.sendLargeCommunity ):
      config.sendCommunityPresent = True

   if lbw:
      additive = lbw[ 'additiveOrDivide' ].get( 'additive', {} )
      lbwVal = additive.get( 'lbwVal' )
      divide = lbw[ 'additiveOrDivide' ].get( 'divide', {} )
      equal = ( divide.get( 'equalOrRatio' ) == 'equal' )
      ratio = ( divide.get( 'equalOrRatio' ) == 'ratio' )
      if not remove:
         # Add or replace existing send lbw community configuration.
         if additive:
            if not lbwVal:
               config.sendCommunityLinkBw = \
                  SendCommunityLinkBandwidthStateEnum.\
                  sendCommunityLinkBandwidthAdditive
            else:
               config.sendCommunityLinkBw = \
                  SendCommunityLinkBandwidthStateEnum.\
                  sendCommunityLinkBandwidthAdditiveRef
               config.sendCommunityLinkBwAdditiveRef = getLbwCommValue( lbwVal )
               config.sendCommunityLinkBwAdditiveRefDisplay = lbwVal
         elif divide:
            if equal:
               config.sendCommunityLinkBw = \
                  SendCommunityLinkBandwidthStateEnum.\
                  sendCommunityLinkBandwidthDivideEqual
            elif ratio:
               config.sendCommunityLinkBw = \
                  SendCommunityLinkBandwidthStateEnum.\
                  sendCommunityLinkBandwidthDivideRatio
         config.sendCommunityLinkBwPresent = True
      else:
         # Only remove send lbw community configuration if we
         # have an exact match w/ existing configuration.
         if ( additive and not lbwVal and
              ( config.sendCommunityLinkBw == # pylint: disable=consider-using-in
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthAdditive or
                config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthAdditiveRef ) ) or \
            ( additive and lbwVal and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthAdditiveRef ) and
              ( config.sendCommunityLinkBwAdditiveRefDisplay == lbwVal ) ) or \
            ( divide and equal and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthDivideEqual ) ) or \
            ( divide and ratio and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthDivideRatio ) ):
            config.sendCommunityLinkBw = config.sendCommunityLinkBwDefault
            config.sendCommunityLinkBwPresent = False
   elif isReplace:
      # Remove existing send lbw community configuration.
      config.sendCommunityLinkBw = config.sendCommunityLinkBwDefault
      config.sendCommunityLinkBwPresent = False

   # If the send lbw community configuration is no longer 'additiveRef',
   # then reset the corresponding 'additiveRef' attributes to default.
   if config.sendCommunityLinkBw != \
      SendCommunityLinkBandwidthStateEnum.\
      sendCommunityLinkBandwidthAdditiveRef:
      config.sendCommunityLinkBwAdditiveRef = \
         config.sendCommunityLinkBwAdditiveRefDefault
      config.sendCommunityLinkBwAdditiveRefDisplay = \
         config.sendCommunityLinkBwAdditiveRefDisplayDefault

# class SetNeighborSendCommunityCmd( BgpNEDCmdBaseClass ):
def SetNeighborSendCommunityCmd_handleNormal( mode, args ):
   setNeighborSendCommunity( mode, args[ 'PEER' ] )

def SetNeighborSendCommunityCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      isPresent = ( noOrDefault != noOrDefault.DEFAULT and
                    config.isPeerGroupPeer )
      config.sendCommunityPresent = isPresent
      config.sendCommunityLinkBwPresent = isPresent
      config.sendCommunity = config.sendCommunityDefault
      config.sendStandardCommunity = config.sendStandardCommunityDefault
      config.sendExtendedCommunity = config.sendExtendedCommunityDefault
      config.sendLargeCommunity = config.sendLargeCommunityDefault
      config.sendCommunityLinkBw = config.sendCommunityLinkBwDefault
      config.sendCommunityLinkBwAdditiveRef = \
         config.sendCommunityLinkBwAdditiveRefDefault
      config.sendCommunityLinkBwAdditiveRefDisplay = \
         config.sendCommunityLinkBwAdditiveRefDisplayDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborSendCommunityCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborSendCommunityCmd", mode, args )

def SetNeighborSendCommunityCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborSendCommunityCmd", mode, args )

# class SetNeighborSendCommunityStandardLargeCmd( CliCommand.CliCommandClass ):
def SetNeighborSendCommunityStandardLargeCmd_handler( mode, args ):
   if 'LBW' in args:
      commType = 'standard' if 'standard' in args else 'large'
      mode.addWarning( 'Link bandwidth is not a %s community. '
                        'Please use "neighbor PEER send-community link-bandwidth" '
                        'or "neighbor PEER send-community [ standard ] extended '
                        '[ large ] link-bandwidth" instead' % commType )
   setNeighborSendCommunity( mode, args[ 'PEER' ],
                             standard='standard' in args,
                             large='large' in args,
                             lbw=args.get( 'LBW' ) )

# class SetNeighborSendCommunityExtendedCmd( CliCommand.CliCommandClass ):
def SetNeighborSendCommunityExtendedCmd_handler( mode, args ):
   setNeighborSendCommunity( mode, args[ 'PEER' ],
                             standard='standard' in args,
                             large='large' in args,
                             extended='extended' in args,
                             lbw=args.get( 'LBW' ) )

# class SetNeighborSendCommunityAddOrRemoveCmd( CliCommand.CliCommandClass ):
def SetNeighborSendCommunityAddOrRemoveCmd_handler( mode, args ):
   setNeighborSendCommunity( mode, args[ 'PEER' ],
                             standard='standard' in args,
                             extended='extended' in args,
                             large='large' in args,
                             lbw=args.get( 'LBW' ),
                             add='add' in args,
                             remove='remove' in args )

# class NeighborShutdownCmd( BgpNEDCmdBaseClass ):
def NeighborShutdownCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   msg = args.get( 'MESSAGE' )
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.neighborShutdownMsg = msg if msg else config.neighborShutdownMsgDefault
   config.neighborShutdown = True
   config.neighborShutdownPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborShutdownCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.neighborShutdownPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                         config.isPeerGroupPeer )
      config.neighborShutdownMsg = config.neighborShutdownMsgDefault
      config.neighborShutdown = config.neighborShutdownDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborShutdownCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborShutdownCmd", mode, args )

def NeighborShutdownCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborShutdownCmd", mode, args )

@Tac.memoize
def configTagInputAllocator():
   return Tac.newInstance( "ConfigTag::ConfigTagInputAllocator" )

def removePeerFromCTtoPeerList( tagId, peer, commandTagIdToPeer ):
   commandTagIdToPeer[ tagId ].peer.remove( peer )
   # Remove the tag's entry from the mapping if no more neighbors are
   # associated with it.
   if not commandTagIdToPeer[ tagId ].peer:
      del commandTagIdToPeer[ tagId ]

def NeighborCommandTagCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   globalConfig = configForVrf( mode.vrfName )
   configTag = args.get( 'CONFIG_TAG', None )
   ConfigTagCommon.checkTagInRemovedOrDisassociatedState( mode,
      configTag, featureCheck=True )
   ConfigTagCommon.commandTagConfigCheck( mode, configTag )

   # Extract the tagId associated with the command-tag
   tagId = configTagConfig.configTagEntry[ configTag ].tagId

   # Extract the current tagId present in the neighbor
   curTagId = config.commandTagId

   # Update the command-tag information in the peer config
   config.commandTagId = tagId
      
   # Update the tag->peers mapping maintained in the global bgp config
   commandTagIdToPeer = globalConfig.commandTagIdToPeer
   if tagId not in commandTagIdToPeer:
      commandTagIdToPeer.newMember( tagId )
   commandTagIdToPeer[ tagId ].peer.add( peer )

   if curTagId != config.commandTagIdDefault:
      removePeerFromCTtoPeerList( curTagId, peer, commandTagIdToPeer )

   # Create an input for command-tag if it doesn't exist
   res = configTagInputAllocator().newConfigTagInputEntry( configTagInput,
         configTag, mode.vrfName )
   if not res:
      mode.addErrorAndStop( configTagInputAllocator().errorMsg )

   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborCommandTagCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   globalConfig = configForVrf( mode.vrfName )
   tagId = config.commandTagId
   if tagId:
      config.commandTagId = config.commandTagIdDefault
      removePeerFromCTtoPeerList( tagId, peer, globalConfig.commandTagIdToPeer )

      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborCommandTagCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborCommandTagCmd", mode, args )

def NeighborCommandTagCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "NeighborCommandTagCmd", mode, args )

class NeighborCommandTagState( ConfigTagCommon.ConfigTagDependentBase ):
   def processPeer( self, tagId, peerKey, globalConfig, removeOrDisassociate,
         vrfName=DEFAULT_VRF ):
      peer = globalConfig.neighborConfig[ peerKey ]
      removePeerFromCTtoPeerList( tagId, peerKey, globalConfig.commandTagIdToPeer )
      if removeOrDisassociate == RemoveOrDisassociate.disassociate:
         peer.commandTagId = peer.commandTagIdDefault
         delNeighborConfigIfDefault( peerKey, vrfName=vrfName )
      elif removeOrDisassociate == RemoveOrDisassociate.remove:
         del globalConfig.neighborConfig[ peerKey ]
      else:
         assert False, "unhandled action"

   def processTaggedPeers( self, mode, tag, removeOrDisassociate ):
      # Check if the tag exists in commandTag CLI input.
      if not configTagInput.configTagEntry.get( tag, None ):
         return
      # Extract the tagId associated with the command-tag
      entry = configTagConfig.configTagEntry.get( tag )
      if not entry or entry.tagId == 0:
         # This should never happen. We throw an error and stop if it does.
         mode.addErrorAndStop( ConfigTagCommon.commandTagErrStr( tag ) )
      tagId = entry.tagId
      # Process all the peers associated with the command-tag in the global
      # BGP config. We make use of commandTagIdToPeer which already contains
      # the list of peers associated with the tag.
      if tagId in Globals.bgpConfig.commandTagIdToPeer:
         for peerKey in Globals.bgpConfig.commandTagIdToPeer[ tagId ].peer:
            self.processPeer( tagId, peerKey, Globals.bgpConfig,
               removeOrDisassociate )
      # Process all the peers associated with the command-tag in the per VRF
      # BGP config.
      for vrfConfig in bgpVrfConfigDir.vrfConfig.values():
         if tagId in vrfConfig.commandTagIdToPeer:
            for peerKey in vrfConfig.commandTagIdToPeer[ tagId ].peer:
               self.processPeer( tagId, peerKey, vrfConfig, removeOrDisassociate,
                  vrfName=vrfConfig.name )

   def removeTaggedConfig( self, mode, tag ):
      self.processTaggedPeers( mode, tag, RemoveOrDisassociate.remove )

   def disassociateConfigFromTag( self, mode, tag ):
      self.processTaggedPeers( mode, tag, RemoveOrDisassociate.disassociate )

   def processAndValidateConfig( self, mode, commandTagInfo ):
      pass

ConfigTagCommon.ConfigTagState.registerConfigTagSupportingClass(
      NeighborCommandTagState )

# class NeighborDontCapabilityNegotiateCmd( BgpNEDCmdBaseClass ):
def NeighborDontCapabilityNegotiateCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=True )
   config.dontCapabilityNegotiate = True
   config.dontCapabilityNegotiatePresent = True

def NeighborDontCapabilityNegotiateCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
   if config:
      config.dontCapabilityNegotiate = False
      config.dontCapabilityNegotiatePresent = (
         noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborDontCapabilityNegotiateCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborDontCapabilityNegotiateCmd", mode, args )

def NeighborDontCapabilityNegotiateCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "NeighborDontCapabilityNegotiateCmd", mode, args )

# class NeighborReflectorClientCmd( BgpNEDCmdBaseClass ):
def NeighborReflectorClientCmd_setRrClientValue( peer, vrfName, value ):
   """
   rrClient is
   = notReflector(0) by default, neighbor is not a route reflector client
   = reflector(1) when the neighbor is a route reflector client
   = reflectorMeshed(2) when intra-cluster reflection is disabled.
   """
   config = bgpNeighborConfig( peer, vrfName=vrfName )
   config.rrClient = value
   config.rrClientPresent = True
   delNeighborConfigIfDefault( peer, vrfName=vrfName )

def NeighborReflectorClientCmd_setRrClientDefault( peer, vrfName, noOrDefault ):
   """
   Restores the original config values, maybe overriding peer group
   configuration if the peer belongs to a peer-group.
   """
   config = bgpNeighborConfig( peer, create=False, vrfName=vrfName )
   if config:
      config.rrClientPresent = ( noOrDefault != NoOrDefault.DEFAULT and
         config.isPeerGroupPeer )
      config.rrClient = config.rrClientDefault
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

def NeighborReflectorClientCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   intraClusterDisabled = ( 'meshed' in args )
   NeighborReflectorClientCmd_setRrClientValue( peer, mode.vrfName,
      RRClientEnum.reflectorMeshed if intraClusterDisabled else
      RRClientEnum.reflector )

def NeighborReflectorClientCmd_handleNoOrDefault( mode, args, noOrDefault ):
   # if NED is on: we should expect "reflection intra-cluster in args" and
   # NoOrDefault will be NO. If NED mode is off, differentiating
   # `neighbor ... disabled` from `no neighbor ...` is done with the aid
   # of CliCommand.isNoCmd()
   peer = args[ 'PEER' ]
   if noOrDefault == NoOrDefault.NO and \
      not CliCommand.isNoCmd( args ) and \
      all( arg in args for arg in
         ( 'reflection', 'intra-cluster', 'disabled' ) ):
      NeighborReflectorClientCmd_setRrClientValue( peer, mode.vrfName,
         RRClientEnum.reflectorMeshed )
   else:
      # explicit no or default detected.
      NeighborReflectorClientCmd_setRrClientDefault( peer, mode.vrfName,
         noOrDefault )

def NeighborReflectorClientCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborReflectorClientCmd", mode, args )

def NeighborReflectorClientCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborReflectorClientCmd", mode, args )

# class NeighborStallCmd( BgpNEDCmdBaseClass ):
def NeighborStallCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   fail = 'fail' in args
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.stall = 'stallWriteFail' if fail else 'stallWriteBlock'
   config.stallPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborStallCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.stallPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                              config.isPeerGroupPeer )
      config.stall = config.stallDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborStallCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborStallCmd", mode, args )

def NeighborStallCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborStallCmd", mode, args )

# class NeighborSockFaultCmd( BgpNEDCmdBaseClass ):
def NeighborSockFaultCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   greedyRecvEstabFail = 'greedyrecvestabfail' in args
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=True, vrfName=mode.vrfName )
   if greedyRecvEstabFail:
      config.sockFault = 'greedyRecvEstablishedFault'
   config.sockFaultPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborSockFaultCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.sockFaultPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                  config.isPeerGroupPeer )
      config.sockFault = config.sockFaultDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborSockFaultCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborSockFaultCmd", mode, args )

def NeighborSockFaultCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborSockFaultCmd", mode, args )

# class SetNeighborLocalIp6AddrCmd( BgpNEDCmdBaseClass ):
def SetNeighborLocalIp6AddrCmd_handleNormal( mode, args ):
   v4OrPgPeerKey = args[ 'PEER' ]
   localIp6Addr = args[ 'IP6_ADDR' ]
   validatePeer( mode, v4OrPgPeerKey )
   config = bgpNeighborConfig( v4OrPgPeerKey, vrfName=mode.vrfName )
   config.localIp6Addr = localIp6Addr
   config.localIp6AddrPresent = True
   delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

def SetNeighborLocalIp6AddrCmd_handleNoOrDefault( mode, args, noOrDefault ):
   v4OrPgPeerKey = args[ 'PEER' ]
   config = bgpNeighborConfig( v4OrPgPeerKey, create=False, vrfName=mode.vrfName )
   if config:
      config.localIp6AddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                     config.isPeerGroupPeer )
      config.localIp6Addr = config.localIp6AddrDefault
      delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

def SetNeighborLocalIp6AddrCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLocalIp6AddrCmd", mode, args )

def SetNeighborLocalIp6AddrCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLocalIp6AddrCmd", mode, args )

# class SetNeighborIp6NextHopAddrCmd( BgpNEDCmdBaseClass ):
def SetNeighborIp6NextHopAddrCmd_handleNormal( mode, args ):
   v4OrPgPeerKey = args[ 'PEER' ]
   remoteIp6Addr = args[ 'IP6_ADDR' ]
   validatePeer( mode, v4OrPgPeerKey )
   config = bgpNeighborConfig( v4OrPgPeerKey, vrfName=mode.vrfName )
   config.remoteIp6Addr = remoteIp6Addr
   config.remoteIp6AddrPresent = True
   delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

def SetNeighborIp6NextHopAddrCmd_handleNoOrDefault( mode, args, noOrDefault ):
   v4OrPgPeerKey = args[ 'PEER' ]
   config = bgpNeighborConfig( v4OrPgPeerKey, create=False, vrfName=mode.vrfName )
   if config:
      config.remoteIp6AddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                      config.isPeerGroupPeer )
      config.remoteIp6Addr = config.remoteIp6AddrDefault
      delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

def SetNeighborIp6NextHopAddrCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborIp6NextHopAddrCmd", mode, args )

def SetNeighborIp6NextHopAddrCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborIp6NextHopAddrCmd", mode, args )

# class SetNeighborLocalIp4AddrCmd( BgpNEDCmdBaseClass ):
def SetNeighborLocalIp4AddrCmd_handleNormal( mode, args ):
   v6OrPgPeerKey = args[ 'PEER' ]
   localIp4Addr = args[ 'IP4_ADDR' ]
   validatePeer( mode, v6OrPgPeerKey )
   config = bgpNeighborConfig( v6OrPgPeerKey, vrfName=mode.vrfName )
   config.localIp4Addr = localIp4Addr
   config.localIp4AddrPresent = True
   delNeighborConfigIfDefault( v6OrPgPeerKey, vrfName=mode.vrfName )

def SetNeighborLocalIp4AddrCmd_handleNoOrDefault( mode, args, noOrDefault ):
   v6OrPgPeerKey = args[ 'PEER' ]
   config = bgpNeighborConfig( v6OrPgPeerKey, create=False, vrfName=mode.vrfName )
   if config:
      config.localIp4AddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                     config.isPeerGroupPeer )
      config.localIp4Addr = config.localIp4AddrDefault
      delNeighborConfigIfDefault( v6OrPgPeerKey, vrfName=mode.vrfName )

def SetNeighborLocalIp4AddrCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLocalIp4AddrCmd", mode, args )

def SetNeighborLocalIp4AddrCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLocalIp4AddrCmd", mode, args )

# class SetNeighborAutoLocalAddrPeerCmd( BgpNEDCmdBaseClass ):
def SetNeighborAutoLocalAddrPeerCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.autoLocalAddr = True
   config.autoLocalAddrPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAutoLocalAddrPeerCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.autoLocalAddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                      config.isPeerGroupPeer )
      config.autoLocalAddr = config.autoLocalAddrDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAutoLocalAddrPeerCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborAutoLocalAddrPeerCmd", mode, args )

def SetNeighborAutoLocalAddrPeerCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborAutoLocalAddrPeerCmd", mode, args )

# class SetNeighborAddPathRecvCmd( BgpNEDCmdBaseClass ):
def SetNeighborAddPathRecvCmd_setNeighborAddPathRecv( mode, peer ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
   setattr( config, attr + 'Present', True )
   setattr( config, attr, True )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAddPathRecvCmd_noNeighborAddPathRecv( mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer,
                               vrfName=mode.vrfName,
                               create=( noOrDefault == NoOrDefault.NO ) )
   attr = BgpLib.peerConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
   if config:
      if noOrDefault == NoOrDefault.NO:
         setattr( config, attr, False )
         setattr( config, attr + 'Present', True )
      else:
         # Add-path receive is enabled by default
         setattr( config, attr, True )
         setattr( config, attr + 'Present', False )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAddPathRecvCmd_handleNormal( mode, args ):
   SetNeighborAddPathRecvCmd_setNeighborAddPathRecv( mode, args[ 'PEER' ] )

def SetNeighborAddPathRecvCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborAddPathRecvCmd_noNeighborAddPathRecv( mode,
                                                    args[ 'PEER' ],
                                                    noOrDefault )

def SetNeighborAddPathRecvCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborAddPathRecvCmd", mode, args )

def SetNeighborAddPathRecvCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborAddPathRecvCmd", mode, args )

# class SetNeighborAddPathSendCmd( BgpNEDCmdBaseClass ):
def SetNeighborAddPathSendCmd_validateAddPathSendPrefixList( mode ):
   afList = [ 'ipv4', 'ipv6' ]
   if mode.addrFamily not in afList:
      mode.addErrorAndStop( "Neighbor additional-paths send with prefix list "
                            "can only be configured under ipv4 and ipv6 address "
                            "families" )

def SetNeighborAddPathSendCmd_validateAddPathSend( mode ):
   afList = [ 'evpn', 'vpn-ipv4', 'vpn-ipv6', 'path-selection' ]
   if ( mode.vrfName != DEFAULT_VRF and
        mode.addrFamily in afList ):
      mode.addError( "Neighbor additional-paths send can only be configured under"
                     " %s mode for %s VRF" % ( mode.addrFamily, DEFAULT_VRF ) )
      return False
   elif ( mode.vrfName != DEFAULT_VRF and 'labeled-unicast' in mode.addrFamily ):
      mode.addError( "Neighbor additional-paths send can only be configured under"
                     " labeled-unicast mode for %s VRF" % DEFAULT_VRF )
      return False
   return True

def SetNeighborAddPathSendCmd_setNeighborAddPathSend(
      mode, peer, app='appNone', pathLimit=0, prefixList='' ):
   validatePeer( mode, peer )
   if prefixList:
      SetNeighborAddPathSendCmd_validateAddPathSendPrefixList( mode )
   if not SetNeighborAddPathSendCmd_validateAddPathSend( mode ):
      return
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   sendConfig = addPathSendConfig( config, mode.addrFamily )
   newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app, enable=True,
                           pathLimit=pathLimit, prefixList=prefixList )
   if app not in sendConfig:
      sendConfig.clear()
   sendConfig.addMember( newConfig )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAddPathSendCmd_noNeighborAddPathSend( mode, peer, app='appNone',
                              noOrDefault=NoOrDefault.DEFAULT ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer,
                               vrfName=mode.vrfName,
                               create=( noOrDefault == NoOrDefault.NO ) )
   if config:
      sendConfig = addPathSendConfig( config, mode.addrFamily )
      if noOrDefault == NoOrDefault.NO:
         # if app is appAny then show run will show the no form of this
         # command with "any" at the end for backward compatibility
         # pylint: disable-next=consider-using-in
         assert app == 'appNone' or app == 'appAny'
         # The following setting is to support config inheritance
         newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app,
                                 enable=False, prefixList='' )
         if app not in sendConfig:
            sendConfig.clear()
         sendConfig.addMember( newConfig )
      else:
         sendConfig.clear()
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAddPathSendCmd_handleNormal( mode, args ):
   app = 'appNone'
   pathLimit = 0
   if 'any' in args:
      app = 'appAny'
   elif 'backup' in args:
      app = 'appBackup'
   elif 'limit' in args:
      if 'ecmp' in args:
         app = 'appEcmp'
      else:
         app = 'appLimit'
      pathLimit = args[ 'LIMIT' ]
   elif 'ecmp' in args:
      app = 'appEcmp'
   prefixList = args.get( 'PREFIX_LIST_NAME', '' )
   SetNeighborAddPathSendCmd_setNeighborAddPathSend( mode,
                                                     args[ 'PEER' ],
                                                     app=app,
                                                     pathLimit=pathLimit,
                                                     prefixList=prefixList )

def SetNeighborAddPathSendCmd_handleNoOrDefault( mode, args, noOrDefault ):
   app = 'appNone'
   # For backward compatibility if "any" is specified then the
   # no form of this command will show "any"
   if 'ANY_HIDDEN' in args or 'any' in args:
      app = 'appAny'
   SetNeighborAddPathSendCmd_noNeighborAddPathSend( mode,
                                                    args[ 'PEER' ],
                                                    app=app,
                                                    noOrDefault=noOrDefault )

def SetNeighborAddPathSendCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborAddPathSendCmd", mode, args )

def SetNeighborAddPathSendCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborAddPathSendCmd", mode, args )

# class SetNeighborDefaultOriginateBaseClass( object ):
def SetNeighborDefaultOriginateBaseClass_verifyNoConflictingDefOrigin(
      mode, config, addrFamily ):
   """
   Print an error if a conflicting default-originate config is already present.
   """
   # config in VPN address-families do not conflict with other modes
   if addrFamily in vpnAfTypeCliKeywords:
      return
   attrs = BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      # config in VPN address-families do not conflict with other modes
      if otherAf in vpnAfTypeCliKeywords:
         continue
      if attrIsPresent( config, c ):
         errMsg = "Cannot configure default-originate in mode '%s' while it is "\
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ),
                     configModeCmdForAf( otherAf ) )
         mode.addError( errMsg )
         raise CliCommon.AlreadyHandledError

def SetNeighborDefaultOriginateBaseClass_noDefaultOriginateCommon(
      peer, noOrDefault, vrfName, addrFamily='all' ):
   # The "no" variant of the command is not same as "default" variant in case of
   # peerGroupPeer. Also, we should check for conflicting configuration in case of
   # "no" command for peerGroupPeer (as done in setDefaultOriginateCommon). But
   # the "neighbor <> route-map" command does not have this check. So, I am
   # leaving out the check here as well, till I verify if it is a bug or is
   # intentional. - pranav
   config = bgpNeighborConfig( peer, vrfName=vrfName )
   present = noOrDefault == NoOrDefault.NO and config.isPeerGroupPeer
   attr = BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ].get( addrFamily )
   if config:
      setattr( config, attr + 'Present', present )
      setattr( config, attr, False )
      setattr( config, attr + 'RouteMap',
               getattr( config, attr + 'RouteMapDefault' ) )
      # RCF is not supported in address family all (top level router bgp)
      if addrFamily != 'all':
         setattr( config, attr + 'Rcf',
                  getattr( config, attr + 'RcfDefault' ) )
      setattr( config, attr + 'Always',
               getattr( config, attr + 'AlwaysDefault' ) )
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

def SetNeighborDefaultOriginateBaseClass_defaultOriginateCommon(
      mode, peer, addrFamily, policyName, always ):
   config = bgpNeighborConfig( peer, mode.vrfName )
   SetNeighborDefaultOriginateBaseClass_verifyNoConflictingDefOrigin(
         mode, config, addrFamily )
   attr = BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ].get( addrFamily )

   setattr( config, attr, True )
   setattr( config, attr + 'Present', True )
   rmPolicyName = getattr( config, attr + 'RouteMapDefault' )
   if addrFamily != 'all':
      rcfPolicyName = getattr( config, attr + 'RcfDefault' )
   if policyName is not None:
      policyType = 'Rcf' if re.search( r'\(\)', policyName ) else 'RouteMap'
      if policyType == 'Rcf':
         rcfPolicyName = policyName.replace( '()', '' )
      else:
         rmPolicyName = policyName

   # As we only allow one default originate policy to be in place at any given time
   # We must check to what type of policy is being set so we can set the other type
   # back to its default value first. Otherwise it is possible for both route-map and
   # Rcf policy to exist briefly at the same time when propagating the changes.
   # If no policyName is set, we don't care about order as we set both route-map and
   # Rcf back to their default values.
   if policyName is None or policyType == 'RouteMap':
      # RCF is not supported in address family all (top level router bgp)
      if addrFamily != 'all':
         setattr( config, attr + 'Rcf', rcfPolicyName )
      setattr( config, attr + 'RouteMap', rmPolicyName )
   else:
      setattr( config, attr + 'RouteMap', rmPolicyName )
      setattr( config, attr + 'Rcf', rcfPolicyName )

   setattr( config, attr + 'Always', always )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborDefaultOriginateBaseClass_handleNormal( mode, args ):
   addrFamily = mode.addrFamily if mode.addrFamily else 'all'
   SetNeighborDefaultOriginateBaseClass_defaultOriginateCommon(
      mode, args[ 'PEER' ], addrFamily,
      args.get( 'MAP_NAME', args.get( 'FUNCTION', args.get( 'POLICY_TYPE' ) ) ),
      'always' in args )

def SetNeighborDefaultOriginateBaseClass_handleNoOrDefault(
      mode, args, noOrDefault ):
   SetNeighborDefaultOriginateBaseClass_noDefaultOriginateCommon(
         args[ 'PEER' ], noOrDefault, mode.vrfName, mode.addrFamily )

# class SetNeighborDefaultOriginateCmd( BgpNEDCmdBaseClass ):
def SetNeighborDefaultOriginateCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborDefaultOriginateBaseClass", mode, args )

def SetNeighborDefaultOriginateCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborDefaultOriginateBaseClass", mode, args )

# class SetVpnNeighborDefaultOriginateHiddenCmd( BgpNEDCmdBaseClass ):
def SetVpnNeighborDefaultOriginateHiddenCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborDefaultOriginateBaseClass", mode, args )

def SetVpnNeighborDefaultOriginateHiddenCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborDefaultOriginateBaseClass", mode, args )

# class SetVpnNeighborDefaultOriginateCmd( BgpCmdBaseClass ):
def SetVpnNeighborDefaultOriginateCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborDefaultOriginateBaseClass", mode, args )

def SetVpnNeighborDefaultOriginateCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "SetNeighborDefaultOriginateBaseClass", mode, args )

# class SetNeighborEnforceFirstAsCmd( BgpNEDCmdBaseClass ):
def SetNeighborEnforceFirstAsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.enforceFirstAs = True
   config.enforceFirstAsPresent = True

def SetNeighborEnforceFirstAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer,
                               create=( noOrDefault != NoOrDefault.DEFAULT ),
                               vrfName=mode.vrfName )
   if config:
      if noOrDefault == NoOrDefault.DEFAULT:
         # Enforce First AS is enabled by default
         config.enforceFirstAs = True
         config.enforceFirstAsPresent = False
      else:
         config.enforceFirstAs = False
         config.enforceFirstAsPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborEnforceFirstAsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborEnforceFirstAsCmd", mode, args )

def SetNeighborEnforceFirstAsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborEnforceFirstAsCmd", mode, args )

# class NeighborUpdateSrcCommon( object ):
def NeighborUpdateSrcCommon_setUpdateSrcConfig( config, intf=None, addr=None ):
   if intf:
      config.updateSrcAddrPresent = False
      config.updateSrcAddr = config.updateSrcAddrDefault
      config.updateSrcIntf = intf
      config.updateSrcIntfPresent = True
   elif addr:
      config.updateSrcIntfPresent = False
      config.updateSrcIntf = config.updateSrcIntfDefault
      config.updateSrcAddr = IpGenAddr( str( addr ) )
      config.updateSrcAddrPresent = True

def NeighborUpdateSrcCommon_noOrDefaultUpdateSrcConfig( config, noOrDefault ):
   if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
      config.updateSrcAddr = config.updateSrcAddrDefault
      config.updateSrcAddrPresent = True
      config.updateSrcIntf = config.updateSrcIntfDefault
      config.updateSrcIntfPresent = True
   else:
      config.updateSrcAddrPresent = False
      config.updateSrcAddr = config.updateSrcAddrDefault
      config.updateSrcIntfPresent = False
      config.updateSrcIntf = config.updateSrcIntfDefault

# class NeighborUpdateSrcCmd( BgpNEDCmdBaseClass, NeighborUpdateSrcCommon ):
def NeighborUpdateSrcCmd_handleNormal( mode, args ):
   addr = args.get( 'V4_ADDR' ) or args.get( 'V6_ADDR' )
   if addr:
      v4 = 'V4_ADDR' in args
      if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
         mode.addWarning( "%s source address is only supported in "
                           "multi-agent mode" % ( "IPv4" if v4 else "IPv6" ) )
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )

   if addr:
      NeighborUpdateSrcCommon_setUpdateSrcConfig( config, addr=addr )
   else:
      NeighborUpdateSrcCommon_setUpdateSrcConfig( config,
                                    intf=args[ 'UPDATE_SRC_INTF' ].name )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborUpdateSrcCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      NeighborUpdateSrcCommon_noOrDefaultUpdateSrcConfig( config, noOrDefault )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborUpdateSrcCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborUpdateSrcCmd", mode, args )

def NeighborUpdateSrcCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborUpdateSrcCmd", mode, args )

# class SetNeighborEbgpMultiHopTtlCmd( BgpNEDCmdBaseClass ):
def SetNeighborEbgpMultiHopTtlCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   ebgpMultiHopTtl = args.get( 'TTL' )
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if ebgpMultiHopTtl:
      config.ebgpMultiHop = ebgpMultiHopTtl
   else:
      # if ebgp-multihop is enabled default TTL is 255
      config.ebgpMultiHop = config.ebgpMultiHopEnabledTtlDefault
   config.ebgpMultiHopPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborEbgpMultiHopTtlCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.ebgpMultiHopPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                       config.isPeerGroupPeer )
      config.ebgpMultiHop = config.ebgpMultiHopDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborEbgpMultiHopTtlCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborEbgpMultiHopTtlCmd", mode, args )

def SetNeighborEbgpMultiHopTtlCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborEbgpMultiHopTtlCmd", mode, args )

# class SetNeighborTtlMaxHopsCmd( BgpNEDCmdBaseClass ):
def SetNeighborTtlMaxHopsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   bgpTtlMaxHops = args[ 'NUM_HOPS' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.bgpTtlSecMaxHop = bgpTtlMaxHops
   config.bgpTtlSecMaxHopPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborTtlMaxHopsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.bgpTtlSecMaxHopPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                          config.isPeerGroupPeer )
      config.bgpTtlSecMaxHop = config.bgpTtlSecMaxHopDefault
      config.bgpTtlSecMaxHopLog = config.bgpTtlSecMaxHopLogDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborTtlMaxHopsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborTtlMaxHopsCmd", mode, args )

def SetNeighborTtlMaxHopsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborTtlMaxHopsCmd", mode, args )

# class SetNeighborLinkbwDelayCmd( BgpNEDCmdBaseClass ):
def SetNeighborLinkbwDelayCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   linkbwDelay = args[ 'DELAY' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.linkbwDelay = linkbwDelay
   config.linkbwDelayPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkbwDelayCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.linkbwDelayPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                    config.isPeerGroupPeer )
      config.linkbwDelay = config.linkbwDelayDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkbwDelayCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLinkbwDelayCmd", mode, args )

def SetNeighborLinkbwDelayCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLinkbwDelayCmd", mode, args )

# class SetNeighborWeightCmd( BgpNEDCmdBaseClass ):
def SetNeighborWeightCmd_setWeight( mode, peer, weight ):
   validatePeer( mode, peer )
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=vrfName )
   if haveConflictingWeight( mode, config, addrFamily ):
      return
   attr = BgpLib.peerConfigAttrsAfMap[ 'weight' ].get( addrFamily )
   if weight is None:
      weight = getattr( config, attr + 'Default' )
   setattr( config, attr, weight )
   setattr( config, attr + 'Present', True )
   delNeighborConfigIfDefault( peer, vrfName=vrfName )

def SetNeighborWeightCmd_noWeight( mode, peer, noOrDefault ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=vrfName, create=False )
   attr = BgpLib.peerConfigAttrsAfMap[ 'weight' ].get( addrFamily )
   if config:
      setattr( config,
               attr + 'Present',
               ( noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer ) )
      setattr( config, attr, getattr( config, attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

def SetNeighborWeightCmd_handleNormal( mode, args ):
   SetNeighborWeightCmd_setWeight( mode, args[ 'PEER' ], args[ 'WEIGHT' ] )

def SetNeighborWeightCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborWeightCmd_noWeight( mode, args[ 'PEER' ], noOrDefault )

def SetNeighborWeightCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborWeightCmd", mode, args )

def SetNeighborWeightCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborWeightCmd", mode, args )

# class SetNeighborAllowAsCmd( BgpNEDCmdBaseClass ):
def SetNeighborAllowAsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   allowAs = args.get( 'NUM_AS' )
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if allowAs is None:
      # if allowAs is enabled default allowAs loop count is 3
      config.allowAs = config.allowAsEnabledDefault
   else:
      config.allowAs = allowAs
   config.allowAsPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAllowAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.allowAsPresent = noOrDefault != NoOrDefault.DEFAULT
   config.allowAs = config.allowAsDefault
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborAllowAsCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborAllowAsCmd", mode, args )

def SetNeighborAllowAsCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborAllowAsCmd", mode, args )

# class SetNeighborPassiveCmd( BgpNEDCmdBaseClass ):
def SetNeighborPassiveCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.passive = True
   config.passivePresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborPassiveCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.passivePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                config.isPeerGroupPeer )
      config.passive = config.passiveDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborPassiveCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborPassiveCmd", mode, args )

def SetNeighborPassiveCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborPassiveCmd", mode, args )

# class SetNeighborRemotePortCmd( BgpNEDCmdBaseClass ):
def SetNeighborRemotePortCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   remotePort = args[ 'PORT_NUM' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.remotePort = remotePort
   config.remotePortPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborRemotePortCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.remotePortPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                   config.isPeerGroupPeer )
      config.remotePort = config.remotePortDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborRemotePortCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborRemotePortCmd", mode, args )

def SetNeighborRemotePortCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborRemotePortCmd", mode, args )

# class SetNeighborTransportPmtudCmd( BgpCmdBaseClass ):
def SetNeighborTransportPmtudCmd_setNeighPathMtuDiscovery( mode, peer ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.pathMtuDiscovery = True
   config.pathMtuDiscoveryPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborTransportPmtudCmd_disabledNeighPathMtuDiscovery( mode, peer ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.pathMtuDiscovery = False
   config.pathMtuDiscoveryPresent = True
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborTransportPmtudCmd_defaultNeighPathMtuDiscovery( mode, peer ):
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.pathMtuDiscoveryPresent = False
   config.pathMtuDiscovery = config.pathMtuDiscoveryDefault
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborTransportPmtudCmd_handleNoOrDefault( mode, args, noOrDefault ):
   if noOrDefault == NoOrDefault.DEFAULT:
      SetNeighborTransportPmtudCmd_defaultNeighPathMtuDiscovery(
         mode,
         args.get( 'PEER' ) )
   else:
      SetNeighborTransportPmtudCmd_disabledNeighPathMtuDiscovery(
         mode,
         args.get( 'PEER' ) )

def SetNeighborTransportPmtudCmd_handleNormal( mode, args ):
   SetNeighborTransportPmtudCmd_setNeighPathMtuDiscovery(
      mode,
      args.get( 'PEER' ) )

def SetNeighborTransportPmtudCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborTransportPmtudCmd", mode, args )

def SetNeighborTransportPmtudCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborTransportPmtudCmd", mode, args )

# class SetNeighborPeerGroupCmd( BgpNEDCmdBaseClass ):
def SetNeighborPeerGroupCmd_handleNormal( mode, args ):
   v4v6PeerKey = args[ 'PEER' ]
   pgKey = PeerConfigKey( args[ 'PEER_GROUP' ] )
   validatePeer( mode, v4v6PeerKey )
   peerGroup = bgpNeighborConfig( pgKey, vrfName=mode.vrfName )
   peer = bgpNeighborConfig( v4v6PeerKey, vrfName=mode.vrfName )
   peer.peerGroupKey = peerGroup.key
   delNeighborConfigIfDefault( v4v6PeerKey, vrfName=mode.vrfName )

def SetNeighborPeerGroupCmd_handleNoOrDefault( mode, args, noOrDefault ):
   v4v6PeerKey = args[ 'PEER' ]
   config = bgpNeighborConfig( v4v6PeerKey, create=False, vrfName=mode.vrfName )
   if config:
      if config.isPeerGroupPeer:
         config.peerGroupKey = PeerConfigKeyType()
         RoutingBgpCliHandler.resetPresents( config )
      delNeighborConfigIfDefault( v4v6PeerKey, vrfName=mode.vrfName )

def SetNeighborPeerGroupCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborPeerGroupCmd", mode, args )

def SetNeighborPeerGroupCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborPeerGroupCmd", mode, args )

# class ConfigureNeighborPeerGroupCmd( CliCommand.CliCommandClass ):
def ConfigureNeighborPeerGroupCmd_verifyAncestorPeerGroups(
      pgName, mode, ancestorPeerGroupNames ):
   # Verify the peer group ancestors of pgName do not also have ancestors.
   validPgNames = []
   for apgName in ancestorPeerGroupNames:
      # Cannot have itself as an ancestor
      if apgName == pgName:
         mode.addErrorAndStop(
            "Peer group %s cannot have itself as an ancestor" % pgName )
      # No duplicates allowed in the ancestor peer groups
      if apgName in validPgNames:
         mode.addErrorAndStop(
            "Peer group %s has duplicate ancestor peer group %s" % (
               pgName, apgName ) )
      apgKey = PeerConfigKey( apgName )
      apgConfig = bgpNeighborConfig( apgKey, create=False, vrfName=mode.vrfName )
      # Ancestor cannot have ancestors
      if apgConfig is not None and apgConfig.isPeerGroupWithAncestors:
         mode.addErrorAndStop( "Peer group %s already has ancestors" % apgName )
      validPgNames.append( apgName )

def ConfigureNeighborPeerGroupCmd_verifyPeerGroupIsNonAncestor( apgKey, mode ):
   # Verify the peer group apgName is not already an ancestor of another peer
   # group.
   for pg in Globals.bgpConfig.neighborConfig.values():
      if pg.isPeerGroupWithAncestors:
         if apgKey in pg.ancestorPeerGroupConfig.ancestorPeerGroupKeys.values():
            mode.addErrorAndStop( "Peer group %s is already an ancestor "
                                    "in peer group %s" % ( apgKey.group,
                                                         pg.key.group ) )

def ConfigureNeighborPeerGroupCmd_handler( mode, args ):
   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd and \
      'GROUPS' in args:
      msg = 'Ancestor peer groups are only supported in multi-agent mode.'
      errFunc = mode.addErrorAndStop if mode.session.isInteractive() else \
         mode.addWarning
      errFunc( msg )

   pgKey = PeerConfigKey( args[ 'GROUP' ] )
   ancestorPeerGroupNames = args.get( 'GROUPS', [] )
   # Initial checks before creating anything for peer groups which specify
   # ancestor peer groups.
   # - the ancestor peer groups cannot also have ancestors
   # - the peer group that is adding ancestor peer groups cannot already
   #   be an ancestor in another peer group.
   if ancestorPeerGroupNames:
      ConfigureNeighborPeerGroupCmd_verifyAncestorPeerGroups(
         pgKey.group, mode, ancestorPeerGroupNames )
      ConfigureNeighborPeerGroupCmd_verifyPeerGroupIsNonAncestor( pgKey, mode )
   pgConfig = bgpNeighborConfig( pgKey, vrfName=mode.vrfName )
   # Update the peer group's current ancestors and with the new ancestors and
   # trim the ancestors at the end of the current queue.
   ancestorPeerGroupConfig = Globals.AncestorPgConfigType()
   # Start key index at 1 due to nominal default key restriction
   for idx, apgName in enumerate( ancestorPeerGroupNames, 1 ):
      apgKey = PeerConfigKey( apgName )
      bgpNeighborConfig( apgKey, vrfName=mode.vrfName )
      ancestorPeerGroupConfig.ancestorPeerGroupKeys[ idx ] = apgKey
   pgConfig.ancestorPeerGroupConfig = ancestorPeerGroupConfig

def ConfigureNeighborPeerGroupCmd_noOrDefaultHandler( mode, args ):
   pgKey = PeerConfigKey( args[ 'GROUP' ] )
   config = bgpNeighborConfig( pgKey, create=False, vrfName=mode.vrfName )
   peergroupName = pgKey.group
   if config:
      # if this peer-group is being used by a configured listen range or neighbor
      # interface config in any VRF prevent peer-group config from being removed
      if isDynamicPeerGroupByName( peergroupName ):
         mode.addError( 'Cannot remove peer-group %s while a listen-range '
                        'referencing the peer-group is still configured' %
                        peergroupName )
         return
      if isNeighIntfConfigPeerGroup( peergroupName ):
         mode.addError( 'Cannot remove peer-group %s while a neighbor '
                        'interface config referencing the peer-group '
                        'is still configured' % peergroupName )
         return
      # cleanup all references to this peer-group for all configured static peers
      for peer in Globals.bgpConfig.neighborConfig.values():
         if peer.isPeerGroupPeer and peer.peerGroupKey == config.key:
            peer.peerGroupKey = PeerConfigKeyType()
            RoutingBgpCliHandler.resetPresents( peer )
            delNeighborConfigIfDefault( peer.key, vrfName=DEFAULT_VRF )
         elif peer.isPeerGroupWithAncestors:
            # Delete the reference to the deleted peer group in any other peer
            # group that has it as an ancestor.
            apgConfig = Globals.AncestorPgConfigType()
            peerKeys = peer.ancestorPeerGroupConfig.ancestorPeerGroupKeys.values()
            peerKeys = list( peerKeys )
            if config.key in peerKeys:
               peerKeys.remove( config.key )
               # Start key index at 1 due to nominal default key restriction
               apgConfig.ancestorPeerGroupKeys.update( enumerate( peerKeys, 1 ) )
               peer.ancestorPeerGroupConfig = apgConfig

      for vrfConfig in bgpVrfConfigDir.vrfConfig.values():
         for peer in vrfConfig.neighborConfig.values():
            if peer.isPeerGroupPeer and peer.peerGroupKey == config.key:
               peer.peerGroupKey = PeerConfigKeyType()
               RoutingBgpCliHandler.resetPresents( peer )
               RoutingBgpCliHandler. \
                  delNeighborConfigIfDefault( peer.key, vrfName=vrfConfig.name )
      # delete the peer-group config in the end
      del Globals.bgpConfig.neighborConfig[ config.key ]

# class SetNeighborBfdIntervalCmd( BgpCmdBaseClass ):
def SetNeighborBfdIntervalCmd_isSingleAgentAndInteractive( mode ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = 'per peer BFD interval is only supported in multi-agent mode'
      if mode.session.isInteractive():
         mode.addError( msg )
         return True
      mode.addWarning( msg )
   return False

def SetNeighborBfdIntervalCmd_noBfdInterval( mode, peer ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.bfdIntervalConfigPresent = False
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborBfdIntervalCmd_setBfdInterval( mode, peer, minTx, minRx, mult ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.bfdIntervalConfigPresent = True
   config.bfdIntervalConfig = Tac.Value( 'Bfd::BfdIntervalConfig',
                                          minTx, minRx, mult )

def SetNeighborBfdIntervalCmd_handleNoOrDefault( mode, args, noOrDefault ):
   if SetNeighborBfdIntervalCmd_isSingleAgentAndInteractive( mode ):
      return
   SetNeighborBfdIntervalCmd_noBfdInterval( mode, args[ 'PEER' ] )

def SetNeighborBfdIntervalCmd_handleNormal( mode, args ):
   if SetNeighborBfdIntervalCmd_isSingleAgentAndInteractive( mode ):
      return
   SetNeighborBfdIntervalCmd_setBfdInterval( mode,
                                             args[ 'PEER' ],
                                             args[ 'INTERVAL' ],
                                             args[ 'MINRX' ],
                                             args[ 'MULTIPLIER' ] )

def SetNeighborBfdIntervalCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborBfdIntervalCmd", mode, args )

def SetNeighborBfdIntervalCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborBfdIntervalCmd", mode, args )

# class SetNeighborBfdWithDampingCmd( BgpNEDCmdBaseClass ):
def SetNeighborBfdWithDampingCmd_noBfd( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.bfdEnabledStatePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
      config.bfdEnabledState = config.bfdEnabledStateDefault
      config.bfdCBitEnabled = config.bfdCBitEnabledDefault
      config.bfdDampingEnabled = config.bfdDampingEnabledDefault
      config.bfdDampingTime = config.bfdDampingTimeDefault
      config.bfdDampingMultiplier = config.bfdDampingMultiplierDefault
      config.bfdDampingTimePresent = False
      config.bfdDampingMultiplierPresent = False
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborBfdWithDampingCmd_setBfd( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.bfdEnabledStatePresent = True
   config.bfdEnabledState = True
   config.bfdCBitEnabled = 'c-bit' in args
   config.bfdDampingEnabled = 'damping' in args
   config.bfdDampingTimePresent = 'TIME' in args
   config.bfdDampingTime = args.get( 'TIME', config.bfdDampingTimeDefault )
   config.bfdDampingMultiplierPresent = 'MULTIPLIER' in args
   config.bfdDampingMultiplier = args.get( 'MULTIPLIER',
                                           config.bfdDampingMultiplierDefault )

def SetNeighborBfdWithDampingCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborBfdWithDampingCmd_noBfd(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborBfdWithDampingCmd_handleNormal( mode, args ):
   SetNeighborBfdWithDampingCmd_setBfd( mode, args )

def SetNeighborBfdWithDampingCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborBfdWithDampingCmd", mode, args )

def SetNeighborBfdWithDampingCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborBfdWithDampingCmd", mode, args )

# class SetNeighborLinkBwAutoBase( object ):
def SetNeighborLinkBwAutoBase_setLinkBwAuto( mode, peer, percent=None ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if config:
      config.linkBw = LinkBandwidthGenerationStateEnum.linkBandwidthAutoGeneration
      config.linkBwPresent = True
      config.linkBwDef = config.linkBwDefDefault
      if percent:
         config.linkBwAutoPercentPresent = True
         config.linkBwAutoPercent = percent
      else:
         config.linkBwAutoPercentPresent = False
         config.linkBwAutoPercent = config.linkBwAutoPercentDefault
      config.linkBwDefDisplay = config.linkBwDefDisplayDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwAutoBase_noLinkBwAuto( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.linkBwPresent = (
            noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
      config.linkBwAutoPercentPresent = False
      config.linkBw = config.linkBwDefault
      config.linkBwAutoPercent = config.linkBwAutoPercentDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

# class SetNeighborLinkBwAutoPercentCmd( BgpNEDCmdBaseClass ):
def SetNeighborLinkBwAutoPercentCmd_handleNormal( mode, args ):
   SetNeighborLinkBwAutoBase_setLinkBwAuto(
      mode,
      args[ 'PEER' ],
      percent=args.get( 'PERCENT' ) )

def SetNeighborLinkBwAutoPercentCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborLinkBwAutoBase_noLinkBwAuto( mode, args[ 'PEER' ], noOrDefault )

def SetNeighborLinkBwAutoPercentCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLinkBwAutoPercentCmd", mode, args )

def SetNeighborLinkBwAutoPercentCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborLinkBwAutoPercentCmd", mode, args )

# class SetNeighborLinkBwAdjustCmd( BgpNEDCmdBaseClass ):
def SetNeighborLinkBwAdjustCmd_setLinkBwAdjust( mode, peer, percent ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if config:
      if percent:
         config.linkBwAdjustPercentPresent = True
         config.linkBwAdjustPercent = percent
      else:
         config.linkBwAdjustPercentPresent = False
         config.linkBwAdjustPercent = config.linkBwAdjustPercentDefault
      config.linkBwAdjustPresent = True
      config.linkBwAdjust = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwAdjustCmd_noLinkBwAdjust( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.linkBwAdjustPresent = (
            noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
      )
      config.linkBwAdjustPercentPresent = False
      config.linkBwAdjust = config.linkBwAdjustDefault
      config.linkBwAdjustPercent = config.linkBwAdjustPercentDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwAdjustCmd_handleNormal( mode, args ):
   SetNeighborLinkBwAdjustCmd_setLinkBwAdjust(
      mode,
      args[ 'PEER' ],
      args.get( 'PERCENT' ) )

def SetNeighborLinkBwAdjustCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborLinkBwAdjustCmd_noLinkBwAdjust(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborLinkBwAdjustCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLinkBwAdjustCmd", mode, args )

def SetNeighborLinkBwAdjustCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLinkBwAdjustCmd", mode, args )

# class SetNeighborLinkBwCmd( BgpNEDCmdBaseClass ):
def SetNeighborLinkBwCmd_setLinkBw( mode, peer ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if config:
      config.linkBw = LinkBandwidthGenerationStateEnum.linkBandwidthProcessing
      config.linkBwPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwCmd_noLinkBw( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.linkBwPresent = (
            noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
      config.linkBw = config.linkBwDefault
      config.linkBwDef = config.linkBwDefDefault
      config.linkBwAutoPercent = config.linkBwAutoPercentDefault
      config.linkBwDefDisplay = config.linkBwDefDisplayDefault
      config.linkBwAdjustPresent = False
      config.linkBwAdjust = config.linkBwAdjustDefault
      config.linkBwAdjustPercentPresent = False
      config.linkBwAutoPercentPresent = False
      config.linkBwAdjustPercent = config.linkBwAdjustPercentDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwCmd_handleNormal( mode, args ):
   SetNeighborLinkBwCmd_setLinkBw( mode, args[ 'PEER' ] )

def SetNeighborLinkBwCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborLinkBwCmd_noLinkBw( mode, args[ 'PEER' ], noOrDefault )

def SetNeighborLinkBwCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLinkBwCmd", mode, args )

def SetNeighborLinkBwCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLinkBwCmd", mode, args )

# class SetNeighborLinkBwDefaultCmd( BgpNEDCmdBaseClass ):
def SetNeighborLinkBwDefaultCmd_setLinkBwDefault( mode, peer, lbwVal ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if config:
      config.linkBw = (
            LinkBandwidthGenerationStateEnum.linkBandwidthDefaultGeneration )
      config.linkBwPresent = True
      config.linkBwDefDisplay = lbwVal
      config.linkBwDef = getLbwCommValue( lbwVal )
      config.linkBwAutoPercent = config.linkBwAutoPercentDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwDefaultCmd_noLinkBwDefault( mode, peer, noOrDefault ):
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.linkBwPresent = (
            noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
      config.linkBw = config.linkBwDefault
      config.linkBwDef = config.linkBwDefDefault
      config.linkBwDefDisplay = config.linkBwDefDisplayDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborLinkBwDefaultCmd_handleNormal( mode, args ):
   SetNeighborLinkBwDefaultCmd_setLinkBwDefault(
      mode,
      args[ 'PEER' ],
      args[ 'COMM' ] )

def SetNeighborLinkBwDefaultCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborLinkBwDefaultCmd_noLinkBwDefault(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborLinkBwDefaultCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborLinkBwDefaultCmd", mode, args )

def SetNeighborLinkBwDefaultCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborLinkBwDefaultCmd", mode, args )

# class SetNeighborNextHopAfV6OriginateCmd( BgpNEDCmdBaseClass ):
def SetNeighborNextHopAfV6OriginateCmd_setNeighborV6NextHopAf(
      mode, peer, originate ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if mode.addrFamily == 'ipv4':
      if originate:
         config.v6NextHopAfV4Uni = (
               ExtendedNextHopCapabilityEnum.isEnabledWithOriginate )
      else:
         config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isEnabled
      config.v6NextHopAfV4UniPresent = True
   elif mode.addrFamily == 'ipv6':
      mode.addError(
            'IPv6 nexthop address family may not be specified in IPv6 mode' )
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborNextHopAfV6OriginateCmd_noNeighborV6NextHopAf(
      mode, noOrDefault, peer ):
   validatePeer( mode, peer )
   if mode.addrFamily == 'ipv4':
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isDisabled
         config.v6NextHopAfV4UniPresent = ( noOrDefault != NoOrDefault.DEFAULT )
   elif mode.addrFamily == 'ipv6':
      mode.addError(
            'IPv6 nexthop address family may not be specified in IPv6 mode' )
   else:
      raise ValueError()
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborNextHopAfV6OriginateCmd_handleNormal( mode, args ):
   SetNeighborNextHopAfV6OriginateCmd_setNeighborV6NextHopAf(
      mode,
      args[ 'PEER' ],
      ( 'originate' in args ) )

def SetNeighborNextHopAfV6OriginateCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborNextHopAfV6OriginateCmd_noNeighborV6NextHopAf(
      mode,
      noOrDefault,
      args[ 'PEER' ] )

def SetNeighborNextHopAfV6OriginateCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborNextHopAfV6OriginateCmd", mode, args )

def SetNeighborNextHopAfV6OriginateCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborNextHopAfV6OriginateCmd", mode, args )

# class NeighborAigpSessionCommand( BgpNEDCmdBaseClass ):
def NeighborAigpSessionCommand_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      mode.addWarning(
         "AIGP-session is only supported in multi-agent mode" )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'aigpSession' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr + "Present", True )
      setattr( config, attr, True )

def NeighborAigpSessionCommand_handleNoOrDefault( mode, args, noOrDefault ):
   # Explicit 'no' case
   #
   # Note that, because there are per-peer-type globals for
   # aigp-session config, we need to keep the explicit 'no' at the
   # peer level (even when the peer is not in a peer group).
   #
   # There are per-peer-type global configs for aigp-session: one for
   # iBGP peers, one for confed peers, and one for eBGP peers.
   #
   # For example, when the global for iBGP peers is set to true, if
   # the aigp-session for an iBGP peer is not configured at the peer
   # level, the aigp-session is enabled. On the other hand, when 'no'
   # is configured at the peer level, it overrides the global
   # default.
   #
   # In order to override the per-peer-type global defaults, set the
   # config to false explicitly.
   explicitNo = ( noOrDefault == NoOrDefault.NO )
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=explicitNo,
                                 vrfName=mode.vrfName )
   if config:
      attr = BgpLib.peerConfigAttrsAfMap[ 'aigpSession' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr + "Present", explicitNo )
         setattr( config, attr, getattr( config, attr + 'Default' ) )

      if not explicitNo:
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborAigpSessionCommand_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "NeighborAigpSessionCommand", mode, args )

def NeighborAigpSessionCommand_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "NeighborAigpSessionCommand", mode, args )

# class SetNeighborDefaultEncapCmd( BgpCmdBaseClass ):
def SetNeighborDefaultEncapCmd_setDefaultEncap(
      mode, mpls=None, vxlan=None, srv6=None, intf=None, locator=None, dps=None ):
   config = configForVrf( mode.vrfName )
   config.afEvpnMplsNexthopSelfSrcIntf = (
      config.afEvpnMplsNexthopSelfSrcIntfDefault )
   config.afEvpnSrv6Locator = config.afEvpnSrv6LocatorDefault

   if intf:
      config.afEvpnEncap = mpls
      config.afEvpnMplsNexthopSelfSrcIntf = intf
   elif locator:
      config.afEvpnEncap = srv6
      config.afEvpnSrv6Locator = locator
   else:
      config.afEvpnEncap = vxlan or mpls or dps

def SetNeighborDefaultEncapCmd_noDefaultEncap( mode ):
   config = configForVrf( mode.vrfName )
   config.afEvpnEncap = config.afEvpnEncapDefault
   config.afEvpnMplsNexthopSelfSrcIntf = (
         config.afEvpnMplsNexthopSelfSrcIntfDefault )
   config.afEvpnSrv6Locator = config.afEvpnSrv6LocatorDefault

def SetNeighborDefaultEncapCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborDefaultEncapCmd_noDefaultEncap( mode )

def SetNeighborDefaultEncapCmd_handleNormal( mode, args ):
   SetNeighborDefaultEncapCmd_setDefaultEncap(
      mode,
      mpls=args.get( 'mpls' ),
      vxlan=args.get( 'vxlan' ),
      srv6=args.get( 'srv6' ),
      intf=args.get( 'INTF' ),
      locator=args.get( 'LOCATOR' ),
      dps=args.get( 'dps' ), )

def SetNeighborDefaultEncapCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborDefaultEncapCmd", mode, args )

def SetNeighborDefaultEncapCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborDefaultEncapCmd", mode, args )

# class SetNeighborEncapCmd( BgpCmdBaseClass ):
def SetNeighborEncapCmd_setPeerEncap( mode, peer, mpls=None, vxlan=None, intf=None,
                                      dps=None ):
   peerConfig = bgpNeighborConfig( peer, mode.vrfName )
   peerConfig.afEvpnEncapPresent = True
   if intf:
      peerConfig.afEvpnEncap = mpls
      peerConfig.afEvpnMplsNexthopSelfSrcIntf = intf
      peerConfig.afEvpnMplsNexthopSelfSrcIntfPresent = True
   else:
      peerConfig.afEvpnEncap = vxlan or mpls or dps
      peerConfig.afEvpnMplsNexthopSelfSrcIntf = (
            peerConfig.afEvpnMplsNexthopSelfSrcIntfDefault )
      peerConfig.afEvpnMplsNexthopSelfSrcIntfPresent = False

def SetNeighborEncapCmd_noOrDefaultPeerEncap( mode, peer ):
   peerConfig = bgpNeighborConfig( peer, mode.vrfName, create=False )
   if peerConfig:
      peerConfig.afEvpnEncap = peerConfig.afEvpnEncapDefault
      peerConfig.afEvpnEncapPresent = False
      peerConfig.afEvpnMplsNexthopSelfSrcIntf = (
            peerConfig.afEvpnMplsNexthopSelfSrcIntfDefault )
      peerConfig.afEvpnMplsNexthopSelfSrcIntfPresent = False

def SetNeighborEncapCmd_handleNormal( mode, args ):
   SetNeighborEncapCmd_setPeerEncap(
      mode,
      peer=args[ 'PEER' ],
      mpls=args.get( 'mpls' ),
      vxlan=args.get( 'vxlan' ),
      intf=args.get( 'INTF' ),
      dps=args.get( 'dps' ) )

def SetNeighborEncapCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborEncapCmd_noOrDefaultPeerEncap(
      mode,
      peer=args[ 'PEER' ] )

def SetNeighborEncapCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborEncapCmd", mode, args )

def SetNeighborEncapCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborEncapCmd", mode, args )

def SetEvpnNeighborDefaultOismMplsCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afEvpnMplsMulticastPmsiTunnel = 'mLdpP2mpLsp'

def SetEvpnNeighborDefaultOismMplsCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afEvpnMplsMulticastPmsiTunnel = 'noTunnel'

# class PeerOutDelayCmdMixin( object ):
def PeerOutDelayCmdMixin_setOutDelay(
      mode, peer, outDelay, changes=False, withdrawals=False ):
   validatePeer( mode, peer )
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=vrfName )
   if haveConflictingOutDelay( mode, config, addrFamily ):
      return
   attr = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ].get( addrFamily )
   if not outDelay:
      outDelay = getattr( config, attr + 'Default' )
   if changes and withdrawals:
      outDelayApply = OutDelayApplyEnum.outDelayApplyChangesWithdrawals
   elif changes:
      outDelayApply = OutDelayApplyEnum.outDelayApplyChanges
   else:
      outDelayApply = OutDelayApplyEnum.outDelayApplyInitial
   config.outDelayApply = outDelayApply
   setattr( config, attr, outDelay )
   setattr( config, attr + 'Present', True )
   delNeighborConfigIfDefault( peer, vrfName=vrfName )

def PeerOutDelayCmdMixin_noOutDelay( mode, peer, noOrDefault ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=vrfName, create=False )
   attr = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ].get( addrFamily )
   if config:
      if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
         setattr( config, attr + 'Present', True )
      else:
         setattr( config, attr + 'Present', False )
      setattr( config, attr, getattr( config, attr + 'Default' ) )
      config.outDelayApply = OutDelayApplyEnum.outDelayApplyInitial
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

# class SetNeighborOutDelayCmd( PeerOutDelayCmdMixin, BgpNEDCmdBaseClass ):
def SetNeighborOutDelayCmd_handleNormal( mode, args ):
   PeerOutDelayCmdMixin_setOutDelay(
      mode,
      args[ 'PEER' ],
      args[ 'DELAY' ],
      changes=( 'changes' in args ),
      withdrawals=( 'withdrawals' in args ) )

def SetNeighborOutDelayCmd_handleNoOrDefault( mode, args, noOrDefault ):
   PeerOutDelayCmdMixin_noOutDelay( mode, args[ 'PEER' ], noOrDefault )

def SetNeighborOutDelayCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborOutDelayCmd", mode, args )

def SetNeighborOutDelayCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "SetNeighborOutDelayCmd", mode, args )

# class SetNeighborOutDelayHiddenCmd( CliCommand.CliCommandClass ):
def SetNeighborOutDelayHiddenCmd_handler( mode, args ):
   PeerOutDelayCmdMixin_setOutDelay( mode, args[ 'PEER' ], args[ 'DELAY' ] )

def SetNeighborOutDelayHiddenCmd_noHandler( mode, args ):
   PeerOutDelayCmdMixin_noOutDelay( mode, args[ 'PEER' ], NoOrDefault.NO )

def SetNeighborOutDelayHiddenCmd_defaultHandler( mode, args ):
   PeerOutDelayCmdMixin_noOutDelay( mode, args[ 'PEER' ], NoOrDefault.DEFAULT )

# class SetNeighborNextHopSelfAfSrcIntfCmd( BgpNEDCmdBaseClass ):
def SetNeighborNextHopSelfAfSrcIntfCmd_verifyAgentModelForArgs( mode, args ):
   msg = "v4-mapped-v6 is only supported in multi-agent ipv6 labeled-unicast mode"
   if ( 'v4-mapped-v6' in args and mode.addrFamily == 'ipv6 labeled-unicast'
         and getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent ):
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )

def SetNeighborNextHopSelfAfSrcIntfCmd_setNexthopSelfSourceInterface(
      mode, peer, intf, v4MappedV6 ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if mode.addrFamily == 'ipv6 labeled-unicast':
      v4MappedV6Attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfV4MappedV6' ].get(
            mode.addrFamily )
      setattr( config, v4MappedV6Attr, v4MappedV6 )
   attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ].get( mode.addrFamily )
   setattr( config, attr, intf.name )
   setattr( config, attr + 'Present', True )

def SetNeighborNextHopSelfAfSrcIntfCmd_noNexthopSelfSourceInterface(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
   attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ].get( mode.addrFamily )
   if config:
      setattr( config, attr, config.nhSelfSrcIntfDefault )
      if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
         setattr( config, attr + 'Present', True )
      else:
         setattr( config, attr + 'Present', False )

      if mode.addrFamily == 'ipv6 labeled-unicast':
         v4MappedV6Attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfV4MappedV6' ].get(
               mode.addrFamily )
         setattr( config, v4MappedV6Attr, False )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborNextHopSelfAfSrcIntfCmd_handleNormal( mode, args ):
   SetNeighborNextHopSelfAfSrcIntfCmd_verifyAgentModelForArgs( mode, args )
   SetNeighborNextHopSelfAfSrcIntfCmd_setNexthopSelfSourceInterface(
      mode,
      args[ 'PEER' ],
      args[ 'INTF' ],
      'v4-mapped-v6' in args )

def SetNeighborNextHopSelfAfSrcIntfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborNextHopSelfAfSrcIntfCmd_verifyAgentModelForArgs( mode, args )
   SetNeighborNextHopSelfAfSrcIntfCmd_noNexthopSelfSourceInterface(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborNextHopSelfAfSrcIntfCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborNextHopSelfAfSrcIntfCmd", mode, args )

def SetNeighborNextHopSelfAfSrcIntfCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborNextHopSelfAfSrcIntfCmd", mode, args )

# class SetNeighborInterfaceCmd( BgpCmdBaseClass ):
def SetNeighborInterfaceCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )

   # Filter out the non-existing interfaces that the IntfRange matcher reports.
   for intf in IntfCli.Intf.getAll( mode, args[ 'INTFS' ], config=True ):
      intfId = IntfId( intf )
      peerConfig = config.neighIntfConfig.get( intfId )
      if not peerConfig:
         peerConfig = config.neighIntfConfig.newMember( intfId )
      asnOrPeerFilter = Tac.Value( "Routing::Bgp::AsnOrPeerFilter" )
      if 'AS_NUM' in args:
         asnOrPeerFilter.asn = args[ 'AS_NUM' ]
      elif 'PEER_FILTER' in args:
         asnOrPeerFilter.hasAsn = False
         asnOrPeerFilter.peerFilter = args[ 'PEER_FILTER' ]
      else:
         asnOrPeerFilter.hasAsn = False
         asnOrPeerFilter.asn = 0
         asnOrPeerFilter.peerFilter = ""
      peerConfig.asnOrPeerFilter = asnOrPeerFilter
      peerConfig.peerGroup = PeerConfigKey( args[ 'PG' ] )

   protocolModel = getEffectiveProtocolModel( mode )
   if( protocolModel != ProtoAgentModel.multiAgent and 'PEER_FILTER' in args ):
      if mode.session.isInteractive():
         mode.addError( "peer-filter is only supported in multi-agent mode" )
      return
   if( protocolModel == ProtoAgentModel.ribd and 'AS_NUM' not in args ):
      if mode.session.isInteractive():
         mode.addError( "remote-as option is required with ribd routing protocol"
                        " model" )
      return

   bgpNeighborConfig( PeerConfigKey( args[ 'PG' ] ), vrfName=mode.vrfName )

def SetNeighborInterfaceCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )

   for intf in args[ 'INTFS' ]:
      del config.neighIntfConfig[ intf ]

def SetNeighborInterfaceCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborInterfaceCmd", mode, args )

def SetNeighborInterfaceCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborInterfaceCmd", mode, args )

# class RouteRefreshConfigCmd( BgpNEDCmdBaseClass ):
def RouteRefreshConfigCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   timeoutSecs = args.get( 'TIMEOUT' )
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.enhRtRefreshIn = True
   config.enhRtRefreshInPresent = True
   if timeoutSecs is not None:
      # '0' is a valid value
      config.enhRtRefreshStalePathTimePresent = True
      config.enhRtRefreshStalePathTime = timeoutSecs
   else:
      config.enhRtRefreshStalePathTimePresent = False
      config.enhRtRefreshStalePathTime = config.enhRtRefreshStalePathTimeDefault
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def RouteRefreshConfigCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if not config:
      return
   # 'peer' can be either a peer, peer group, or a peer group peer
   # (a peer group member).
   #
   # For a peer group peer, "default" means fall back to using
   # the peer group's config, so 'enhRtRefreshInPresent' # is set to False.
   # "no" means explicitly disabling the feature, regardless of the peer
   # group configuration. For this reason, 'enhRtRefreshInPresent' is set to
   # True and 'enhRtRefreshIn' is set to False.
   if noOrDefault == NoOrDefault.NO and config.isPeerGroupPeer:
      # Peer group peer: explicitly disabled.
      config.enhRtRefreshInPresent = True
      config.enhRtRefreshIn = False
   else:
      # In all other cases, simply remove the configuration.
      config.enhRtRefreshInPresent = False
      config.enhRtRefreshIn = config.enhRtRefreshInDefault
   config.enhRtRefreshStalePathTimePresent = False
   config.enhRtRefreshStalePathTime = config.enhRtRefreshStalePathTimeDefault
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def RouteRefreshConfigCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "RouteRefreshConfigCmd", mode, args )

def RouteRefreshConfigCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "RouteRefreshConfigCmd", mode, args )

# class SetNeighborNextHopUnchangedCmd( BgpNEDCmdBaseClass ):
def SetNeighborNextHopUnchangedCmd_setNeighborNexthopUnchanged( mode, peer ):
   validatePeer( mode, peer )
   addrFamily = mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
   setattr( config, attr, True )
   setattr( config, attr + 'Present', True )

def SetNeighborNextHopUnchangedCmd_noNeighborNexthopUnchanged(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   addrFamily = mode.addrFamily
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName,
                               create=( noOrDefault != NoOrDefault.DEFAULT ) )
   attr = BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
   if config:
      if noOrDefault != NoOrDefault.DEFAULT:
         setattr( config, attr + 'Present', True )
         setattr( config, attr, False )
      else:
         setattr( config, attr + 'Present', False )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborNextHopUnchangedCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborNextHopUnchangedCmd_noNeighborNexthopUnchanged(
      mode,
      args[ 'PEER' ],
      noOrDefault )

def SetNeighborNextHopUnchangedCmd_handleNormal( mode, args ):
   SetNeighborNextHopUnchangedCmd_setNeighborNexthopUnchanged(
      mode,
      args[ 'PEER' ] )

def SetNeighborNextHopUnchangedCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborNextHopUnchangedCmd", mode, args )

def SetNeighborNextHopUnchangedCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborNextHopUnchangedCmd", mode, args )

# class NeighborRouteTargetExportFilterDisabledCmd( BgpCmdBaseClass ):
def NeighborRouteTargetExportFilterDisabledCmd_setFilterDisabled(
      mode, peer, vpnNlriType, value ):
   collectionType = 'vrfAf' if mode.addrFamily != 'all' else 'vrf'
   validatePeer( mode, peer )
   if collectionType == 'vrfAf':
      ipVersion = int( mode.addrFamily[ -1 ] )
      if vpnNlriType.ipVersion != ipVersion:
         mode.addError( '%s export is not applicable for ipv%d routes' %
               ( bgpTokens.vpnNlriTypeKeyword( vpnNlriType ), ipVersion ) )
         return
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   coll = { 'vrf': config.routeTargetExportFilterDisabled,
            'vrfAf': config.routeTargetExportFilterDisabledAf }[ collectionType ]
   if value == 'isInvalid':
      del coll[ vpnNlriType ]
   else:
      coll[ vpnNlriType ] = value

def NeighborRouteTargetExportFilterDisabledCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   vpnNlriType = args[ 'VPN_NLRI_TYPE' ]
   NeighborRouteTargetExportFilterDisabledCmd_setFilterDisabled(
      mode, peer, vpnNlriType, 'isFalse' )

def NeighborRouteTargetExportFilterDisabledCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   vpnNlriType = args[ 'VPN_NLRI_TYPE' ]
   NeighborRouteTargetExportFilterDisabledCmd_setFilterDisabled(
      mode, peer, vpnNlriType,
      'isTrue' if noOrDefault == NoOrDefault.NO else 'isInvalid' )

def NeighborRouteTargetExportFilterDisabledCmd_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "NeighborRouteTargetExportFilterDisabledCmd", mode, args )

def NeighborRouteTargetExportFilterDisabledCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "NeighborRouteTargetExportFilterDisabledCmd", mode, args )

# class NeighborNexthopLuOriginateCmd( BgpCmdBaseClass ):
def NeighborNexthopLuOriginateCmd_setNeighborNexthopLuOriginate(
      mode, peer, thirdParty ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   addrFamily = mode.addrFamily
   afAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ].get( addrFamily )
   setattr( config, afAttr, not thirdParty )
   setattr( config, afAttr + 'Present', True )
   afDynAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginateDyn' ].get(
         addrFamily )
   setattr( config, afDynAttr, thirdParty )

def NeighborNexthopLuOriginateCmd_noNeighborNexthopLuOriginate(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                 vrfName=mode.vrfName )
   if config:
      addrFamily = mode.addrFamily
      afAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ].get(
                  addrFamily )
      setattr( config, afAttr, False )
      setattr( config, afAttr + 'Present', noOrDefault != NoOrDefault.DEFAULT )
      afDynAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginateDyn' ].get(
            addrFamily )
      setattr( config, afDynAttr, False )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborNexthopLuOriginateCmd_handleNormal( mode, args ):
   NeighborNexthopLuOriginateCmd_setNeighborNexthopLuOriginate(
      mode, args[ 'PEER' ], 'third-party' in args )

def NeighborNexthopLuOriginateCmd_handleNoOrDefault( mode, args, noOrDefault ):
   NeighborNexthopLuOriginateCmd_noNeighborNexthopLuOriginate(
      mode, args[ 'PEER' ], noOrDefault )

def NeighborNexthopLuOriginateCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborNexthopLuOriginateCmd", mode, args )

def NeighborNexthopLuOriginateCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "NeighborNexthopLuOriginateCmd", mode, args )

#  class NeighborNexthopLuPeerSetOriginateCmd( BgpCmdBaseClass ):
def NeighborNexthopLuPeerSetOriginateCmd_setNeighborNexthopLuPeerSetOriginate(
      mode, peer, peerSet ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   addrFamily = mode.addrFamily
   peerSetAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSet' ].\
               get( addrFamily )
   peerSetOrigAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSetOriginate' ].\
               get( addrFamily )
   if getattr( config, peerSetAttr + 'Present' ) and \
      getattr( config, peerSetAttr ) != peerSet:
      mode.addError( "An EPE peer can be configured to one peer set only" )
      return
   setattr( config, peerSetAttr, peerSet )
   setattr( config, peerSetAttr + 'Present', True )
   setattr( config, peerSetOrigAttr, True )

def NeighborNexthopLuPeerSetOriginateCmd_noNeighborNexthopLuPeerSetOriginate(
      mode, peer, peerSet, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                 vrfName=mode.vrfName )
   if config:
      addrFamily = mode.addrFamily
      peerSetAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSet' ].\
                  get( addrFamily )
      peerSetOrigAttr = \
         BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSetOriginate' ].\
                  get( addrFamily )

      # NoOrDefault.DEFAULT:
      #   'default neighbor PEER next-hop labeled-unicast peer-set PEER_SET
      #      originate' being executed
      #   'no neighbor PEER next-hop labeled-unicast peer-set PEER_SET originate'
      #      being executed
      # NoOrDefault.No:
      #   'neighbor PEER next-hop labeled-unicast peer-set PEER_SET originate
      #      disabled' being executed
      if noOrDefault == NoOrDefault.DEFAULT:
         peerSetAttrPresent = False
         peerSetValue = ''
         peerSetOrigValue = False
      elif noOrDefault == NoOrDefault.NO:
         peerSetAttrPresent = True
         peerSetOrigValue = False
         peerSetValue = peerSet

      setattr( config, peerSetAttr, peerSetValue )
      setattr( config, peerSetAttr + 'Present', peerSetAttrPresent )
      setattr( config, peerSetOrigAttr, peerSetOrigValue )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborNexthopLuPeerSetOriginateCmd_handleNormal( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "EPE peer set feature is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )
   NeighborNexthopLuPeerSetOriginateCmd_setNeighborNexthopLuPeerSetOriginate(
      mode, args[ 'PEER' ], args[ 'PEER_SET' ] )

def NeighborNexthopLuPeerSetOriginateCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "EPE peer set feature is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )
   NeighborNexthopLuPeerSetOriginateCmd_noNeighborNexthopLuPeerSetOriginate(
      mode, args[ 'PEER' ], args[ 'PEER_SET' ], noOrDefault )

def NeighborNexthopLuPeerSetOriginateCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborNexthopLuPeerSetOriginateCmd", mode, args )

def NeighborNexthopLuPeerSetOriginateCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "NeighborNexthopLuPeerSetOriginateCmd", mode, args )

# class NeighborNhLuOrigLfibBackupIpFwdCmd( BgpCmdBaseClass ):
def NeighborNhLuOrigLfibBackupIpFwdCmd_setNeighborNhLuOrigLfibBackupIpFwd(
      mode, peer ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   addrFamily = mode.addrFamily
   afAttr = BgpLib.peerConfigAttrsAfMap[ 'nhLuOrigLfibBackupIpFwd' ].\
                                                   get( addrFamily )
   setattr( config, afAttr, True )
   setattr( config, afAttr + 'Present', True )

def NeighborNhLuOrigLfibBackupIpFwdCmd_noNeighborNhLuOrigLfibBackupIpFwd(
      mode, peer, noOrDefault ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                 vrfName=mode.vrfName )
   if config:
      addrFamily = mode.addrFamily
      afAttr = BgpLib.peerConfigAttrsAfMap[ 'nhLuOrigLfibBackupIpFwd' ].\
                                                      get( addrFamily )
      setattr( config, afAttr, False )
      # The optimization of treating explicit disable for non peer-group
      # peers as implicit disable (described before the class definition
      # of BgpCmdBaseClass in RoutingBgpCli.py) cannot be applied here.
      # The reason is that the attribute 'nhLuOrigLfibBackupIpFwd' is
      # configurable at both the neighbor and instance levels.
      setattr( config, afAttr + 'Present', noOrDefault != NoOrDefault.DEFAULT )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborNhLuOrigLfibBackupIpFwdCmd_handleNormal( mode, args ):
   NeighborNhLuOrigLfibBackupIpFwdCmd_setNeighborNhLuOrigLfibBackupIpFwd(
      mode, args[ 'PEER' ] )

def NeighborNhLuOrigLfibBackupIpFwdCmd_handleNoOrDefault( mode, args, noOrDefault ):
   NeighborNhLuOrigLfibBackupIpFwdCmd_noNeighborNhLuOrigLfibBackupIpFwd(
      mode, args[ 'PEER' ], noOrDefault )

def NeighborNhLuOrigLfibBackupIpFwdCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborNhLuOrigLfibBackupIpFwdCmd", mode, args )

def NeighborNhLuOrigLfibBackupIpFwdCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "NeighborNhLuOrigLfibBackupIpFwdCmd", mode, args )

# class NeighborAsSetsCmd( BgpCmdBaseClass ):
def NeighborAsSetsCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   # If other actions besides treat-as-withdraw are supported in future, must catch
   # action in if statements as well
   if mode.addrFamily == 'ipv4':
      config.rejectAsSetsV4Uni = True
      config.rejectAsSetsV4UniPresent = True
   elif mode.addrFamily == 'ipv6':
      config.rejectAsSetsV6Uni = True
      config.rejectAsSetsV6UniPresent = True

def NeighborAsSetsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   # No and default behave the same (do not reject)
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
   if config:
      if mode.addrFamily == 'ipv4':
         config.rejectAsSetsV4Uni = config.rejectAsSetsV4UniDefault
         config.rejectAsSetsV4UniPresent = False
      elif mode.addrFamily == 'ipv6':
         config.rejectAsSetsV6Uni = config.rejectAsSetsV6UniDefault
         config.rejectAsSetsV6UniPresent = False
      RoutingBgpCliHandler.delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def NeighborAsSetsCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      NeighborAsSetsCmd_handleNormal,
      NeighborAsSetsCmd_handleNoOrDefault,
      mode,
      args )

def NeighborAsSetsCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      NeighborAsSetsCmd_handleNoOrDefault,
      mode,
      args )

# class NoNeighborCmd( CliCommand.CliCommandClass ):
def NoNeighborCmd_noOrDefaultHandler( mode, args ):
   config = bgpNeighborConfig( args[ 'PEER' ], create=False,
                                 vrfName=mode.vrfName )
   if config:
      del configForVrf( mode.vrfName ).neighborConfig[ config.key ]

# class SetNeighborSoftReconfigInboundCmd( BgpNEDCmdBaseClass ):
def SetNeighborSoftReconfigInboundCmd_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   allRoutes = 'all' in args
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   if allRoutes:
      config.softReconfigInbound = SoftReconfigInboundStateEnum.sciAll
      config.softReconfigInboundPresent = True
   else:
      config.softReconfigInbound = SoftReconfigInboundStateEnum.sciNormal
      # Setting only needs to be marked present if this is a peer-group peer
      # because it may override the peer-group setting.
      config.softReconfigInboundPresent = config.isPeerGroupPeer
   delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborSoftReconfigInboundCmd_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer,
                                 create=( noOrDefault != NoOrDefault.DEFAULT ),
                                 vrfName=mode.vrfName )
   if config:
      if noOrDefault == NoOrDefault.DEFAULT:
         config.softReconfigInbound = config.softReconfigInboundDefault
         config.softReconfigInboundPresent = False
      else:
         config.softReconfigInbound = SoftReconfigInboundStateEnum.sciNone
         config.softReconfigInboundPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def SetNeighborSoftReconfigInboundCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborSoftReconfigInboundCmd", mode, args )

def SetNeighborSoftReconfigInboundCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborSoftReconfigInboundCmd", mode, args )

# class NeighborV4MappedV6NextHopTranslationCommand( BgpCmdBaseClass ):
def NeighborV4MappedV6NextHopTranslationCommand_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "v4-mapped-v6 translation is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )
   af = mode.addrFamily
   attr = 'v4MappedV6NextHopTranslation'
   attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
   if not attr:
      mode.addError( "Invalid address family %s" % peer )
      return
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   setattr( config, attr + "Present", True )
   setattr( config, attr, True )

def NeighborV4MappedV6NextHopTranslationCommand_handleNoOrDefault(
      mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "v4-mapped-v6 translation is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )
   af = mode.addrFamily
   attr = 'v4MappedV6NextHopTranslation'
   attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
   if not attr:
      mode.addError( "Invalid address family %s" % peer )
      return
   if noOrDefault == NoOrDefault.DEFAULT:
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         setattr( config, attr + "Present", False )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )
   else:
      # Explicitly "disabled"
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      setattr( config, attr + "Present", True )
      setattr( config, attr, False )

def NeighborV4MappedV6NextHopTranslationCommand_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "NeighborV4MappedV6NextHopTranslationCommand", mode, args )

def NeighborV4MappedV6NextHopTranslationCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "NeighborV4MappedV6NextHopTranslationCommand", mode, args )

# class NeighborRFDCmd( BgpCmdBaseClass ):
def NeighborRFDCmd_handleNormal( mode, args ):
   addrFamily = mode.addrFamily
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   emitWarningIfRfdNotSupportedInRibdMode( mode )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ "rfdPolicyAttachment" ].get( addrFamily )
   rfdPolicyName = ""
   rfdMonitorOnly = False
   rfdPolicyName = args[ "RFD_POLICY_NAME" ]
   rfdMonitorOnly = 'monitor-only' in args
   rfdLogOptions = args.get( "RFD_LOG_OPTIONS", [] )
   rfdLogPathEvents = 'path-events' in rfdLogOptions
   rfdLogConfigEvents = 'config-events' in rfdLogOptions
   rfdPolicy = Tac.Value( "Routing::Bgp::RfdPolicyAttachment",
      rfdPolicyName, rfdMonitorOnly, rfdLogPathEvents, rfdLogConfigEvents )
   setattr( config, attr, rfdPolicy )
   setattr( config, attr + "Present", True )

def NeighborRFDCmd_handleNoOrDefault( mode, args, noOrDefault ):
   addrFamily = mode.addrFamily
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   emitWarningIfRfdNotSupportedInRibdMode( mode )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ "rfdPolicyAttachment" ].get( addrFamily )
   if config:
      rfdPolicyDefault = getattr( config, attr + "Default" )
      setattr( config, attr, rfdPolicyDefault )
      if noOrDefault == NoOrDefault.DEFAULT:
         # no or default case
         setattr( config, attr + "Present", False )
      else:
         # disabled case
         setattr( config, attr + "Present", True )

def NeighborRFDCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborRFDCmd", mode, args )

def NeighborRFDCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "NeighborRFDCmd", mode, args )

# class NeighborMultipathDisabledCmd( BgpCmdBaseClass ):
def NeighborMultipathDisabledCmd_handleNormal( mode, args ):
   addrFamily = mode.addrFamily
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = ( "Enabling/disabling neighbor multi-path is only supported in "
              "multi-agent mode" )
      if mode.session.isInteractive():
         mode.addError( msg )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'multipath' ].get( addrFamily )
   setattr( config, attr, True )
   setattr( config, attr + 'Present', True )

def NeighborMultipathDisabledCmd_handleNoOrDefault( mode, args, noOrDefault ):
   addrFamily = mode.addrFamily
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = ( "Enabling/disabling neighbor multi-path is only supported in "
              "multi-agent mode" )
      if mode.session.isInteractive():
         mode.addError( msg )
   config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                 vrfName=mode.vrfName )
   if config:
      attr = BgpLib.peerConfigAttrsAfMap[ 'multipath' ].get( addrFamily )
      if noOrDefault == NoOrDefault.DEFAULT:
         # Multipath is enabled by default
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         setattr( config, attr + 'Present', False )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )
      else:
         setattr( config, attr, False )
         setattr( config, attr + 'Present', True )

def NeighborMultipathDisabledCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "NeighborMultipathDisabledCmd", mode, args )

def NeighborMultipathDisabledCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "NeighborMultipathDisabledCmd", mode, args )

# Enable/disable generation of a BGP-LS EPE PeerNode segment for a peer (see
# AID11677).
def setNeighborPeerSegmentNode( mode, args, enabled=True, noOrDefault=None ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "Link State peer segments are only supported in multi-agent routing" \
            " protocol mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )

   # This command is only available in the default VRF.
   assert mode.vrfName == DEFAULT_VRF

   peer = args[ 'PEER' ]

   # Grab the config for the peer - only create it if we need to explicitly
   # enable/disable the PeerNode segment.
   createConfig = enabled or noOrDefault != NoOrDefault.DEFAULT
   peerConfig = bgpNeighborConfig( peer, mode.vrfName, create=createConfig )
   if not peerConfig:
      return

   attr = 'linkStateEpePeerSegmentNode'
   attrDefault = f'{attr}Default'
   attrPresent = f'{attr}Present'

   if enabled:
      # Config should be present and explicitly enabled.
      setattr( peerConfig, attr, True )
      setattr( peerConfig, attrPresent, True )
   elif noOrDefault == NoOrDefault.NO:
      # Config should be present, but explicitly disabled.
      setattr( peerConfig, attr, False )
      setattr( peerConfig, attrPresent, True )
   else:
      # Config should not be present (i.e. default state).
      setattr( peerConfig, attr, getattr( peerConfig, attrDefault ) )
      setattr( peerConfig, attrPresent, False )

def SetNeighborPeerSegmentNode_handleNormal( mode, args ):
   setNeighborPeerSegmentNode( mode, args, enabled=True )

def SetNeighborPeerSegmentNode_handleNoOrDefault( mode, args, noOrDefault ):
   setNeighborPeerSegmentNode( mode, args, enabled=False, noOrDefault=noOrDefault )

def SetNeighborPeerSegmentNode_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNeighborPeerSegmentNode", mode, args )

def SetNeighborPeerSegmentNode_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNeighborPeerSegmentNode", mode, args )

# Delay inbound route processing
def setRibInDelay( mode, args, enabled=True, noOrDefault=None ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      msg = "Rib-in delay is only supported in multi-agent routing" \
            " protocol mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )

   peer = args[ 'PEER' ]

   # Grab the config for the peer - only create it if we need to explicitly
   # set the rib-in delay.
   createConfig = enabled or noOrDefault != NoOrDefault.DEFAULT
   peerConfig = bgpNeighborConfig( peer, mode.vrfName, create=createConfig )
   if not peerConfig:
      return

   attr = 'ribInDelay'
   attrDefault = f'{attr}Default'
   attrPresent = f'{attr}Present'

   if enabled:
      # Config should be present and the in-delay is explicitly set.
      delay = args[ 'DELAY' ]
      setattr( peerConfig, attr, delay )
      setattr( peerConfig, attrPresent, True )
   elif noOrDefault == NoOrDefault.NO:
      # This is the explicit "disabled" case. Just set to the default.
      setattr( peerConfig, attr, getattr( peerConfig, attrDefault ) )
      setattr( peerConfig, attrPresent, True )
   else:
      # Config should not be present (i.e. default state).
      setattr( peerConfig, attr, getattr( peerConfig, attrDefault ) )
      setattr( peerConfig, attrPresent, False )

def RibInDelayCmd_handleNormal( mode, args ):
   setRibInDelay( mode, args, enabled=True )

def RibInDelayCmd_handleNoOrDefault( mode, args, noOrDefault ):
   setRibInDelay( mode, args, enabled=False, noOrDefault=noOrDefault )

def RibInDelayCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RibInDelayCmd", mode, args )

def RibInDelayCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RibInDelayCmd", mode, args )

# Command for recevied BGP attribute filtering - see aid/10304
def setNeighborRcvdBgpAttributeFilter( mode, args, enabled=True, noOrDefault=None ):
   peer = args.get( 'PEER' )
   if peer is None:
      # this command is the "neighbor default ..." variant
      config = configForVrf( mode.vrfName )
      warningPeerToken = '*'
      shouldSetDisabled = mode.vrfName != DEFAULT_VRF
   else:
      # otherwise it is specific to a peer or peer group
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      warningPeerToken = peer.stringValue
      if peer.type == 'peerGroup':
         warningPeerToken = f'peer-group {warningPeerToken}'
      shouldSetDisabled = True
   if mode.vrfName != DEFAULT_VRF:
      warningPeerToken += f' vrf {mode.vrfName}'
   else:
      warningPeerToken += ' vrf all'

   mode.addWarning( f'To make this command effective for established peerings, '
      f'reset the session(s) by issuing "clear ip bgp {warningPeerToken}"' )

   disabled = 'disabled' in args

   if enabled:
      bgpAttrs = args.get( 'BGP_ATTRIBUTES_SET' )
      config.rcvdBgpAttrDiscard.clear()
      config.rcvdBgpAttrDiscard.update( dict.fromkeys( bgpAttrs, True ) )
      if shouldSetDisabled:
         config.rcvdBgpAttrDiscardDisabled = False
   else:
      config.rcvdBgpAttrDiscard.clear()
      if shouldSetDisabled:
         config.rcvdBgpAttrDiscardDisabled = disabled

def SetNbrRcvdBgpAttrFilteringCmd_handleNormal( mode, args ):
   setNeighborRcvdBgpAttributeFilter( mode, args, enabled=True )

def SetNbrRcvdBgpAttrFilteringCmd_handleNoOrDefault( mode, args, noOrDefault ):
   setNeighborRcvdBgpAttributeFilter( mode, args, enabled=False )

def SetNbrRcvdBgpAttrFilteringCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNbrRcvdBgpAttrFilteringCmd", mode, args )

def SetNbrRcvdBgpAttrFilteringCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNbrRcvdBgpAttrFilteringCmd", mode, args )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpVrfConfigDir
   global configTagConfig
   global configTagInput

   bgpVrfConfigDir = ConfigMount.mount(
      entityManager, 'routing/bgp/vrf/config', 'Routing::Bgp::VrfConfigDir', 'w' )
   configTagConfig = ConfigMount.mount(
      entityManager, 'configTag/config', 'ConfigTag::ConfigTagConfig', 'w' )
   configTagInput = ConfigMount.mount(
      entityManager, 'configTag/input/cli', 'ConfigTag::ConfigTagInput', 'w' )
