#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from TypeFuture import TacLazyType
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.TunnelCli import isDyTunIntfId
from CliPlugin.TunnelCli import appendEntryLabelsToViaModels, getDyTunTidFromIntfId
from CliPlugin.TunnelCli import getTunnelIndexFromId, getViaModels
from CliPlugin.TunnelCli import (
   readMountTunnelTable,
   TunnelTableIdentifier,
)
from CliPlugin.TunnelFibModel import (
   getFecFromIntfId,
)
from CliPlugin.TunnelModels import (
   IpVia,
)
import SmashLazyMount
from TunnelLib import getTunnelIdWithAf
import six

readerInfo = SmashLazyMount.mountInfo( 'reader' )

bgpLuTunnelTable = None
bgpLuForwardingTunnelTable = None
bgpUdpTunnelTable = None
BgpCliModels = CliDynamicPlugin( "BgpCliModels" )

FecIdIntfId = TacLazyType( "Arnet::FecIdIntfId" )
MetricTypeString = TacLazyType( "Arnet::MetricTypeString" )

# -------------------------------------------------------------------------------
# "show bgp labeled-unicast tunnel [<tunnel-index>]"
#
# HIDDEN
# "show bgp labeled-unicast tunnel [<tunnel-index>] raw"
# -------------------------------------------------------------------------------

def getViaModelList( bgpTunnelTableEntry, flattenLuForwarding, labels ):
   vias = []
   useMplsVias = False
   if flattenLuForwarding:
      for via in six.itervalues( bgpTunnelTableEntry.via ):
         if isDyTunIntfId( via.intfId ):
            tid = getDyTunTidFromIntfId( via.intfId )
            tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tid )
            TunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
            if tacTid.tunnelType() == TunnelType.bgpLuForwardingTunnel:
               useMplsVias = True
   for via in six.itervalues( bgpTunnelTableEntry.via ):
      viaModels = getViaModels( via, bgpLuForwardingTunnelTable,
                                flattenLuForwarding=flattenLuForwarding,
                                useMplsVias=useMplsVias )
      if viaModels:
         vias.extend( appendEntryLabelsToViaModels( viaModels, labels ) )
   return vias

def getBgpLuTunnelTableEntryModel( tunnelId, flattenLuForwarding=True ):
   vias = []

   bgpLuTunnelTableEntry = bgpLuTunnelTable.entry.get( tunnelId )
   if bgpLuTunnelTableEntry:
      labels = []
      for mplsStackIndex in\
            reversed( range( bgpLuTunnelTableEntry.labels.stackSize ) ):
         label = bgpLuTunnelTableEntry.labels.labelStack( mplsStackIndex )
         labels.append( str( label ) )
      vias = getViaModelList( bgpLuTunnelTableEntry, flattenLuForwarding, labels )
      bgpMetricType = MetricTypeString().metricTypeCode(
         bgpLuTunnelTableEntry.bgpMetricType )
      return BgpCliModels.BgpLuTunnelTableEntry(
         endpoint=bgpLuTunnelTableEntry.tep, vias=vias, labels=labels,
         contribution=bgpLuTunnelTableEntry.contribution,
         bgpMetric=bgpLuTunnelTableEntry.bgpMetric,
         bgpMetric2=bgpLuTunnelTableEntry.bgpMetric2,
         bgpMetricType=bgpMetricType,
         bgpPref=bgpLuTunnelTableEntry.bgpPref,
         bgpPref2=bgpLuTunnelTableEntry.bgpPref2 )
   return None

def getBgpLuTunnelTable( tunnelIndex=None, flattenLuForwarding=True ):
   bgpLuEntries = {}

   if tunnelIndex is None:
      for tunnelId in bgpLuTunnelTable.entry:
         bgpLuEntryModel = getBgpLuTunnelTableEntryModel(
            tunnelId=tunnelId, flattenLuForwarding=flattenLuForwarding )
         if bgpLuEntryModel:
            bgpLuEntries[ getTunnelIndexFromId( tunnelId ) ] = bgpLuEntryModel
   else:
      tunnelId = getTunnelIdWithAf( tunnelIndex, 'bgpLuTunnel',
                                    bgpLuTunnelTable )
      bgpLuEntryModel = getBgpLuTunnelTableEntryModel(
         tunnelId=tunnelId,
         flattenLuForwarding=flattenLuForwarding )
      if bgpLuEntryModel:
         bgpLuEntries[ getTunnelIndexFromId( tunnelId ) ] = bgpLuEntryModel

   return BgpCliModels.BgpLuTunnelTable( entries=bgpLuEntries )

def showBgpLuTunnelTable( mode, tunnelIndex=None ):
   return getBgpLuTunnelTable( tunnelIndex=tunnelIndex, flattenLuForwarding=True )

def showBgpLuTunnelTableNoFlattening( mode, tunnelIndex=None ):
   return getBgpLuTunnelTable( tunnelIndex=tunnelIndex, flattenLuForwarding=False )

# -------------------------------------------------------------------------------
# HIDDEN:
# "show bgp labeled-unicast forwarding [<tunnel-index>]"
# "show bgp labeled-unicast forwarding [<tunnel-index>] raw"
# -------------------------------------------------------------------------------
def getBgpLuForwardingTunnelTableEntryModel( tunnelId, flattenLuForwarding=True ):
   vias = []

   bgpLuForwardingTunnelTableEntry = bgpLuForwardingTunnelTable.entry.get( tunnelId )
   if bgpLuForwardingTunnelTableEntry:
      labels = []
      for mplsStackIndex in\
            reversed( range( bgpLuForwardingTunnelTableEntry.labels.stackSize ) ):
         label = bgpLuForwardingTunnelTableEntry.labels.labelStack( mplsStackIndex )
         labels.append( str( label ) )
      vias = getViaModelList( bgpLuForwardingTunnelTableEntry, flattenLuForwarding,
                              labels )
      return BgpCliModels.BgpLuForwardingTunnelTableEntry( vias=vias,
                                                                  labels=labels )
   return None

def getBgpLuForwardingTunnelTable( tunnelIndex=None, flattenLuForwarding=True ):
   bgpLuForwardingEntries = {}

   if tunnelIndex is None:
      for tunnelId in bgpLuForwardingTunnelTable.entry:
         bgpLuForwardingEntryModel = getBgpLuForwardingTunnelTableEntryModel(
            tunnelId=tunnelId,
            flattenLuForwarding=flattenLuForwarding )
         if bgpLuForwardingEntryModel:
            bgpLuForwardingEntries[ getTunnelIndexFromId( tunnelId ) ] = \
               bgpLuForwardingEntryModel
   else:
      tunnelId = getTunnelIdWithAf( tunnelIndex, 'bgpLuForwardingTunnel',
                                    bgpLuForwardingTunnelTable )
      bgpLuForwardingEntryModel = getBgpLuForwardingTunnelTableEntryModel(
         tunnelId=tunnelId, flattenLuForwarding=flattenLuForwarding )
      if bgpLuForwardingEntryModel:
         bgpLuForwardingEntries[ getTunnelIndexFromId( tunnelId ) ] = \
            bgpLuForwardingEntryModel

   return BgpCliModels.BgpLuForwardingTunnelTable(
      entries=bgpLuForwardingEntries )

def showBgpLuForwardingTunnelTable( mode, tunnelIndex=None ):
   return getBgpLuForwardingTunnelTable( tunnelIndex=tunnelIndex,
                                         flattenLuForwarding=True )

def showBgpLuForwardingTunnelTableNoFlattening( mode, tunnelIndex=None ):
   return getBgpLuForwardingTunnelTable( tunnelIndex=tunnelIndex,
                                         flattenLuForwarding=False )

# -------------------------------------------------------------------------------
# "show bgp udp tunnel"
# -------------------------------------------------------------------------------
def payloadTypeToStr( payloadType ):
   if payloadType == 'ipvx':
      return "IP"
   elif payloadType == 'ipv4':
      return "IPv4"
   elif payloadType == 'ipv6':
      return "IPv6"
   elif payloadType == 'mpls':
      return "MPLS"
   return None

def getViaModelsFromTunnelVia( tunnelVia ):
   tunnelVias = []
   # Handle HFEC entry
   if FecIdIntfId.isFecIdIntfId( tunnelVia.intfId ):
      fec = getFecFromIntfId( tunnelVia.intfId )
      if fec is None:
         return None
      for fecVia in fec.via.values():
         addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
         tunnelVias.append(
            IpVia( nexthop=addr, interface=fecVia.intfId, type='ip' ) )

   # Handle entry that includes vias within it
   else:
      tunnelVias.append(
         IpVia( nexthop=tunnelVia.nexthop, interface=tunnelVia.intfId, type='ip' ) )
   return tunnelVias

def getBgpUdpTunnelTableEntryModel( tunnelId ):
   vias = []
   endpoint = sourceAddr = payloadType = ''
   tos = 0

   bgpUdpTunnelTableEntry = bgpUdpTunnelTable.entry.get( tunnelId )
   if bgpUdpTunnelTableEntry:
      for tunnelVia in bgpUdpTunnelTableEntry.tunnelVia.values():
         tunnelVias = getViaModelsFromTunnelVia( tunnelVia )
         if not tunnelVias:
            continue
         gueTunnelEncap = bgpUdpTunnelTable.gueTunnelEncap.get( tunnelVia.encapId )
         if gueTunnelEncap:
            gueTunnelInfo = gueTunnelEncap.encapInfo
            endpoint = gueTunnelInfo.dst
            sourceAddr = gueTunnelInfo.src
            tos = int( gueTunnelInfo.tos )
            payloadType = gueTunnelInfo.payloadType
            vias.extend( tunnelVias )

      if vias:
         return BgpCliModels.BgpUdpTunnelTableEntry(
            vias=vias, endpoint=endpoint, sourceAddr=sourceAddr, tos=tos,
            payloadType=payloadType )
   return None

def showBgpUdpTunnelTable():
   bgpUdpTunnelEntries = {}

   for tunnelId in bgpUdpTunnelTable.entry:
      bgpUdpTunnelEntryModel = getBgpUdpTunnelTableEntryModel( tunnelId=tunnelId )
      if bgpUdpTunnelEntryModel:
         bgpUdpTunnelEntries[ getTunnelIndexFromId( tunnelId ) ] = \
            bgpUdpTunnelEntryModel
   return BgpCliModels.BgpUdpTunnelTable( entries=bgpUdpTunnelEntries )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpLuTunnelTable
   global bgpLuForwardingTunnelTable
   global bgpUdpTunnelTable

   bgpLuTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.bgpLuTunnelTable, entityManager )
   bgpLuForwardingTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.bgpLuForwardingTunnelTable, entityManager )
   bgpUdpTunnelTable = SmashLazyMount.mount(
      entityManager, 'tunnel/gueTunnelFib', 'Tunnel::TunnelFib::TunnelFib',
      readerInfo )
