#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
   AmiResponseException,
   EmptyResponseException,
   showRibCapiCommand,
)
from CliPlugin.VrfCli import (
   generateVrfCliModel,
   getVrfNames,
   VrfExecCmdDec,
)
from IpLibConsts import DEFAULT_VRF
from socket import AF_INET, AF_INET6

AutoAggrModels = CliDynamicPlugin( "AutoAggrModels" )

#-------------------------------------------------------------------------------
# "show bgp auto-aggregation ipv[4|6] [prefix] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
autoAggrVrfModel = generateVrfCliModel( AutoAggrModels.AutoAggregation,
                                        'Per VRF auto aggregation summary' )

def doShowAutoAggr( mode, v6=False, prefix=None, detail=None, vrfName=None ):
   args = {}
   args[ 'protocol' ] = 0
   args[ 'af' ] = AF_INET6 if v6 else AF_INET

   if prefix:
      if v6:
         # prefix type is Arnet::Ip6AddrWithMask
         args[ 'addrv6' ] = str( prefix.address )
         args[ 'maskv6' ] = prefix.len
      else:
         addr, mlen = prefix.split( '/' )
         args[ 'addrv4' ] = addr
         args[ 'maskv4' ] = int( mlen )

   if detail:
      args[ 'detail' ] = 1

   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      model = showRibCapiCommand()( mode, AutoAggrModels.AutoAggregation,
                                    'MIO_DGET_AUTO_AGGR', args=args, highPerf=True )
      model._vrf = vrfName # pylint: disable=protected-access
      return model
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowAutoAggrIpv4', ribdModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=autoAggrVrfModel )
def doShowAutoAggrIpv4( mode, prefix=None, detail=None, vrfName=None ):
   return doShowAutoAggr( mode, prefix=prefix, detail=detail,
                          vrfName=vrfName )

@ArBgpShowOutput( 'doShowAutoAggrIpv6', ribdModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=autoAggrVrfModel )
def doShowAutoAggrIpv6( mode, prefix=None, detail=None, vrfName=None ):
   return doShowAutoAggr( mode, v6=True, prefix=prefix, detail=detail,
                          vrfName=vrfName )

def handlerShowBgpAutoAggrIpv4( mode, args ):
   return doShowAutoAggrIpv4(
      mode,
      prefix=args.get( "PREFIX" ),
      detail=args.get( "detail" ),
      vrfName=args.get( "VRF" ) )

def handlerShowBgpAutoAggrIpv6( mode, args ):
   return doShowAutoAggrIpv6(
      mode,
      prefix=args.get( "PREFIX" ),
      detail=args.get( "detail" ),
      vrfName=args.get( "VRF" ) )
