#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable=anomalous-backslash-in-string
# pylint: disable=inconsistent-return-statements
# pylint: disable=singleton-comparison
# pylint: disable=unused-import
# pylint: disable=ungrouped-imports

from Toggles import gatedToggleLib
from Toggles import (
   IsisToggleLib,
   RcfLibToggleLib
)
import Toggles.IgpLibToggleLib
import Tac, LazyMount, SharedMem, Smash
import ConfigMount
import CliParser, BasicCli
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import IraVrfCli
from CliPlugin import ConfigConvert
from CliPlugin.RcfCliLib import rcfFunctionMatcher
from CliPlugin.RouteMapCli import mapNameMatcher

from CliPlugin.TunnelCli import (
   readMountTunnelTable,
   TunnelTableIdentifier,
)

from CliToken.RouteMapCliTokens import CommonTokens as RouteMapMatchers
from CliMode.Isis import RoutingIsisMode, RoutingIsisAfMode, RoutingIsisSrMplsMode, \
      RoutingIsisMetricProfileMode, RoutingIsisTeMode, RoutingIsisAreaProxyMode, \
      RoutingIsisAfSrv6Mode
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED

# pylint: disable=cyclic-import
import CliMatcher

import CliCommand
from TypeFuture import TacLazyType

from CliPlugin.RoutingIsisCli import (
   tilfaKw,
   authKw,
   backupKw,
   ipv6Kw,
   lspKw,
   modeKw,
   rfc8202Kw,
   rxDisabledKw,
   unicastKw,
   indexValueMatcher,
   afMatcher,
   ipv4IPMatcher,
   ipv6IPMatcher,
   level12MatcherForConfig,
   levelMatcherForConfig,
   metricValueMatcher,
   isisInstanceByVrfName,
   hostnameRegex,
   instanceNamePattern,
   AuthenticationModeExpression,
   authData,
   authKeySyntax,
   noOrDefaultAuthSyntax,
   FrrRerouteExpression,
   frrSrlgData,
)

# Exception not raised when try fails. Suppress pylint warning
# pylint: disable-msg=bare-except
# pylint: disable-msg=protected-access
# pylint: disable-msg=broad-except
l3Config = None
tilfaTunnelTable = None
isisSrV4BindingTable = None
isisSrV6BindingTable = None
isisConfig = None
isisStatusDir = None
isisClearReqDir = None
isisClearRespDir = None
srSysdbStatusDir = None
routingHardwareStatusCommon = None
routingHardwareStatus = None
routing6Config = None
routing6HardwareStatus = None
allIntfStatusDir = None
ipConfig  = None
ip6Config = None
vrfInfoDir = None
vrf6InfoDir = None
entityManager = None
linkReadyStatusVrfColl = None
ldpProtoConfigColl = None
ldpConfigColl = None
mplsRoutingConfig = None
mplsStatus = None
isisSystemIdHostnameMap = None
trapConfig = None
flexAlgoConfig = None
teConfig = None
sharedSecretConfig = None
U32_MAX_VALUE = 0xFFFFFFFF

AreaLeaderPriority = TacLazyType( 'Routing::Isis::AreaLeaderPriority' )
CsnpGenInterval = TacLazyType( 'Routing::Isis::CsnpGenInterval' )
GrHoldTime = TacLazyType( 'Routing::Isis::GrHoldTime' )
GrTimerT2 = TacLazyType( 'Routing::Isis::GrTimerT2' )
LabelBindingTable = TacLazyType( 'Mpls::LabelBindingTable' )
LspGenIntervalMsec = TacLazyType( 'Routing::Isis::LspGenIntervalMsec' )
LspGenIntervalSec = TacLazyType( 'Routing::Isis::LspGenIntervalSec' )
LspRefreshInterval = TacLazyType( 'Routing::Isis::LSPRefreshInterval' )
MaxLSPLifetime = TacLazyType( 'Routing::Isis::MaxLSPLifetime' )
MaxLSPSize = TacLazyType( 'Routing::Isis::MaxLSPSize' )
MinLspRemainingLifetime = TacLazyType( 'Routing::Isis::MinLspRemainingLifetime' )
OverloadStartupDelay = TacLazyType( 'Routing::Isis::OverloadStartupDelay' )
PrefixRange = TacLazyType( 'Routing::Isis::PrefixRange' )
Speed = TacLazyType( 'Routing::Isis::Speed' )
SpeedUnitType = TacLazyType( 'Routing::Isis::SpeedUnitType' )
SpfIntervalSec = TacLazyType( 'Routing::Isis::SpfIntervalSec' )
SpfIntervalMsec = TacLazyType( 'Routing::Isis::SpfIntervalMsec' )
TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )
MaxRedistRoutesLimit = TacLazyType( 'Routing::Isis::MaxRedistRoutesLimit' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )

routerIdKw = CliMatcher.KeywordMatcher( 'router-id',
         helpdesc='Configure router ID for IS-IS routing processes' )
ipv6AddrFormatKw = CliMatcher.KeywordMatcher( 'ipv6',
         helpdesc='IS-IS router ID in IPv6 address format' )
routerIdV4Matcher = IpAddrMatcher.IpAddrMatcher(
         "IS-IS router ID in IP address format" )
routerIdV6Matcher = Ip6AddrMatcher.Ip6AddrMatcher(
         "IS-IS router ID in IPv6 address format" )
ipv4IPMatcher = IpAddrMatcher.ipAddrWithMaskExpr( 'IP address',
      'Subnet\'s mask value', 'IPv4 address prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
ipv6IPMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 address prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
proxyNodePhpAttachedKw = CliMatcher.KeywordMatcher( 'attached-flag',
      helpdesc='Proxy-node segments attached-flag configuration' )
proxyNodeAttachedFlagKw = CliMatcher.KeywordMatcher( 'inspect',
      helpdesc='Inspect attached flag of proxy-node segments' )


#------------------------------------------------------------------------------
# Configuration keywords used by CLI parser
#------------------------------------------------------------------------------
bgpKw = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP routes' )
indexKw = CliMatcher.KeywordMatcher( 'index',
      helpdesc='Index to be assigned as Adj-SID for adjacency on this interface' )
redistributeKw = CliMatcher.KeywordMatcher( 'redistribute',
      helpdesc='Redistribute routes with IS-IS' )
redistributedKw = CliMatcher.KeywordMatcher( 'redistributed',
      helpdesc='IS-IS redistributed route specific configuration' )

#------------------------------------------------------------------------------
# Show keywords used by CLI parser
#------------------------------------------------------------------------------

# Authentication related keywords

#------------------------------------------------------------------------------
# Common patterns
#------------------------------------------------------------------------------

netRegex = \
           r'(?P<areaid>(?:[\da-fA-F]{2}(?:\.[\da-fA-F]{4}){0,6})|' \
           r'((?:[\da-fA-F]{4})(?:\.[\da-fA-F]{4}){0,5}))' \
           r'\.(?P<systemid>[\da-fA-F]{4}\.[\da-fA-F]{4}\.[\da-fA-F]{4})' \
           r'\.00$'

netPartialRegex = \
    r'(?:[\da-fA-F]{0,2}\.)?' \
    r'(?:[\da-fA-F]{4}\.){0,8}' \
    r'(?:[\da-fA-F]{0,4}|' \
    r'[\da-fA-F]{4}\.0{0,2})$'

netKw = CliMatcher.PatternMatcher( pattern=netRegex,
        partialPattern=netPartialRegex, helpname='NET',
        helpdesc="NET in HH.HHHH.HHHH.HHHH.HHHH.HHHH."
                 "HHHH.HHHH.HHHH.HHHH.00 format" )

areaRegex = \
      r'(?:[\da-fA-F]{2}(?:\.[\da-fA-F]{4}){0,6})|' \
      r'((?:[\da-fA-F]{4})(?:\.[\da-fA-F]{4}){0,5})$'
areaPartialRegex = \
    r'(?:[\da-fA-F]{0,2}\.)?' \
    r'(?:[\da-fA-F]{4}\.){0,5}' \
    r'(?:[\da-fA-F]{0,4})'
areaMatcher = CliMatcher.PatternMatcher( pattern=areaRegex,
               partialPattern=areaPartialRegex, helpname='AREA',
               helpdesc='Area address in HH.HHHH.HHHH.HHHH.HHHH.HHHH.HHHH format' )

# Restricting the character list allowed in hostname as we would like to support
# filtering of LSP database based on LSPID with hostname in the system Id portion.
# This requires us to take LSPID as input which will be validated in cliribd as per
# CCTYPE_STRING. Without this we could end up in situations where we configure
# hostname, but fail to filter the LSP database stating invalid input.
# Hostnames will be restricted to alphanumeric and any of [,_:-./#%+]
hostnameKw = CliMatcher.PatternMatcher( hostnameRegex,
        helpdesc='IS hostname',
        helpname='HOSTNAME' )

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without routing support
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default VRF IS-IS instance
#-------------------------------------------------------------------------------
def isisNonDefaultVrfGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.vrfName != DEFAULT_VRF:
      return CliParser.guardNonDefaultVrf

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default IS-IS instance (instance ID != 0)
#-------------------------------------------------------------------------------
def isisNonDefaultInstanceGuard( mode, token ):
   if isinstance( mode, IntfCli.IntfConfigMode ):
      return None
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.instanceId != 0:
      return "not supported in non-default IS-IS instance"

#-------------------------------------------------------------------------------
# Guard to prevent configuration in default IS-IS instance (instance ID 0)
#-------------------------------------------------------------------------------
def isisDefaultInstanceGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.instanceId == 0:
      return "not supported in default IS-IS instance"

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default VRF IS-IS instance and in
# non-default IS-IS instance (instance ID != 0)
#-------------------------------------------------------------------------------
def isisNonDefaultVrfAndInstanceGuard( mode, token ):
   return isisNonDefaultVrfGuard( mode, token ) or \
          isisNonDefaultInstanceGuard( mode, token )
#-------------------------------------------------------------------------------
# Segment routing common infrastructure
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# Isis commands in "show tech-support"
#-------------------------------------------------------------------------------
# In the future if we add commands that fork rib to produce their output, then
# we should split the list of commands into two. One list for commands that don't
# fork ribd using GATED_PROTO_SH_TECH_TS and the second one for commands that fork
# using GATED_PROTO_FORK_SH_TECH_TS

#-------------------------------------------------------------------------------
# router isis config mode. A new instance of this mode is created when the
# user enters "router isis".
#-------------------------------------------------------------------------------
class RouterIsisMode( RoutingIsisMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'ISIS instance configuration'

   def __init__( self, parent, session, instanceConfig ):
      RoutingIsisMode.__init__( self, ( instanceConfig.instanceName,
                                        instanceConfig.instanceId,
                                        instanceConfig.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.instanceName = instanceConfig.instanceName
      self.instanceId = instanceConfig.instanceId
      self.vrfName = instanceConfig.vrfName

#-------------------------------------------------------------------------------
# ISIS area proxy config mode. A new instance of this mode is created when the
# user enters "area proxy".
#-------------------------------------------------------------------------------
class RouterIsisAreaProxyMode( RoutingIsisAreaProxyMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'ISIS area proxy configuration'

   def __init__( self, parent, session ):
      # parent is an object of RouterIsisMode which inherits from
      # RoutingIsisMode and that has vrfName which we are using below.
      self.instanceName = parent.instanceName
      RoutingIsisAreaProxyMode.__init__( self, ( parent.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# ISIS-TE config mode. A new instance of this mode is created when the
# user enters "traffic-engineering".
#-------------------------------------------------------------------------------
class RouterIsisTeMode( RoutingIsisTeMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'ISIS traffic-engineering configuration'

   def __init__( self, parent, session ):
      # parent is an object of RouterIsisMode which inherits from
      # RoutingIsisMode and that has vrfName which we are using below.
      self.instanceName = parent.instanceName
      RoutingIsisTeMode.__init__( self, ( parent.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "[no|default] router isis <instance-name> [vrf <vrf-name>]" command,
# in "config" mode.
# -------------------------------------------------------------------------------
MAX_ISIS_INSTANCES_PER_VRF = 1

def isisInstanceNameById( instanceConfig, vrf=DEFAULT_VRF ):
   instDict = {}
   for  i in instanceConfig.values():
      if i.vrfName == vrf:
         instDict[ i.instanceId ] = i.instanceName
   return instDict

def anyInstanceInVrf( instanceConfig, vrf=DEFAULT_VRF ):
   return any( i.vrfName == vrf for i in instanceConfig.values() )

def getInstanceConfig( mode, instanceName, instanceId, vrfName ):
   """Get ISIS Instance config object for supplied instanceName and
   vrfName. Create it if it doesn't exist already"""
   instanceConfig = isisConfig.instanceConfig.get( instanceName )

   # Existing instance config validations.
   if instanceConfig:
      if vrfName and instanceConfig.vrfName != vrfName:
         mode.addError( 'IS-IS instance %s is already configured in %s VRF' % \
                           ( instanceName, instanceConfig.vrfName ) )
         return None
      if instanceId and instanceConfig.instanceId != instanceId:
         mode.addError( 'IS-IS instance %s is already configured '
                        'with instance ID %d' % ( instanceName,
                                                  instanceConfig.instanceId ) )
         return None

      return instanceConfig

   # New instance config validations.
   instanceId = 0 if instanceId is None else instanceId
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( 'VRF name %s is reserved' % vrfName )
      return None

   vrfName = DEFAULT_VRF if vrfName is None else vrfName
   instanceByVrfName = isisInstanceByVrfName( isisConfig.instanceConfig )
   if vrfName == DEFAULT_VRF:
      instanceNameById = isisInstanceNameById( isisConfig.instanceConfig )
      if instanceId in instanceNameById:
         if instanceId == 0:
            mode.addError( 'More than %d IS-IS instance in default VRF '
                           'without instance ID is not supported' % \
                              MAX_ISIS_INSTANCES_PER_VRF )
            return None
         else:
            mode.addError( 'IS-IS instance %s is already configured with '
                           'instance ID %d' % ( instanceNameById[ instanceId ],
                                                instanceId ) )
            return None
   else:
      if vrfName in instanceByVrfName:
         mode.addError( 'More than %d IS-IS instance in non-default VRF '
                        'is not supported' % MAX_ISIS_INSTANCES_PER_VRF )
         return None

   if vrfName != DEFAULT_VRF:
      maxVrfs = routingHardwareStatusCommon.vrfCapability.maxVrfs
      instanceByVrfName.pop( DEFAULT_VRF, None )
      if maxVrfs != -1 and len( instanceByVrfName ) >= maxVrfs:
         mode.addError( 'More than %d IS-IS VRF instances are not supported' % \
                           maxVrfs )
         return None

   IraVrfCli.addAgentVrfEntry( vrfName, "Isis" )
   return isisConfig.instanceConfig.newMember( instanceName, instanceId,
                                               vrfName )

def gotoRouterIsisMode( mode, instanceName, instanceId, vrfName ):
   """Handler for command -
     'router isis <instance-name> [instance-id <instance-id> | vrf <vrf-name>]'"""
   # We do not check if IP routing or IPv6 routing is enabled here.
   # We do it when we enter the address-family ipv4 | ipv6 CLI

   # Request trap resouces if multi-instance
   if instanceId:
      trapConfig.features.add( TrapFeatureName.isisMi )

   # Get the instanceConfig object, create it if not present
   instanceConfig = getInstanceConfig( mode, instanceName, instanceId, vrfName )
   if instanceConfig is None:
      return

   childMode = mode.childMode( RouterIsisMode, instanceConfig=instanceConfig )
   mode.session_.gotoChildMode( childMode )

def anyMultiInstance( instanceConfig ):
   return any( i.instanceId for i in instanceConfig.values() )

def delRouterIsisMode( mode, instanceName ):
   """Handler for 'no|default router isis <instance-name>' command"""
   if instanceName in isisConfig.instanceConfig:
      vrfName = isisConfig.instanceConfig[ instanceName ].vrfName
      del isisConfig.instanceConfig[ instanceName ]
      if not anyInstanceInVrf( isisConfig.instanceConfig, vrfName ):
         IraVrfCli.removeAgentVrfEntry( vrfName, "Isis" )
      # Trap registration is only for isis-mi support and should be removed only
      # if all of the isis multi-instances are deleted.
      if not anyMultiInstance( isisConfig.instanceConfig ):
         trapConfig.features.remove( TrapFeatureName.isisMi )

class RouterIsisAfIpv6Modelet( CliParser.Modelet ):
   '''
   This class is used to add commands specific to ipv6 address family
   modes in router isis config mode.
   '''
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.addrFamily == 'ipv6'

   def clearConfig( self ):
      pass

class RouterIsisAfIpv4Modelet( CliParser.Modelet ):
   '''
   This class is used to add commands specific to ipv4 address-family
   mode in router isis config mode.
   '''
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.addrFamily == 'ipv4'

   def clearConfig( self ):
      pass

class RouterIsisAfMode( RoutingIsisAfMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'ISIS address family configuration'

   def __init__( self, parent, session, addrFamily, modifier="unicast" ):
      # parent is an object of RouterIsisMode which inherits from
      # RoutingIsisMode and that has vrfName which we are using below.
      self.instanceName = parent.instanceName
      RoutingIsisAfMode.__init__( self, ( parent.vrfName, addrFamily, modifier,
                                          parent.instanceId ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      return self.addrFamily


class RouterIsisMetricProfileMode( RoutingIsisMetricProfileMode,
                                    BasicCli.ConfigModeBase ):
   name = "IS-IS Metric Profile configuration"
   unitSpeedtoSppedMapping = { 'mbps' : 1, 'gbps' : 1000 }

   def __init__( self, parent, session, metricProfileName, metricProfileConfig ):
      self.instanceName = parent.instanceName
      self.metricProfileConfig = metricProfileConfig
      RoutingIsisMetricProfileMode.__init__( self, ( parent.vrfName,
                                                     metricProfileName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterIsisAfSrv6Mode( RoutingIsisAfSrv6Mode, BasicCli.ConfigModeBase ):
   name = 'ISIS segment-routing ipv6 configuration'

   def __init__( self, parent, session ):
      RoutingIsisAfSrv6Mode.__init__( self, parent.instanceName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterIsisSrMplsMode( RoutingIsisSrMplsMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'ISIS segment-routing mpls configuration'

   def __init__( self, parent, session ):
      # parent is an object of RouterIsisMode which inherits from RoutingIsisMode
      RoutingIsisSrMplsMode.__init__( self, parent.instanceName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )


#--------------------------------------------------------------------------
# [no|default] router isis <INST> [ instance-id <INST_ID> |vrf <VRFNAME>]
#--------------------------------------------------------------------------
instanceNamePattern = '.{1,100}'

def routerIsisModeCmd( mode, args ):
   gotoRouterIsisMode( mode, args[ 'INST' ],
                       args.get( 'ID' ),
                       args.get( 'VRF' ) )

def routerIsisModeCmdNo( mode, args ):
   delRouterIsisMode( mode, args[ 'INST' ] )

# The "[no|default] route preference rfc7775" command
# under 'router isis <>' mode
class IsisRoutePrefRfc7775( CliCommand.CliCommandClass ):
   syntax = 'route preference rfc7775'
   noOrDefaultSyntax = 'route preference ...'
   data = {
            'route' : 'IS-IS route specific configuration',
            'preference' : 'Route preference order mechanism ',
            'rfc7775' : 'Use RFC7775 defined route preference order'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setRoutePreference"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRoutePreference"
RouterIsisMode.addCommandClass( IsisRoutePrefRfc7775 )

#----------------------------------------------------------------------------------
# The "[no|default] lsp rfc8202 rx disabled" command
# under 'router isis <> instance-id <>' mode
#----------------------------------------------------------------------------------
class IsisLspRfc8202RxDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp rfc8202 rx disabled'
   noOrDefaultSyntax = syntax
   data = {
      'lsp' : lspKw,
      'rfc8202' : CliCommand.Node( matcher=rfc8202Kw,
                                   guard=isisDefaultInstanceGuard ),
      'rx' : 'Configure receive side parameters',
      'disabled' : 'Disable IID-TLV check when receiving LSPs'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setIsisLspRfc8202RxDisabled"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_noIsisLspRfc8202RxDisabled"
RouterIsisMode.addCommandClass( IsisLspRfc8202RxDisabledCmd )

#----------------------------------------------------------------------------------
# The "[no|default] rfc8202 disabled" command
# under 'router isis <> instance-id <>' mode
#----------------------------------------------------------------------------------
class IsisRfc8202DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'rfc8202 disabled'
   noOrDefaultSyntax = syntax
   data = {
      'rfc8202' : CliCommand.Node( matcher=rfc8202Kw,
                                   guard=isisDefaultInstanceGuard ),
      'disabled' : 'Disabled all rfc8202 functionality'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setIsisRfc8202Disabled"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noIsisRfc8202Disabled"
RouterIsisMode.addCommandClass( IsisRfc8202DisabledCmd )

#----------------------------------------------------------------------------------
# The "[no|default] lsp purge origination-identification [ LEVEL ]" command
# under 'router isis <>' mode
#----------------------------------------------------------------------------------
class IsisLspPoiTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp purge origination-identification [ LEVEL ]'
   noOrDefaultSyntax = 'lsp purge origination-identification ...'
   data = {
            'lsp' : lspKw,
            'purge' : 'Configure purge specific parameters',
            'origination-identification' : 'Enable Origination identification',
            'LEVEL' : levelMatcherForConfig,
          }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setPurgeOrigination"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noPurgeOrigination"

RouterIsisMode.addCommandClass( IsisLspPoiTypeCmd )

#------------------------------------------------------------------------------
# [no|default] is-type LEVEL under 'router isis <>' mode
#------------------------------------------------------------------------------
class RouterIsisIsTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'is-type LEVEL'
   noOrDefaultSyntax = 'is-type ...'
   data = {
      'is-type' : 'Set level of the instance',
      'LEVEL' : level12MatcherForConfig,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setIsType"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noIsType"

RouterIsisMode.addCommandClass( RouterIsisIsTypeCmd )

#-------------------------------------------------------------------------------
# [no | default] address-family AF [ unicast ]
#-------------------------------------------------------------------------------
class RouterIsisAddressFamilyCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family AF [ unicast ]'
   noOrDefaultSyntax = syntax
   data = {
      'address-family' : 'Enable address family and enter its config mode',
      'AF' : afMatcher,
      'unicast' : unicastKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_gotoRouterIsisAfMode"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_delRouterIsisAfMode"

RouterIsisMode.addCommandClass( RouterIsisAddressFamilyCmd )

#-------------------------------------------------------------------------------
# "[no | default] multi-topology"
# command, in config-router-isis-af mode.
#-------------------------------------------------------------------------------
mtWarningMsg = "Multi Topology is not enabled"
mtV4WarningMsg = "IPv4 address family is not configured in router IS-IS mode"
RouterIsisAfMode.addModelet( RouterIsisAfIpv6Modelet )

class RouterIsisAfIpv6MtCmd( CliCommand.CliCommandClass ):
   syntax = 'multi-topology'
   noOrDefaultSyntax = syntax
   data = {
      'multi-topology' : 'Enable multi topology'
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfIpv6Modelet_multiTopologyIs"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfIpv6Modelet_multiTopologyDel"
RouterIsisAfIpv6Modelet.addCommandClass( RouterIsisAfIpv6MtCmd )

#---------------------------------------------------------------------------------
# [no|default] traffic-engineering
#---------------------------------------------------------------------------------
trafficengineeringKw = CliMatcher.KeywordMatcher( 'traffic-engineering',
                                  helpdesc='Enter traffic engineering config mode' )

class RouterIsisTrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering'
   noOrDefaultSyntax = 'traffic-engineering'
   data = {
      'traffic-engineering' : CliCommand.Node( matcher=trafficengineeringKw,
                                               guard=isisNonDefaultVrfGuard )
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_gotoRouterIsisTeMode"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_delRouterIsisTeMode"
RouterIsisMode.addCommandClass( RouterIsisTrafficEngineeringCmd )

#---------------------------------------------------------------------------------
# [no|default] segment-routing mpls
#---------------------------------------------------------------------------------
segmentRoutingKw = CliMatcher.KeywordMatcher( 'segment-routing',
                                       helpdesc='Enter segment routing config mode' )
mplsKw = CliMatcher.KeywordMatcher( 'mpls',
                                    helpdesc='segment routing in MPLS dataplane' )
class RouterIsisSegmentRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing mpls'
   noOrDefaultSyntax = syntax
   data = {
      'segment-routing' : segmentRoutingKw,
      'mpls' : CliCommand.Node( matcher=mplsKw, guard=isisNonDefaultVrfGuard )
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_gotoRouterIsisSrMplsMode"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_delRouterIsisSrMplsMode"
RouterIsisMode.addCommandClass( RouterIsisSegmentRoutingCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] area proxy
#---------------------------------------------------------------------------------
class RouterIsisAreaProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'area proxy'
   noOrDefaultSyntax = syntax
   data = {
      'area' : 'Set area parameters',
      'proxy' : 'Area proxy configuration',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_gotoRouterIsisAreaProxyMode"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_delRouterIsisAreaProxyMode"

RouterIsisMode.addCommandClass( RouterIsisAreaProxyCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] shutdown
#---------------------------------------------------------------------------------

class RoutingIsisAreaProxyShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable area proxy'
   }
   handler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_shutdown"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_noShutdown"

RouterIsisAreaProxyMode.addCommandClass( RoutingIsisAreaProxyShutdownCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] net <NET>
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyNetCmd( CliCommand.CliCommandClass ):
   syntax = 'net NET'
   noOrDefaultSyntax = 'net ...'
   data = {
      'net' : 'Configure Network Entity Title (NET) '
              'for the proxy area',
      'NET' : netKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_setNet"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_noNet"

RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyNetCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] is-hostname <HOSTNAME>
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'is-hostname HOSTNAME'
   noOrDefaultSyntax = 'is-hostname ...'
   data = {
      'is-hostname' : 'Configure hostname for the proxy area',
      'HOSTNAME' : hostnameKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_setHostname"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_noHostname"

RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyHostnameCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] area segment (ipv4|ipv6) index <value>
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyAreaSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'area segment ( IPADDR | IP6ADDR ) index INDEX'
   noOrDefaultSyntax = 'area segment ( IPADDR | IP6ADDR ) ...'
   data = {
      'area' : 'Configure area parameters',
      'segment' : 'Configure area segment SID for the proxy area',
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'index' : indexKw,
      'INDEX' : indexValueMatcher,
   }
   handler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_setAreaSegment"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_noAreaSegment"

RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyAreaSegmentCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] router-id ipv4 [ROUTER_IDV4]
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyRouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ( ( ipv4 ROUTER_IDV4 ) | ( ipv6 ROUTER_IDV6 ) )'
   noOrDefaultSyntax = 'router-id ( ipv4 | ipv6 ) ... '
   data = {
      'router-id' : routerIdKw,
      'ipv4' : 'IS-IS router ID in IP address format',
      'ROUTER_IDV4' : routerIdV4Matcher,
      'ipv6' : ipv6AddrFormatKw,
      'ROUTER_IDV6' : routerIdV6Matcher
   }
   handler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_setRouterId"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAreaProxyMode_noRouterId"

RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyRouterIdCmd )

#------------------------------------------------------------------------------
# [no|default] redistribute <static|connected|bgp> [include leaked]
#              [route-map <route-map-name>]
#------------------------------------------------------------------------------
protocolMatcher = CliMatcher.EnumMatcher( {
   'static'    : 'Static routes',
   'connected' : 'Connected interface routes',
} )

class RedistributeBaseCommand( CliCommand.CliCommandClass ):
   data = {
      'redistribute' : redistributeKw,
      'route-map'    : RouteMapMatchers.routeMapApplication,
      'MAPNAME'      : mapNameMatcher,
      # Contingent upon the toggle, but can add anyway.
      'include'      : 'Include leaked routes',
      'leaked'       : 'Include leaked routes',
   }

class RedistributeStaticCommand( RedistributeBaseCommand ):
   if RcfLibToggleLib.toggleRcfRedistStaticToIsisEnabled():
      _policySyntax = ( "[ ( route-map MAPNAME ) | "
                       "( rcf FUNCTION_NAME ) ]" )
   else:
      _policySyntax = "[ route-map MAPNAME ] "

   syntax = ( 'redistribute static [ include leaked ] '
             f'{ _policySyntax }' )

   noOrDefaultSyntax = 'redistribute static ...'
   data = RedistributeBaseCommand.data | {
      'static': 'Static routes',
      'rcf': 'Routing control functions configuration to filter and modify paths',
      'FUNCTION_NAME': rcfFunctionMatcher,
   }

   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistributeStatic"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistributeStatic"

RouterIsisMode.addCommandClass( RedistributeStaticCommand )

class RedistributeConnectedCommand( RedistributeBaseCommand ):
   syntax = 'redistribute connected [ include leaked ] [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute connected ...'
   data = RedistributeBaseCommand.data | {
      'connected': 'Connected interface routes',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistributeCon"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistributeCon"

RouterIsisMode.addCommandClass( RedistributeConnectedCommand )

#-----------------------------------------------------------------------------------
# [no|default] redistribute ospf [include leaked]
#              match <internal|external|nssa-exeternal> [route-map <route-map-name>]
#-----------------------------------------------------------------------------------
ospfKw = CliMatcher.KeywordMatcher( 'ospf',
      helpdesc='Routes learned by the OSPF protocol' )
ospfv3Kw = CliMatcher.KeywordMatcher( 'ospfv3',
      helpdesc='Redistribution of OSPFv3 routes' )
matchKw = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Routes learned by the OSPF protocol' )
externalKw = CliMatcher.KeywordMatcher( 'external',
      helpdesc='OSPF routes learned from external sources' )
internalKw = CliMatcher.KeywordMatcher( 'internal',
      helpdesc='OSPF routes learned from internal sources' )
nssaExternalKw = CliMatcher.KeywordMatcher( 'nssa-external',
      helpdesc='Redistribute OSPF nssa - external routes' )
includeKw = CliMatcher.KeywordMatcher( 'include',
      helpdesc='Include leaked routes' )
leakedKw = CliMatcher.KeywordMatcher( 'leaked',
      helpdesc='Include leaked routes' )
matchRoute = 'match ( external | internal | nssa-external )'

redistributeCmdData = {
   'redistribute' : redistributeKw,
   'route-map' : RouteMapMatchers.routeMapApplication,
   'MAPNAME' : mapNameMatcher
}

isisRedistributeCmdData = {
   'isis' : 'IS-IS routes',
   'into' : 'into IS-IS level',
}
isisRedistributeCmdData.update( redistributeCmdData )

class OspfRedistributeCmdData( CliCommand.CliCommandClass ):
   data = {
      'ospfv3' : ospfv3Kw,
      'match' : matchKw,
      'external' : externalKw,
      'internal' : internalKw,
      'nssa-external' : nssaExternalKw,
   }
   data.update( redistributeCmdData )

class RouterIsisModeRedistributionCmd( OspfRedistributeCmdData ):
   syntax = 'redistribute ( ( ospf [ include leaked ] ) | ospfv3 ) ' + \
            matchRoute + ' [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute ( ( ospf [ include leaked ] ) | ospfv3 ) ' + \
                       matchRoute + '...'
   data = { 'ospf' : ospfKw }
   data.update( OspfRedistributeCmdData.data )
   data.update( { 'include' : includeKw,
                  'leaked' : leakedKw } )

   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistribute2"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistribute2"
RouterIsisMode.addCommandClass( RouterIsisModeRedistributionCmd )

#-----------------------------------------------------------------------------------
# [no|default] redistribute bgp [include leaked] [route-map <route-map-name>]
#-----------------------------------------------------------------------------------
bgpCliCommandNode = CliCommand.Node( matcher=bgpKw,
                                     guard=isisNonDefaultInstanceGuard )

class RedistributeBgpCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute bgp [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute bgp ...'
   data = {
      'bgp'     : bgpCliCommandNode
   }
   data.update( redistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistribute3"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistribute3"

RouterIsisMode.addCommandClass( RedistributeBgpCommand )

#-----------------------------------------------------------------------------------
# [no|default] route redistributed limit [level-1 | level-2] <Max Limit>
#-----------------------------------------------------------------------------------
maxRedistRoutesLimitMatcher = CliMatcher.IntegerMatcher(
        MaxRedistRoutesLimit.min + 1, MaxRedistRoutesLimit.max,
        helpdesc='Maximum number of redistributed routes into IS-IS' )
class RedistributedRouteCountLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'route redistributed limit [ level-1 | level-2 ] MAX_LIMIT'
   noOrDefaultSyntax = 'route redistributed limit [ level-1 | level-2 ] ...'
   data = {
      'route'         : 'IS-IS route specific configuration',
      'redistributed' : redistributedKw,
      'limit'         : 'Limit the number of redistributed routes into IS-IS',
      'level-1'       : 'Configure at level 1',
      'level-2'       : 'Configure at level 2',
      'MAX_LIMIT'     : maxRedistRoutesLimitMatcher,
   }

   handler = "RoutingIsisCliHandler.RouterIsisMode_setMaxRedistRoutesLimit"
   noHandler = "RoutingIsisCliHandler.RouterIsisMode_noMaxRedistRoutesLimit"
   defaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noMaxRedistRoutesLimit"

RouterIsisMode.addCommandClass( RedistributedRouteCountLimitCmd )

#-------------------------------------------------------------------------------
# "[no|default] mpls ldp sync default" in 'router isis <>' mode
#-------------------------------------------------------------------------------
defaultKw = CliCommand.guardedKeyword( 'default',
        helpdesc='Set global default',
        guard=isisNonDefaultVrfGuard )

class RouterIsisMplsLDPSyncCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp sync default'
   noOrDefaultSyntax = 'mpls ldp sync default ...'
   data = {
      'mpls' : 'Mpls ldp sync configuration',
      'ldp' : 'LDP configuration',
      'sync' : 'Sync configuration',
      'default' : defaultKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setMplsLdpSync"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noSetMplsLdpSync"

RouterIsisMode.addCommandClass( RouterIsisMplsLDPSyncCmd )

#-------------------------------------------------------------------------------
# "[no|default] max-lsp-lifetime <time>" under 'router isis <>' mode
#-------------------------------------------------------------------------------
class RouterIsisMaxLSPLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'max-lsp-lifetime MAX_LIFETIME'
   noOrDefaultSyntax = 'max-lsp-lifetime ...'
   data = {
      'max-lsp-lifetime' : 'Set maximum lifetime of LSPs',
      'MAX_LIFETIME' : CliMatcher.IntegerMatcher(
            MaxLSPLifetime.min, MaxLSPLifetime.max,
            helpdesc='LSP lifetime value in seconds' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setMaxLSPLifetime"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noMaxLSPLifetime"

RouterIsisMode.addCommandClass( RouterIsisMaxLSPLifetimeCmd )

#-------------------------------------------------------------------------------
# "[no|default] lsp size maximum <size>" under 'router isis <>' mode
#-------------------------------------------------------------------------------
class RouterIsisMaxLSPSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp size maximum MAX_LSP_SIZE'
   noOrDefaultSyntax = 'lsp size maximum ...'
   data = {
      'lsp' : lspKw,
      'size' : 'Configure LSP size attribute',
      'maximum' : 'Configure LSP size maximum',
      'MAX_LSP_SIZE' : CliMatcher.IntegerMatcher(
            MaxLSPSize.min, MaxLSPSize.max,
            helpdesc='LSP size value in bytes' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setMaxLSPSize"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noMaxLSPSize"

RouterIsisMode.addCommandClass( RouterIsisMaxLSPSizeCmd )

#---------------------------------------------------------------------------------
# '[no|default] is-hostname <hostname>' under 'router isis <>' mode
#---------------------------------------------------------------------------------
class RouterIsisIsHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'is-hostname HOSTNAME'
   noOrDefaultSyntax = 'is-hostname ...'
   data = {
      'is-hostname' : 'Configure hostname of Intermediate System',
      'HOSTNAME' : hostnameKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setIsisHostname"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noIsisHostname"

RouterIsisMode.addCommandClass( RouterIsisIsHostnameCmd )

# TE CLI
#----------------------------------------------------
#"[no | default] shutdown" command, in isis - te mode
#----------------------------------------------------
class RouterIsisTeShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Traffic Engineering state in IS-IS'
   }
   handler = "RoutingIsisCliHandler.RouterIsisTeMode_shutdown"
   defaultHandler = handler
   noHandler = "RoutingIsisCliHandler.RouterIsisTeMode_noShutdown"

RouterIsisTeMode.addCommandClass( RouterIsisTeShutdownCmd )

#------------------------------------------------------------------------------
# [no|default] is-type LEVEL under isis-te mode
#------------------------------------------------------------------------------
class RouterIsisTeIsTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'is-type LEVEL'
   noOrDefaultSyntax = 'is-type ...'
   data = {
      'is-type' : 'Set level of the instance',
      'LEVEL' : level12MatcherForConfig,
   }
   handler = "RoutingIsisCliHandler.RouterIsisTeMode_setIsType"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisTeMode_noIsType"

RouterIsisTeMode.addCommandClass( RouterIsisTeIsTypeCmd )

#---------------------------------------------------------------------------------
# [no|default] shutdown
#---------------------------------------------------------------------------------
class RouterIsisShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown'
   data = {
      'shutdown' : 'Disable this IS-IS instance'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setShutdown"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noShutdown"
RouterIsisMode.addCommandClass( RouterIsisShutdownCmd )

#---------------------------------------------------------------------------------
# [no|default] log-adjacency-changes
#---------------------------------------------------------------------------------
class RouterIsisLogAdjacencyChangesCmd( CliCommand.CliCommandClass ):
   syntax = 'log-adjacency-changes'
   noOrDefaultSyntax = syntax
   data = {
      'log-adjacency-changes' : 'Enable logging of adjacency events'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setLogAdjacencyChanges"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noLogAdjacencyChanges"
RouterIsisMode.addCommandClass( RouterIsisLogAdjacencyChangesCmd )

#---------------------------------------------------------------------------------
# [no|default] log duplicate-system-id disabled
# command in "router isis" config mode
#---------------------------------------------------------------------------------
class RoutingIsisLogDuplicateSystemIDDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'log duplicate-system-id disabled'
   noOrDefaultSyntax = syntax
   data = {
      'log' : 'Toggle logging of selected features',
      'duplicate-system-id' : 'Disable duplicate system ID detection',
      'disabled' : 'Disables syslog'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setLogDuplicateSystemIDDisabled"
   noOrDefaultHandler = \
              "RoutingIsisCliHandler.RouterIsisMode_noLogDuplicateSystemIDDisabled"

RouterIsisMode.addCommandClass( RoutingIsisLogDuplicateSystemIDDisabledCmd )

#---------------------------------------------------------------------------------
# [no|default] hello padding [disabled] command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
class RouterIsisHelloPaddingCmd( CliCommand.CliCommandClass ):
   syntax = 'hello padding [ disabled ]'
   noOrDefaultSyntax = 'hello padding ...'
   data = {
      'hello' : 'Configure hello packets',
      'padding' : 'Configure hello padding for all interfaces',
      'disabled' : 'Disable hello padding for all adjacency states'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setHelloPadding"
   defaultHandler = "RoutingIsisCliHandler.RouterIsisMode_setHelloPadding"
   noHandler = "RoutingIsisCliHandler.RouterIsisMode_noHelloPadding"

RouterIsisMode.addCommandClass( RouterIsisHelloPaddingCmd )

#-------------------------------------------------------------------------------
# "[no|default] net <NET>" command in "router isis <n>" config mode.
#-------------------------------------------------------------------------------

class RouterIsisNetCmd( CliCommand.CliCommandClass ):
   syntax = 'net NET'
   noOrDefaultSyntax = 'net ...'
   data = {
      'net' : 'Configure Network Entity Title (NET) '
              'for this IS-IS instance',
      'NET' : netKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setNet"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noNet"

RouterIsisMode.addCommandClass( RouterIsisNetCmd )

#-------------------------------------------------------------------------------
# "[no|default] area address <AREA>" command in "router isis <n>" config mode.
#-------------------------------------------------------------------------------
class RouterIsisAreaAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'area address AREA'
   noOrDefaultSyntax = syntax
   data = {
      'area' : 'Set area parameters',
      'address' : 'Configure area address',
      'AREA' : areaMatcher,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAreaAddress"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noAreaAddress"
RouterIsisMode.addCommandClass( RouterIsisAreaAddressCmd )

#---------------------------------------------------------------------------------
# [no] bfd all-interfaces in address-family v4 and v6 config mode
#---------------------------------------------------------------------------------
RouterIsisAfMode.addModelet( RouterIsisAfIpv4Modelet )

class RouterIsisAfBfdAllIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd all-interfaces'
   noOrDefaultSyntax = 'bfd ...'
   data = {
      'bfd' : 'Enable BFD',
      'all-interfaces' : 'Enable BFD on all interfaces'
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setBfd"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noBfd"
RouterIsisAfMode.addCommandClass( RouterIsisAfBfdAllIntfCmd )

#---------------------------------------------------------------------------------
# [no|default] spf-interval <max-wait> [ initial-wait [hold-time] ] command in
# "router isis <n>" config mode
#---------------------------------------------------------------------------------
class RouterIsisSpfIntervalWithoutUnits( CliCommand.CliCommandClass ):
   syntax = 'spf-interval MAX_WAIT [ INITIAL_WAIT [ HOLD_TIME ] ]'
   noOrDefaultSyntax = 'spf-interval ...'
   data = {
      'spf-interval': 'Set SPF interval config',
      'MAX_WAIT': CliMatcher.IntegerMatcher(
            SpfIntervalSec.min,
            SpfIntervalSec.max,
            helpdesc='Maximum interval between two SPFs in seconds' ),
      'INITIAL_WAIT': CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min,
            SpfIntervalMsec.max,
            helpdesc='Initial wait interval for SPF in milliseconds' ),
      'HOLD_TIME': CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min,
            SpfIntervalMsec.max,
            helpdesc='Hold interval between the first and'
                     ' second SPF runs in milliseconds' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setSpfInterval"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noSpfInterval"

class RouterIsisSpfInterval( CliCommand.CliCommandClass ):
   syntax = '''spf-interval
         ( MAX_WAIT [ INITIAL_WAIT [ HOLD_TIME ] ] ) |
         ( MAX_WAIT MAX_WAIT_SEC_KW [ INITIAL_WAIT INITIAL_WAIT_MILLI_SEC_KW
                              [ HOLD_TIME HOLD_TIME_MILLI_SEC_KW ] ] ) |
         ( MAX_WAIT_MS MAX_WAIT_MILLI_SEC_KW [ INITIAL_WAIT INITIAL_WAIT_MILLI_SEC_KW
                                      [ HOLD_TIME HOLD_TIME_MILLI_SEC_KW ] ] )
   '''
   noOrDefaultSyntax = 'spf-interval ...'
   data = {
      'spf-interval' : 'Set SPF interval config',
      'MAX_WAIT' : CliMatcher.IntegerMatcher(
            SpfIntervalSec.min,
            SpfIntervalSec.max,
            helpdesc='Maximum interval between two SPFs in seconds' ),
      'MAX_WAIT_MS': CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min, SpfIntervalMsec.max,
            helpdesc='Maximum interval between two SPFs in milliseconds' ),
      'MAX_WAIT_SEC_KW': CliMatcher.KeywordMatcher( 'seconds',
            helpdesc='Configure maximum interval between two SPFs in seconds' ),
      'MAX_WAIT_MILLI_SEC_KW': CliMatcher.KeywordMatcher( 'milliseconds',
            helpdesc='Configure maximum interval between two SPFs in milliseconds' ),
      'INITIAL_WAIT' : CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min,
            SpfIntervalMsec.max,
            helpdesc='Initial wait interval for SPF in milliseconds' ),
      'INITIAL_WAIT_MILLI_SEC_KW': CliMatcher.KeywordMatcher( 'milliseconds',
            helpdesc='Configure initial wait interval for SPF in milliseconds' ),
      'HOLD_TIME' : CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min,
            SpfIntervalMsec.max,
            helpdesc='Hold interval between the first and'
                     ' second SPF runs in milliseconds' ),
      'HOLD_TIME_MILLI_SEC_KW': CliMatcher.KeywordMatcher( 'milliseconds',
            helpdesc='Configure hold interval between the first and'
                     ' second SPF runs in milliseconds' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setSpfInterval"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noSpfInterval"

if gatedToggleLib.toggleIsisSpfMaxIntervalMilliSecEnabled():
   RouterIsisMode.addCommandClass( RouterIsisSpfInterval )
else:
   RouterIsisMode.addCommandClass( RouterIsisSpfIntervalWithoutUnits )

#---------------------------------------------------------------------------------
# [no] spf partial prefix command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
class RouterIsisSpfPartial( CliCommand.CliCommandClass ):
   syntax = 'spf partial prefix'
   noOrDefaultSyntax = syntax
   data = {
      'spf' : 'Configure SPF',
      'partial' : 'Configure partial SPF computation methods',
      'prefix' : 'Partial SPF computation on prefix only changes',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setSpfPartial"
   defaultHandler = "RoutingIsisCliHandler.RouterIsisMode_setSpfPartial"
   noHandler = "RoutingIsisCliHandler.RouterIsisMode_noSpfPartial"
RouterIsisMode.addCommandClass( RouterIsisSpfPartial )

#---------------------------------------------------------------------------------
# [no] set-attached-bit route-map <rmName>  command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
setAttachedBitKw = CliCommand.guardedKeyword( 'set-attached-bit',
                   helpdesc='Configure conditionally setting attach bit',
                   guard=isisNonDefaultInstanceGuard )

class RouterIsisAttachedBitCmd( CliCommand.CliCommandClass ):
   syntax = 'set-attached-bit route-map RM_NAME'
   noOrDefaultSyntax = 'set-attached-bit ...'
   data = {
      'set-attached-bit': setAttachedBitKw,
      'route-map': RouteMapMatchers.routeMapApplication,
      'RM_NAME': mapNameMatcher,
      }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAttachedBit"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noSetAttachedBit"
RouterIsisMode.addCommandClass( RouterIsisAttachedBitCmd )

#---------------------------------------------------------------------------------
# [no|default] "maximum-paths <paths>" command, in 'address-family mode'
#---------------------------------------------------------------------------------
def afMaxEcmpRangeFn( mode, context ):
   '''
   This code can be called when the startup-config is being parsed,
   which happens before the ForwardingAgent has discovered all the hardware
   and populated Sysdb. Allowing any value in this case.
   '''
   ipv4 = mode.addrFamily == "ipv4"
   status = routingHardwareStatus if ipv4 else routing6HardwareStatus
   if status and status.maxLogicalProtocolEcmp:
      return ( 1, routingHardwareStatus.maxLogicalProtocolEcmp )
   return ( 1, 0xffffffff )

class RouterIsisAfMaxPathsCmd( CliCommand.CliCommandClass ):
   syntax = 'maximum-paths PATHS'
   noOrDefaultSyntax = 'maximum-paths ...'
   data = {
      'maximum-paths' : 'Maximum number of next hops in an ECMP route',
      'PATHS' : CliMatcher.DynamicIntegerMatcher( afMaxEcmpRangeFn,
         helpdesc='Maximum number of next hops in an ECMP route' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setMaxEcmp"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noMaxEcmp"

RouterIsisAfMode.addCommandClass( RouterIsisAfMaxPathsCmd )

#---------------------------------------------------------------------------------
# "[no|default] lsp match flag attached action none | (install default-route)"
# command in 'address-family mode'.
#---------------------------------------------------------------------------------
class RouterIsisAFLspMatcheFlagCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp match flag attached action ( none | ( install default-route ) )'
   noOrDefaultSyntax = 'lsp match flag attached action ...'
   data = {
      'lsp' : lspKw,
      'match' : 'Match on some LSP parameter',
      'flag' : 'Flag to match',
      'attached' : 'Match on attached bit in LSP',
      'action' : 'Action to be taken',
      'none' : 'Skip attached bit processing',
      'install' : 'Install based on attached bit',
      'default-route' : 'Install default route'
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setIgnoreAttachedBit"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noIgnoreAttachedBit"

RouterIsisAfMode.addCommandClass( RouterIsisAFLspMatcheFlagCmd )

#---------------------------------------------------------------------------------
# [no|default] lsp flooding dynamic [ LEVEL ]
# command in "router isis <n>" config mode
#---------------------------------------------------------------------------------

class RouterIsisLspFloodingDynamicCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp flooding dynamic [ LEVEL ]'
   noOrDefaultSyntax = syntax

   data = {
      'lsp'      : lspKw,
      'flooding' : 'Set LSP flooding related parameters',
      'dynamic'  : 'Enable Dynamic Flooding',
      'LEVEL' : levelMatcherForConfig,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_dynamicFlooding"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noDynamicFlooding"

RouterIsisMode.addCommandClass( RouterIsisLspFloodingDynamicCmd )

#---------------------------------------------------------------------------------
# [no|default] passive <intfName>  command in "router isis <n>" config mode
#
# deprecated:
# [no|default] passive-interface <intfName>  command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
passiveIntfDeprecatedKw = CliMatcher.KeywordMatcher( 'passive-interface',
                      helpdesc='Suppress routing updates on an interface' )

class RouterIsisPassiveCmd( CliCommand.CliCommandClass ):
   syntax = 'passive | passive-interface INTFS'
   noOrDefaultSyntax = syntax
   data = {
      'passive' : 'Suppress routing updates on an interface',
      'passive-interface' : CliCommand.Node( matcher=passiveIntfDeprecatedKw,
                                             deprecatedByCmd='passive' ),
      'INTFS' : IntfRange.IntfRangeMatcher(
         explicitIntfTypes=IntfRange.intfTypesWithRoutingProtoSupport ),
   }
   handler = "RoutingIsisCliHandler.setPassiveIntf"
   noOrDefaultHandler = "RoutingIsisCliHandler.noPassiveIntf"

RouterIsisMode.addCommandClass( RouterIsisPassiveCmd )

#--------------------------------------------------------------------------------
# [no|default] timers lsp generation < max-wait > [ initial-wait ] [ hold-wait ]
# command in "router isis <n>" config mode
#--------------------------------------------------------------------------------
class RouterIsisTimersLSPGenerationCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp generation MAX_WAIT [ INITIAL_WAIT [ HOLD_WAIT ] ]'
   noOrDefaultSyntax = 'timers lsp generation ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'generation' : 'LSP generation interval timer parameters',
      'MAX_WAIT' : CliMatcher.IntegerMatcher(
            LspGenIntervalSec.min,
            LspGenIntervalSec.max,
            helpdesc='Maximum interval (in seconds) between generating two LSPs' ),
      'INITIAL_WAIT' : CliMatcher.IntegerMatcher(
            LspGenIntervalMsec.min,
            LspGenIntervalMsec.max,
            helpdesc='Initial wait time (in milliseconds) before generating LSPs' ),
      'HOLD_WAIT' : CliMatcher.IntegerMatcher(
            LspGenIntervalMsec.min,
            LspGenIntervalMsec.max,
            helpdesc='Wait time (in milliseconds) between generating '
                     'the first and second LSPs' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setLspGenInterval"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noLspGenInterval"

RouterIsisMode.addCommandClass( RouterIsisTimersLSPGenerationCmd )

#-------------------------------------------------------------------------------
# [no|default] timers lsp out-delay <LSP out-delay>
#-------------------------------------------------------------------------------
class RouterIsisTimersLSPOutDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp out-delay OUT_DELAY'
   noOrDefaultSyntax = 'timers lsp out-delay ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'out-delay' : 'Configure out-delay timer',
      'OUT_DELAY' : CliMatcher.IntegerMatcher( 1, 65000,
            helpdesc='Transmit delay (in milliseconds) for link state packets' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setOutDelayTimer"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noOutDelayTimer"

RouterIsisMode.addCommandClass( RouterIsisTimersLSPOutDelayCmd )

#-------------------------------------------------------------------------------
# [no|default] timers csnp generation interval <n> seconds
# in 'router isis' global mode
#-------------------------------------------------------------------------------
class RouterIsisCsnpGenIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers csnp generation interval INTERVAL seconds'
   noOrDefaultSyntax = 'timers csnp generation interval ...'
   data = {
      'timers' : 'IS-IS timers',
      'csnp' : 'CSN Packet timers',
      'generation' : 'CSNP generation interval timer parameters',
      'interval' : 'Configure CSNP transmission interval',
      'INTERVAL' : CliMatcher.IntegerMatcher(
            CsnpGenInterval.min,
            CsnpGenInterval.max,
            helpdesc='Transmit frequency (in seconds) for CSN packets' ),
      'seconds' : 'seconds',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setCsnpGenInterval"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noCsnpGenInterval"
RouterIsisMode.addCommandClass( RouterIsisCsnpGenIntervalCmd )

#-------------------------------------------------------------------------------
# [no|default] timers csnp generation p2p disabled
# in 'router isis' global mode
#-------------------------------------------------------------------------------
class RouterIsisP2pInstPeriodicCsnpCmd( CliCommand.CliCommandClass ):
   syntax = 'timers csnp generation p2p disabled'
   noOrDefaultSyntax = syntax
   data = {
      'timers' : 'IS-IS timers',
      'csnp' : 'CSN Packet timers',
      'generation' : 'CSNP generation interval timer parameters',
      'p2p' : 'Configuration for P2P links',
      'disabled' : 'Disable periodic CSN packets',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setP2pInstPeriodicCsnp"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noP2pInstPeriodicCsnp"

RouterIsisMode.addCommandClass( RouterIsisP2pInstPeriodicCsnpCmd )

#------------------------------------------------------------------------------------
# timers local-convergence-delay [<delay_in_seconds>] protected-prefixes
# no|default timers local-convergence-delay protected-prefixes
# in 'router isis' global mode
#------------------------------------------------------------------------------------
convergenceDelayMin = 1
convergenceDelayMax = 65000
delayMatcher = CliMatcher.IntegerMatcher(
                  convergenceDelayMin, convergenceDelayMax,
                  helpdesc='Delay in milliseconds (default 10000 milliseconds)' )

class RouterIsisAndAfTimersLocalConvergenceDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'timers local-convergence-delay [ delay ] protected-prefixes'
   noOrDefaultSyntax = 'timers local-convergence-delay protected-prefixes'
   data = {
      'timers' : 'IS-IS timers',
      'local-convergence-delay' : 'Set local convergence delay',
      'delay' : delayMatcher,
      'protected-prefixes' : 'micro-loop prevention for routers over protected '
                             'links',
   }

   handler = "RoutingIsisCliHandler.RouterIsisMode_setMicroLoopConvergenceDelay"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_noMicroLoopConvergenceDelay"
RouterIsisMode.addCommandClass( RouterIsisAndAfTimersLocalConvergenceDelayCmd )

#------------------------------------------------------------------------------------
# timers local-convergence-delay [<delay_in_seconds>] protected-prefixes
# no|default timers local-convergence-delay protected-prefixes
# in 'router-isis-af' mode
#------------------------------------------------------------------------------------
class RouterIsisAndAfTimersLocalConvergenceDelayAfCmd( CliCommand.CliCommandClass ):
   syntax = 'timers local-convergence-delay [ delay ] protected-prefixes'
   noOrDefaultSyntax = 'timers local-convergence-delay protected-prefixes'
   data = {
      'timers' : 'IS-IS timers',
      'local-convergence-delay' : 'Set local convergence delay',
      'delay' : delayMatcher,
      'protected-prefixes' : 'micro-loop prevention for routers over protected '
                             'links',
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setMicroLoopConvergenceDelay"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfMode_noMicroLoopConvergenceDelay"
RouterIsisAfMode.addCommandClass( RouterIsisAndAfTimersLocalConvergenceDelayAfCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers lsp refresh <IS-IS LSP refresh-interval>"
#-------------------------------------------------------------------------------
class RouterIsisTimersLSPRefreshIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp refresh REFRESH_INTERVAL'
   noOrDefaultSyntax = 'timers lsp refresh ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'refresh' : 'Configure LSP refresh interval timer',
      'REFRESH_INTERVAL' : CliMatcher.IntegerMatcher(
            LspRefreshInterval.min,
            LspRefreshInterval.max,
            helpdesc='Interval (in seconds) between two LSP refreshes' )
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setLSPRefreshInterval"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noLSPRefreshInterval"

RouterIsisMode.addCommandClass( RouterIsisTimersLSPRefreshIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers lsp min-remaining-lifetime <n seconds>"
#-------------------------------------------------------------------------------
class RouterIsisTimersLSPMinRemainingLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp min-remaining-lifetime MIN_REMAINING_LIFETIME'
   noOrDefaultSyntax = 'timers lsp min-remaining-lifetime ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'min-remaining-lifetime' : 'Configure minimum remaining lifetime '
                                 'for non self-originating LSPs',
      'MIN_REMAINING_LIFETIME' : CliMatcher.IntegerMatcher(
            MinLspRemainingLifetime.min + 60,
            MinLspRemainingLifetime.max,
            helpdesc='Minimum remaining lifetime for LSPs (in seconds)' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setMinLspRemainingLifetime"
   noHandler = "RoutingIsisCliHandler.RouterIsisMode_noMinLspRemainingLifetime"
   defaultHandler = "RoutingIsisCliHandler.RouterIsisMode_setMinLspRemainingLifetime"

RouterIsisMode.addCommandClass( RouterIsisTimersLSPMinRemainingLifetimeCmd )

#---------------------------------------------------------------------------------
# area leader priority <priority> [ LEVEL ]
# area leader [ LEVEL ] [ disabled ]
# [no|default] area leader priority [ LEVEL ]
# [no|default] area leader [ LEVEL ] [ disabled ]
# command in "router isis" config mode
#---------------------------------------------------------------------------------
class RouterIsisAreaLeaderPriorityCmd( CliCommand.CliCommandClass ):
   syntax = '''area leader
               [ ( priority PRIORITY [ LEVEL ] )
               | ( [ LEVEL ] [ disabled ] ) ]'''
   noOrDefaultSyntax = 'area leader ...'
   data = {
      'area'     : 'Set area parameters',
      'leader'   : 'Set area leader parameters',
      'priority' : 'Set the area leader priority',
      'PRIORITY' : CliMatcher.IntegerMatcher( AreaLeaderPriority.min,
                                              AreaLeaderPriority.max,
                                              helpdesc='Area leader priority' ),
      'LEVEL' : levelMatcherForConfig,
      'disabled' : 'Disable becoming area leader',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAreaLeaderPriority"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_defaultAreaLeaderPriority"

RouterIsisMode.addCommandClass( RouterIsisAreaLeaderPriorityCmd )

#---------------------------------------------------------------------------------
# [no|default] metric profile <profileName>
# command in "router isis" config mode
#---------------------------------------------------------------------------------
_metricProfileDesc = 'Define metric profile'
profileNameMatcher = CliMatcher.PatternMatcher( instanceNamePattern,
      helpname='WORD', helpdesc='IS-IS metric profile name' )

class RouterIsisMetricProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'metric profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'metric' : _metricProfileDesc,
      'profile' : _metricProfileDesc,
      'PROFILE_NAME' : profileNameMatcher
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_gotoIsisMetricProfileMode"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_delIsisMetricProfileMode"

RouterIsisMode.addCommandClass( RouterIsisMetricProfileCmd )

#---------------------------------------------------------------------------------
# [no|default] router-id ipv4 <ipv4Addr>
# [no|default] router-id ipv6 <ipv6Addr>  hidden for now since it has no effect
# commands in "router isis" config mode
#---------------------------------------------------------------------------------
class RouterIsisRouterIdV4Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv4 ROUTER_IDV4'
   noOrDefaultSyntax = 'router-id ipv4 ...'
   data = {
      'router-id' : routerIdKw,
      'ipv4' : 'IS-IS router ID in IP address format',
      'ROUTER_IDV4' : routerIdV4Matcher
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setRouterIdV4"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRouterIdV4"

RouterIsisMode.addCommandClass( RouterIsisRouterIdV4Cmd )

class RouterIsisRouterIdV6Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv6 ROUTER_IDV6'
   noOrDefaultSyntax = 'router-id ipv6 ...'
   data = {
      'router-id' : routerIdKw,
      'ipv6' : CliCommand.Node( matcher=ipv6AddrFormatKw, hidden=True ),
      'ROUTER_IDV6' : routerIdV6Matcher
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setRouterIdV6"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRouterIdV6"

RouterIsisMode.addCommandClass( RouterIsisRouterIdV6Cmd )
metricMatcher = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Sets metric for Metric Profile' )
speedValueMatcher = CliMatcher.IntegerMatcher(
      Speed.min,
      Speed.max,
      helpdesc='Interface speed' )
unitsMatcher = CliMatcher.EnumMatcher( {
      SpeedUnitType.mbps : 'Megabits per second',
      SpeedUnitType.gbps : 'Gigabits per second',
} )

#---------------------------------------------------------------------------------
# [no|default] metric <metricValue>
# command in "router isis metric profile" config mode
#---------------------------------------------------------------------------------
class RouterIsisMetricValueCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRIC_VALUE'
   noOrDefaultSyntax = 'metric ...'
   data = {
      'metric' : metricMatcher,
      'METRIC_VALUE' : metricValueMatcher
   }
   handler = "RoutingIsisCliHandler.RouterIsisMetricProfileMode_setMetricValue"
   noOrDefaultHandler = handler

RouterIsisMetricProfileMode.addCommandClass( RouterIsisMetricValueCmd )

#---------------------------------------------------------------------------------
# [no|default] metric ratio <speed> <mbps|gbps>
# command in "router isis metric profile" config mode
#---------------------------------------------------------------------------------
class RouterIsisMetricRatioCmd( CliCommand.CliCommandClass ):
   syntax = 'metric ratio SPEED UNITS'
   noOrDefaultSyntax = 'metric ratio ...'
   data = {
      'metric' : metricMatcher,
      'ratio' : 'Sets metric inversely proportional to given <speed>',
      'SPEED' : speedValueMatcher,
      'UNITS' : unitsMatcher,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMetricProfileMode_setMetricRatio"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMetricProfileMode_delMetricRatio"

RouterIsisMetricProfileMode.addCommandClass( RouterIsisMetricRatioCmd )

#---------------------------------------------------------------------------------
# [no|default] metric <metricValue> if speed <= <speed> <mbps|gbps>
# command in "router isis metric profile" config mode
#---------------------------------------------------------------------------------
_metricRuleDesc = 'Given metric is used when interface speed matches condition'

class RouterIsisMetricValueSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRIC_VALUE if speed <= SPEED UNITS'
   noOrDefaultSyntax = syntax
   data = {
      'metric' : metricMatcher,
      'METRIC_VALUE' : metricValueMatcher,
      'if' : _metricRuleDesc,
      'speed' : _metricRuleDesc,
      '<=' : _metricRuleDesc,
      'SPEED' : speedValueMatcher,
      'UNITS' : unitsMatcher,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMetricProfileMode_addMetricRule"
   noOrDefaultHandler = handler

RouterIsisMetricProfileMode.addCommandClass( RouterIsisMetricValueSpeedCmd )

#-------------------------------------------------------------------------------
# Segment-Routing specific CLI
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# [no|default] shutdown
#-------------------------------------------------------------------------------
class RoutingIsisSrMplsShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable segment routing',
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_srShutdown"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_noSrShutdown"
RouterIsisSrMplsMode.addCommandClass( RoutingIsisSrMplsShutdownCmd )

#-------------------------------------------------------------------------------
# [no|default] segment path verification prefix-segment adjacency-segment
#-------------------------------------------------------------------------------
class RoutingIsisSrMplsSegmentPathVerificationCmd( CliCommand.CliCommandClass ):
   syntax = 'segment path verification prefix-segment adjacency-segment'
   noOrDefaultSyntax = syntax
   data = {
      'segment' : 'Configuration for segments',
      'path' : 'Configuration for the path to the segment',
      'verification' : 'Enable path verification for segments',
      'prefix-segment' : 'Enable path verification for prefix-segments',
      'adjacency-segment' : 'Enable path verification for adjacency-segments',
   }
   handler = \
            "RoutingIsisCliHandler.RouterIsisSrMplsMode_setSrSegmentPathVerification"
   noOrDefaultHandler = \
            "RoutingIsisCliHandler.RouterIsisSrMplsMode_noSrSegmentPathVerification"

RouterIsisSrMplsMode.addCommandClass( RoutingIsisSrMplsSegmentPathVerificationCmd )

#-------------------------------------------------------------------------------
# [no|default] router-id <router-id>
#-------------------------------------------------------------------------------
# router-id configuration is not supposed to be under the RouterIsisSrMplsMode.
# Make it hidden for backward compatibility.
srRouterIdKw = CliMatcher.KeywordMatcher( 'router-id',
      helpdesc='Configure the router ID for IS-IS Segment Routing process' )
routerIdMatcher = IpAddrMatcher.IpAddrMatcher(
      "IS-IS SR router ID in IP address format" )

class RouterIsisRouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ROUTER_ID'
   noOrDefaultSyntax = 'router-id [ ROUTER_ID ]'
   data = {
      'router-id' : CliCommand.Node( matcher=srRouterIdKw, hidden=True ),
      'ROUTER_ID' : routerIdMatcher
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setSrRouterId"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_noSrRouterId"

RouterIsisSrMplsMode.addCommandClass( RouterIsisRouterIdCmd )

#-------------------------------------------------------------------------------
# [no|default] adjacency-segment allocation [none|sr-peers [backup-eligible]|
# all-interfaces [backup-eligible] ]
#-------------------------------------------------------------------------------
srAdjacencySegmentKw = CliMatcher.KeywordMatcher( 'adjacency-segment',
                                    helpdesc='Adjacency Segment allocation options' )
allocationKw = CliMatcher.KeywordMatcher( 'allocation',
                                        helpdesc='Allocation of adjacency segments' )
noneKw = CliMatcher.KeywordMatcher( 'none',
                          helpdesc='Disable automatic adjacency segment allocation' )
srPeersKw = CliMatcher.KeywordMatcher( 'sr-peers',
          helpdesc='Allocate adjacency segments to IS-IS adjacencies with SR peers' )
allInterfacesKw = CliMatcher.KeywordMatcher( 'all-interfaces',
                    helpdesc='Allocate adjacency segments to all IS-IS adjacencies' )

class RouterIsisSrMplsAdjSegAllocationCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency-segment allocation '\
            '( none | ( sr-peers | all-interfaces [ backup-eligible ] ) )'
   noOrDefaultSyntax = 'adjacency-segment allocation ...'
   data = {
      'adjacency-segment' : srAdjacencySegmentKw,
      'allocation' : allocationKw,
      'none' : noneKw,
      'sr-peers' : srPeersKw,
      'all-interfaces' : allInterfacesKw,
      'backup-eligible' : backupKw
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_adjacencySegmentAllocation"
   noOrDefaultHandler = \
            "RoutingIsisCliHandler.RouterIsisSrMplsMode_noAdjacencySegmentAllocation"
RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsAdjSegAllocationCmd )

#-------------------------------------------------------------------------------
# [no|default] adjacency-segment sid reuse timeout <time>  XXX: Hidden
#-------------------------------------------------------------------------------
sidKw = CliMatcher.KeywordMatcher( 'sid', helpdesc='Adjacency segment SID' )
reUseKw = CliMatcher.KeywordMatcher( 'reuse',
                                     helpdesc='Reserve the SID for reuse' )
timeoutKw = CliMatcher.KeywordMatcher( 'timeout',
                                     helpdesc='Set the reserve timeout for the SID' )
adjSegmentSidReuseTimeout = CliMatcher.IntegerMatcher( 0, U32_MAX_VALUE - 1,
                                             helpdesc='Reserve timeout in seconds' )
adjSegmentSidReuseTimeoutInf = CliMatcher.KeywordMatcher( 'infinite',
                                                  helpdesc='Reserve SID infinitely' )
class RouterIsisSrMplsAdjSegSidReuseTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency-segment sid reuse timeout ( TIMEOUT | infinite )'
   noOrDefaultSyntax = 'adjacency-segment sid reuse timeout ...'
   data = {
      'adjacency-segment' : srAdjacencySegmentKw,
      'sid' : CliCommand.Node( matcher=sidKw, hidden=True ),
      'reuse' : reUseKw,
      'timeout' : timeoutKw,
      'TIMEOUT' : adjSegmentSidReuseTimeout,
      'infinite' : adjSegmentSidReuseTimeoutInf
   }
   handler = \
         "RoutingIsisCliHandler.RouterIsisSrMplsMode_adjacencySegmentSidReuseTimeout"
   noOrDefaultHandler = \
       "RoutingIsisCliHandler.RouterIsisSrMplsMode_noAdjacencySegmentSidReuseTimeout"
RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsAdjSegSidReuseTimeoutCmd )

#-------------------------------------------------------------------------------
# [no|default] prefix-segment <prefix> index <index> | label <label>
#                     [ flex-algo <name> ] [ explicit-null | no-php ]
#-------------------------------------------------------------------------------
srPrefixSegmentKw = CliMatcher.KeywordMatcher( 'prefix-segment',
      helpdesc='Configure a prefix segment' )
indexKw = CliMatcher.KeywordMatcher( 'index',
      helpdesc='Prefix segment identifier' )
indexMatcher = CliMatcher.IntegerMatcher( MplsLabel.min,
                              MplsLabel.max - MplsLabel.unassignedMin + 1,
                              helpdesc='Index value' )
srLabelMatcher = CliMatcher.IntegerMatcher( MplsLabel.unassignedMin, MplsLabel.max,
                                  helpdesc="Absolute sid for the prefix" )
flexAlgoNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( [ fad.name for fad in flexAlgoConfig.definition.values() ]
         if flexAlgoConfig else [] ),
      helpdesc='Algorithm name' )
class RouterIsisSrMplsPrefixSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'prefix-segment ( IPADDR | IP6ADDR ) ( index INDEX ) '
   noOrDefaultSyntax = 'prefix-segment ( IPADDR | IP6ADDR ) '
   if IsisToggleLib.toggleIsisPrefixSegmentAbsoluteSidEnabled():
      syntax = syntax + '| ( label LABEL ) '
   if gatedToggleLib.togglePrefixSidInFlexAlgoEnabled():
      syntax = syntax + '[ flex-algo FAD_NAME ] '
      noOrDefaultSyntax = noOrDefaultSyntax + \
         '[ ( index INDEX ) | ( label LABEL ) [ flex-algo FAD_NAME ] ] '
   syntax = syntax + '[ explicit-null | no-php ]'
   noOrDefaultSyntax = noOrDefaultSyntax + '...'
   data = {
      'prefix-segment' : srPrefixSegmentKw,
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'index' : indexKw,
      'INDEX' : indexMatcher,
      'label': 'Prefix segment label',
      'LABEL': srLabelMatcher,
      'flex-algo': 'Flexible algorithm',
      'FAD_NAME': flexAlgoNameMatcher,
      'explicit-null': 'Set explicit-null flag',
      'no-php': 'Set no-php flag',
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setPrefixSegment"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_noPrefixSegment"

RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsPrefixSegmentCmd )

proxyNodeSegmentKw = CliMatcher.KeywordMatcher( 'proxy-node-segment',
                  helpdesc='Configure a node segment on behalf of another node' )
proxyNodeAttachedKw = CliMatcher.KeywordMatcher( 'attached',
            helpdesc='Directly connected prefix' )

class RouterIsisSrMplsProxyNodeSegmentAttachedFlagCmd( CliCommand.CliCommandClass ):
   syntax = 'proxy-node-segment ( ( attached-flag inspect ) | \
            ( ( IPADDR | IP6ADDR ) index INDEX [ LEVEL ] [ attached ] ) )'
   noOrDefaultSyntax = 'proxy-node-segment ( ( attached-flag ... ) | \
            ( ( IPADDR | IP6ADDR ) ... ) )'
   data = {
      'proxy-node-segment' : proxyNodeSegmentKw,
      'attached-flag' : proxyNodePhpAttachedKw,
      'inspect' : proxyNodeAttachedFlagKw,
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'index' : indexKw,
      'INDEX' : indexMatcher,
      'LEVEL' : level12MatcherForConfig,
      'attached' : proxyNodeAttachedKw,
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setProxyNodeSegment"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisSrMplsMode_noProxyNodeSegment"

#-------------------------------------------------------------------------------
# [no|default] proxy-node-segment <host prefix> index <index> [ range <range> ]
#                                             [ LEVEL ] [ attached ]
#-------------------------------------------------------------------------------
def getProxyNodeRange( mode, context ):
   minVal = PrefixRange.min
   if 'INDEX' in context.sharedResult:
      maxIndex = MplsLabel.max - MplsLabel.unassignedMin + 1
      startIndex = context.sharedResult[ 'INDEX' ]
   else:
      maxIndex = MplsLabel.max
      startIndex = context.sharedResult[ 'LABEL' ]
   availableRangeMaxValue = maxIndex - startIndex + 1
   maxVal = min( availableRangeMaxValue, PrefixRange.max )
   return ( minVal, maxVal )

proxyNodeRangeMatcher = CliMatcher.DynamicIntegerMatcher( getProxyNodeRange,
                                               helpdesc='Number of proxy node SIDs' )
rangeKw = CliMatcher.KeywordMatcher( 'range',
      helpdesc='Specify a range of proxy node SIDs' )
class RouterIsisSrMplsProxyNodeSegmentRangeCmd( CliCommand.CliCommandClass ):
   syntax = '''
   proxy-node-segment
   ( ( attached-flag inspect )
   | ( ( IPADDR | IP6ADDR ) index INDEX [ range RANGE ] [ LEVEL ] [ attached ] ) )'''
   noOrDefaultSyntax = '''
   proxy-node-segment
   ( ( attached-flag ... )
   | ( ( IPADDR | IP6ADDR ) ... ) )'''
   data = {
      'proxy-node-segment': proxyNodeSegmentKw,
      'attached-flag': proxyNodePhpAttachedKw,
      'inspect': proxyNodeAttachedFlagKw,
      'IPADDR': ipv4IPMatcher,
      'IP6ADDR': ipv6IPMatcher,
      'index': indexKw,
      'INDEX': CliCommand.Node( indexMatcher, storeSharedResult=True ),
      'range': rangeKw,
      'RANGE': proxyNodeRangeMatcher,
      'LEVEL': level12MatcherForConfig,
      'attached': proxyNodeAttachedKw,
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setProxyNodeSegment"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisSrMplsMode_noProxyNodeSegment"

class RouterIsisSrMplsProxyNodeSegmentLabelRangeCmd( CliCommand.CliCommandClass ):
   syntax = '''
   proxy-node-segment
   ( ( attached-flag inspect )
   | ( ( IPADDR | IP6ADDR ) ( index INDEX ) | ( label LABEL ) [ range RANGE ]
     [ LEVEL ] [ attached ] ) )'''
   noOrDefaultSyntax = '''
   proxy-node-segment
   ( ( attached-flag ... )
   | ( ( IPADDR | IP6ADDR ) ... ) )'''
   data = {
      'proxy-node-segment': proxyNodeSegmentKw,
      'attached-flag': proxyNodePhpAttachedKw,
      'inspect': proxyNodeAttachedFlagKw,
      'IPADDR': ipv4IPMatcher,
      'IP6ADDR': ipv6IPMatcher,
      'index': indexKw,
      'INDEX': CliCommand.Node( indexMatcher, storeSharedResult=True ),
      'label': 'Proxy Node segment label',
      'LABEL': CliCommand.Node( srLabelMatcher, storeSharedResult=True ),
      'range': rangeKw,
      'RANGE': proxyNodeRangeMatcher,
      'LEVEL': level12MatcherForConfig,
      'attached': proxyNodeAttachedKw,
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setProxyNodeSegment"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisSrMplsMode_noProxyNodeSegment"

conflictKw = CliMatcher.KeywordMatcher( 'conflict',
            helpdesc='Proxy-Node range segment conflicts' )
ignoreKw = CliMatcher.KeywordMatcher( 'ignore',
            helpdesc='Ignore conflicts with other segments' )

class RouterIsisSrMplsProxyNodeRangeConflictIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = '''
   proxy-node-segment
   ( ( attached-flag inspect )
   | ( ( IPADDR | IP6ADDR ) ( index INDEX ) | ( label LABEL )
     [ range RANGE [ conflict ignore ] ] [ LEVEL ] [ attached ] ) )'''
   noOrDefaultSyntax = '''
   proxy-node-segment
   ( ( attached-flag ... )
   | ( ( IPADDR | IP6ADDR ) ... ) )'''
   data = {
      'proxy-node-segment': proxyNodeSegmentKw,
      'attached-flag': proxyNodePhpAttachedKw,
      'inspect': proxyNodeAttachedFlagKw,
      'IPADDR': ipv4IPMatcher,
      'IP6ADDR': ipv6IPMatcher,
      'index': indexKw,
      'INDEX': CliCommand.Node( indexMatcher, storeSharedResult=True ),
      'label': 'Proxy Node segment label',
      'LABEL': CliCommand.Node( srLabelMatcher, storeSharedResult=True ),
      'range': rangeKw,
      'RANGE': proxyNodeRangeMatcher,
      'LEVEL': level12MatcherForConfig,
      'attached': proxyNodeAttachedKw,
      'conflict': conflictKw,
      'ignore': ignoreKw,
   }
   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setProxyNodeSegment"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisSrMplsMode_noProxyNodeSegment"

if IsisToggleLib.toggleIsisProxyNodeIgnoreConflictEnabled():
   RouterIsisSrMplsMode.addCommandClass(
         RouterIsisSrMplsProxyNodeRangeConflictIgnoreCmd )
elif IsisToggleLib.toggleIsisProxyNodeAbsoluteSidRangeEnabled():
   RouterIsisSrMplsMode.addCommandClass(
         RouterIsisSrMplsProxyNodeSegmentLabelRangeCmd )
elif gatedToggleLib.toggleIsisProxyNodeSidRangeEnabled():
   RouterIsisSrMplsMode.addCommandClass(
                           RouterIsisSrMplsProxyNodeSegmentRangeCmd )
else:
   RouterIsisSrMplsMode.addCommandClass(
                           RouterIsisSrMplsProxyNodeSegmentAttachedFlagCmd )

#-------------------------------------------------------------------------------
# '[no|default] flex-algo <name> [level-1|level-2|level-1-2] [advertised]
#-------------------------------------------------------------------------------

def flexAlgoDefinitionNames( mode ):
   # Look through the defined set of algorithms in the TE configuration
   # and return the list of names found there.
   # pylint: disable-next=unnecessary-comprehension
   algoIds = [ algoId for algoId in flexAlgoConfig.definition ]
   names = [ flexAlgoConfig.definition[ algoId ].name for algoId in algoIds ]
   return names

class RouterIsisSrMplsFlexAlgoCmd( CliCommand.CliCommandClass ):
   syntax = 'flex-algo NAME [ LEVEL ] [ advertised ]'
   noOrDefaultSyntax = syntax
   data = {
      'flex-algo' : 'Participate in a SR flexible algorithm',
      'NAME' : CliMatcher.DynamicNameMatcher( flexAlgoDefinitionNames,
                                              helpdesc='Algorithm name' ),
      'LEVEL' : level12MatcherForConfig,
      'advertised' : 'Advertise the definition to the network',
   }

   handler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_setFlexAlgo"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisSrMplsMode_noFlexAlgo"

RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsFlexAlgoCmd )

#-------------------------------------------------------------------------------
# segment-routing ipv6 specific CLI
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# '[no|default] disabled' under 'segement-routing ipv6' mode
#-------------------------------------------------------------------------------
class RouterIsisAfSrv6DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled' : 'Disable segment routing over ipv6',
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfSrv6Mode_srv6Disabled"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfSrv6Mode_noSrv6Disabled"
RouterIsisAfSrv6Mode.addCommandClass( RouterIsisAfSrv6DisabledCmd )

#-------------------------------------------------------------------------------
# '[no|default] locator LOCATOR' under 'segement-routing ipv6' mode
#-------------------------------------------------------------------------------

def srv6LocatorNames( mode ):
   config = isisConfig.instanceConfig[ mode.instanceName ]
   return config.srv6Locator

class RouterIsisAfSrv6LocatorCmd( CliCommand.CliCommandClass ):
   syntax = 'locator LOCATOR'
   noOrDefaultSyntax = syntax
   data = {
      'locator' : 'SRv6 Locator configuration',
      'LOCATOR' : CliMatcher.DynamicNameMatcher( srv6LocatorNames,
                      helpdesc='Locator name' )
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfSrv6Mode_gotoIsisAfSrv6LocatorMode"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfSrv6Mode_delIsisAfSrv6LocatorMode"

RouterIsisAfSrv6Mode.addCommandClass( RouterIsisAfSrv6LocatorCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute bgp' under 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
class RouterIsisAfModeRedistributeCmd( OspfRedistributeCmdData ):
   _ospfV3RedistRule = 'ospfv3 ' + matchRoute
   syntax = 'redistribute ( bgp | ( ' + _ospfV3RedistRule + ' ) ) ' + \
            '[ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute ( bgp | ( ' + _ospfV3RedistRule + ') ) ...'
   data = {
      'bgp' : bgpCliCommandNode
   }
   data.update( OspfRedistributeCmdData.data )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRedistribute4"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noRedistribute4"
RouterIsisAfMode.addCommandClass( RouterIsisAfModeRedistributeCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute user [route-map <route-map-name>] '
# 'under 'router isis' and 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
class RedistributeUserCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute user [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute user ...'
   data = {
      'user' : 'EosSdk routes'
   }
   data.update( redistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistribute5"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistribute5"

if IsisToggleLib.toggleEosSdkRedistributionIsisEnabled():
   RouterIsisMode.addCommandClass( RedistributeUserCmd )

# -------------------------------------------------------------------------------
# [no|default] redistribute <static|connected> [include leaked]
#              [route-map <route-map-name>]
# under 'address-family ipv[4|6]' mode
# -------------------------------------------------------------------------------

class RedistributeProtocolAfCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute PROTOCOL [ include leaked ] [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute PROTOCOL ...'
   data = {
      'redistribute': redistributeKw,
      'PROTOCOL': protocolMatcher,
      'route-map': RouteMapMatchers.routeMapApplication,
      'MAPNAME': mapNameMatcher,
      'include': 'Include leaked routes',
      'leaked': 'Include leaked routes',
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRedistributeStaticConnected"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfMode_noRedistributeStaticConnected"

if IsisToggleLib.toggleIsisPerAfRedistributeStaticConnectedEnabled():
   RouterIsisAfMode.addCommandClass( RedistributeProtocolAfCommand )

#-------------------------------------------------------------------------------
# '[no|default] redistribute dynamic [route-map <route-map-name>] '
# 'under 'router isis' and 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
class RedistributeDynamicCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute dynamic [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute dynamic ...'
   data = {
      'dynamic' : 'Dynamic policy routes'
   }
   data.update( redistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistribute6"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistribute6"

RouterIsisMode.addCommandClass( RedistributeDynamicCmd )

#-----------------------------------------------------------------------------------
# [no|default] tunnel source-protocol bgp ipv4 labeled-unicast [ rcf <func-name> ]
# under 'address-family ipv4' mode
#-----------------------------------------------------------------------------------
class RedistributeBgpLuCommand( CliCommand.CliCommandClass ):
   syntax = 'tunnel source-protocol bgp ipv4 labeled-unicast [ rcf FUNCTION ]'
   noOrDefaultSyntax = syntax
   data = {
         'tunnel': 'Tunnel protocols',
         'source-protocol': 'Configure the tunnel source',
         'bgp' : 'BGP routes',
         'ipv4' : 'IPv4 related',
         'labeled-unicast' : 'Labeled-unicast sub-address family',
         'rcf' : 'Routing control functions configuration to filter and '
                 'modify paths',
         'FUNCTION' : rcfFunctionMatcher,
   }
   data.update( redistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setBgpLuDirectRedistributeV4"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfMode_noBgpLuDirectRedistributeV4"

RouterIsisAfIpv4Modelet.addCommandClass( RedistributeBgpLuCommand )

#-------------------------------------------------------------------------------
# '[no|default] redistribute dhcp' under 'address-family ipv6' mode
#-------------------------------------------------------------------------------
class RouterIsisAfIpv6RedistributeDhcp( CliCommand.CliCommandClass ):
   syntax = 'redistribute dhcp [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute dhcp ...'
   data = {
      'redistribute' : redistributeKw,
      'dhcp' : 'DHCPv6 routes',
      'route-map' : RouteMapMatchers.routeMapApplication,
      'MAPNAME' : mapNameMatcher
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfIpv6Modelet_setRedistribute7"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfIpv6Modelet_noRedistribute7"

RouterIsisAfIpv6Modelet.addCommandClass( RouterIsisAfIpv6RedistributeDhcp )

class SummaryL1ToL2Expression( CliCommand.CliExpression ):
   """Common syntax for isis leaked route summarization for both ipv4 and ipv6
   address-family mode"""
   expression = 'redistribute isis level-1 into level-2 summary-address'
   data = {
      'redistribute' : redistributeKw,
      'isis' : 'IS-IS routes',
      'level-1' : 'level-1 routes',
      'into' : 'into IS-IS level',
      'level-2' : 'into level-2 routes',
      'summary-address' : 'Summary route configuration'
   }

#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1 into level-2 summary-address V4PREFIX'
# under 'address-family ipv4' mode
#-------------------------------------------------------------------------------
class RouterIsisAf4ModeSummaryAddressL1ToL2LeakCmd( CliCommand.CliCommandClass ):
   syntax = 'SUMMARY_L1_TO_L2_EXPR V4PREFIX'
   noOrDefaultSyntax = 'SUMMARY_L1_TO_L2_EXPR V4PREFIX'
   data = {
      "SUMMARY_L1_TO_L2_EXPR" : SummaryL1ToL2Expression,
      "V4PREFIX" : ipv4IPMatcher
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfIpv4Modelet_setLeakSummaryAddress"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfIpv4Modelet_setLeakSummaryAddress_no"

RouterIsisAfIpv4Modelet.addCommandClass(
   RouterIsisAf4ModeSummaryAddressL1ToL2LeakCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1 into level-2 summary-address V6PREFIX'
# under 'address-family ipv6' mode
#-------------------------------------------------------------------------------
class RouterIsisAf6ModeSummaryAddressL1ToL2LeakCmd( CliCommand.CliCommandClass ):
   syntax = 'SUMMARY_L1_TO_L2_EXPR V6PREFIX'
   noOrDefaultSyntax = 'SUMMARY_L1_TO_L2_EXPR V6PREFIX'
   data = {
      'SUMMARY_L1_TO_L2_EXPR' : SummaryL1ToL2Expression,
      'V6PREFIX' : ipv6IPMatcher
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfIpv6Modelet_setLeakSummaryAddress"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfIpv6Modelet_setLeakSummaryAddress_no"

RouterIsisAfIpv6Modelet.addCommandClass(
   RouterIsisAf6ModeSummaryAddressL1ToL2LeakCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1|level-2 into level-2|level-1 route-map <>'
# under 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
class RouterIsisAfModeRedistributeLeakL1ToL2Cmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis level-1 into level-2 route-map MAPNAME'
   noOrDefaultSyntax = 'redistribute isis level-1 into level-2 route-map ...'
   data = {
      'level-1' : 'level-1 routes',
      'level-2' : 'into level-2 routes'
   }
   data.update( isisRedistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRouteLeak1"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noRouteLeak1"

RouterIsisAfMode.addCommandClass( RouterIsisAfModeRedistributeLeakL1ToL2Cmd )

class RouterIsisAfModeRedistributeLeakL2ToL1Cmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis level-2 into level-1 route-map MAPNAME'
   noOrDefaultSyntax = 'redistribute isis level-2 into level-1 route-map ...'
   data = {
      'level-2' : 'level-2 routes',
      'level-1' : 'into level-1 routes'
   }
   data.update( isisRedistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRouteLeak2"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noRouteLeak2"

RouterIsisAfMode.addCommandClass( RouterIsisAfModeRedistributeLeakL2ToL1Cmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute isis instance [route-map <>]'
# under 'router isis' and 'address-family ipv[4|6]' modes
#-------------------------------------------------------------------------------
class RedistributeIsisInstanceCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis instance [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute isis instance ...'
   data = {
      'instance' : 'routes from another IS-IS instance',
   }
   data.update( isisRedistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisMode_setRedistribute8"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noRedistribute8"
RouterIsisMode.addCommandClass( RedistributeIsisInstanceCmd )

#----------------------------------------------------------------------------------
# [no|default] "distance <1-255> [ LEVEL ]" command, in address-family mode
#----------------------------------------------------------------------------------
distanceMin = 1
distanceMax = 255

class RouterIsisAFDistanceCmd( CliCommand.CliCommandClass ):
   syntax = 'distance DISTANCE_VALUE [ LEVEL ]'
   noOrDefaultSyntax = 'distance [ DISTANCE_VALUE ] [ LEVEL ]'
   data = {
      'distance' : 'Protocol instance administrative distance',
      'DISTANCE_VALUE' : CliMatcher.IntegerMatcher( distanceMin, distanceMax,
                            helpdesc='Administrative distance value' ),
      'LEVEL' : CliMatcher.EnumMatcher( {
            'level-1' : 'Level-1 administrative distance',
            'level-2' : 'Level-2 administrative distance',
      } ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setDistance"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noDistance"

RouterIsisAfMode.addCommandClass( RouterIsisAFDistanceCmd )

# [no] authentication mode ( md5 | sha key-id <id> | text | shared-secret profile
# <profileName> algorithm ( md5 | sha-1 | sha-224 | sha-256 | sha-384 | sha-512 ) )
# [ level-1 | level-2 | ( rx-disabled [ level-1 | level-2 ] ) ]
#---------------------------------------------------------------------------------

class RouterIsisModeAuthenticationMode( CliCommand.CliCommandClass ):
   syntax = 'authentication mode AUTHMODE [ rx-disabled ] [ LEVEL ]'
   noOrDefaultSyntax = 'authentication mode [ AUTHMODE [ rx-disabled ] [ LEVEL ] ]'
   data = {
      'authentication' : authKw,
      'mode' : modeKw,
      'AUTHMODE' : AuthenticationModeExpression,
      'rx-disabled' : rxDisabledKw,
      'LEVEL' : levelMatcherForConfig,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAuthMode"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noAuthMode"

RouterIsisMode.addCommandClass( RouterIsisModeAuthenticationMode )


#
#---------------------------------------------------------------------------------
# [no] authentication [key-id <id> algorithm SHA]
#       key [ [0|7|8a] < key-string > [ LEVEL ] ]
#---------------------------------------------------------------------------------

class RouterIsisAuthenticationKeyCmd( CliCommand.CliCommandClass ):
   syntax = authKeySyntax
   noOrDefaultSyntax = noOrDefaultAuthSyntax
   data = authData
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAuthKey"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noAuthKey"

RouterIsisMode.addCommandClass( RouterIsisAuthenticationKeyCmd )

#-----------------------------------------------------------------------------
# [no| default ] maximum-sid-depth base-mpls-imposition
#-----------------------------------------------------------------------------
maxSidDepthKw = CliMatcher.KeywordMatcher( 'maximum-sid-depth',
                                    helpdesc='Advertise maximum SID depth sub-tlv ' )
class  RouterIsisMaximumSidDepthCmd( CliCommand.CliCommandClass ):
   syntax = 'maximum-sid-depth MSD_TYPE'
   noOrDefaultSyntax = syntax
   data = {
           'maximum-sid-depth' : maxSidDepthKw,
           'MSD_TYPE' : CliMatcher.EnumMatcher( {
                        'base-mpls-imposition' : 'Advertise MSD type base MPLS'
                                                 ' imposition, number of MPLS'
                                                 ' labels which can be imposed',
                        })
          }
   noOrDefaultHandler = handler = \
                                "RoutingIsisCliHandler.RouterIsisMode_setMaxSidDepth"
RouterIsisMode.addCommandClass( RouterIsisMaximumSidDepthCmd )

#-------------------------------------------------------------------------------
# Adds isis-specific CLI commands to the "config-if" mode for routed ports.
# Enable isis commands under config-if only on routed and loopback ports
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# SrConfig stores all common sr configuration for OSPF and ISIS (only node segment
# for now)
#-------------------------------------------------------------------------------
srConfig = None

#-------------------------------------------------------------------------------
# IntfConfig for ISIS configuration, is created when one of its attributes
# is configured. It is deleted when all the attributes are at their defaults
# What this means is that after the last "no isis..." command is run on the
# interface, we delete the intfConfig object
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis enable"
# command, in config-if mode.
# "[no|default] isis instance <INSTANCE_NAME>"
# command, in loopback config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis rfc8202 disabled
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis metric <value> | <maximum>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 metric <value> | <maximum>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis metric profile <profileName>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 metric profile <profileName>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis multi-topology address-family ( ipv4 | ipv6 ) [ unicast ]"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis circuit-type LEVEL"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis area proxy boundary"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis priority"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis hello-interval <interval>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis hello-multiplier <multiplier>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis lsp tx interval <interval>"
# command, in config-if mode.
#
# legacy:
# "[no|default] isis lsp-interval <interval>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#"[no|default] isis passive
# command in config-if mode
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis network point-to-point"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#---------------------------------------------------------------------------------
# [no] isis authentication mode [ < text|md5|sha key-id <keyid> > [rx-disabled]
# [ LEVEL ] ]
# in intf config mode
#---------------------------------------------------------------------------------

#
#---------------------------------------------------------------------------------
# [no] isis authentication [key-id <id> algorithm SHA ]
#       key [ [0|7] < key-string > [ LEVEL ] ]
#---------------------------------------------------------------------------------

# Helper function to check for a conflicting prefix/proxy/node segment

# Helper function to check for a conflicting SR SID

#-------------------------------------------------------------------------------
# "[no|default] isis bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------


#-------------------------------------------------------------------------------
# "[no|default] advertise high-metrics [include redistributed]
#                       [ on-startup ( DELAY | wait-for-bgp [timeout DELAY ] ) ]"
# command, in isis config mode.
#-------------------------------------------------------------------------------
advertiseMatcher = CliMatcher.KeywordMatcher( 'advertise',
                              helpdesc='Control routing advertisements' )

class RouterIsisAdvertiseHighMetricCmd( CliCommand.CliCommandClass ):
   syntax = '''advertise high-metrics
                  [ include redistributed ]
                  [ on-startup
                     ( DELAY |
                     ( wait-for-bgp [ timeout DELAY ] ) ) ]'''
   noOrDefaultSyntax = syntax
   data = {
      'advertise': advertiseMatcher,
      'high-metrics': 'Advertise high metric value on all interfaces',
      'include': 'Advertise high-metrics for redistributed routes',
      'redistributed': 'Advertise high-metrics for redistributed routes',
      'on-startup': 'Advertise at startup for a specified number of seconds or'
      ' wait for BGP convergence',
      'wait-for-bgp': 'Wait for BGP convergence',
      'timeout': 'Timeout for BGP convergence',
      'DELAY': CliMatcher.IntegerMatcher( 1,
                                          OverloadStartupDelay.max,
                                          helpdesc='Number of seconds' ),
      }
   handler = "RoutingIsisCliHandler.RouterIsisMode_advertiseHighMetric"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noAdvertiseHighMetric"
RouterIsisMode.addCommandClass( RouterIsisAdvertiseHighMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart" command, in isis config mode
#-------------------------------------------------------------------------------
gracefulRestartMatcher = CliMatcher.KeywordMatcher(
      'graceful-restart', helpdesc='Enable graceful restart mode' )

class RouterIsisGracefulRestartCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart' : gracefulRestartMatcher,
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setGracefulRestart"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noGracefulRestart"

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart-helper" command, in isis config mode
#-------------------------------------------------------------------------------

class RouterIsisGracefulRestartHelperCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart-helper'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart-helper' : 'Enable graceful restart helper',
   }
   noOrDefaultHandler = handler = "RoutingIsisCliHandler.RouterIsisMode_setGrHelper"

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartHelperCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart t2" command, in isis config mode
#-------------------------------------------------------------------------------
class RouterIsisGracefulRestartT2Cmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart t2 [ LEVEL ] SECONDS'
   noOrDefaultSyntax = 'graceful-restart t2 [ LEVEL ] ...'
   data = {
      'graceful-restart' : gracefulRestartMatcher,
      't2' : 'Set LSP database sync wait time',
      'LEVEL' : CliMatcher.EnumMatcher( {
            'level-1' : 'Set t2 for level-1 only',
            'level-2' : 'Set t2 for level-2 only',
      } ),
      'SECONDS' : CliMatcher.IntegerMatcher(
            GrTimerT2.min,
            GrTimerT2.max,
            helpdesc='Number of seconds' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setGrTimerT2"
   noOrDefaultHandler = handler

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartT2Cmd )

#--------------------------------------------------------------------------------
# "[no|default] graceful-restart restart-hold-time <n seconds>"
#  command in isis config mode
#--------------------------------------------------------------------------------
class RouterIsisGracefulRestartHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart restart-hold-time SECONDS'
   noOrDefaultSyntax = 'graceful-restart restart-hold-time ...'
   data = {
      'graceful-restart' : gracefulRestartMatcher,
      'restart-hold-time' : 'Set hold time when restarting',
      'SECONDS' : CliMatcher.IntegerMatcher(
            GrHoldTime.min + 5,
            GrHoldTime.max,
            helpdesc='Number of seconds' ),
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setGrHoldTime"
   noOrDefaultHandler = handler

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartHoldTimeCmd )

#--------------------------------------------------------------------------------
# "[no|default] database publish"  in 'router isis <>' mode
#--------------------------------------------------------------------------------
class RouterIsisDatabasePublishCmd( CliCommand.CliCommandClass ):
   syntax = 'database publish'
   noOrDefaultSyntax = syntax
   data = {
      'database' : 'IS-IS Link State Database',
      'publish' : 'Publish database to shared memory',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setPublishLsdb"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noPublishLsdb"

RouterIsisMode.addCommandClass( RouterIsisDatabasePublishCmd )

#--------------------------------------------------------------------------------
# "[no|default] advertise ip-reachability [ passive-only ]" in
# 'router isis <>' mode
# deprecated:
# "[no|default] advertise passive-only" hidden for backward compatibility
#-------------------------------------------------------------------------------
class RouterIsisAdvertiseIpReachabilityCmd( CliCommand.CliCommandClass ):
   syntax = '''advertise ( PASSIVE_ONLY_DEPR_KW
                         | ( ip-reachability [ passive-only ] ) )'''
   noOrDefaultSyntax = syntax
   data = {
      'advertise': advertiseMatcher,
      # Only connected prefixes of passive interfaces
      'PASSIVE_ONLY_DEPR_KW': CliCommand.hiddenKeyword( 'passive-only',
                                                        alias='passive-only' ),
      'ip-reachability': 'Advertise IP reachability TLVs',
      'passive-only': 'Advertise IP reachability TLVs only for passive interfaces'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAdvertisePassiveOnly"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_noSetAdvertisePassiveOnly"

RouterIsisMode.addCommandClass( RouterIsisAdvertiseIpReachabilityCmd )

def configCovertIsisAdvertisePassiveOnly( mode ):
   isisConfig.advertisePassiveOnlyUseNewSyntax = True

ConfigConvert.registerConfigConvertCallback(
               configCovertIsisAdvertisePassiveOnly )

# --------------------------------------------------------------------------------
# "[no|default] advertise interface-address [ passive-only ]"  in
# 'router isis <>' mode
# --------------------------------------------------------------------------------
class RouterIsisAdvertiseInterfaceAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'advertise interface-address [ passive-only ]'
   noOrDefaultSyntax = 'advertise interface-address ...'
   data = {
      'advertise': advertiseMatcher,
      'interface-address': 'Advertise interface address TLVs',
      'passive-only': 'Advertise interface address TLVs only for passive interfaces'
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAdvertiseInterfaceAddress"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_noSetAdvertiseInterfaceAddress"

RouterIsisMode.addCommandClass( RouterIsisAdvertiseInterfaceAddressCmd )

#-------------------------------------------------------------------------------
# "[no|default] adjacency address-family match disabled" command, in isis config mode
#-------------------------------------------------------------------------------
class RouterIsisAdjAddrFamilyMatchDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency address-family match disabled'
   noOrDefaultSyntax = syntax
   data = {
      'adjacency' : 'Configure parameters for adjacency formation',
      'address-family' : 'Configure address-family related parameters' \
             ' for adjacency formation',
      'match' : 'Configure address-family match check related parameters' \
             ' for adjacency formation',
      'disabled' : 'Relax address-family match check for bringing up adjacency',
   }
   handler = "RoutingIsisCliHandler.RouterIsisMode_setAdjAddrFamilyMatchDisabled"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisMode_noAdjAddrFamilyMatchDisabled"

RouterIsisMode.addCommandClass( RouterIsisAdjAddrFamilyMatchDisableCmd )

#-----------------------------------------------------------------------------
#   [no|default] "metric <value>"
#   command, in 'address-family mode'
#----------------------------------------------------------------------------
# We don't support "metric maximum' because we can achieve
# the similar behavior through set overload bit
class RouterIsisAfMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRICVALUE'
   noOrDefaultSyntax = 'metric ...'
   data = {
      'metric' : 'Set IS-IS metric for all interfaces',
      'METRICVALUE' : metricValueMatcher,
   }
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setMetricAllIntf"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noMetricAllIntf"

RouterIsisAfMode.addCommandClass( RouterIsisAfMetricCmd )

#-------------------------------------------------------------------------------
# [no|default] "fast-reroute ti-lfa mode ( ( PROTECTION [ LEVEL ] ) | disabled )"
# in 'address-family mode'
#-------------------------------------------------------------------------------

class RouterIsisAfFastRerouteCmd( CliCommand.CliCommandClass ):
   syntax = 'FRR_REROUTE_EXPRESSION'
   noOrDefaultSyntax = 'fast-reroute ti-lfa mode ...'
   data = {
      'FRR_REROUTE_EXPRESSION' : FrrRerouteExpression,
      }
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_frrProtection"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noFrrProtection"

RouterIsisAfMode.addCommandClass( RouterIsisAfFastRerouteCmd )

#-------------------------------------------------------------------------------
# "[no] fast-reroute ti-lfa srlg [strict]
# in 'address-family mode'
#-------------------------------------------------------------------------------

class RouterIsisAfFrrSrlgCmd( CliCommand.CliCommandClass ):
   syntax = '''fast-reroute ti-lfa srlg [ strict ]'''
   noOrDefaultSyntax = '''fast-reroute ti-lfa srlg ...'''

   data = frrSrlgData
   handler = "RoutingIsisCliHandler.RouterIsisAfMode_frrSrlgProtection"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noFrrSrlgProtection"

RouterIsisAfMode.addCommandClass( RouterIsisAfFrrSrlgCmd )

#-------------------------------------------------------------------------------
# [no|default] igp shortcut disabled
# in IPv4 'address family mode'
#-------------------------------------------------------------------------------
igpKw = CliMatcher.KeywordMatcher( 'igp', helpdesc='IGP configuration' )
shortcutKw = CliMatcher.KeywordMatcher( 'shortcut', helpdesc='IGP shortcut '
                                        'configuration for IS-IS instance' )
disabledKw = CliCommand.guardedKeyword( 'disabled',
                                        helpdesc='Disable IGP shortcut',
                                        guard=isisNonDefaultVrfAndInstanceGuard )

class RouterIsisAfIgpShortcutDisabledCmd( CliCommand.CliCommandClass ):
   syntax = "igp shortcut disabled"
   noOrDefaultSyntax = "igp shortcut disabled"
   data = {
      "igp" : igpKw,
      "shortcut": shortcutKw,
      "disabled": disabledKw
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_igpShortcutDisableIs"
   noOrDefaultHandler = \
             "RoutingIsisCliHandler.RouterIsisAfMode_igpShortcutDisableDel"

RouterIsisAfIpv4Modelet.addCommandClass( RouterIsisAfIgpShortcutDisabledCmd )

#-------------------------------------------------------------------------------
# [no|default] igp shortcut
# in IPv6 'address family mode'
#-------------------------------------------------------------------------------
class RouterIsisAfIgpShortcutCmd( CliCommand.CliCommandClass ):
   syntax = 'igp shortcut'
   noOrDefaultSyntax = syntax
   data = {
      'igp' : igpKw,
      'shortcut' : CliCommand.Node( matcher=shortcutKw,
                                    guard=isisNonDefaultVrfAndInstanceGuard )
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_igpShortcutIs"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_igpShortcutIs"

RouterIsisAfIpv6Modelet.addCommandClass( RouterIsisAfIgpShortcutCmd )

#-------------------------------------------------------------------------------
# [no|default] segment-routing ipv6
# in IPv6 'address family mode'
#-------------------------------------------------------------------------------

class RouterIsisAfIpv6Srv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing ipv6'
   noOrDefaultSyntax = syntax
   data = {
      'segment-routing': CliCommand.guardedKeyword( 'segment-routing',
                                       'Segment routing configuration',
                                       guard=isisNonDefaultVrfAndInstanceGuard),
      'ipv6' : ipv6Kw
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_gotoIsisAfSrv6Mode"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_delIsisAfSrv6Mode"

if gatedToggleLib.toggleIsisSrv6Enabled():
   RouterIsisAfIpv6Modelet.addCommandClass( RouterIsisAfIpv6Srv6Cmd )

#-----------------------------------------------------------------------------------
# [no|default] next-hop tunnel rsvp < tunnel-name > prefix-list < prefix-list-name >
# in 'address family mode'
#-----------------------------------------------------------------------------------

# Can be extended to include other tunnel types in future
tunnelTypes = {
      'rsvp': 'Tunnel type RSVP'
}

class RouterIsisAfTrafficSteeringCmd( CliCommand.CliCommandClass ):
   syntax = "next-hop tunnel TUNNEL_TYPE TUNNEL_NAME prefix-list PREFIX_LIST_NAME"
   noOrDefaultSyntax = "next-hop tunnel TUNNEL_TYPE TUNNEL_NAME ..."
   data = {
      'next-hop': 'IGP nexthop',
      'tunnel': 'Tunnel whose outgoing interface will replace IGP nexthop',
      'TUNNEL_TYPE': CliMatcher.EnumMatcher( tunnelTypes ),
      'TUNNEL_NAME': CliMatcher.PatternMatcher(
         '.+', helpname='WORD', helpdesc='Tunnel name' ),
      'prefix-list': 'Prefix list',
      'PREFIX_LIST_NAME': CliMatcher.PatternMatcher(
         '.+', helpname='WORD', helpdesc='Prefix-list name' )
   }

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_trafficSteeringIs"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noTrafficSteering"

if gatedToggleLib.toggleRsvpTrafficSteeringIsisEnabled():
   RouterIsisAfIpv4Modelet.addCommandClass( RouterIsisAfTrafficSteeringCmd )

#-------------------------------------------------------------------------------
# "[no|default] set-overload-bit
#                  [on-startup <<delay> | wait-for-bgp [timeout <delay>]> ]"
# command, in isis config mode
#-------------------------------------------------------------------------------
class RouterIsisSetOverloadBitCmd( CliCommand.CliCommandClass ):
   syntax = '''set-overload-bit
               [ on-startup ( DELAY | ( wait-for-bgp [ timeout DELAY ] ) ) ]'''
   noOrDefaultSyntax = 'set-overload-bit ...'
   data = {
      'set-overload-bit' : 'Control the overload bit settings',
      'on-startup' : 'Overload at startup for <time> seconds or '
                     'wait for BGP convergence',
      'wait-for-bgp' : 'Wait for BGP convergence',
      'timeout' : 'Timeout in seconds for BGP convergence',
      'DELAY' : CliMatcher.IntegerMatcher(
            1,
            OverloadStartupDelay.max,
            helpdesc='Number of seconds' ),
   }

   handler = "RoutingIsisCliHandler.RouterIsisMode_setOverloadBit"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisMode_noSetOverloadBit"

RouterIsisMode.addCommandClass( RouterIsisSetOverloadBitCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis hello padding"
# command, in config-if mode.
#-------------------------------------------------------------------------------

#---------------------------------------------------------------------------------
# [no|default] adjacency-segment AF p2p [multiple]
# {<label <val>| index <val> global>} [backup-eligible]
#---------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] route-tag <tag-no>   XXX: Hidden
#------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] fast-reroute ti-lfa mode ( {link-protection |
# node-protection} [ LEVEL ] | disabled )
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] fast-reroute ti-lfa srlg [strict] [disabled]
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# New Cli Parser
#-------------------------------------------------------------------------------


#-------------------------------------------------------------------------------
# Flex Algo Show commands
#
# show isis flex-algo
#    [ { ( algorithm NAME ) | ( system SYSTEM ) | LEVEL |
#          advertised | routers } ]
#
# Funkiness warning: the algorithm, system, and level can always be specified.
# Routers is both a qualifier and a stand-alone command.
# Ordering is not important.
# If an algorithm name is specified then we return details.
# If 'routers' is specified alone, then we issue that command, otherwise, we
# request a summary.
#-------------------------------------------------------------------------------

# Helper function used by the CLI models

#-------------------------------------------------------------------------------
# Segment Routing Show Commands
#-------------------------------------------------------------------------------

class RedistributeUserAfCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute user [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute user ...'
   data = {
      'user' : 'EosSdk routes'
   }
   data.update( redistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRedistribute5"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noRedistribute5"

if IsisToggleLib.toggleEosSdkRedistributionIsisEnabled():
   RouterIsisAfMode.addCommandClass( RedistributeUserAfCmd )

class RedistributeDynamicAfCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute dynamic [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute dynamic ...'
   data = {
      'dynamic' : 'Dynamic policy routes'
   }
   data.update( redistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRedistribute6"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noRedistribute6"

RouterIsisAfMode.addCommandClass( RedistributeDynamicAfCmd )

class RedistributeIsisInstanceAfCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis instance [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute isis instance ...'
   data = {
      'instance' : 'routes from another IS-IS instance',
   }
   data.update( isisRedistributeCmdData )

   handler = "RoutingIsisCliHandler.RouterIsisAfMode_setRedistribute8"
   noOrDefaultHandler = "RoutingIsisCliHandler.RouterIsisAfMode_noRedistribute8"

RouterIsisAfMode.addCommandClass( RedistributeIsisInstanceAfCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entMan ):
   global routingHardwareStatusCommon
   global isisConfig, isisStatusDir, routingHardwareStatus
   global routing6Config, routing6HardwareStatus, allIntfStatusDir
   global ipConfig, ip6Config
   global isisClearReqDir, isisClearRespDir
   global vrfInfoDir, vrf6InfoDir, srSysdbStatusDir
   global linkReadyStatusVrfColl, ldpProtoConfigColl, ldpConfigColl
   global entityManager
   global mplsRoutingConfig
   global l3Config
   global mplsStatus
   global isisSystemIdHostnameMap
   global tilfaTunnelTable
   global isisSrV4BindingTable
   global isisSrV6BindingTable
   global srConfig
   global trapConfig
   global flexAlgoConfig
   global teConfig
   global sharedSecretConfig

   entityManager = entMan
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )

   l3Config = ConfigMount.mount( entityManager, "l3/config", "L3::Config", 'w' )
   tilfaTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.tiLfaTunnelTable, entityManager )
   isisConfig = ConfigMount.mount( entityManager, 'routing/isis/config',
                                   'Routing::Isis::Config', 'w' )
   isisStatusDir = LazyMount.mount( entityManager, 'routing/isis/status',
                                    'Tac::Dir', 'ri' )
   isisClearReqDir = LazyMount.mount( entityManager, 'routing/isis/clear/request',
                                      'Routing::Isis::ClearRequestNode', 'w' )
   isisClearRespDir = LazyMount.mount( entityManager, 'routing/isis/clear/response',
                                       'Tac::Dir', 'ri' )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager, "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routing6Config = LazyMount.mount( entityManager, "routing6/config",
                                     "Routing6::Config", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
                                             "routing6/hardware/status",
                                             "Routing6::Hardware::Status", "r" )
   srSysdbStatusDir = LazyMount.mount( entityManager,
                                    "segmentrouting/isis",
                                    "Tac::Dir", "ri" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   vrfInfoDir = LazyMount.mount( entityManager, "routing/vrf/routingInfo/status",
                                "Tac::Dir", "ri" )
   vrf6InfoDir = LazyMount.mount( entityManager, "routing6/vrf/routingInfo/status",
                                "Tac::Dir", "ri" )
   linkReadyStatusVrfColl = LazyMount.mount( entityManager,
                                  "mpls/ldp/linkReadyStatus",
                                  "Ldp::LdpLinkReadyStatusVrfColl", "r" )
   ldpProtoConfigColl = LazyMount.mount( entityManager,
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpConfigColl = LazyMount.mount( entityManager,
                                    "mpls/ldp/ldpConfigColl",
                                    "Ldp::LdpConfigColl", "r" )
   mplsRoutingConfig = LazyMount.mount( entityManager, 'routing/mpls/config',
                                                      "Mpls::Config", "r" )
   mplsStatus = LazyMount.mount( entityManager, 'mpls/status',
                                 "Mpls::Api::Status", "r" )
   isisSystemIdHostnameMap = shmemEm.doMount( 'routing/isis/hostnamemap/default',
                                              "IsisExportImpl::IsisHostnameMap",
                                              smi )
   isisSrV4BindingTable = shmemEm.doMount( "mpls/labelBindingTables/isisV4",
                                           "CommonLibSmash::LabelBindingTable",
                                           Smash.mountInfo( 'keyshadow' ) )
   isisSrV6BindingTable = shmemEm.doMount( "mpls/labelBindingTables/isisV6",
                                           "CommonLibSmash::LabelBindingTable",
                                           Smash.mountInfo( 'keyshadow' ) )
   srConfig = ConfigMount.mount( entityManager, "routing/sr/config",
                                 "Routing::SegmentRoutingCli::Config", 'w' )
   trapConfig = ConfigMount.mount( entityManager, "hardware/trap/config/trapConfig",
                                   "Arnet::TrapConfig", 'w' )
   flexAlgoConfig = LazyMount.mount( entityManager, 'te/flexalgo/config',
                                     'FlexAlgo::Config', 'r' )
   teConfig = LazyMount.mount( entityManager, 'te/config',
                               'TrafficEngineering::Config', 'r' )
   sharedSecretConfig = LazyMount.mount( entityManager,
                        "mgmt/security/sh-sec-prof/config",
                        "Mgmt::Security::SharedSecretProfile::Config", "r" )
