#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import itertools
import ipaddress
import os
import re
import sys
from functools import partial

# The ones with "pilint" are to satisfy "a4 submit" that barfs about invalid option
# replace back with "pylint" when checking in workspace (I guess py2 vs py3 pylint)
# pylint: disable=anomalous-backslash-in-string
# pylint: disable=cell-var-from-loop
# pilint: disable=chained-comparison
# pilint: disable=consider-using-from-import
# pilint: disable=consider-using-f-string
# pilint: disable=consider-using-in
# pylint: disable=inconsistent-return-statements
# pylint: disable=singleton-comparison
# pylint: disable=ungrouped-imports
# pylint: disable=unused-import
# pilint: disable=use-a-generator
# pilint: disable=useless-object-inheritance
# This one is required by "a4 submit" only!
# pylint: disable=len-as-condition

from Toggles import IpRibLibToggleLib
import Tac, LazyMount, SharedMem, Smash, SmashLazyMount
import ConfigMount
import CliParser, BasicCli, Arnet, CliToken
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from ReversibleSecretCli import decodeKey, generateSecretEntity, getDefaultSecret
import CliPlugin.SegmentRoutingCli
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import IraIp6Cli
from CliPlugin import IraIp6IntfCli
from CliPlugin import IraIpCli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraVrfCli
from CliPlugin import IsisCliHelper
from CliPlugin import IsisCliModels
from CliPlugin import MplsModel
from CliPlugin import TechSupportCli
from CliPlugin.MplsCli import mplsNodeForShow, mplsForShowNode
from CliPlugin.RcfCliLib import rcfFunctionMatcher
from CliPlugin.RouteMapCli import mapNameMatcher
from CliPlugin.RoutingCommon import GATED_PROTO_SH_TECH_TS
from CliPlugin.IsisCliModels import (
   IsisMplsLdpSyncInterfaceModel,
   IsisMplsLdpSyncModel,
   TilfaTunnelTableEntry,
   TilfaTunnelTable
)

from CliPlugin.TunnelCli import (
   getTunnelIdFromIndex,
   getTunnelIndexFromId,
   readMountTunnelTable,
   TunnelTableIdentifier,
   tokenTunnelMatcher,
   tunnelIndexMatcher
)

from CliPlugin.TunnelModels import (
   MplsVia as TunnelMplsVia
)

from CliToken.Router import routerMatcherForConfig as routerKw
from CliToken.RouteMapCliTokens import CommonTokens as RouteMapMatchers
from CliMode.Isis import RoutingIsisMode, RoutingIsisAfMode, RoutingIsisSrMplsMode, \
      RoutingIsisMetricProfileMode, RoutingIsisTeMode, RoutingIsisAreaProxyMode, \
      RoutingIsisAfSrv6Mode, RoutingIsisAfSrv6LocatorMode
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED, ALL_VRF_NAME
import BasicCliUtil
from BasicCliUtil import notAPrefixOf
from CliParserCommon import namePattern

from IsisCliLib import LVL_KW_TO_TAC_TYPE_MAP

from CliModel import cliPrinted
# pylint: disable=cyclic-import

from RibCapiLib import AmiResponseException
from RibCapiLib import showRibCapiCommand
from RibCapiLib import EmptyResponseException
import CliCommand
from CliPrint import CliPrint
from TypeFuture import TacLazyType

from CliPlugin.RoutingIsisCli import (
   isisConfig,
   isisStatusDir,
   routingHardwareStatusCommon,
   routingHardwareStatus,
   routing6HardwareStatus,
   ipConfig,
   ip6Config,
   isisClearReqDir,
   isisClearRespDir,
   vrfInfoDir,
   vrf6InfoDir,
   srSysdbStatusDir,
   linkReadyStatusVrfColl,
   ldpProtoConfigColl,
   ldpConfigColl,
   entityManager,
   mplsRoutingConfig,
   l3Config,
   mplsStatus,
   isisSystemIdHostnameMap,
   tilfaTunnelTable,
   isisSrV4BindingTable,
   isisSrV6BindingTable,
   srConfig,
   trapConfig,
   flexAlgoConfig,
   getAuthModeData,
   getConflictingConfigFoundDict
)

from CliDynamicSymbol import loadDynamicPlugin
submodes = loadDynamicPlugin( "RoutingIsisCli" )

# ---- common (handler & syntax) TODO: not copy.paste ---

def isisConfiguration():
   return isisConfig

trailingByte = r'\.[0-9a-fA-F]{2}'
trailingRegEx = trailingByte + '$'
mtWarningMsg = "Multi Topology is not enabled"
mtV4WarningMsg = "IPv4 address family is not configured in router IS-IS mode"

cprinter = CliPrint().lib

# Exception not raised when try fails. Suppress pylint warning
# pylint: disable-msg=bare-except
# pylint: disable-msg=protected-access
# pylint: disable-msg=broad-except
U32_MAX_VALUE = 0xFFFFFFFF

AdjacencyLoggingType = TacLazyType( 'Routing::Isis::AdjacencyLoggingType' )
AdjacencySegmentKey = TacLazyType( 'Mpls::SegmentRouting::AdjacencySegmentKey' )
AreaLeaderPriority = TacLazyType( 'Routing::Isis::AreaLeaderPriority' )
ArnetAddressFamily = TacLazyType( 'Arnet::AddressFamily' )
AuthAlgo = TacLazyType( 'Routing::Isis::AuthAlgo' )
AuthKey = TacLazyType( 'Routing::Isis::AuthKey' )
AuthKeyid = TacLazyType( 'Routing::Isis::AuthKeyid' )
AuthMode = TacLazyType( 'Routing::Isis::AuthMode' )
ClearAdjRequest = TacLazyType( 'Routing::Isis::ClearAdjRequest' )
ClearCountersRequest = TacLazyType( 'Routing::Isis::ClearCountersRequest' )
ClearDatabaseRequest = TacLazyType( 'Routing::Isis::ClearDatabaseRequest' )
ClearRequest = TacLazyType( 'Routing::Isis::ClearRequest' )
CommonLibConsumerSm = TacLazyType( 'CommonLibSmash::CommonLibConsumerSm' )
CsnpGenInterval = TacLazyType( 'Routing::Isis::CsnpGenInterval' )
FlexAlgoApplication = TacLazyType( 'Routing::Isis::FlexAlgoApplication' )
FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
GrHoldTime = TacLazyType( 'Routing::Isis::GrHoldTime' )
GrTimerT2 = TacLazyType( 'Routing::Isis::GrTimerT2' )
HelloInterval = TacLazyType( 'Routing::Isis::HelloInterval' )
HelloMultiplier = TacLazyType( 'Routing::Isis::HelloMultiplier' )
HelloPadding = TacLazyType( 'Routing::Isis::HelloPadding' )
InstanceStatus = TacLazyType( 'Routing::Isis::InstanceStatus' )
InterfaceType = TacLazyType( 'Routing::Isis::InterfaceType' )
IntfHelloPadding = TacLazyType( 'Routing::Isis::IntfHelloPadding' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
IsisAddressFamily = TacLazyType( 'Routing::Isis::AddressFamily' )
IsisIntfBfdConfig = TacLazyType( 'Routing::Isis::IsisIntfBfdConfig' )
IsisMplsBindingsHelper = TacLazyType( 'Isis::Cli::IsisMplsBindingsHelper' )
LabelBindingTable = TacLazyType( 'Mpls::LabelBindingTable' )
LabelBindingTableColl = TacLazyType( 'Mpls::LabelBindingTableColl' )
Level = TacLazyType( 'Routing::Isis::Level' )
LspGenIntervalMsec = TacLazyType( 'Routing::Isis::LspGenIntervalMsec' )
LspGenIntervalSec = TacLazyType( 'Routing::Isis::LspGenIntervalSec' )
LspGenThrottleTimer = TacLazyType( 'Routing::Isis::LspGenThrottleTimer' )
LspInterval = TacLazyType( 'Routing::Isis::LspInterval' )
LspRefreshInterval = TacLazyType( 'Routing::Isis::LSPRefreshInterval' )
MaxLSPLifetime = TacLazyType( 'Routing::Isis::MaxLSPLifetime' )
MaxLSPSize = TacLazyType( 'Routing::Isis::MaxLSPSize' )
Metric = TacLazyType( 'Routing::Isis::Metric' )
MetricRule = TacLazyType( 'Routing::Isis::MetricRule' )
MinLspRemainingLifetime = TacLazyType( 'Routing::Isis::MinLspRemainingLifetime' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
OverloadStartupDelay = TacLazyType( 'Routing::Isis::OverloadStartupDelay' )
PrefixSegment = TacLazyType( 'Routing::Isis::PrefixSegment' )
PrefixSegmentInfo = TacLazyType( 'Routing::Isis::PrefixSegmentInfo' )
PrefixRange = TacLazyType( 'Routing::Isis::PrefixRange' )
Priority = TacLazyType( 'Routing::Isis::Priority' )
ProtectionMode = TacLazyType( 'Routing::Isis::ProtectionMode' )
ProtectionConfig = TacLazyType( 'Routing::Isis::ProtectionConfig' )
ProtectionSrlg = TacLazyType( 'Routing::Isis::ProtectionSrlg' )
Redistribute = TacLazyType( 'Routing::Isis::Redistribute' )
RouteTag = TacLazyType( 'Routing::Isis::RouteTag' )
SegmentIdentifier = TacLazyType( 'Mpls::SegmentRouting::SegmentIdentifier' )
Speed = TacLazyType( 'Routing::Isis::Speed' )
SpeedUnitType = TacLazyType( 'Routing::Isis::SpeedUnitType' )
SpfIntervalSec = TacLazyType( 'Routing::Isis::SpfIntervalSec' )
SpfIntervalMsec = TacLazyType( 'Routing::Isis::SpfIntervalMsec' )
SpfThrottleTimer = TacLazyType( 'Routing::Isis::SpfThrottleTimer' )
SrAdjAllocationType = TacLazyType(
   'Routing::SegmentRoutingCli::SrAdjAllocationType' )
SrDataPlane = TacLazyType( 'Routing::SegmentRoutingCli::SrDataPlane' )
TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )
InstanceKey = TacLazyType( 'Routing::Isis::InstanceKey' )
IntfConfig = TacLazyType( 'Routing::Isis::IntfConfig' )
nullPrefix = Tac.Type( 'Arnet::Prefix' ).nullPrefix
srSidInvalid = Tac.Type( 'Routing::SegmentRoutingCli::Constants' ).srSidInvalid
RouterId = TacLazyType( 'Mpls::RouterId' )
TristateU32 = TacLazyType( 'Ark::TristateU32' )
MaxRedistRoutesLimit = TacLazyType( 'Routing::Isis::MaxRedistRoutesLimit' )
AreaIdIndex = TacLazyType( 'Routing::Isis::AreaIdIndex' )
SrDataPlaneEnum = TacLazyType( "Routing::SegmentRoutingCli::SrDataPlane" )
ClearResponseErrorCode = TacLazyType( 'Routing::Isis::ClearResponseErrorCode' )

SHA_KW_TO_TAC_TYPE_MAP = {
      'sha-1' : AuthAlgo.sha1,
      'sha-224' : AuthAlgo.sha224,
      'sha-256' : AuthAlgo.sha256,
      'sha-384' : AuthAlgo.sha384,
      'sha-512' : AuthAlgo.sha512,
}

SHARED_SECRET_ALGO_KW_TO_TAC_TYPE_MAP = {
      'md5' : AuthMode.md5,
      'sha-1' : AuthMode.sha_1,
      'sha-224' : AuthMode.sha_224,
      'sha-256' : AuthMode.sha_256,
      'sha-384' : AuthMode.sha_384,
      'sha-512' : AuthMode.sha_512
}

#------------------------------------------------------------------------------
# Common patterns
#------------------------------------------------------------------------------

netRegex = \
           r'(?P<areaid>(?:[\da-fA-F]{2}(?:\.[\da-fA-F]{4}){0,6})|' \
           r'((?:[\da-fA-F]{4})(?:\.[\da-fA-F]{4}){0,5}))' \
           r'\.(?P<systemid>[\da-fA-F]{4}\.[\da-fA-F]{4}\.[\da-fA-F]{4})' \
           r'\.00$'

def getProtoRouteType( args ):
   if 'bgp' in args:
      proto = 'protoBgp'
   elif 'dhcp' in args:
      proto = 'protoDhcp'
   elif 'user' in args:
      proto = 'protoEosSdk'
   elif 'dynamic' in args:
      proto = 'protoDynamic'
   else:
      proto = args.get( 'ospf' ) or args.get( 'ospfv3' )
   routeType = ( args.get( 'external' ) or args.get( 'internal' ) or
                 args.get( 'nssa-external' ) )
   mapName = args.get( 'MAPNAME' )
   return proto, routeType, mapName

# Callback hook to check if a given ip address can be assigned to an interface
def isisSrIpAddressAllowed( intfName, newAddrWithMask=None,
                              secondary=False, mode=None ):
   # If the interface:
   # - is not an isis interface 
   # - or it doesn't have a node-segment config  
   # - or the adddress isn't a /32
   # There is nothing to validate, return true
   intfConfig = _getIntfConfig( intfName )
   srIntfConfig = _getSrIntfConfig( intfName )
   if ( intfConfig is None or
        srIntfConfig is None or
        newAddrWithMask.len != 32 or
        ( srIntfConfig.srNodeSegmentIndex == srSidInvalid and
          not srIntfConfig.srV4NodeSegment and
          not srIntfConfig.srV4NodeSegmentLabel  ) or
        secondary ):
      return [ True, None ]

   # The interface has a node segment configuration, so verify that the
   # prefix doesn't conflict with any other known prefix segments
   prefix = Arnet.IpGenPrefix( str( newAddrWithMask.subnet ) )
   prefixConflictHelper = getConflictingConfigFoundDict()[ 'prefixConflict' ]
   if prefixConflictHelper( intfConfig.instanceName, intfName, prefix ):
      msg = "Conflicts with the IP address of another IS-IS SR "
      msg += "proxy-node/prefix segment"
      return [ False, msg ]
   return [ True, None ]

IraIpIntfCli.canSetIntfIpHook.addExtension( isisSrIpAddressAllowed )

# Callback hook to check if a given ipv6 address can be assigned to an interface
def isisSrIp6AddressAllowed( intfName, ip6Addr=None, mode=None ):
   srIntfConfig = _getSrIntfConfig( intfName )
   intfConfig = _getIntfConfig( intfName )
   # If the interface:
   # - is not an isis interface 
   # - or it doesn't have a node-segment config  
   # - or the adddress isn't a /128
   # There is nothing to validate, return true
   if ( intfConfig is None or
        srIntfConfig is None or
        ip6Addr.len != 128 or
        ( srIntfConfig.srV6NodeSegmentIndex == srSidInvalid and
          not srIntfConfig.srV6NodeSegment and
          not srIntfConfig.srV6NodeSegmentLabel ) ):
      return [ True, None ]

   # The interface has a node segment configuration, so verify that the
   # prefix doesn't conflict with any other known prefix segments
   prefix = Arnet.IpGenPrefix( str( ip6Addr ) )
   prefixConflictHelper = getConflictingConfigFoundDict()[ 'prefixConflict' ]
   if prefixConflictHelper( intfConfig.instanceName, intfName, prefix ):
      msg = "Conflicts with the IPv6 address of another IS-IS SR "
      msg += "proxy-node/prefix segment"
      return [ False, msg ]
   return [ True, None ]

IraIp6IntfCli.canSetIntfIpHook.addExtension( isisSrIp6AddressAllowed )

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without routing support
#-------------------------------------------------------------------------------
def routingSupportedGuard( mode, token ):
   if routingHardwareStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default VRF IS-IS instance
#-------------------------------------------------------------------------------
def isisNonDefaultVrfGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.vrfName != DEFAULT_VRF:
      return CliParser.guardNonDefaultVrf

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default IS-IS instance (instance ID != 0)
#-------------------------------------------------------------------------------
def isisNonDefaultInstanceGuard( mode, token ):
   if isinstance( mode, IntfCli.IntfConfigMode ):
      return None
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.instanceId != 0:
      return "not supported in non-default IS-IS instance"

#-------------------------------------------------------------------------------
# Guard to prevent configuration in default IS-IS instance (instance ID 0)
#-------------------------------------------------------------------------------
def isisDefaultInstanceGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.instanceId == 0:
      return "not supported in default IS-IS instance"

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default VRF IS-IS instance and in
# non-default IS-IS instance (instance ID != 0)
#-------------------------------------------------------------------------------
def isisNonDefaultVrfAndInstanceGuard( mode, token ):
   return isisNonDefaultVrfGuard( mode, token ) or \
          isisNonDefaultInstanceGuard( mode, token )

#-------------------------------------------------------------------------------
# Address family string to enum
#-------------------------------------------------------------------------------
def addrFamilyStrToEnum( addrFamilyStr ):
   if addrFamilyStr == "ipv4":
      return IsisAddressFamily.addressFamilyIPv4
   elif addrFamilyStr == "ipv6":
      return IsisAddressFamily.addressFamilyIPv6
   elif addrFamilyStr == "both":
      return IsisAddressFamily.addressFamilyBoth
   else:
      return IsisAddressFamily.addressFamilyNone

#-------------------------------------------------------------------------------
# Segment routing common infrastructure
#-------------------------------------------------------------------------------
def nodeIndexRangeFn( mode, context ):
   intfConfig = _getIntfConfig( mode.intf.name ) if hasattr( mode, 'intf' ) else None

   if intfConfig is not None and intfConfig.instanceName != '':
      instanceName = intfConfig.instanceName
      instanceConfig = isisConfig.instanceConfig.get( instanceName )
      instanceStatus = None

      if instanceConfig is not None:
         # obtain vrfName from instanceConfig and use this to obtain
         # instanceStatus
         vrfName = instanceConfig.vrfName
         isisStatus = isisStatusDir.get( vrfName )

         if isisStatus is not None:
            instanceStatus = isisStatus.instanceStatus.get( instanceName )
      else:
         # in the case of re-config, cannot use the isisConfig.instanceConfig
         # as instance config won't exist, so take brute force approach and
         # iterate through the Status dir vrfs looking for matching instance
         for vrfName in isisStatusDir:
            isisStatus = isisStatusDir[ vrfName ]
            intfNames = [ x.intfName for x in isisStatus.intfStatus ]
            if intfConfig.name in intfNames:
               instanceStatus = isisStatus.instanceStatus.get( instanceName )
               break

      if instanceStatus is not None:
         return ( 0, instanceStatus.srSrgbRange - 1 )

   return ( 0, InstanceStatus.srSrgbRangeDefault - 1 )

def addAddrFamilyToConfig( currentAf, addrFamily ):
   if currentAf == addrFamilyStrToEnum( "none" ):
      newAf = addrFamilyStrToEnum( addrFamily )
   elif currentAf == addrFamilyStrToEnum( "ipv4" ) and addrFamily == "ipv6":
      newAf = addrFamilyStrToEnum( "both" )
   elif currentAf == addrFamilyStrToEnum( "ipv6" ) and addrFamily == "ipv4":
      newAf = addrFamilyStrToEnum( "both" )
   else:
      newAf = currentAf
   return newAf

def delAddrFamilyFromConfig( currentAf, addrFamily ):
   if currentAf == addrFamilyStrToEnum( "both" ) and addrFamily == "ipv4":
      newAf = addrFamilyStrToEnum( "ipv6" )
   elif currentAf == addrFamilyStrToEnum( "both" ) and addrFamily == "ipv6":
      newAf = addrFamilyStrToEnum( "ipv4" )
   elif currentAf == addrFamilyStrToEnum( addrFamily ):
      newAf = addrFamilyStrToEnum( "none" )
   else:
      newAf = currentAf
   return newAf

def _redistributeConfig( instanceConfig, addrFamily ):
   redistributeConfigDict = { 'both' : instanceConfig.redistributeConfig,
                              'ipv4' : instanceConfig.redistributeConfigV4,
                              'ipv6' : instanceConfig.redistributeConfigV6 }
   return redistributeConfigDict[ addrFamily ]

def haveConflictingRedistribute( mode, config, attr, proto ):
   attrConfigModeMap = { 'redistributeConfig' : 'router isis',
                     'redistributeConfigV4' : 'address-family ipv4',
                     'redistributeConfigV6' : 'address-family ipv6'
                     }
   protoStr = { 'protoBgp' : 'bgp',
                'protoOspf3' : 'ospfv3 match internal',
                'protoOspf3Ase': 'ospfv3 match external',
                'protoOspf3Nssa' : 'ospfv3 match nssa-external',
                'protoIsis' : 'isis',
                'protoEosSdk' : 'user',
                'protoDynamic': 'dynamic',
                'protoStatic': 'static',
                'protoDirect': 'connected' }
   conflict = None

   if attr != 'redistributeConfig':
      if proto in config.redistributeConfig:
         conflict = 'redistributeConfig'
   else:
      if proto in config.redistributeConfigV4:
         conflict = 'redistributeConfigV4'
      elif proto in config.redistributeConfigV6:
         conflict = 'redistributeConfigV6'
   
   if conflict:
      errorMsg = "Cannot configure 'redistribute %s' in mode '%s' " \
          "while it is configured in mode '%s'" \
          % ( protoStr[ proto ], attrConfigModeMap[ attr ],
              attrConfigModeMap[ conflict ] )
      mode.addError( errorMsg )
      return True
   return False

afRedistConfigAttrMap = { 'ipv4' : 'redistributeConfigV4',
                          'ipv6' : 'redistributeConfigV6',
                          'both' : 'redistributeConfig'
                          }

def haveConflictMicroLoopProtected( config, addrFamily ):
   msg = None
   if addrFamily != 'both':
      if config.microLoopPreventionProtectedConfig:
         msg = 'both'
   else:
      if config.microLoopPreventionProtectedConfigV4:
         msg = 'ipv4'
      elif config.microLoopPreventionProtectedConfigV6:
         msg = 'ipv6'
   return msg

def haveConflictingMicroLoopConfig( mode, config, addrFamily ):
   cmdStr = 'timers local-convergence-delay protected-prefixes'
   addrFamilyToMode = { 'ipv4' : 'address-family ipv4',
                        'ipv6' : 'address-family ipv6',
                        'both' : 'router isis' }
   conflictMsg = haveConflictMicroLoopProtected( config, addrFamily )
   if conflictMsg:
      errorMsg = "Cannot configure '%s' in mode '%s' " \
                 "while it is configured in mode '%s'" \
                 % ( cmdStr, addrFamilyToMode[ addrFamily ],
                     addrFamilyToMode[ conflictMsg ] )
      mode.addError( errorMsg )
      return True
   return False

def routerIsisModeCmd( mode, args ):
   gotoRouterIsisMode( mode, args[ 'INST' ],
                       args.get( 'ID' ),
                       args.get( 'VRF' ) )

def routerIsisModeCmdNo( mode, args ):
   delRouterIsisMode( mode, args[ 'INST' ] )

def delIsisRedistAf( instanceConfig, addrFamily ):
   redistConfig = _redistributeConfig( instanceConfig, addrFamily )
   redistConfig.clear()

def delIsisDistanceAf( instanceConfig, addrFamily ):
   if addrFamily == 'ipv4':
      instanceConfig.distanceV4L1 = instanceConfig.distanceDefault
      instanceConfig.distanceV4L2 = instanceConfig.distanceDefault
   elif addrFamily == 'ipv6':
      instanceConfig.distanceV6L1 = instanceConfig.distanceDefault
      instanceConfig.distanceV6L2 = instanceConfig.distanceDefault

def delIsisTrafficSteering( instanceConfig, addrFamily ):
   if addrFamily == 'ipv4':
      instanceConfig.trafficSteeringRsvpV4.clear()

ospfRouteTypeProtoMap = {
      'ospf' : {
         'internal' : 'protoOspf',
         'external' : 'protoOspfAse',
         'nssa-external' : 'protoOspfNssa'
      },
      'ospfv3' : {
         'internal' : 'protoOspf3',
         'external' : 'protoOspf3Ase',
         'nssa-external' : 'protoOspf3Nssa'
      },
}

def setRedistributeCommon( mode, instanceName, proto, addrFamily='both',
      routeType=None, mapName=None, rcfName=None, includeLeaked=False ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   attrStr = afRedistConfigAttrMap[ addrFamily ]

   if proto in ospfRouteTypeProtoMap:
      proto = ospfRouteTypeProtoMap[ proto ][ routeType ]

   if haveConflictingRedistribute( mode, instanceConfig, attrStr, proto ):
      return 
   redistributeConfig = _redistributeConfig( instanceConfig, addrFamily )

   redistribute = Redistribute( proto, includeLeaked=includeLeaked )

   if mapName:
      assert not rcfName
      redistribute.routeMap = mapName

   if rcfName:
      assert not mapName
      # Remove possible () on RCF name
      rcfName = rcfName.removesuffix( '()' )

      redistribute.rcfFunction = rcfName

   if proto in redistributeConfig:
      oldRedistCfg = redistributeConfig[ proto ]
      if redistribute == oldRedistCfg:
         # Config hasn't changed, nothing to do
         return
      del redistributeConfig[ proto ]

   redistributeConfig.addMember( redistribute )

def noRedistributeCommon( instanceName, addrFamily, proto, routeType=None ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   redistributeConfig = _redistributeConfig( instanceConfig, addrFamily )
   if proto in ospfRouteTypeProtoMap:
      proto = ospfRouteTypeProtoMap[ proto ][ routeType ]
   del redistributeConfig[ proto ]

def setMicroLoopProtectedCommon( mode, instanceName, addrFamily='both',
                                 negate=False ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   enable = not negate
   if enable and haveConflictingMicroLoopConfig(
         mode, instanceConfig, addrFamily ):
      return
   if addrFamily == 'both':
      instanceConfig.microLoopPreventionProtectedConfig = enable
   elif addrFamily == 'ipv4':
      instanceConfig.microLoopPreventionProtectedConfigV4 = enable
   else:
      instanceConfig.microLoopPreventionProtectedConfigV6 = enable

def setMicroLoopConvergenceDelayCommon( mode, instanceName, delay, addrFamily='both',
                                        negate=False ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   if negate:
      delay = instanceConfig.microLoopConvergenceDelayDefault

   if addrFamily == 'both':
      instanceConfig.microLoopConvergenceDelay = delay
   elif addrFamily == 'ipv4':
      instanceConfig.microLoopConvergenceDelayV4 = delay
   else:
      instanceConfig.microLoopConvergenceDelayV6 = delay

def getAuthKeyData( args ):
   authKeyId = args.get( 'ID' )
   authLevel = args.get( 'LEVEL' )
   algorithm = args.get( 'SHA' )
   shaApadRfc5310 = 'rfc-5310' in args
   authKey = {}
   if 'PASS' in args:
      authKey[ 'PASS' ] = args[ 'PASS' ]
   if 'UPASS' in args:
      authKey[ 'UPASS' ] = args[ 'UPASS' ]
   if 'EPASS' in args:
      authKey[ 'EPASS' ] = args[ 'EPASS' ]
   return authKeyId, authLevel, algorithm, authKey, shaApadRfc5310

def setAuthModeCommon( configHandle, authLevel, authModeVal, keyid,
                       authRxDisabled, sharedSecretProfileName ):

   if authRxDisabled is not None:
      authRxDisabled = True
   else:
      authRxDisabled = configHandle.authRxDisabledDefault

   if authLevel != 'level-2':
      configHandle.authModeL1 = authModeVal
      configHandle.authModeKeyidL1 = keyid
      configHandle.authRxDisabledL1 = authRxDisabled
      configHandle.sharedSecretProfileNameL1 = sharedSecretProfileName
   if authLevel != 'level-1':
      configHandle.authModeL2 = authModeVal
      configHandle.authModeKeyidL2 = keyid
      configHandle.authRxDisabledL2 = authRxDisabled
      configHandle.sharedSecretProfileNameL2 = sharedSecretProfileName

def noAuthModeCommon( configHandle, authLevel, authModeVal, keyid,
                      sharedSecretProfileName ):
   authRxDisabledDefault = configHandle.authRxDisabledDefault
   authModeNone = AuthMode.noAuth
   idDefault = AuthKeyid().idDefault
   profileNameDefault = configHandle.profileNameDefault
   if authModeVal == None and authLevel == None:
      configHandle.authModeL1 = authModeNone
      configHandle.authModeKeyidL1 = idDefault
      configHandle.authRxDisabledL1 = authRxDisabledDefault
      configHandle.sharedSecretProfileNameL1 = profileNameDefault
      configHandle.authModeL2 = authModeNone
      configHandle.authModeKeyidL2 = idDefault
      configHandle.authRxDisabledL2 = authRxDisabledDefault
      configHandle.sharedSecretProfileNameL2 = profileNameDefault
      return
   if authLevel == None:
      if ( configHandle.authModeL1 == authModeVal ) and \
         ( configHandle.authModeKeyidL1 == keyid ) and \
         ( configHandle.sharedSecretProfileNameL1 == sharedSecretProfileName ):
         configHandle.authModeL1 = authModeNone
         configHandle.authModeKeyidL1 = idDefault
         configHandle.authRxDisabledL1 = authRxDisabledDefault
         configHandle.sharedSecretProfileNameL1 = profileNameDefault
      if ( configHandle.authModeL2 == authModeVal ) and \
         ( configHandle.authModeKeyidL2 == keyid ) and \
         ( configHandle.sharedSecretProfileNameL2 == sharedSecretProfileName ):
         configHandle.authModeL2 = authModeNone
         configHandle.authModeKeyidL2 = idDefault
         configHandle.authRxDisabledL2 = authRxDisabledDefault
         configHandle.sharedSecretProfileNameL2 = profileNameDefault
      return
   if ( configHandle.authModeL1 == authModeVal ) and \
      ( configHandle.authModeKeyidL1 == keyid ) and \
      ( configHandle.sharedSecretProfileNameL1 == sharedSecretProfileName ) and \
      ( authLevel == 'level-1' ):
      configHandle.authModeL1 = authModeNone
      configHandle.authModeKeyidL1 = idDefault
      configHandle.authRxDisabledL1 = authRxDisabledDefault
      configHandle.sharedSecretProfileNameL1 = profileNameDefault
   if ( configHandle.authModeL2 == authModeVal ) and \
      ( configHandle.authModeKeyidL2 == keyid ) and \
      ( configHandle.sharedSecretProfileNameL2 == sharedSecretProfileName ) and \
      ( authLevel == 'level-2' ):
      configHandle.authModeL2 = authModeNone
      configHandle.authModeKeyidL2 = idDefault
      configHandle.authRxDisabledL2 = authRxDisabledDefault
      configHandle.sharedSecretProfileNameL2 = profileNameDefault

def setAuthKeyCommon( mode, configHandle, authLevel, authKey, keyid, algorithm,
                      shaApadRfc5310 ):
   # process all levels applicable
   testAuthLevel = [ authLevel ] if authLevel else [ 'level-1', 'level-2' ]
   passwd = {}
   for level in testAuthLevel:
      # decrypt password
      if 'PASS' in authKey:
         passwd[ level ] = authKey[ 'PASS' ]
      else:
         if 'UPASS' in authKey:
            passwd[ level ] = generateSecretEntity( authKey[ 'UPASS' ] )
         else:
            postfix = str( configHandle.authModeL1 ) if level == 'level-1' else \
                      str( configHandle.authModeL2 )
            key = configHandle.instanceName + '_' + postfix
            try:
               # We need a try-except clause here, as the encrypted password
               # specified needs to be base64 encoded for decryption to succeed.
               passwd[ level ] = generateSecretEntity(
                                            decodeKey( authKey[ 'EPASS' ],
                                            key=key.encode(),
                                            algorithm='MD5' ) )
            except Exception as e:
               error = "%s" % e
               mode.addError( error.capitalize() )
               return

   idDefault = AuthKeyid()
   if keyid is None:
      keyid = idDefault
   if keyid in configHandle.authKeyColl:
      authKeyEntry = Tac.nonConst( configHandle.authKeyColl[ keyid ] )
   else:
      authKeyEntry = AuthKey( keyid )
   if algorithm:
      authKeyEntry.algorithm = SHA_KW_TO_TAC_TYPE_MAP[ algorithm ]
   # save password for authentication level if all decryptions worked
   for level in testAuthLevel:
      if level == 'level-1':
         authKeyEntry.authKeyL1 = passwd[ level ]
         authKeyEntry.shaApadRfc5310L1 = shaApadRfc5310
      elif level == 'level-2':
         authKeyEntry.authKeyL2 = passwd[ level ]
         authKeyEntry.shaApadRfc5310L2 = shaApadRfc5310
   configHandle.authKeyColl.addMember( authKeyEntry )

def noAuthKeyCommon( configHandle, authLevel, keyid ):
   if keyid is None:
      keyid = AuthKeyid()
   if keyid not in configHandle.authKeyColl:
      return
   authKeyEntry = Tac.nonConst( configHandle.authKeyColl[ keyid ] )
   if authLevel != 'level-2':
      authKeyEntry.authKeyL1 = getDefaultSecret()
   if authLevel != 'level-1':
      authKeyEntry.authKeyL2 = getDefaultSecret()

   if authKeyEntry.authKeyL1.clearTextEqual( authKeyEntry.authKeyInvalid ) and \
      authKeyEntry.authKeyL2.clearTextEqual( authKeyEntry.authKeyInvalid ):
      del configHandle.authKeyColl[keyid]
   else:
      configHandle.authKeyColl.addMember( authKeyEntry )

#-------------------------------------------------------------------------------
# router isis config mode. A new instance of this mode is created when the
# user enters "router isis".
#-------------------------------------------------------------------------------

def RouterIsisMode_setNet( mode, args ):
   net = re.match( netRegex, args[ 'NET' ] )
   ( areaId, systemId ) = net.group( 'areaid', 'systemid' )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if systemId == instanceConfig.systemIdInvalid:
      mode.addError( "System Id '%s' is invalid." % systemId )
      return
   instanceConfig.areaIdColl[ 0 ] = areaId
   instanceConfig.systemId = systemId

def RouterIsisMode_noNet( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.systemId = instanceConfig.systemIdInvalid
   instanceConfig.areaIdColl[ 0 ] = instanceConfig.areaIdInvalid

def RouterIsisMode_setAreaAddress( mode, args ):
   """
   Handler to add area address in areaIdColl.
   area address (area) will be added in the first available slot in areaIdColl
   Also, we will check empty slots from index 1 and not from index 0, as index 0
   in areaIdColl is corresponds to the area address configured via NET CLI.
   """
   area = args[ 'AREA' ]
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   # Check for duplicate
   # Index 0 in areaIdColl is corresponds to area address configured via
   # NET CLI. So, start checking for available/duplicate slot from index 1
   if area in list( instanceConfig.areaIdColl.values() )[ 1 : ]:
      return
   for index in range( 1, AreaIdIndex.max + 1 ):
      if instanceConfig.areaIdColl[ index ] == instanceConfig.areaIdInvalid:
         instanceConfig.areaIdColl[ index ] = area
         return
   # All the spot are full in areaIdColl
   mode.addError( 'Maximum %d area addresses can be configured via area '
                  'address CLI' % AreaIdIndex.max )

def RouterIsisMode_noAreaAddress( mode, args ):
   area = args[ 'AREA' ]
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   # Index 0 in areaIdColl is corresponding to the area address configured via
   # NET CLI. So, start from index 1
   for index in range( 1, AreaIdIndex.max + 1 ):
      if instanceConfig.areaIdColl[ index ] == area:
         instanceConfig.areaIdColl[ index ] = instanceConfig.areaIdInvalid
         return

def RouterIsisMode_setIsType( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.level = LVL_KW_TO_TAC_TYPE_MAP[ args[ 'LEVEL' ] ]

def RouterIsisMode_noIsType( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.level = instanceConfig.levelDefault

def RouterIsisMode_gotoRouterIsisAfMode( mode, args ):
   """Handler for 'address-family AF [unicast]"""
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   addrFamily = args[ 'AF' ]
   if addrFamily == "ipv6":
      IraIp6Cli.warnIfRoutingDisabled( mode, instanceConfig.vrfName )
      if not routing6HardwareStatus.routingSupported:
         mode.addWarning( "IPv6 Hardware forwarding is not "
                          "supported on this platform." )
         mode.addWarning( "This means that all IPv6 traffic will be "
                          "routed in software." )
   elif addrFamily == "ipv4":
      IraIpCli.warnIfRoutingDisabled( mode, instanceConfig.vrfName )

   instanceConfig.addressFamily = addAddrFamilyToConfig( 
         instanceConfig.addressFamily, addrFamily )
   childMode = mode.childMode( submodes.RouterIsisAfMode,
                               addrFamily=addrFamily )
   mode.session_.gotoChildMode( childMode )

def RouterIsisMode_delRouterIsisAfMode( mode, args ):
   """Handler for '[no|default] address-family AF [unicast]"""
   addrFamily = args[ 'AF' ]
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   childMode = mode.childMode( submodes.RouterIsisAfMode,
                               addrFamily=addrFamily )
   RouterIsisAfMode_clearConfig( childMode )
   instanceConfig.addressFamily = delAddrFamilyFromConfig( 
         instanceConfig.addressFamily, addrFamily )

def RouterIsisMode_getOrCreateMetricProfileConfig( mode, name ):
   profile, instanceName = findMetricProfileConfig( name )
   if instanceName and instanceName != mode.instanceName:
      if mode.session_.isInteractive():
         #TODO: Confirm that we should print error msg here
         mode.addError( "Metric profile %s is already defined in %s" \
            % ( name, instanceName ) )
      return None
   if not profile:
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      profile = instanceConfig.metricProfiles.newMember( name )
   return profile

def RouterIsisMode_gotoIsisMetricProfileMode( mode, args ):
   """Handler for 'metric profile <profileName>' command"""
   profileName = args[ 'PROFILE_NAME' ]
   metricProfileConfig = RouterIsisMode_getOrCreateMetricProfileConfig( mode, 
                                                                        profileName )

   if not metricProfileConfig:
      return

   childMode = mode.childMode( submodes.RouterIsisMetricProfileMode,
                                 metricProfileName=profileName,
                                 metricProfileConfig=metricProfileConfig )
   mode.session_.gotoChildMode( childMode )

def RouterIsisMode_delIsisMetricProfileMode( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   del instanceConfig.metricProfiles[ args[ 'PROFILE_NAME' ] ]

def RouterIsisMode_gotoRouterIsisSrMplsMode( mode, args ):
   """Handler for 'segment-routing mpls'"""
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srDataPlane = SrDataPlane.srDataPlaneMpls
   childMode = mode.childMode( submodes.RouterIsisSrMplsMode )
   mode.session_.gotoChildMode( childMode )

def RouterIsisMode_delRouterIsisSrMplsMode( mode, args ):
   """Handler for [no|default] segment-routing mpls """
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   # Set all segment routing related attributes to default
   instanceConfig.srDataPlane = SrDataPlane.srDataPlaneNone
   instanceConfig.srEnabled = instanceConfig.srEnabledDefault
   instanceConfig.srSegmentPathVerification = \
         instanceConfig.srSegmentPathVerificationDefault
   instanceConfig.srAdjSegmentSidReuseTimeout = \
                              instanceConfig.srAdjSegmentSidReuseTimeoutDefault
   instanceConfig.srAdjSegmentAlloc = instanceConfig.srAdjSegmentAllocDefault
   instanceConfig.srRouterId = instanceConfig.routerIdV4Default
   instanceConfig.srPrefixSegments.clear()


def RouterIsisMode_gotoRouterIsisTeMode( mode, args ):
   """Handler for 'traffic-engineering'"""
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisTeMode = True
   childMode = mode.childMode( submodes.RouterIsisTeMode )
   mode.session_.gotoChildMode( childMode )

def RouterIsisMode_delRouterIsisTeMode( mode, args ):
   """Handler for [no|default] traffic-engineering """
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisTeMode = instanceConfig.isisTeModeDefault
   childMode = mode.childMode( submodes.RouterIsisTeMode )
   RouterIsisTeMode_clearConfig( childMode )

def RouterIsisMode_gotoRouterIsisAreaProxyMode( mode, args ):
   '''
   Handler for 'area proxy'
   '''
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisAreaProxyMode = True
   childMode = mode.childMode( submodes.RouterIsisAreaProxyMode )
   mode.session_.gotoChildMode( childMode )

def RouterIsisMode_delRouterIsisAreaProxyMode( mode, args ):
   '''
   Handler for [ no | default ] area proxy
   '''
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisAreaProxyMode = instanceConfig.isisAreaProxyModeDefault
   childMode = mode.childMode( submodes.RouterIsisAreaProxyMode )
   RouterIsisAreaProxyMode_clearConfig( childMode )

def RouterIsisMode_setRedistribute( mode, proto, routeType=None, mapName=None,
                     rcfName=None, includeLeaked=False ):
   setRedistributeCommon( mode, mode.instanceName, proto, addrFamily='both',
                          routeType=routeType, mapName=mapName,
                          rcfName=rcfName, includeLeaked=includeLeaked )

def RouterIsisMode_noRedistribute( mode, proto, routeType=None ):
   noRedistributeCommon( mode.instanceName, 'both', proto, routeType=routeType )

def RouterIsisMode_setMicroLoopConvergenceDelay( mode, args ):
   """
   Handler for [no|default] timers local-convergence-delay protected-prefixes
   """
   setMicroLoopProtectedCommon( mode, mode.instanceName, addrFamily='both' )
   if 'delay' in args:
      delay = args.get( 'delay' )
      setMicroLoopConvergenceDelayCommon( mode, mode.instanceName,
                                          delay, addrFamily='both' )

def RouterIsisMode_noMicroLoopConvergenceDelay( mode, args ):
   setMicroLoopProtectedCommon( mode, mode.instanceName,
                                addrFamily='both', negate=True )
   setMicroLoopConvergenceDelayCommon( mode, mode.instanceName, None,
                                       addrFamily='both', negate=True )

def RouterIsisMode_setMplsLdpSync( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.mplsLdpSync = True

def RouterIsisMode_noSetMplsLdpSync( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.mplsLdpSync = instanceConfig.mplsLdpSyncDefault

def RouterIsisMode_setMaxLSPSize( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.maxLSPSize = args[ 'MAX_LSP_SIZE' ]

def RouterIsisMode_noMaxLSPSize( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.maxLSPSize = instanceConfig.maxLSPSizeDefault
         
def RouterIsisMode_setMaxLSPLifetime( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.maxLSPLifetime = args[ 'MAX_LIFETIME' ]

def RouterIsisMode_noMaxLSPLifetime( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.maxLSPLifetime = instanceConfig.maxLSPLifetimeDefault

def RouterIsisMode_setLSPRefreshInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspRefreshInterval = args[ 'REFRESH_INTERVAL' ]

def RouterIsisMode_noLSPRefreshInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspRefreshInterval = \
         instanceConfig.lspRefreshIntervalDefault

def RouterIsisMode_setIsisHostname( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisHostname = args[ 'HOSTNAME' ]

def RouterIsisMode_noIsisHostname( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisHostname = instanceConfig.isisHostnameInvalid
   
def RouterIsisMode_setShutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.shutdown = True

def RouterIsisMode_noShutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.shutdown = False

def RouterIsisMode_setLogAdjacencyChanges( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   value = AdjacencyLoggingType.adjacencyLoggingTypeNormal
   instanceConfig.adjacencyLogging = value

def RouterIsisMode_noLogAdjacencyChanges( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   value = AdjacencyLoggingType.adjacencyLoggingTypeNone
   instanceConfig.adjacencyLogging = value

def RouterIsisMode_setLogDuplicateSystemIDDisabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.duplicateSystemIDLogging = False

def RouterIsisMode_noLogDuplicateSystemIDDisabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.duplicateSystemIDLogging =\
         instanceConfig.duplicateSystemIDLoggingDefault

# set-overload-bit and set-overload-bit on-startup <delay> commands are
# mutually exclusive. The setting of one should reset the other. This is
# similar to its implementation in GateD as well
def RouterIsisMode_setOverloadBit( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if 'wait-for-bgp' in args:
      instanceConfig.overloadWaitForBgp = True
      instanceConfig.overloadBit = False
      instanceConfig.overloadStartupDelay = args.get( 'DELAY',
          instanceConfig.overloadWaitForBgpTimeoutDefault )
   elif 'DELAY' in args:
      instanceConfig.overloadBit = False
      instanceConfig.overloadStartupDelay = args[ 'DELAY' ]
      instanceConfig.overloadWaitForBgp = False
   else:
      instanceConfig.overloadStartupDelay = \
          instanceConfig.overloadStartupDelayInvalid
      instanceConfig.overloadBit = True
      instanceConfig.overloadWaitForBgp = False

def RouterIsisMode_noSetOverloadBit( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.overloadBit = False
   instanceConfig.overloadWaitForBgp = False

   instanceConfig.overloadStartupDelay = \
       instanceConfig.overloadStartupDelayInvalid

def RouterIsisMode_setGracefulRestart( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.gracefulRestart = True

def RouterIsisMode_noGracefulRestart( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.gracefulRestart = False

def RouterIsisMode_setGrHelper( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   no = False
   if CliCommand.isNoCmd( args ):
      no = True
   instanceConfig.grHelper = not no

def RouterIsisMode_setGrTimerT2( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   t2 = args.get( 'SECONDS', instanceConfig.grTimerT2Default )
   level = args.get( 'LEVEL' )
   if level != 'level-2':
      instanceConfig.grTimerT2L1 = t2
   if level != 'level-1':
      instanceConfig.grTimerT2L2 = t2

def RouterIsisMode_setGrHoldTime( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   seconds = args.get( 'SECONDS', instanceConfig.grHoldTimeDefault )
   instanceConfig.grHoldTime = seconds

def RouterIsisMode_setPublishLsdb( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.publishLsdb = True

def RouterIsisMode_noPublishLsdb( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.publishLsdb = False

def RouterIsisMode_advertiseHighMetric( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   inclRedis = 'redistributed' in args
   waitForBgp = 'wait-for-bgp' in args
   delay = 'DELAY' in args
   instanceConfig.advertiseHighMetricInclRedis = inclRedis
   instanceConfig.advertiseHighMetricWaitForBgp = waitForBgp
   if waitForBgp:
      defaultTimeout = instanceConfig.advertiseHighMetricWaitForBgpTimeoutDefault
   else:
      defaultTimeout = instanceConfig.advertiseHighMetricStartupDelayInvalid
   instanceConfig.advertiseHighMetricStartupDelay = args.get( 'DELAY', 
                                                               defaultTimeout )
   instanceConfig.advertiseHighMetric = not ( inclRedis or waitForBgp or delay )

def RouterIsisMode_noAdvertiseHighMetric( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.advertiseHighMetric = False
   instanceConfig.advertiseHighMetricInclRedis = False
   instanceConfig.advertiseHighMetricWaitForBgp = False
   instanceConfig.advertiseHighMetricStartupDelay = \
         instanceConfig.advertiseHighMetricStartupDelayInvalid

def RouterIsisMode_setAdvertisePassiveOnly( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.advertisePassiveOnly = 'passive-only' in args
   isisConfig.advertisePassiveOnlyUseNewSyntax = 'ip-reachability' in args

def RouterIsisMode_noSetAdvertisePassiveOnly( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.advertisePassiveOnly = False
   isisConfig.advertisePassiveOnlyUseNewSyntax = 'ip-reachability' in args

def RouterIsisMode_setAdvertiseInterfaceAddress( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.advertiseIntfAddrPassiveOnly = 'passive-only' in args

def RouterIsisMode_noSetAdvertiseInterfaceAddress( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.advertiseIntfAddrPassiveOnly = False

def RouterIsisMode_setSpfInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   spfIntervalsUnitsConfigured = 'MAX_WAIT_SEC_KW' in args or \
                                 'MAX_WAIT_MILLI_SEC_KW' in args
   if 'MAX_WAIT_MILLI_SEC_KW' in args:
      confMSec = args.get( 'MAX_WAIT_MS' )
      maxWaitSpfInterval = confMSec
      maxWaitIntConfiguredInMsec = True
   else:
      confSec = args.get( 'MAX_WAIT' )
      maxWaitSpfInterval = confSec * 1000
      maxWaitIntConfiguredInMsec = False

   holdSpfInterval = args.get( 'HOLD_TIME', instanceConfig.spfHoldIntDefault )
   startSpfInterval = args.get( 'INITIAL_WAIT',
                                instanceConfig.spfStartIntDefault )

   if startSpfInterval > maxWaitSpfInterval:
      mode.addError( "SPF initial wait interval cannot exceed max wait interval" )
      return
   if holdSpfInterval > maxWaitSpfInterval:
      mode.addError( "SPF hold interval cannot exceed max wait interval" )
      return

   instanceConfig.spfInterval = SpfThrottleTimer(
                    startSpfInterval, holdSpfInterval, maxWaitSpfInterval,
                    spfIntervalsUnitsConfigured, maxWaitIntConfiguredInMsec )

def RouterIsisMode_noSpfInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.spfInterval = instanceConfig.spfIntervalDefault

def RouterIsisMode_setSpfPartial( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.spfFlagsPrcEnabled = True

def RouterIsisMode_noSpfPartial( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.spfFlagsPrcEnabled = False

def RouterIsisMode_setAuthMode( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   authModeVal, keyid, authRxDisabled, authLevel, \
        sharedSecretProfileName = getAuthModeData( args )
   setAuthModeCommon( instanceConfig, authLevel, authModeVal, keyid,
                      authRxDisabled, sharedSecretProfileName )

def RouterIsisMode_noAuthMode( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   authModeVal, keyid, _, authLevel, \
         sharedSecretProfileName = getAuthModeData( args )
   noAuthModeCommon( instanceConfig, authLevel, authModeVal, keyid,
                     sharedSecretProfileName )


def RouterIsisMode_setAuthKey( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   keyid, authLevel, algorithm, authKey, shaApadRfc5310 = getAuthKeyData( args )
   setAuthKeyCommon( mode, instanceConfig, authLevel, authKey, keyid, algorithm,
                     shaApadRfc5310 )

def RouterIsisMode_noAuthKey( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   keyid, authLevel, _, _, _ = getAuthKeyData( args )
   noAuthKeyCommon( instanceConfig, authLevel, keyid )

def RouterIsisMode_setAttachedBit( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.attachBitRouteMap = args[ 'RM_NAME' ]

def RouterIsisMode_noSetAttachedBit( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.attachBitRouteMap = instanceConfig.attachBitRouteMapInvalid

def RouterIsisMode_setHelloPadding( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if args.get( 'disabled' ):
      instanceConfig.helloPadding = HelloPadding.helloPaddingAlwaysOff
   else:
      instanceConfig.helloPadding = HelloPadding.helloPaddingOn

def RouterIsisMode_noHelloPadding( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.helloPadding = HelloPadding.helloPaddingTillAdjUp

def RouterIsisMode_setLspGenInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   maxWaitLspGenInterval = args[ 'MAX_WAIT' ]
   holdLspGenInterval = args.get( 'HOLD_WAIT', 
         instanceConfig.lspGenHoldIntDefault )
   startLspGenInterval = args.get( 'INITIAL_WAIT', 
         instanceConfig.lspGenStartIntDefault )

   if startLspGenInterval > maxWaitLspGenInterval * 1000:
      mode.addError( "LSP initial wait interval cannot exceed max wait interval" )
      return
   if holdLspGenInterval > maxWaitLspGenInterval * 1000:
      mode.addError( "LSP hold interval cannot exceed max wait interval" )
      return

   instanceConfig.lspGenInterval = LspGenThrottleTimer(
         startLspGenInterval, holdLspGenInterval, maxWaitLspGenInterval )

def RouterIsisMode_setCsnpGenInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.csnpGenInterval = args[ 'INTERVAL' ]

def RouterIsisMode_noCsnpGenInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.csnpGenInterval = instanceConfig.csnpGenIntervalDefault

def RouterIsisMode_setP2pInstPeriodicCsnp( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.p2pInstPeriodicCsnp = False

def RouterIsisMode_noP2pInstPeriodicCsnp( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.p2pInstPeriodicCsnp = instanceConfig.p2pInstPeriodicCsnpDefault

def RouterIsisMode_noLspGenInterval( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspGenInterval = instanceConfig.lspGenIntervalDefault

def RouterIsisMode_setOutDelayTimer( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspOutDelayTimerConfig = args[ 'OUT_DELAY' ]
     
def RouterIsisMode_noOutDelayTimer( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspOutDelayTimerConfig = instanceConfig.lspOutDelayTimerDefault

def RouterIsisMode_setMinLspRemainingLifetime( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.minLspRemainingLifetime = args.get( 'MIN_REMAINING_LIFETIME',
         instanceConfig.minLspRemainingLifetimeDefault )

def RouterIsisMode_noMinLspRemainingLifetime( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.minLspRemainingLifetime = \
         instanceConfig.minLspRemainingLifetimeDisabled

def RouterIsisMode_setRouterIdV4( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   routerIdV4 = args[ 'ROUTER_IDV4' ]
   if routerIdV4 == instanceConfig.routerIdV4Default:
      mode.addError( "%s is not a valid router ID" % routerIdV4 )
      return
   instanceConfig.routerIdV4 = routerIdV4

def RouterIsisMode_noRouterIdV4( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.routerIdV4 = instanceConfig.routerIdV4Default

def RouterIsisMode_setRouterIdV6( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   routerIdV6 = args[ 'ROUTER_IDV6' ]
   if routerIdV6 == instanceConfig.routerIdV6Default:
      mode.addError( "%s is not a valid router ID" % routerIdV6 )
      return
   instanceConfig.routerIdV6 = routerIdV6

def RouterIsisMode_noRouterIdV6( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.routerIdV6 = instanceConfig.routerIdV6Default

def RouterIsisMode_setPurgeOrigination( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   level = args.get( 'LEVEL' )
   instanceConfig.poiEnabledL1 = level != 'level-2'
   instanceConfig.poiEnabledL2 = level != 'level-1'

def RouterIsisMode_noPurgeOrigination( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.poiEnabledL1 = False
   instanceConfig.poiEnabledL2 = False

def RouterIsisMode_setAreaLeaderPriority( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   priority = args.get( 'PRIORITY', instanceConfig.areaLeaderPriorityDefault )
   level = args.get( 'LEVEL' )
   applyToL1 = level != 'level-2'
   applyToL2 = level != 'level-1'
   if 'disabled' in args:
      if applyToL1:
         instanceConfig.areaLeaderEnabledL1 = False
      if applyToL2:
         instanceConfig.areaLeaderEnabledL2 = False
   else:
      if applyToL1:
         instanceConfig.areaLeaderEnabledL1 = True
         instanceConfig.areaLeaderPriorityL1 = priority
      if applyToL2:
         instanceConfig.areaLeaderEnabledL2 = True
         instanceConfig.areaLeaderPriorityL2 = priority

def RouterIsisMode_defaultAreaLeaderPriority( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   level = args.get( 'LEVEL' )
   default = instanceConfig.areaLeaderPriorityDefault
   if level != 'level-2':
      instanceConfig.areaLeaderEnabledL1 = True
      instanceConfig.areaLeaderPriorityL1 = default
   if level != 'level-1':
      instanceConfig.areaLeaderEnabledL2 = True
      instanceConfig.areaLeaderPriorityL2 = default

def RouterIsisMode_dynamicFlooding( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   level = args.get( 'LEVEL' )
   if level != 'level-2':
      instanceConfig.dynamicFloodingL1 = True
   if level != 'level-1':
      instanceConfig.dynamicFloodingL2 = True

def RouterIsisMode_noDynamicFlooding( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   level = args.get( 'LEVEL' )
   if level != 'level-2':
      instanceConfig.dynamicFloodingL1 = False
   if level != 'level-1':
      instanceConfig.dynamicFloodingL2 = False

def RouterIsisMode_setRoutePreference( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.routePrefRfc7775 = True

def RouterIsisMode_noRoutePreference( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.routePrefRfc7775 = False

def RouterIsisMode_setIsisLspRfc8202RxDisabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspRfc8202RxDisabled = True

def RouterIsisMode_noIsisLspRfc8202RxDisabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.lspRfc8202RxDisabled = False

def RouterIsisMode_setIsisRfc8202Disabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.rfc8202Disabled = True

def RouterIsisMode_noIsisRfc8202Disabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.rfc8202Disabled = False

def RouterIsisMode_setAdjAddrFamilyMatchDisabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.matchAdjacencyNlpid = False

def RouterIsisMode_noAdjAddrFamilyMatchDisabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.matchAdjacencyNlpid = True

def RouterIsisMode_setMaxRedistRoutesLimit( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   maxRedistRoutesLimit = args.get( 'MAX_LIMIT',
                        instanceConfig.maxRedistRoutesLimitDefault )
   if 'level-1' not in args:
      instanceConfig.maxRedistRoutesLimitL2 = maxRedistRoutesLimit
   if 'level-2' not in args:
      instanceConfig.maxRedistRoutesLimitL1 = maxRedistRoutesLimit

def RouterIsisMode_noMaxRedistRoutesLimit( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   defaultRouteLimit = instanceConfig.maxRedistRoutesLimitDefault
   if 'level-1' not in args:
      instanceConfig.maxRedistRoutesLimitL2 = defaultRouteLimit
   if 'level-2' not in args:
      instanceConfig.maxRedistRoutesLimitL1 = defaultRouteLimit

def RouterIsisMode_setMaxSidDepth( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   no = CliCommand.isNoCmd( args )
   msdType = args.get( 'MSD_TYPE' )
   if msdType == 'base-mpls-imposition':
      instanceConfig.msdBaseMplsImposition = not no

#-------------------------------------------------------------------------------
# ISIS area proxy config mode. A new instance of this mode is created when the
# user enters "area proxy".
#-------------------------------------------------------------------------------

def RouterIsisAreaProxyMode_shutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisAreaProxyEnabled = False

def RouterIsisAreaProxyMode_noShutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisAreaProxyEnabled = True

def RouterIsisAreaProxyMode_setNet( mode, args ):
   net = re.match( netRegex, args[ 'NET' ] )
   ( areaId, systemId ) = net.group( 'areaid', 'systemid' )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if systemId == instanceConfig.systemIdInvalid:
      mode.addError( "System Id '%s' is invalid." % systemId )
      return
   if instanceConfig.systemIdInvalid == instanceConfig.systemId:
      mode.addError( "The primary system id must be configured first." )
      return
   if systemId == instanceConfig.systemId:
      mode.addError( "Proxy system id cannot be the same as primary system id." )
      return
   instanceConfig.proxyAreaId = areaId
   instanceConfig.proxySystemId = systemId

def RouterIsisAreaProxyMode_noNet( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.proxyAreaId = instanceConfig.areaIdInvalid
   instanceConfig.proxySystemId = instanceConfig.systemIdInvalid

def RouterIsisAreaProxyMode_setHostname( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.proxyHostname = args[ 'HOSTNAME' ]

def RouterIsisAreaProxyMode_noHostname( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.proxyHostname = instanceConfig.isisHostnameInvalid

def RouterIsisAreaProxyMode_setAreaSegment( mode, args ):
   prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
   prefix = Arnet.IpGenPrefix( prefix.stringValue )
   labelType = args.get( 'label', 'index' )
   isValue = labelType == 'label'
   value = args.get( 'LABEL' ) or args.get( 'INDEX' )
   sid = SegmentIdentifier( value, isValue )
   if not prefix.isHost:
      mode.addError( "An area segment can be configured for a /32 IPv4"
                     " prefix or /128 IPv6 prefix only" )
      return
   prefixConflictHelper = getConflictingConfigFoundDict()[ 'prefixConflict' ]
   if prefixConflictHelper( mode.instanceName, None, prefix, mode=mode ):
      mode.addError( "area segment conflicts with a prefix/node segment"
                     " configured for the prefix %s" % prefix )
      return

   # Check for label-label conflicts
   if isValue:
      labelConflictHelper = getConflictingConfigFoundDict()[ "labelConflict" ]
      if labelConflictHelper( mode.instanceName, None, prefix, value, mode=mode ):
         mode.addError( "Two prefixes cannot be configured with the same label" )
         return
   else:
      # Check for index-index conflicts
      sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
      if sidConflictHelper( mode.instanceName, None, prefix, value, mode=mode ):
         mode.addError( "Two prefixes cannot be configured with the same SID" )
         return

   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if prefix in instanceConfig.areaSegment:
      instanceConfig.areaSegment[ prefix ] = sid
   else:
      # Allow only one (prefix, area SID) per address family. The new prefix
      # overwrites the existing one in the same address family.
      for p in instanceConfig.areaSegment:
         if p.af == prefix.af:
            del instanceConfig.areaSegment[ p ]
            break
      instanceConfig.areaSegment[ prefix ] = sid

def RouterIsisAreaProxyMode_noAreaSegment( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
   prefix = Arnet.IpGenPrefix( prefix.stringValue )
   del instanceConfig.areaSegment[ prefix ]

def RouterIsisAreaProxyMode_setRouterId( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if 'ipv4' in args:
      routerIdV4 = args[ 'ROUTER_IDV4' ]
      if routerIdV4 == instanceConfig.routerIdV4Default:
         mode.addError( "%s is not a valid router ID" % routerIdV4 )
         return
      instanceConfig.proxyRouterIdV4 = routerIdV4
   elif 'ipv6' in args:
      routerIdV6 = args[ 'ROUTER_IDV6' ]
      if routerIdV6 == instanceConfig.routerIdV6Default:
         mode.addError( "%s is not a valid router ID" % routerIdV6 )
         return
      instanceConfig.proxyRouterIdV6 = routerIdV6

def RouterIsisAreaProxyMode_noRouterId( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if 'ipv4' in args:
      instanceConfig.proxyRouterIdV4 = instanceConfig.routerIdV4Default
   elif 'ipv6' in args:
      instanceConfig.proxyRouterIdV6 = instanceConfig.routerIdV6Default

def RouterIsisAreaProxyMode_clearConfig( mode ):
   RouterIsisAreaProxyMode_shutdown( mode, None )
   RouterIsisAreaProxyMode_noNet( mode, None )
   RouterIsisAreaProxyMode_noHostname( mode, None )
   RouterIsisAreaProxyMode_noRouterId( mode, { 'ipv4' : 'ipv4' } )
   RouterIsisAreaProxyMode_noRouterId( mode, { 'ipv6' : 'ipv6' } )

#-------------------------------------------------------------------------------
# ISIS-TE config mode. A new instance of this mode is created when the
# user enters "traffic-engineering".
#-------------------------------------------------------------------------------

def RouterIsisTeMode_shutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisTeEnabled = instanceConfig.isisTeEnabledDefault

def RouterIsisTeMode_noShutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisTeEnabled = True

def RouterIsisTeMode_setIsType( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisTeLevel = LVL_KW_TO_TAC_TYPE_MAP[ args[ 'LEVEL' ] ]

def RouterIsisTeMode_noIsType( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.isisTeLevel = instanceConfig.isisTeLevelDefault

def RouterIsisTeMode_clearConfig( mode ):
   RouterIsisTeMode_shutdown( mode, None )
   RouterIsisTeMode_noIsType( mode, None )

#-------------------------------------------------------------------------------
# "[no|default] router isis <instance-name> [vrf <vrf-name>]" command,
# in "config" mode.
# -------------------------------------------------------------------------------
MAX_ISIS_INSTANCES_PER_VRF = 1

def isisInstanceByVrfName( instanceConfig ):
   instDict = {}
   for i in instanceConfig.values():
      instDict.setdefault( i.vrfName, [] ).append( i.instanceName )
   return instDict

def isisInstanceNameById( instanceConfig, vrf=DEFAULT_VRF ):
   instDict = {}
   for  i in instanceConfig.values():
      if i.vrfName == vrf:
         instDict[ i.instanceId ] = i.instanceName
   return instDict

def anyInstanceInVrf( instanceConfig, vrf=DEFAULT_VRF ):
   return any( i.vrfName == vrf for i in instanceConfig.values() )

def getInstanceConfig( mode, instanceName, instanceId, vrfName ):
   """Get ISIS Instance config object for supplied instanceName and
   vrfName. Create it if it doesn't exist already"""
   instanceConfig = isisConfig.instanceConfig.get( instanceName )

   # Existing instance config validations.
   if instanceConfig:
      if vrfName and instanceConfig.vrfName != vrfName:
         mode.addError( 'IS-IS instance %s is already configured in %s VRF' % \
                           ( instanceName, instanceConfig.vrfName ) )
         return None
      if instanceId and instanceConfig.instanceId != instanceId:
         mode.addError( 'IS-IS instance %s is already configured '
                        'with instance ID %d' % ( instanceName,
                                                  instanceConfig.instanceId ) )
         return None

      return instanceConfig

   # New instance config validations.
   instanceId = 0 if instanceId is None else instanceId
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( 'VRF name %s is reserved' % vrfName )
      return None

   vrfName = DEFAULT_VRF if vrfName is None else vrfName
   instanceByVrfName = isisInstanceByVrfName( isisConfig.instanceConfig )
   if vrfName == DEFAULT_VRF:
      instanceNameById = isisInstanceNameById( isisConfig.instanceConfig )
      if instanceId in instanceNameById:
         if instanceId == 0:
            mode.addError( 'More than %d IS-IS instance in default VRF '
                           'without instance ID is not supported' % \
                              MAX_ISIS_INSTANCES_PER_VRF )
            return None
         else:
            mode.addError( 'IS-IS instance %s is already configured with '
                           'instance ID %d' % ( instanceNameById[ instanceId ],
                                                instanceId ) )
            return None
   else:
      if vrfName in instanceByVrfName:
         mode.addError( 'More than %d IS-IS instance in non-default VRF '
                        'is not supported' % MAX_ISIS_INSTANCES_PER_VRF )
         return None

   if vrfName != DEFAULT_VRF:
      maxVrfs = routingHardwareStatusCommon.vrfCapability.maxVrfs
      instanceByVrfName.pop( DEFAULT_VRF, None )
      if maxVrfs != -1 and len( instanceByVrfName ) >= maxVrfs:
         mode.addError( 'More than %d IS-IS VRF instances are not supported' % \
                           maxVrfs )
         return None

   IraVrfCli.addAgentVrfEntry( vrfName, "Isis" )
   return isisConfig.instanceConfig.newMember( instanceName, instanceId,
                                               vrfName )

def gotoRouterIsisMode( mode, instanceName, instanceId, vrfName ):
   """Handler for command -
     'router isis <instance-name> [instance-id <instance-id> | vrf <vrf-name>]'"""
   # We do not check if IP routing or IPv6 routing is enabled here.
   # We do it when we enter the address-family ipv4 | ipv6 CLI

   # Request trap resouces if multi-instance
   if instanceId:
      trapConfig.features.add( TrapFeatureName.isisMi )

   # Get the instanceConfig object, create it if not present
   instanceConfig = getInstanceConfig( mode, instanceName, instanceId, vrfName )
   if instanceConfig is None:
      return

   childMode = mode.childMode( submodes.RouterIsisMode, 
                               instanceConfig=instanceConfig )
   mode.session_.gotoChildMode( childMode )

def anyMultiInstance( instanceConfig ):
   return any( i.instanceId for i in instanceConfig.values() )

def delRouterIsisMode( mode, instanceName ):
   """Handler for 'no|default router isis <instance-name>' command"""
   if instanceName in isisConfig.instanceConfig:
      vrfName = isisConfig.instanceConfig[ instanceName ].vrfName
      del isisConfig.instanceConfig[ instanceName ]
      if not anyInstanceInVrf( isisConfig.instanceConfig, vrfName ):
         IraVrfCli.removeAgentVrfEntry( vrfName, "Isis" )
      # Trap registration is only for isis-mi support and should be removed only 
      # if all of the isis multi-instances are deleted.
      if not anyMultiInstance( isisConfig.instanceConfig ):
         trapConfig.features.remove( TrapFeatureName.isisMi )

def deleteVrfHook( vrfName ):
   instanceByVrfName = isisInstanceByVrfName( isisConfig.instanceConfig )
   if vrfName in instanceByVrfName:
      return ( True, "IS-IS configuration for instance %s has been disabled" % \
                  instanceByVrfName[ vrfName ][ 0 ] )
   else:
      return ( True, None )

def getProtectionMode( protection ):
   protectionMode = None
   if protection == 'disabled' or protection is None:
      protectionMode = ProtectionMode.protectionDisabled
   elif protection == 'link-protection':
      protectionMode = ProtectionMode.linkProtection
   else:
      protectionMode = ProtectionMode.nodeProtection
   return protectionMode

def getSrlgProtection( srlg=False, srlgStrict=False ):
   srlgVal = ProtectionSrlg.srlgDefault
   if srlg == 'disabled':
      srlgVal = ProtectionSrlg.srlgDisabled
   elif srlg == 'strict':
      srlgVal = ProtectionSrlg.srlgStrict
   elif srlg == 'loose':
      srlgVal = ProtectionSrlg.srlgLoose
   return srlgVal

def RouterIsisAfMode_setMetricAllIntf( mode, args ):
   metricVal = args[ 'METRICVALUE' ]
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.metricAllIntfV4 = metricVal
   elif mode.addrFamily == 'ipv6':
      instanceConfig.metricAllIntfV6 = metricVal

def RouterIsisAfMode_noMetricAllIntf( mode, args=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.metricAllIntfV4 = instanceConfig.metricAllIntfDefault
   elif mode.addrFamily == 'ipv6':
      instanceConfig.metricAllIntfV6 = instanceConfig.metricAllIntfDefault
   
def RouterIsisAfMode_setMaxEcmp( mode, args ):
   maxEcmp = args.get( 'PATHS' )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.maxEcmpV4 = maxEcmp
   elif mode.addrFamily == 'ipv6':
      instanceConfig.maxEcmpV6 = maxEcmp

def RouterIsisAfMode_noMaxEcmp( mode, args=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.maxEcmpV4 = instanceConfig.maxEcmpV4Invalid
   elif mode.addrFamily == 'ipv6':
      instanceConfig.maxEcmpV6 = instanceConfig.maxEcmpV6Invalid

def RouterIsisAfMode_setRedistribute( mode, proto, routeType=None, mapName=None,
                                      includeLeaked=False ):
   if proto == 'protoDhcp' and mode.addrFamily == 'ipv4':
      mode.addError( 'Redistributing DHCP routes not supported in IPv4 '
                           'address family mode' )
   else:
      setRedistributeCommon( mode, mode.instanceName, proto,
                             addrFamily=mode.addrFamily,
                             routeType=routeType, mapName=mapName,
                             includeLeaked=includeLeaked )

def RouterIsisAfMode_noRedistribute( mode, proto, routeType=None ):
   noRedistributeCommon( mode.instanceName, mode.addrFamily, proto,
                         routeType=routeType )

def RouterIsisAfMode_setBgpLuDirectRedistributeV4( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.bgpLuDirectRedistributeV4 = True
   rcfName = ''
   if 'FUNCTION' in args:
      rcfName = args.get( 'FUNCTION' )
      if rcfName.endswith( '()' ):
         # Remove possible () on RCF name
         rcfName = rcfName[ : -2 ]
   instanceConfig.bgpLuDirectRedistRcfV4 = rcfName

def RouterIsisAfMode_noBgpLuDirectRedistributeV4( mode, args=None ):
   if mode.addrFamily == 'ipv4':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.bgpLuDirectRedistributeV4 = False
      instanceConfig.bgpLuDirectRedistRcfV4 = ''

def RouterIsisAfMode_setMicroLoopConvergenceDelay( mode, args ):
   """
   Handler for [no|default] timers local-convergence-delay protected-prefixes
   """
   setMicroLoopProtectedCommon( mode, mode.instanceName,
                                addrFamily=mode.addrFamily )
   if 'delay' in args:
      delay = args.get( 'delay' )
      setMicroLoopConvergenceDelayCommon( mode, mode.instanceName,
                                          delay, addrFamily=mode.addrFamily )

def RouterIsisAfMode_noMicroLoopConvergenceDelay( mode, args=None ):
   setMicroLoopProtectedCommon( mode, mode.instanceName,
                                addrFamily=mode.addrFamily, negate=True )
   setMicroLoopConvergenceDelayCommon( mode, mode.instanceName, None,
                                       addrFamily=mode.addrFamily, negate=True )

def RouterIsisAfMode_setRouteLeak( mode, isLevel=None, mapName=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      if isLevel == 'level-1':
         instanceConfig.leakRouteMapL1ToL2V4 = mapName
      elif isLevel == 'level-2':
         instanceConfig.leakRouteMapL2ToL1V4 = mapName
   elif mode.addrFamily == 'ipv6':
      if isLevel == 'level-1':
         instanceConfig.leakRouteMapL1ToL2V6 = mapName
      elif isLevel == 'level-2':
         instanceConfig.leakRouteMapL2ToL1V6 = mapName

def RouterIsisAfMode_noRouteLeak( mode, isLevel=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   RouterIsisAfMode_setRouteLeak( mode, isLevel, instanceConfig.leakRouteMapInvalid )

def RouterIsisAfMode_setLeakSummaryAddress( mode, args ):
   if 'V4PREFIX' in args:
      prefix = args[ 'V4PREFIX' ].stringValue
      collection = isisConfig.instanceConfig[ 
                      mode.instanceName ].summaryAddressL1ToL2LeakV4
   else:
      prefix = args[ 'V6PREFIX' ].stringValue
      collection = isisConfig.instanceConfig[ 
                      mode.instanceName ].summaryAddressL1ToL2LeakV6

   if prefix == '0.0.0.0/0' or prefix == '::/0': # pylint: disable=consider-using-in
      mode.addError( "Summary address for %s is not supported" % prefix )
      return
   prefix = Arnet.IpGenPrefix( prefix )
   summAddr = Tac.Value( 'Routing::Isis::SummaryAddress', prefix )
   collection.addMember( summAddr )

def RouterIsisAfMode_setDistance( mode, args ):
   """Set the protocol addmnistrative distance.

   :param valueAndLevel: dictionary of distance value and level arguments
   """
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   distanceValue = args[ 'DISTANCE_VALUE' ]
   level = args.get( 'LEVEL' )
   applyToL1 = level != 'level-2'
   applyToL2 = level != 'level-1'
   if mode.addrFamily == 'ipv4':
      if applyToL1:
         instanceConfig.distanceV4L1 = distanceValue
      if applyToL2:
         instanceConfig.distanceV4L2 = distanceValue
   elif mode.addrFamily == 'ipv6':
      if applyToL1:
         instanceConfig.distanceV6L1 = distanceValue
      if applyToL2:
         instanceConfig.distanceV6L2 = distanceValue

def RouterIsisAfMode_noDistance( mode, args ):
   """Reset the protocol addmnistrative distance to the default values.

   :param valueAndLevel: dictionary of distance value and level arguments
   """
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   args[ 'DISTANCE_VALUE' ] = instanceConfig.distanceDefault
   RouterIsisAfMode_setDistance( mode, args )

def RouterIsisAfMode_clearConfig( mode ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   RouterIsisAfMode_noMaxEcmp( mode )
   delIsisRedistAf( instanceConfig, mode.addrFamily )
   delIsisDistanceAf( instanceConfig, mode.addrFamily )
   delIsisTrafficSteering( instanceConfig, mode.addrFamily )
   RouterIsisAfMode_noBfd( mode )
   RouterIsisAfMode_multiTopologyDel( mode )
   RouterIsisAfMode_igpShortcutDel( mode )
   RouterIsisAfMode_igpShortcutDisableDel( mode )
   RouterIsisAfMode_noBgpLuDirectRedistributeV4( mode )
   RouterIsisAfMode_noIgnoreAttachedBit( mode )
   RouterIsisAfMode_noMetricAllIntf( mode )
   RouterIsisAfMode_noFrrProtection( mode )
   RouterIsisAfMode_noFrrSrlgProtection( mode )
   RouterIsisAfMode_noMicroLoopConvergenceDelay( mode )
   RouterIsisAfMode_delIsisAfSrv6Mode( mode )
   for modeletName in mode.modeletMap:
      if modeletName != mode.__class__:
         mode.modeletMap[ modeletName ].clearConfig()

def RouterIsisAfMode_setIgnoreAttachedBit( mode, args ):
   ignore = 'none' in args
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.ignoreAttachedBitV4 = ignore
   else:
      instanceConfig.ignoreAttachedBitV6 = ignore

def RouterIsisAfMode_noIgnoreAttachedBit( mode, args=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.ignoreAttachedBitV4 = \
            instanceConfig.ignoreAttachedBitV4Default
   else:
      instanceConfig.ignoreAttachedBitV6 = \
            instanceConfig.ignoreAttachedBitV6Default

def RouterIsisAfMode_setProtection( mode, protection, protectionLevel ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   protectionMode = getProtectionMode( protection )
   protectionConfig = ProtectionConfig( protectionMode )
   protectionConfig.level = LVL_KW_TO_TAC_TYPE_MAP[ protectionLevel ]

   if mode.addrFamily == 'ipv4':
      instanceConfig.protectionConfigAllIntfV4 = protectionConfig
   elif mode.addrFamily == 'ipv6':
      instanceConfig.protectionConfigAllIntfV6 = protectionConfig

def RouterIsisAfMode_noFrrProtection( mode, args=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.protectionConfigAllIntfV4 \
         = ProtectionConfig( ProtectionMode.protectionDefault )
   elif mode.addrFamily == 'ipv6':
      instanceConfig.protectionConfigAllIntfV6 \
         = ProtectionConfig( ProtectionMode.protectionDefault )

def RouterIsisAfMode_frrProtection( mode, args ):
   protection = args.get( 'PROTECTION' )
   if protection:
      protectionLevel = args.get( 'LEVEL', 'level-1-2' )
      RouterIsisAfMode_setProtection( mode, protection, protectionLevel )
   else:
      RouterIsisAfMode_noFrrProtection( mode )

def RouterIsisAfMode_frrSrlgProtection( mode, args ):
   srlg = 'strict' if 'strict' in args else 'loose'
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   srlgVal = getSrlgProtection( srlg )
   if mode.addrFamily == 'ipv4':
      instanceConfig.protectionSrlgV4 = srlgVal
   elif mode.addrFamily == 'ipv6':
      instanceConfig.protectionSrlgV6 = srlgVal

def RouterIsisAfMode_noFrrSrlgProtection( mode, args=None ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if mode.addrFamily == 'ipv4':
      instanceConfig.protectionSrlgV4 = instanceConfig.protectionSrlgDefault
   elif mode.addrFamily == 'ipv6':
      instanceConfig.protectionSrlgV6 = instanceConfig.protectionSrlgDefault

def RouterIsisAfMode_igpShortcutDisableIs( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.igpShortcutV4 = False

def RouterIsisAfMode_igpShortcutDisableDel( mode, args=None ):
   if mode.addrFamily == 'ipv4':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.igpShortcutV4 = True

def RouterIsisAfMode_igpShortcutIs( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   no = CliCommand.isNoOrDefaultCmd( args )
   instanceConfig.igpShortcutV6 = not no

def RouterIsisAfMode_igpShortcutDel( mode, args=None ):
   if mode.addrFamily == 'ipv6':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.igpShortcutV6 = False

def RouterIsisAfMode_trafficSteeringIs( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   # Only supported for IPv4 address family and tunnel type rsvp as of now
   assert mode.addrFamily == 'ipv4'
   assert args[ 'TUNNEL_TYPE' ] == 'rsvp'
   tunnel = args[ 'TUNNEL_NAME' ]
   prefix_list = args[ 'PREFIX_LIST_NAME' ]
   instanceConfig.trafficSteeringRsvpV4[ tunnel ] = prefix_list

def RouterIsisAfMode_noTrafficSteering( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   # Only supported for IPv4 address family and tunnel type rsvp as of now
   assert mode.addrFamily == 'ipv4'
   assert args[ 'TUNNEL_TYPE' ] == 'rsvp'
   tunnel = args[ 'TUNNEL_NAME' ]
   del instanceConfig.trafficSteeringRsvpV4[ tunnel ]

def RouterIsisAfMode_setBfd( mode, args ):
   if mode.addrFamily == 'ipv4':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.bfdEnabledV4 = True
   elif mode.addrFamily == 'ipv6':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.bfdEnabledV6 = True

def RouterIsisAfMode_noBfd( mode, args=None ):
   if mode.addrFamily == 'ipv4':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.bfdEnabledV4 = False
   elif mode.addrFamily == 'ipv6':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.bfdEnabledV6 = False

def RouterIsisAfMode_multiTopologyIs( mode ):
   if mode.addrFamily == 'ipv6':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.multiTopology = True

def RouterIsisAfMode_multiTopologyDel( mode ):
   if mode.addrFamily == 'ipv6':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.multiTopology = False

def RouterIsisAfMode_gotoIsisAfSrv6Mode( mode, args ):
   assert mode.addrFamily == 'ipv6'
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srv6Configured = True
   childMode = mode.childMode( submodes.RouterIsisAfSrv6Mode )
   mode.session_.gotoChildMode( childMode )

def RouterIsisAfMode_delIsisAfSrv6Mode( mode, args=None ):
   if mode.addrFamily == 'ipv6':
      instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
      instanceConfig.srv6Configured = instanceConfig.srv6ConfiguredDefault
      instanceConfig.srv6Enabled = instanceConfig.srv6EnabledDefault
      instanceConfig.srv6Locator.clear()

def RouterIsisMetricProfileMode_setMetricValue( mode, args ):
   mode.metricProfileConfig.metric = args.get( 'METRIC_VALUE', 
         mode.metricProfileConfig.metricDefault )

def RouterIsisMetricProfileMode_setMetricRatio( mode, args ):
   mr = Tac.nonConst( mode.metricProfileConfig.metricRatio )
   mr.speed = args[ 'SPEED' ] 
   mr.speedUnit = args[ 'UNITS' ]
   mr.active = mr.speed != 0
   mode.metricProfileConfig.metricRatio = mr

def RouterIsisMetricProfileMode_delMetricRatio( mode, args ):
   mr = Tac.nonConst( mode.metricProfileConfig.metricRatio )
   mr.active = False
   mode.metricProfileConfig.metricRatio = mr

def RouterIsisMetricProfileMode_addMetricRule( mode, args ):
   metricValue = args[ 'METRIC_VALUE' ]
   speed = args[ 'SPEED' ]
   unit = args[ 'UNITS' ]
   newRule = MetricRule( metricValue, speed, unit )
   RouterIsisMetricProfileMode__delMetricRule( mode, newRule.realSpeed )
   if not CliCommand.isNoOrDefaultCmd( args ):
      mode.metricProfileConfig.metricRules.addMember( newRule )

def RouterIsisMetricProfileMode__delMetricRule( mode, realSpeed ):
   r = mode.metricProfileConfig.metricRules.get( realSpeed )
   if r:
      del mode.metricProfileConfig.metricRules[ realSpeed ]

def RouterIsisAfSrv6Mode_srv6Disabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srv6Enabled = False

def RouterIsisAfSrv6Mode_noSrv6Disabled( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srv6Enabled = instanceConfig.srv6EnabledDefault

def RouterIsisAfSrv6Mode_gotoIsisAfSrv6LocatorMode( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   locatorName = args[ 'LOCATOR' ]
   instanceConfig.srv6Locator.newMember( locatorName )
   childMode = mode.childMode( RouterIsisAfSrv6LocatorMode,
                               locatorName=locatorName )
   mode.session_.gotoChildMode( childMode )

def RouterIsisAfSrv6Mode_delIsisAfSrv6LocatorMode( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   del instanceConfig.srv6Locator[ args[ 'LOCATOR' ] ]

class RouterIsisAfSrv6LocatorMode( RoutingIsisAfSrv6LocatorMode,
                                   BasicCli.ConfigModeBase ):
   name = 'ISIS segment-routing ipv6 locator configuration'

   def __init__( self, parent, session, locatorName ):
      RoutingIsisAfSrv6LocatorMode.__init__( self, locatorName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def RouterIsisSrMplsMode_srShutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srEnabled = instanceConfig.srEnabledDefault

def RouterIsisSrMplsMode_noSrShutdown( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srEnabled = True

def RouterIsisSrMplsMode_setSrSegmentPathVerification( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srSegmentPathVerification = True

def RouterIsisSrMplsMode_noSrSegmentPathVerification( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srSegmentPathVerification = \
            instanceConfig.srSegmentPathVerificationDefault

def RouterIsisSrMplsMode_setSrRouterId( mode, args ):
   # Ideally we should never have the router-id command under the "router isis"/
   # "segment-routing mpls" submode. Users should configure router-id under the
   # "router general" mode or the "router isis" mode. Let them know about it.
   mode.addWarning( "Router ID configured under segment-routing mode will be "
                    "overridden by the general or the ISIS router ID" )
   routerId = args[ 'ROUTER_ID' ]
   if routerId == "0.0.0.0":
      mode.addError( "%s is not a valid router ID" % routerId )
      return

   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srRouterId = routerId

def RouterIsisSrMplsMode_noSrRouterId( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srRouterId = instanceConfig.routerIdV4Default

def RouterIsisSrMplsMode_setSrGlobalBlock( mode, srBase, srRange ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srSrgbBase = srBase
   instanceConfig.srSrgbRange = srRange

def RouterIsisSrMplsMode_noSrGlobalBlock( mode ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srSrgbBase = instanceConfig.srSrgbBaseDefault
   instanceConfig.srSrgbRange = instanceConfig.srSrgbRangeDefault

def RouterIsisSrMplsMode_adjacencySegmentAllocation( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   srAllocation = SrAdjAllocationType.srAdjacencyAllocationNone
   backup = args.get( 'backup-eligible' )
   if 'sr-peers' in args and backup:
      srAllocation = SrAdjAllocationType.srAdjacencyAllocationSrOnlyBackup
   elif 'sr-peers' in args:
      srAllocation = SrAdjAllocationType.srAdjacencyAllocationSrOnly
   elif 'all-interfaces' in args and backup:
      srAllocation = SrAdjAllocationType.srAdjacencyAllocationAllBackup
   elif 'all-interfaces' in args:
      srAllocation = SrAdjAllocationType.srAdjacencyAllocationAll
   instanceConfig.srAdjSegmentAlloc = srAllocation

def RouterIsisSrMplsMode_noAdjacencySegmentAllocation( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srAdjSegmentAlloc = \
       instanceConfig.srAdjSegmentAllocDefault

def RouterIsisSrMplsMode_adjacencySegmentSidReuseTimeout( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if 'infinite' in args:
      instanceConfig.srAdjSegmentSidReuseTimeout = U32_MAX_VALUE
   else:
      instanceConfig.srAdjSegmentSidReuseTimeout = args[ 'TIMEOUT' ]

def RouterIsisSrMplsMode_noAdjacencySegmentSidReuseTimeout( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   instanceConfig.srAdjSegmentSidReuseTimeout = \
                           instanceConfig.srAdjSegmentSidReuseTimeoutDefault

def RouterIsisSrMplsMode_setPrefixSegment( mode, args ):
   prefix = args.get( 'IPADDR' )
   if prefix is None:
      prefix = args.get( 'IP6ADDR' )
   
   labelType = args.get( 'label', 'index' )
   label = args.get( 'LABEL', PrefixSegmentInfo.labelInvalid )
   index = args.get( 'INDEX', PrefixSegmentInfo.indexInvalid )
   algoName = args.get( 'FAD_NAME', '' )
   
   noPhp = 'no-php' in args
   explicitNull = 'explicit-null' in args
   prefix = Arnet.IpGenPrefix( prefix.stringValue )
   prefixConflictHelper = getConflictingConfigFoundDict()[ 'prefixConflict' ]
   if prefixConflictHelper( mode.instanceName, None, prefix, mode=mode,
         algoName=algoName ):
      mode.addError( "prefix segment conflicts with a proxy/node segment"
                     " configured for the prefix %s" % prefix )
      return
   
   # Check for label-label conflicts
   if labelType == 'label':
      labelConflictHelper = getConflictingConfigFoundDict()[ "labelConflict" ]
      if labelConflictHelper( mode.instanceName, None, prefix, label, mode=mode,
                  algoName=algoName ):
         mode.addError( "Two prefixes cannot be configured with the same label" )
         return
   else:
      sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
      if sidConflictHelper( mode.instanceName, None, prefix, index, mode=mode,
                  algoName=algoName ):
         mode.addError( "Two prefixes cannot be configured with the same SID" )
         return
   
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   algoPrefix = Tac.Value( "Routing::Isis::AlgoNamePrefix", algoName=algoName,
                           prefix=prefix )
   prefixSegment = PrefixSegment( algoPrefix )
   prefixSegment.index = index
   prefixSegment.label = label
   prefixSegment.isNoPhp = noPhp
   prefixSegment.isExplicitNull = explicitNull
   if explicitNull:
      # no-PHP flag needs to be set when E flag is set,
      # because if P-Flag is unset, the received E-Flag is ignored
      prefixSegment.isNoPhp = True
   instanceConfig.srPrefixSegments.addMember( prefixSegment )

def RouterIsisSrMplsMode_noPrefixSegment( mode, args ):
   prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
   algoName = args.get( 'FAD_NAME', '' )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   algoPrefix = Tac.Value( "Routing::Isis::AlgoNamePrefix", algoName=algoName,
                           prefix=Arnet.IpGenPrefix( prefix.stringValue ) )
   if algoName:
      labelType = args.get( 'label', 'index' )
      label = args.get( 'LABEL', PrefixSegmentInfo.labelInvalid )
      index = args.get( 'INDEX', PrefixSegmentInfo.indexInvalid )
      if algoPrefix in instanceConfig.srPrefixSegments:
         prefixSegment = instanceConfig.srPrefixSegments[ algoPrefix ]
         # delete the segment only if the sid value is also matching
         if ( labelType == 'label' and prefixSegment.label != label ) or \
            ( labelType == 'index' and prefixSegment.index != index ):
            return
   del instanceConfig.srPrefixSegments[ algoPrefix ]

def RouterIsisSrMplsMode_setProxyNodeSegment( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   attachedFlagKnobPresent = 'attached-flag' in args
   ignoreConflictPresent = 'conflict' in args and 'ignore' in args

   if attachedFlagKnobPresent:
      instanceConfig.inspectProxyAttachedFlag = True
   else:
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      prefix = Arnet.IpGenPrefix( prefix.stringValue )
      rangeVal = args.get( 'RANGE', PrefixRange.min )
      labelType = args.get( 'label', 'index' )
      label = args.get( 'LABEL', PrefixSegmentInfo.labelInvalid )
      index = args.get( 'INDEX', PrefixSegmentInfo.indexInvalid )

      if not prefix.isHost:
         mode.addErrorAndStop( "A proxy-node segment can be configured for a"
                               " /32 IPv4 prefix or /128 IPv6 prefix only" )
      
      if rangeVal > 1 and not _isisSrValidProxyNodePrefix( prefix, rangeVal ):
         mode.addErrorAndStop( "An invalid prefix has been specified for the"
                               "proxy-node segment" )

      # TODO Cleanup passing in mode.instanceName and mode
      prefixConflictHelper = getConflictingConfigFoundDict()[ "prefixConflict" ]
      if prefixConflictHelper( mode.instanceName, None, prefix, proxy=True,
                               mode=mode, rangeVal=rangeVal,
                               ignoreConflict=ignoreConflictPresent ):
         mode.addErrorAndStop( "proxy node segment conflicts with a prefix/node"
                               " segment configured for the prefix %s" % prefix )

      if labelType == 'label':
         labelConflictHelper = getConflictingConfigFoundDict()[ "labelConflict" ]
         if labelConflictHelper( mode.instanceName, None, prefix, label, mode=mode,
                                 rangeVal=rangeVal ):
            mode.addError( "Two prefixes cannot be configured with the same label" )
            return
      else:
         sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
         if sidConflictHelper( mode.instanceName, None, prefix, index, mode=mode,
                               rangeVal=rangeVal ):
            mode.addErrorAndStop( "Two prefixes cannot be configured with"
                                  " the same SID" )

      level = args.get( 'LEVEL', 'level-1-2' )
      algoPrefix = Tac.Value( "Routing::Isis::AlgoNamePrefix", algoName='',
                              prefix=prefix )
      segment = PrefixSegment( algoPrefix )
      segment.level = LVL_KW_TO_TAC_TYPE_MAP[ level ]
      segment.isProxyNode = True
      segment.index = index
      segment.label = label
      #Penultimate hop should pop the label for attached prefix
      #Penultimate hop should not pop the label for non-attached prefix
      segment.isAttached = 'attached' in args
      segment.range = rangeVal
      segment.ignoreConflict = ignoreConflictPresent
      instanceConfig.srPrefixSegments.addMember( segment )

def RouterIsisSrMplsMode_noProxyNodeSegment( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   attachedFlagKnobPresent = 'attached-flag' in args

   if attachedFlagKnobPresent:
      instanceConfig.inspectProxyAttachedFlag = False
   else:
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      algoPrefix = Tac.Value( "Routing::Isis::AlgoNamePrefix", algoName='',
                              prefix=Arnet.IpGenPrefix( prefix.stringValue ) )
      del instanceConfig.srPrefixSegments[ algoPrefix ]

def RouterIsisSrMplsMode_setFlexAlgo( mode, args ):
   name = args[ 'NAME' ]
   level = args.get( 'LEVEL', 'level-1-2' )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   algo = instanceConfig.flexAlgoAlgorithms.newMember( name )
   algo.level = LVL_KW_TO_TAC_TYPE_MAP[ level ]
   algo.advertised = 'advertised' in args

def RouterIsisSrMplsMode_noFlexAlgo( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   del instanceConfig.flexAlgoAlgorithms[ args[ 'NAME' ] ]

#--------------------------------------------------------------------------
# [no|default] router isis <INST> [ instance-id <INST_ID> |vrf <VRFNAME>]
#--------------------------------------------------------------------------
def getCurrentIsis( mode ):
   instNameCollection = {}
   for ( k, v ) in isisConfig.instanceConfig.items():
      instNameCollection[ k ] = ( 'Configure ISIS %s with instance ID %d in %s VRF'
            % ( k, v.instanceId, v.vrfName ) )
   return instNameCollection
#----------------------------------------------------------------------------------
# The "[no|default] lsp rfc8202 rx disabled" command
# under 'router isis <> instance-id <>' mode
#----------------------------------------------------------------------------------
#----------------------------------------------------------------------------------
# The "[no|default] rfc8202 disabled" command
# under 'router isis <> instance-id <>' mode
#----------------------------------------------------------------------------------
#----------------------------------------------------------------------------------
# The "[no|default] lsp purge origination-identification [ LEVEL ]" command
# under 'router isis <>' mode
#----------------------------------------------------------------------------------
#------------------------------------------------------------------------------
# [no|default] is-type LEVEL under 'router isis <>' mode
#------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no | default] address-family AF [ unicast ]
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# "[no | default] multi-topology"
# command, in config-router-isis-af mode.
#-------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] traffic-engineering
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] segment-routing mpls
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] area proxy
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] shutdown
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] net <NET>
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] is-hostname <HOSTNAME>
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] area segment (ipv4|ipv6) index <value>
#---------------------------------------------------------------------------------

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] router-id ipv4 [ROUTER_IDV4]
#---------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# [no|default] redistribute <static|connected|bgp> [include leaked]
#              [route-map <route-map-name>]
#------------------------------------------------------------------------------
PROTO_KW_MAP = {
   'static' : 'protoStatic',
   'connected' : 'protoDirect',
}



#-----------------------------------------------------------------------------------
# [no|default] redistribute ospf [include leaked]
#              match <internal|external|nssa-exeternal> [route-map <route-map-name>]
#-----------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------
# [no|default] redistribute bgp [include leaked] [route-map <route-map-name>]
#-----------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------
# [no|default] route redistributed limit [level-1 | level-2] <Max Limit>
#-----------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# "[no|default] mpls ldp sync default" in 'router isis <>' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# "[no|default] max-lsp-lifetime <time>" under 'router isis <>' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# "[no|default] lsp size maximum <size>" under 'router isis <>' mode
#-------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# '[no|default] is-hostname <hostname>' under 'router isis <>' mode
#---------------------------------------------------------------------------------
# TE CLI
#----------------------------------------------------
#"[no | default] shutdown" command, in isis - te mode
#----------------------------------------------------
#------------------------------------------------------------------------------
# [no|default] is-type LEVEL under isis-te mode
#------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] shutdown
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] log-adjacency-changes
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] log duplicate-system-id disabled
# command in "router isis" config mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] hello padding [disabled] command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# "[no|default] net <NET>" command in "router isis <n>" config mode.
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# "[no|default] area address <AREA>" command in "router isis <n>" config mode.
#-------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no] bfd all-interfaces in address-family v4 and v6 config mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] spf-interval <max-wait> [ initial-wait [hold-time] ] command in
# "router isis <n>" config mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no] spf partial prefix command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no] set-attached-bit route-map <rmName>  command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] "maximum-paths <paths>" command, in 'address-family mode'
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# "[no|default] lsp match flag attached action none | (install default-route)"
# command in 'address-family mode'.
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] lsp flooding dynamic [ LEVEL ]
# command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no|default] passive <intfName>  command in "router isis <n>" config mode
#
# deprecated:
# [no|default] passive-interface <intfName>  command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
def vrfMatches( mode, intfName ):
   instVrf = isisConfig.instanceConfig[ mode.instanceName ].vrfName
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )
   ip6IntfConfig = ip6Config.intf.get( intfName )
   # There's an off-chance that only ip6IntfConfig has the vrf. Take the value from
   # there if present
   intfVrf = ( getattr( ipIntfConfig, 'vrf', '' ) or
               getattr( ip6IntfConfig, 'vrf', '' ) or
               DEFAULT_VRF )

   if intfVrf == instVrf:
      return True

   mode.addError( f"Interface {intfName} does not belong to Vrf {instVrf}"
                  " in which IS-IS instance is configured" )
   return False

def setPassiveIntf( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   acceptableInsts = ( IntfConfig.instanceNameInvalid, instanceConfig.instanceName )
   for intf in args[ 'INTFS' ]:
      if vrfMatches( mode, intf ):
         if not checkNodePrefixConflict( mode, mode.instanceName, intf ):
            intfConfig = _getOrCreateIntfConfig( isisConfiguration(), intf )
            if intfConfig.instanceName in acceptableInsts:
               intfConfig.passive = True
               intfConfig.instanceName = instanceConfig.instanceName
               updateAllInstanceColl( intfConfig )
            else:
               mode.addError( f"Interface {intf} is configured in a different IS-IS "
                              f"instance {intfConfig.instanceName}" )

def noPassiveIntf( mode, args ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   acceptableInsts = ( IntfConfig.instanceNameInvalid, instanceConfig.instanceName )
   for intf in args[ 'INTFS' ]:
      intfConfig = _getIntfConfig( intf )
      if intfConfig and intfConfig.passive and vrfMatches( mode, intf ):
         if intfConfig.instanceName in acceptableInsts:
            intfConfig.passive = False
            intfConfig.instanceName = intfConfig.instanceNameInvalid
            updateAllInstanceColl( intfConfig )
            _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, intf )
         else:
            mode.addError( f"Interface {intf} is configured in a different IS-IS "
                           f"instance {intfConfig.instanceName}" )

#-------------------------------------------------------------------------------
# Metric Profile specific CLI (common parts)
#-------------------------------------------------------------------------------
def findMetricProfileConfig( name ):
   for iName, instanceCfg in isisConfig.instanceConfig.items():
      profile = instanceCfg.metricProfiles.get( name )
      if profile :
         return profile, iName
   return None, None

#-------------------------------------------------------------------------------
# Segment-Routing specific CLI
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] shutdown
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] segment path verification prefix-segment adjacency-segment
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] router-id <router-id>
#-------------------------------------------------------------------------------
# router-id configuration is not supposed to be under the RouterIsisSrMplsMode.
# Make it hidden for backward compatibility.
#-------------------------------------------------------------------------------
# [no|default] adjacency-segment allocation [none|sr-peers [backup-eligible]|
# all-interfaces [backup-eligible] ]
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] adjacency-segment sid reuse timeout <time>  XXX: Hidden
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] prefix-segment <prefix> index <index> [ explicit-null | no-php ]
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# [no|default] proxy-node-segment <host prefix> index <index> [ LEVEL ]
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] flex-algo <name> [level-1|level-2|level-1-2] [advertised]
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# segment-routing ipv6 specific CLI
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# '[no|default] disabled' under 'segement-routing ipv6' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] locator LOCATOR' under 'segement-routing ipv6' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute bgp' under 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute user [route-map <route-map-name>] '
# 'under 'router isis' and 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute dynamic [route-map <route-map-name>] '
# 'under 'router isis' and 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------
# [no|default] tunnel source-protocol bgp ipv4 labeled-unicast [ rcf <func-name> ] 
# under 'address-family ipv4' mode
#-----------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute dhcp' under 'address-family ipv6' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1 into level-2 summary-address V4PREFIX'
# under 'address-family ipv4' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1 into level-2 summary-address V6PREFIX'
# under 'address-family ipv6' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1|level-2 into level-2|level-1 route-map <>'
# under 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# '[no|default] redistribute isis instance [route-map <>]'
# under 'router isis' and 'address-family ipv[4|6]' modes
#-------------------------------------------------------------------------------
#----------------------------------------------------------------------------------
# [no|default] "distance <1-255> [ LEVEL ]" command, in address-family mode
#----------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# [no] authentication mode [ < text|md5|sha key-id <id> > [rx-disable] [ LEVEL ] ]
#---------------------------------------------------------------------------------
#
#---------------------------------------------------------------------------------
# [no] authentication [key-id <id> algorithm SHA]
#       key [ [0|7] < key-string > [ LEVEL ] ]
#---------------------------------------------------------------------------------
#-----------------------------------------------------------------------------
# [no| default ] maximum-sid-depth base-mpls-imposition
#-----------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# Adds isis-specific CLI commands to the "config-if" mode for routed ports.
# Enable isis commands under config-if only on routed and loopback ports
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# Associate the interface config modelets with the "config-if" mode.
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# SrConfig stores all common sr configuration for OSPF and ISIS (only node segment
# for now)
#-------------------------------------------------------------------------------
def _getSrIntfConfig( intfName ):
   return srConfig.intfConfig.get( intfName, None )

#-------------------------------------------------------------------------------
# IntfConfig for ISIS configuration, is created when one of its attributes
# is configured. It is deleted when all the attributes are at their defaults
# What this means is that after the last "no isis..." command is run on the
# interface, we delete the intfConfig object
#-------------------------------------------------------------------------------
def _getIntfConfig( intfName ):
   return isisConfiguration().intfConfig.get( intfName, None )

def _getOrCreateIntfConfig( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      intfConfig = config.intfConfig.newMember( intfName )
   return intfConfig

def _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName ):
   """Delete the intfConfig collection element if all the attributes of intfConfig
   have default values. This needs to be called by command handlers for "no" form
   of the isis interface config commands. Also when we add a new attribute to
   intfConfig, it needs to be added below in the comparison against default values
   """
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      return
   if ( ( intfConfig.instanceName == intfConfig.instanceNameInvalid ) and
        ( not intfConfig.multiInstanceName ) and
        ( intfConfig.lspInterval == intfConfig.lspIntervalDefault ) and
        ( intfConfig.priority == intfConfig.priorityDefault ) and
        ( not intfConfig.confMetric.isSet ) and
        ( not intfConfig.metricProfile ) and
        ( not intfConfig.metricProfileV6 ) and
        ( not intfConfig.confMetricV6.isSet ) and
        ( intfConfig.mtAddressFamily == intfConfig.mtAddressFamilyDefault ) and
        ( intfConfig.helloInterval == intfConfig.helloIntervalDefault ) and
        ( intfConfig.helloMultiplier == intfConfig.helloMultiplierDefault ) and
        ( intfConfig.interfaceType == intfConfig.interfaceTypeDefault ) and
        ( intfConfig.passive == intfConfig.passiveDefault ) and
        ( len( intfConfig.srSingleAdjacencySegment ) == 0 ) and
        ( len( intfConfig.srMultipleAdjacencySegment ) == 0 ) and
        ( intfConfig.circuitType == intfConfig.circuitTypeDefault ) and
        ( intfConfig.bfdV4 == intfConfig.bfdV4Default ) and
        ( intfConfig.bfdV6 == intfConfig.bfdV6Default ) and
        ( intfConfig.authModeL1 == intfConfig.authModeDefault ) and
        ( intfConfig.authModeL2 == intfConfig.authModeDefault ) and
        ( not intfConfig.authKeyColl ) and
        ( intfConfig.helloPadding == intfConfig.helloPaddingDefault ) and
        ( intfConfig.routeTagV4 == intfConfig.routeTagNotConfigured ) and
        ( intfConfig.routeTagV6 == intfConfig.routeTagNotConfigured ) and
        ( not intfConfig.protectionConfigV4 ) and
        ( not intfConfig.protectionConfigV6 ) and
        ( not intfConfig.protectionConfigV4V6 ) and
        ( intfConfig.areaProxyBoundary == intfConfig.areaProxyBoundaryDefault ) and
        ( intfConfig.protectionSrlgV4 == intfConfig.protectionSrlgDefault ) and
        ( intfConfig.protectionSrlgV6 == intfConfig.protectionSrlgDefault ) and
        ( intfConfig.protectionSrlgV4V6 == intfConfig.protectionSrlgDefault ) ):

      del config.intfConfig[ intfName ]

def addIpInterfaceWarnings( mode, addrFamily ):
   '''
   Warnings are issued whenever an IS-IS instance is enabled on an interface
   with missing V4/V6 address configuration. These warnings are also applicable 
   for user-configured MT address-family commands
   @mode - The CLI mode where these warnings get printed
   @addrFamily - The configured address family(ies)
   '''
   ipConfigured = ipConfig.ipIntfConfig.get( mode.intf.name )
   ip6Configured = ip6Config.intf.get( mode.intf.name )

   # When a vrf is associated with an interface with no ip address configured
   # the default address "0.0.0.0/0" is assigned. A warning needs to be given
   # in that case
   ipV4WarningCondition = ( ipConfigured is None
         or ( ipConfigured.addrWithMask ==
              Arnet.AddrWithMask( "0.0.0.0/0" )
              and not ipConfigured.unnumberedIntfId )
         )

   ipV6WarningCondition = ( ip6Configured is None
         or ip6Configured.ipv6Configured() == False )
   v4WarningMsg = "Interface does not have IPv4 address configured"
   v6WarningMsg = "Interface does not have IPv6 address configured"
   bothIpWarningMsg = (
         "Interface does not have IPv4 and IPv6 address configured" )
   noIpWarningMsg = "Interface does not have IPv4 or IPv6 address configured"

   if addrFamily == IsisAddressFamily.addressFamilyNone \
         and ipV4WarningCondition and ipV6WarningCondition :
      mode.addWarning( noIpWarningMsg )
   elif addrFamily == IsisAddressFamily.addressFamilyIPv4 \
         and ipV4WarningCondition:
      mode.addWarning( v4WarningMsg )
   elif addrFamily == IsisAddressFamily.addressFamilyIPv6 \
         and ipV6WarningCondition:
      mode.addWarning( v6WarningMsg )
   elif addrFamily == IsisAddressFamily.addressFamilyBoth:
      if ipV4WarningCondition and ipV6WarningCondition:
         mode.addWarning( bothIpWarningMsg )
      elif ipV4WarningCondition:
         mode.addWarning( v4WarningMsg )
      elif ipV6WarningCondition:
         mode.addWarning( v6WarningMsg )

#-------------------------------------------------------------------------------
# "[no|default] isis enable"
# command, in config-if mode.
# "[no|default] isis instance <INSTANCE_NAME>"
# command, in loopback config-if mode.
#-------------------------------------------------------------------------------
# When an interface is being added to a ISIS instance or being set as
# passive, check for any prefix segment conflicts that could happen
def checkNodePrefixConflict( mode, instanceName, intfName ):
   srIntfConfig = _getSrIntfConfig( intfName )
   if not srIntfConfig:
      # The interface has no node-segment configuration, so there can't
      # be any conflicts, return False
      return False

   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )
   ip6IntfConfig = ip6Config.intf.get( intfName )
   if ipIntfConfig:
      prefix = Arnet.IpGenPrefix( str ( ipIntfConfig.addrWithMask.subnet ) )
   else:
      prefix = None

   # If the interface has v4 node-segment configured, check for v4 prefix conflicts
   v4NodeSids = list( srIntfConfig.srV4NodeSegment )
   v4NodeSidLabels = list( srIntfConfig.srV4NodeSegmentLabel )
   if srIntfConfig.srNodeSegmentIndex != srSidInvalid:
      v4NodeSids.append( srIntfConfig.srNodeSegmentIndex )

   if v4NodeSids or v4NodeSidLabels:
      if prefix != None:
         prefixConflictHelper = getConflictingConfigFoundDict()[ "prefixConflict" ]
         if prefixConflictHelper( instanceName, intfName, prefix, mode=mode ):
            mode.addError( "Node-segment conflicts with a prefix/proxy segment"
                           " configured for the prefix %s" % prefix )
            return True

      sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
      for v4NodeSid in v4NodeSids:
         if sidConflictHelper( instanceName, intfName, None, v4NodeSid, mode=mode ):
            mode.addError( "IPv4 Node-segment index conflicts with an existing"
                           " segment index in the instance %s" % instanceName )
            return True

      labelConflictHelper = getConflictingConfigFoundDict()[ "labelConflict" ]
      for v4NodeSidLabel in v4NodeSidLabels:
         if labelConflictHelper( instanceName, intfName, None, v4NodeSidLabel,
                                 ipv6=False, mode=mode ):
            mode.addError( "IPv4 Node-segment label conflicts with an existing"
                           " segment label in the instance %s" % instanceName )
            return True

   # If the interface has v6 node-segment configured, check for v6 prefix conflicts
   v6NodeSids = list( srIntfConfig.srV6NodeSegment )
   v6NodeSidLabels = list( srIntfConfig.srV6NodeSegmentLabel )
   if srIntfConfig.srV6NodeSegmentIndex != srSidInvalid:
      v6NodeSids.append( srIntfConfig.srV6NodeSegmentIndex )

   if v6NodeSids or v6NodeSidLabels:
      if ip6IntfConfig:
         prefixConflictHelper = getConflictingConfigFoundDict()[ "prefixConflict" ]
         for prefix in ip6IntfConfig.addr:
            prefix = Arnet.IpGenPrefix( str( prefix ) )
            if prefixConflictHelper( instanceName, intfName, prefix, mode=mode ):
               mode.addError( "Node-segment conflicts with a prefix/proxy segment"
                              " configured for the prefix %s" % prefix )
               return True

      sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
      for v6NodeSid in v6NodeSids:
         if sidConflictHelper( instanceName, intfName, None, v6NodeSid, mode=mode ):
            mode.addError( "IPv6 Node-segment index conflicts with an existing"
                           " segment index in the instance %s" % instanceName )
            return True
      
      labelConflictHelper = getConflictingConfigFoundDict()[ "labelConflict" ]
      for v6NodeSidLabel in v6NodeSidLabels:
         if labelConflictHelper( instanceName, intfName, None, v6NodeSidLabel,
                                 ipv6=True, mode=mode ):
            mode.addError( "IPv6 Node-segment label conflicts with an existing"
                           " segment label in the instance %s" % instanceName )
            return True
   
   return False 

def setIsisEnable( mode, args ):
   instanceName = args[ 'INSTANCE_NAME' ]
   if 'RIB_NAME' in args:
      ribFilters = args[ 'RIB_NAME' ]
   else:
      ribFilters = []
   if mode.session_.isInteractive():
      if instanceName not in isisConfig.instanceConfig:
         mode.addWarning( "IS-IS instance %s doesn't exist." % instanceName )
      else:
         instanceConfig = isisConfig.instanceConfig[ instanceName ]
         if instanceConfig.multiTopology and isisConfig.intfConfig.get( 
            mode.intf.name ):
            intfMtAddrFamily = isisConfig.intfConfig[
               mode.intf.name ].mtAddressFamily
            instanceAddrFamily = instanceConfig.addressFamily
            if intfMtAddrFamily == IsisAddressFamily.addressFamilyNone:
               intfMtAddrFamily = instanceAddrFamily

            if instanceAddrFamily == IsisAddressFamily.addressFamilyIPv6 and ( 
               # pylint: disable-next=consider-using-in
               intfMtAddrFamily == IsisAddressFamily.addressFamilyIPv4 
               or intfMtAddrFamily == IsisAddressFamily.addressFamilyBoth ):
               mode.addWarning( "Interface multi topology address family not"
                                " configured in router IS-IS mode" )
            else:
               addIpInterfaceWarnings( mode, intfMtAddrFamily )
         else:
            instanceAddrFamily = instanceConfig.addressFamily
            addIpInterfaceWarnings( mode, instanceAddrFamily )
   if not checkNodePrefixConflict( mode, instanceName, mode.intf.name ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      # Determine if this is multiple instance or just a single instance
      if 'instance' in args:
         intfConfig.multiInstanceName.add( instanceName )
      else:
         prevInstName = intfConfig.instanceName
         intfConfig.instanceName = instanceName
         if IpRibLibToggleLib.toggleIsisConditionalAdvertiseEnabled():
            # Clear the ribFilterName list for the previous instance.
            # Or if instName is unchanged, clear the list if there are no
            # filteredRibs in the new config
            if ( prevInstName in intfConfig.instanceRibFilter and
                 ( prevInstName != instanceName or len( ribFilters ) == 0 ) ):
               intfConfig.instanceRibFilter[ prevInstName ].ribFilterName.clear()
               del intfConfig.instanceRibFilter[ prevInstName ]

            if len( ribFilters ) > 0:
               if instanceName not in intfConfig.instanceRibFilter:
                  intfConfig.instanceRibFilter.newMember( instanceName )
               # Remove ribFilterNames that are not in the updated config
               prevRibFilters =\
                  intfConfig.instanceRibFilter[ instanceName ].ribFilterName
               setDiff = set( prevRibFilters ) - set( ribFilters )
               for ribFilterName in setDiff:
                  del prevRibFilters[ ribFilterName ]

               for ribFilterName in ribFilters:
                  intfConfig.instanceRibFilter[ instanceName ].ribFilterName.add(
                     ribFilterName )
      updateAllInstanceColl( intfConfig )

def noIsisEnable( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   if 'instance' in args:
      instanceName = args[ 'INSTANCE_NAME' ]
      del intfConfig.multiInstanceName[ instanceName ]
   else:
      instanceName = intfConfig.instanceName
      intfConfig.instanceName = intfConfig.instanceNameInvalid
   updateAllInstanceColl( intfConfig )
   if ( IpRibLibToggleLib.toggleIsisConditionalAdvertiseEnabled()
        and instanceName in intfConfig.instanceRibFilter ):
      # Clear the instanceRibFilter before deleting so it triggers the reactor
      # to clear the corresponding entry from isisConfigSm::intfListByRibFilter.
      intfConfig.instanceRibFilter[ instanceName ].ribFilterName.clear()
      del intfConfig.instanceRibFilter[ instanceName ]
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def updateAllInstanceColl( intfConfig ):
   nameUnion = set( intfConfig.multiInstanceName )
   if intfConfig.instanceName:
      nameUnion.add( intfConfig.instanceName )
   addNames = nameUnion - set( intfConfig.allInstanceName )
   delNames = set( intfConfig.allInstanceName ) - nameUnion
   for name in addNames:
      intfConfig.allInstanceName.add( name )
   for name in delNames:
      del intfConfig.allInstanceName[ name ]

#-------------------------------------------------------------------------------
# "[no|default] isis rfc8202 disabled 
# command, in config-if mode.
#-------------------------------------------------------------------------------





def setIsisRfc8202Disabled( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.multiInstanceRfc8202 = False

def noIsisRfc8202Disabled( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.multiInstanceRfc8202 = intfConfig.multiInstanceRfc8202Default
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
# "[no|default] isis metric <value> | <maximum>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfMetric( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   confMetric = args.get( 'METRIC_VALUE', Metric.isisMetricMax )
   intfConfig.confMetric = TristateU32.valueSet( confMetric )
   #This will be removed once OpenConfig BUG572339 is addressed
   intfConfig.metric = confMetric

def noIsisIntfMetric( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.confMetric = TristateU32.valueInvalid()
   #This will be removed once OpenConfig BUG572339 is addressed
   intfConfig.metric = intfConfig.metricNotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )


#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 metric <value> | <maximum>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

def setIsisIntfV6Metric( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   confMetric = args.get( 'METRIC_VALUE', Metric.isisMetricMax )
   intfConfig.confMetricV6 = TristateU32.valueSet( confMetric )
   if mode.session_.isInteractive():
      instanceName = intfConfig.instanceName
      if instanceName in isisConfig.instanceConfig:
         instance = isisConfig.instanceConfig[ instanceName ]
         if not instance.multiTopology:
            mode.addWarning( mtWarningMsg )
   #This will be removed once OpenConfig BUG572339 is addressed
   intfConfig.metricV6 = confMetric

def noIsisIntfV6Metric( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.confMetricV6 = TristateU32.valueInvalid()
   #This will be removed once OpenConfig BUG572339 is addressed
   intfConfig.metricV6 = intfConfig.metricV6NotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
# "[no|default] isis metric profile <profileName>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def getAllMetricProfiles( mode ):
   return sum( ( list( cfg.metricProfiles ) for cfg in
                              isisConfig.instanceConfig.values() ), [] )

def setMetricProfile( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.metricProfile = args.get( 'PROFILE_NAME', '' )

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 metric profile <profileName>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setMetricProfileV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.metricProfileV6 = args.get( 'PROFILE_NAME', '' )

#-------------------------------------------------------------------------------
# "[no|default] isis multi-topology address-family ( ipv4 | ipv6 ) [ unicast ]" 
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfMtAf( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   addrFamily = args[ 'AF' ]
   intfConfig.mtAddressFamily = addAddrFamilyToConfig( 
         intfConfig.mtAddressFamily, addrFamily )

   if mode.session_.isInteractive():
      instanceName = intfConfig.instanceName
      if instanceName in isisConfig.instanceConfig:
         instance = isisConfig.instanceConfig[ instanceName ]
         if not instance.multiTopology:
            mode.addWarning( mtWarningMsg )
         elif instance.addressFamily != IsisAddressFamily.addressFamilyBoth:
            if addrFamily == "ipv4":
               mode.addWarning( mtV4WarningMsg )
         else:
            addIpInterfaceWarnings( mode, addrFamilyStrToEnum( addrFamily ))

def noIsisIntfMtAf( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return 
   intfConfig.mtAddressFamily = delAddrFamilyFromConfig( 
         intfConfig.mtAddressFamily, args[ 'AF' ] )
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
# "[no|default] isis circuit-type LEVEL"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfCircuitType( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.circuitType = LVL_KW_TO_TAC_TYPE_MAP[ args[ 'LEVEL' ] ]

def noIsisIntfCircuitType( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.circuitType = intfConfig.circuitTypeDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
# "[no|default] isis area proxy boundary"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfAreaProxyBoundary( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.areaProxyBoundary = True

def noIntfAreaProxyBoundary( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if not intfConfig:
      return
   intfConfig.areaProxyBoundary = intfConfig.areaProxyBoundaryDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
# "[no|default] isis priority"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfPriority( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.priority = args[ 'PRIORITY_VALUE' ]

def noIsisIntfPriority( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.priority = intfConfig.priorityDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )


#-------------------------------------------------------------------------------
# "[no|default] isis hello-interval <interval>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisHelloInterval( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.helloInterval = args[ 'INTERVAL' ]

def noIsisHelloInterval( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.helloInterval = intfConfig.helloIntervalDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )


#-------------------------------------------------------------------------------
# "[no|default] isis hello-multiplier <multiplier>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisHelloMultiplier( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.helloMultiplier = args[ 'MULTIPLIER' ]

def noIsisHelloMultiplier( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.helloMultiplier = intfConfig.helloMultiplierDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
#"[no|default] isis passive
# command in config-if mode
#-------------------------------------------------------------------------------
def setIntfPassive( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.passive = True 

def noIntfPassive( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return 
   intfConfig.passive = intfConfig.passiveDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#-------------------------------------------------------------------------------
# "[no|default] isis network point-to-point"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfType( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.interfaceType = InterfaceType.interfaceTypeP2P

def noIsisIntfType( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return 
  
   intfConfig.interfaceType = intfConfig.interfaceTypeDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

# Helper function to check that all prefixes in a proxy node range
# are valid
def _isisSrValidProxyNodePrefix( startPrefix, rangeVal ):
   try:
      if startPrefix.af == 'ipv4':
         ipv4Pfx = ipaddress.IPv4Address( startPrefix.v4Addr )
         lastv4Pfx = int( ipv4Pfx ) + rangeVal - 1
         ipaddress.IPv4Address( lastv4Pfx )
      else:
         ipv6Pfx = ipaddress.IPv6Address( startPrefix.v6Addr )
         lastv6Pfx = int( ipv6Pfx ) + rangeVal - 1
         ipaddress.IPv6Address( lastv6Pfx )
   except ipaddress.AddressValueError:
      return False
   return True

#-------------------------------------------------------------------------------
# "[no|default] isis bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfBfdV4( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.bfdV4 = IsisIntfBfdConfig.isisIntfBfdEnabled

def noIntfBfdV4( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   if CliCommand.isDefaultCmd( args ):
      intfConfig.bfdV4 = intfConfig.bfdV4Default
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig,
                                                   mode.intf.name )
   else:
      intfConfig.bfdV4 = IsisIntfBfdConfig.isisIntfBfdDisabled

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfBfdV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.bfdV6 = IsisIntfBfdConfig.isisIntfBfdEnabled

def noIntfBfdV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   if CliCommand.isDefaultCmd( args ):
      intfConfig.bfdV6 = intfConfig.bfdV6Default
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig,
                                                   mode.intf.name )
   else:
      intfConfig.bfdV6 = IsisIntfBfdConfig.isisIntfBfdDisabled


def checkV4NodeSegmentConflict( intfName, instanceName, index ):
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )

   if ipIntfConfig is not None:
      prefix = Arnet.IpGenPrefix( str( ipIntfConfig.addrWithMask.subnet ) )
      prefixConflictHelper = getConflictingConfigFoundDict()[ "prefixConflict" ]
      if prefixConflictHelper( instanceName, intfName,
                               prefix, mode=None ):
         return ( False,
                  "Node-segment conflicts with a prefix/proxy segment" \
                  " configured for the prefix %s" % prefix )
   sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
   if sidConflictHelper( instanceName, intfName, None, index ):
      return ( False,
               "Two prefixes cannot be configured with the same SID" )

   return ( True, None )

def checkV6NodeSegmentConflict( intfName, instanceName, index ):
   ip6IntfConfig = ip6Config.intf.get( intfName )
   if ip6IntfConfig:
      for prefix in ip6IntfConfig.addr:
         prefix = Arnet.IpGenPrefix( str( prefix ) )
         prefixConflictHelper = getConflictingConfigFoundDict()[ "prefixConflict" ]
         if prefixConflictHelper( instanceName, intfName,
                                  prefix, mode=None ):
            return ( False,
                     "Node-segment conflicts with a prefix/proxy segment" \
                     " configured for the prefix %s" % prefix )
   sidConflictHelper = getConflictingConfigFoundDict()[ "sidConflict" ]
   if sidConflictHelper( instanceName, intfName, None, index ):
      return ( False,
               "Two prefixes cannot be configured with the same SID" )

   return ( True, None )

#-------------------------------------------------------------------------------
# "[no|default] isis hello padding"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfHelloPadding( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   if args.get( 'disabled' ):
      intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingAlwaysOff
   elif CliCommand.isNoCmd( args ):
      intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingOff
   else:
      intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingOn

def defaultIsisIntfHelloPadding( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#---------------------------------------------------------------------------------
# [no|default] adjacency-segment AF p2p [multiple] 
# {<label <val>| index <val> global>} [backup-eligible]
#---------------------------------------------------------------------------------
def getAdjSidKey( addrFamily, labelType, value ):
   sidVal = value
   isIpV6 = addrFamily == "ipv6"
   isValue = labelType == 'label'

   sid = SegmentIdentifier( sidVal, isValue )
   key = AdjacencySegmentKey( sid, isIpV6 )
   return key

def setStaticAdjSid( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   addrFamily = args[ 'AF' ]
   labelType = args.get( 'label', 'index' )
   value = args.get( 'LABEL_VALUE' ) or args.get( 'INDEX_VALUE' )
   backup = args.get( 'backup-eligible' )
   key = getAdjSidKey( addrFamily, labelType, value )

   if 'multiple' not in args:
      if key in  intfConfig.srSingleAdjacencySegment:
         sidEntry = intfConfig.srSingleAdjacencySegment[ key ]
      else:
         sidEntry = intfConfig.newSrSingleAdjacencySegment( key )
      sidEntry.backup = bool( backup )

      isIpV6 = addrFamily == "ipv6"
      # delete old adj-SID (if any)
      for adjSidKey in intfConfig.srSingleAdjacencySegment:
         if adjSidKey != key and adjSidKey.isIpV6 == isIpV6:
            del intfConfig.srSingleAdjacencySegment[ adjSidKey ]
   else:
      if key in  intfConfig.srMultipleAdjacencySegment:
         sidEntry = intfConfig.srMultipleAdjacencySegment[ key ]
      else:
         sidEntry = intfConfig.newSrMultipleAdjacencySegment( key )
      sidEntry.backup = bool( backup )

def noStaticAdjSid( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   addrFamily = args[ 'AF' ]
   labelType = args.get( 'label', 'index' )
   if 'multiple' not in args:
      isIpV6 = addrFamily == "ipv6"
      isValue = labelType == "label"
      for adjSidKey in intfConfig.srSingleAdjacencySegment:
         if adjSidKey.isIpV6 == isIpV6 and adjSidKey.sid.isValue == isValue:
            del intfConfig.srSingleAdjacencySegment[ adjSidKey ]
   else:
      value = args.get( 'LABEL_VALUE' ) or args.get( 'INDEX_VALUE' )
      key = getAdjSidKey( addrFamily, labelType, value )
      del intfConfig.srMultipleAdjacencySegment[ key ]

   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

#------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] route-tag <tag-no>   XXX: Hidden
#------------------------------------------------------------------------------

def setIsisIntfRouteTagV4( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfRouteTagVal = args[ 'TAG_VALUE' ]
   #Setting a route tag to 0 removes the tag
   if intfRouteTagVal == RouteTag.isisRouteTagMin:
      noIsisIntfRouteTagV4( mode, args )
      return
   intfConfig.routeTagV4 = intfRouteTagVal

def noIsisIntfRouteTagV4( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.routeTagV4 = intfConfig.routeTagNotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def setIsisIntfRouteTagV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfRouteTagVal = args[ 'TAG_VALUE' ]
   #Setting a route tag to 0 removes the tag
   if intfRouteTagVal == RouteTag.isisRouteTagMin:
      noIsisIntfRouteTagV6( mode, args )
      return
   intfConfig.routeTagV6 = intfRouteTagVal

def noIsisIntfRouteTagV6( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.routeTagV6 = intfConfig.routeTagNotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )


#-------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] fast-reroute ti-lfa mode ( {link-protection |
# node-protection} [ LEVEL ] | disabled )
#-------------------------------------------------------------------------------






#-------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] fast-reroute ti-lfa srlg [strict] [disabled]
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# New Cli Parser
#-------------------------------------------------------------------------------
def isisShowCommandSyntaxWrapper( cmd ):
   '''
   IS-IS show command CLIs can be of two forms:
   show isis [ INSTANCE_NAME ] COMMAND [ ARGS ]
   or
   show isis COMMAND [ ARGS ] [ vrf VRF_NAME ]

   If instance name is specified, we show the information for that instance.
   We do not allow both instance name and vrf name to be specified.
   '''
   template = 'show isis ( %s ) | ( INSTANCE %s ) | ( %s vrf VRFNAME )'
   return template % ( cmd, cmd, cmd )

def isisClearCommandSytaxWrapper( cmd ):
   '''
   IS-IS clear command CLIs can be of two forms:
   clear isis [ INSTANCE_NAME ] COMMAND [ ARGS ]
   or
   clear isis COMMAND [ ARGS ] [ vrf VRF_NAME ]

   If instance name is specified, we clear the information for that instance.
   We do not allow both instance name and vrf name to be specified.
   '''
   template = "clear isis ( [ INSTANCE | ALL_INSTANCE ] %s ) | ( %s vrf VRFNAME )"
   return template % ( cmd, cmd )

def showIsisCommand( *args, **kwargs ):
   '''
   args - Contains CLI mode and callback function
   kwargs - Contains the cmdVrfModel and instDictModelType for the Capi compatible
          commands.

   instDictModelType - Instance models for CAPI compatible commands
   cmdVrfModel - cmd VRF model for CAPI compatible commands

   If vrf name is specified show the information for the Isis instance in that VRF.
   If it doesn't exist report error.
   We do not allow both instance name and vrf name to be specified
   If neither instance name nor vrf name are specified use the VRF name from the
   routing context if it is set. If routing context is not set and there is an
   Isis instance configured in the default VRF, then show information for that
   instance. If there is no Isis instance in the default VRF but there is/are Isis
   instance(s) in non-default VRF(s), show information for one of those instances.
   Sort them by instance name and show the status for first one.
   Otherwise report error.
   '''

   mode = args[ 0 ]

   # callback will be the last entry in *args
   callback = args[ -1 ]
   args = args[ : -1 ]

   # RibCapiLib will not accept these extra arguments - Need to remove them
   for keyword in ( 'show', 'isis', 'vrf' ):
      kwargs.pop( keyword, None )

   # Some callback functions take 'level' as keyword argument
   # and not as 'level-1' or 'level-2'
   level = kwargs.pop( 'LEVEL', None )
   if level is not None:
      kwargs[ 'level' ] = level

   instDictModelType = kwargs.pop( 'instDictModelType', None )
   cmdVrfModel = kwargs.pop( 'cmdVrfModel', None )
   instanceName = kwargs.pop( 'INSTANCE', None )
   vrfName = kwargs.pop( 'VRFNAME', None )

   useCliPrint = kwargs.pop( 'useCliPrint', False )
   if useCliPrint:
      showVrfModelRoutine = isisShowVrfModelUsingCliPrint
   else:
      showVrfModelRoutine = isisShowVrfModel

   sortedInstanceList = [ isisConfig.instanceConfig[ iName ] for iName in 
                         sorted( isisConfig.instanceConfig ) ]

   vrfs = { inst.vrfName for inst in  sortedInstanceList }

   instanceList = sortedInstanceList
   vrfSet = vrfs

   if instanceName:
      instance = isisConfig.instanceConfig[ instanceName ]
      instanceList = [ instance ]
      vrfSet = { instance.vrfName }
   elif vrfName:
      # vrf name is specified
      if vrfName != ALL_VRF_NAME:
         if vrfName not in vrfs:
            mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
            return
         else:
            vrfSet = { vrfName }
   else:
      # If vrf name and instance name isn't specified
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName not in vrfs:
            mode.addError( "No IS-IS instance is configured in routing-context "
                           "VRF: %s" % vrfName )
            return
         vrfSet = { vrfName }
      elif DEFAULT_VRF in vrfs:
         vrfSet = { DEFAULT_VRF }
      else:
         if not isisConfig.instanceConfig:
            return cmdVrfModel() if cmdVrfModel else None
         vrfSet = { sortedInstanceList[0].vrfName }
   # Some vrf name is always set.

   if cmdVrfModel:
      return showVrfModelRoutine( vrfSet, mode, cmdVrfModel, instDictModelType,
                                  callback, instanceList, *args, **kwargs)
   else:
      for instance in instanceList:
         if instance.vrfName in vrfSet:
            invokeCallback( instance, callback, *args, **kwargs )

def getIsisConfigAndStatus( mode, instanceId, instanceName ):
   if not instanceName:
      mode.addError( "Invalid IS-IS instance name" )
      return None, None
   
   instConfig = isisConfig.instanceConfig[ instanceName ]
   
   if not instConfig:
      mode.addError( "IS-IS (%s) Can not find configuration" % instanceName )
      return None, None
  
   isisStatus = isisStatusDir.get( instConfig.vrfName )
   instStatus = isisStatus.instanceStatus[ instanceName ]
   if not instStatus:
      mode.addError( "IS-IS (%s) Can not find status data" % instanceName )
      return None, None
   
   if instConfig.shutdown:
      return None, None
   
   return instConfig, instStatus

#-------------------------------------------------------------------------------
# Show commands
#-------------------------------------------------------------------------------
def invokeCallback( instance, callback, *args, **kwargs ):
   cmdPrefix = 'show isis'
   cmd = cmdPrefix
   mode = args[0]

   isisStatus = isisStatusDir.get( instance.vrfName )
   warnings, statusValid = checkAndShowIsisConfigMismatch( mode, instance,
                                                           isisStatus )
   if warnings:
      for warning in warnings:
         mode.addWarning( warning )
   if statusValid:
      # statusValid is True only if isisStatus is not None and
      # isisStatus.instanceStatus[instance.instanceName] is not None
      instanceId = isisStatus.instanceStatus[
            instance.instanceName ].instanceId
      # The actual show command handlers don't need/accept vrfName as
      # keyword arguments.
      if 'vrfName' in kwargs:
         del kwargs[ 'vrfName' ]
      kwargs[ 'cmdPrefix' ] = cmd
      kwargs[ 'instanceId' ] = instanceId
      kwargs[ 'instanceName' ] = instance.instanceName

      return callback( *args, **kwargs )

def isisShowVrfModelUsingCliPrint( vrfSet, mode, cmdVrfModel, instDictModelType,
                                   callback, instanceList, *args, **kwargs ):
   vrfInstDict = {}
   # Loop for prepopulating warnings for all instances
   for vrf in vrfSet:
      vrfInstDict[ vrf ] = []
   for instance in instanceList:
      isisStatus = isisStatusDir.get( instance.vrfName )
      if instance.vrfName in vrfSet:
         warnings, statusValid = checkAndShowIsisConfigMismatch( mode,
                                                       instance, isisStatus )
         if warnings:
            for warning in warnings:
               mode.addWarning( warning )
         vrfInstDict[ instance.vrfName ] += \
                                    [ { 'instanceName' : instance.instanceName,
                                        'statusValid' :statusValid } ]

   fd = sys.stdout.fileno()
   outputFormat = cprinter.stringToOutputFormat( mode.session_.outputFormat_ )
   p = cprinter.initPrinter( fd, outputFormat, True )
   cprinter.startRender( p )
   cprinter.startDict( p, "vrfs" )
   for vrf in vrfSet:
      cprinter.startDict( p, vrf )
      isisStatus = isisStatusDir.get( vrf )
      if isisStatus is None:
         return
      cprinter.startDict( p, "isisInstances" )
      for inst in vrfInstDict[ vrf ]:
         if inst[ 'statusValid' ]:
            cprinter.startDict( p, inst[ 'instanceName' ] )
            kwargs[ 'vrfName' ] = vrf
            kwargs[ 'instanceId' ] = isisStatus. \
                                     instanceStatus[ inst[ 'instanceName' ] ]. \
                                     instanceId
            kwargs[ 'instanceName' ] = inst[ 'instanceName' ]
            _ = callback( *args, **kwargs )
            cprinter.endDict( p ) # this instance
      cprinter.endDict( p ) # instances
      cprinter.endDict( p ) # this vrf
   cprinter.endDict( p ) # vrfs
   cprinter.endRender( p )
   cprinter.deinitPrinter( p )
   return cliPrinted( cmdVrfModel )

def isisShowVrfModel( vrfSet, mode, cmdVrfModel, instDictModelType, callback,
                      instanceList, *args, **kwargs ):
   def instDictFunc( vrf ):
      isisStatus = isisStatusDir.get( vrf )
      if isisStatus is None:
         return
      for inst in vrfInstDict[ vrf ]:
         instModel = None
         if inst[ 'statusValid' ]:
            kwargs[ 'vrfName' ] = vrf
            kwargs[ 'instanceId' ] = isisStatus. \
                                   instanceStatus[inst[ 'instanceName' ]]. \
                                   instanceId
            kwargs[ 'instanceName' ] = inst[ 'instanceName' ]

            # handling EmptyResponseException raised from RibCapiLib
            try:
               instModel = callback( *args, **kwargs )
            except EmptyResponseException:
               instModel = None
            except AmiResponseException:
               instModel = None
            
         if instModel is None:
            continue
         yield inst[ 'instanceName' ], instModel 
   def vrfListFunc( vrfs ):
      for vrf in vrfs:
         instDictModel = instDictModelType()
         instDictModel.isisInstances = instDictFunc( vrf )
         yield vrf, instDictModel

   vrfInstDict = {}
   #Loop for prepopulating warnings for all instances
   for vrf in vrfSet:
      vrfInstDict[ vrf ] = []

   for instance in instanceList:
      isisStatus = isisStatusDir.get( instance.vrfName )
      if instance.vrfName in vrfSet:
         warnings, statusValid = checkAndShowIsisConfigMismatch( mode,
                                                       instance, isisStatus )
         if warnings:
            for warning in warnings:
               mode.addWarning( warning )
         vrfInstDict[ instance.vrfName ] += \
                                       [ { 'instanceName' : instance.instanceName,
                                           'statusValid' :statusValid } ]

   model = cmdVrfModel()
   model.vrfs = vrfListFunc( vrfSet )
   return model

def _isisConfigVrfIsNotActiveReason( config ):
   return "Vrf %s is not active" % config.vrfName

def _isisConfigInvalidReason( config, status, vrfStatus, vrf6Status ):
   msgs = []
   if not status.netValid:
      msgs.append( "IS-IS Network Entity Title (NET) configuration is not present" )

   v4RoutingStatus = vrfStatus and vrfStatus.routing
   v6RoutingStatus = vrf6Status and vrf6Status.routing
   vrfMessage = ''
   if config.vrfName != DEFAULT_VRF:
      vrfMessage = ' in vrf %s' % config.vrfName

   if status.activeAddressFamily == IsisAddressFamily.addressFamilyNone:
      if config.addressFamily == IsisAddressFamily.addressFamilyNone:
         msgs.append( "IS-IS address family configuration is not present" )
      if config.addressFamily == IsisAddressFamily.addressFamilyIPv4 \
             and not v4RoutingStatus:
         msgs.append( "IPv4 unicast routing is not enabled%s" % vrfMessage )
      elif config.addressFamily == IsisAddressFamily.addressFamilyIPv6 \
             and not v6RoutingStatus:
         msgs.append( "IPv6 unicast routing is not enabled%s" % vrfMessage )
      elif config.addressFamily == IsisAddressFamily.addressFamilyBoth \
             and not v4RoutingStatus and not v6RoutingStatus:
         msgs.append( "Routing is not enabled%s" % vrfMessage )
   return msgs

def _isisBothAfConfigMismatchReason( config, status, vrfStatus, vrf6Status ):
   msg = None
   v4RoutingStatus = vrfStatus and vrfStatus.routing
   v6RoutingStatus = vrf6Status and vrf6Status.routing
   if not v4RoutingStatus:
      msg = 'IS-IS IPv4 and IPv6 address family are configured but IPv4 unicast '
      msg += 'routing is not enabled for instance %s' % status.instanceName 
   if not v6RoutingStatus:
      msg = "IS-IS IPv4 and IPv6 address family are configured but IPv6 unicast "
      msg += "routing is not enabled for instance %s" % status.instanceName
   return msg

def checkAndShowIsisConfigMismatch( mode, config, isisStatus ):
   '''
   Wrapper function that needs to be used to invoke a show command handler, to show
   reason for ISIS to be disabled / not running as configured due to a config
   mismatch.
   If the config is valid it calls the show command handler function.
   '''
   # When this function is called, an instance exists. The status, however, may
   # or may not exist. The status doesn't exist in the case of the isis instance
   # living in a non-default VRF which was either not created or was deleted.
   # If the status is None, we let users know about it printing
   # Vrf <vrfName> is not active
   instanceName = config.instanceName
   warnings = []
   configValid = None

   if isisStatus is not None and instanceName in isisStatus.instanceStatus:
      status = isisStatus.instanceStatus[ instanceName ]
      configValid = status.configValid
      vrfStatus = vrfInfoDir.get( config.vrfName )
      vrf6Status = vrf6InfoDir.get( config.vrfName )
      if not configValid:
         reasons = _isisConfigInvalidReason( config, status, vrfStatus, vrf6Status )
         for r in reasons:
            mode.addError( "IS-IS (%s) is disabled because: %s"
                           % ( instanceName, r ) )
      else:
         if config.addressFamily == IsisAddressFamily.addressFamilyBoth:
            reason = _isisBothAfConfigMismatchReason( config, status, vrfStatus,
                                                      vrf6Status )
            if reason is not None:
               warnings.append( "IS-IS ({}) has incorrect configuration: {}".format(
                  instanceName, reason ) )

   elif isisStatus is None and config.vrfName != DEFAULT_VRF:
      configValid = False
      reason = _isisConfigVrfIsNotActiveReason( config )
      mode.addError( "IS-IS (%s) is disabled because: %s"
                     % ( instanceName, reason ) )

   if config.shutdown:
      warnings.append( "IS-IS (%s) is shutdown" % instanceName )

   return warnings, configValid

# This function allows us to reverse-lookup from the LVL_KW_TO_TAC_TYPE dict.
getLevel = partial( LVL_KW_TO_TAC_TYPE_MAP.reverse().__getitem__ )

#-------------------------------------------------------------------------------
# show isis interface [interface] [detail]
#-------------------------------------------------------------------------------
def showIsisIntfWithName( mode, intf, detail=None, brief=None ):
   callback = showIsisIntf
   instance = None
   # Find the instance Id of isis running on this interface
   intfConfig = _getIntfConfig( intf.name )
   if ( intfConfig is None or not intfConfig.instanceName ):
      mode.addError( "No IS-IS instance is enabled on this interface" )
   elif intfConfig.instanceName in isisConfig.instanceConfig:
      instance = isisConfig.instanceConfig[ intfConfig.instanceName ]
   else:
      mode.addError( "IS-IS instance %s of this interface is not configured"
                     % intfConfig.instanceName )

   if instance is not None:
      args = [ mode ]
      kwargs = { "intf" : intf, "detail" : detail, "brief" : brief }
      return isisShowVrfModel( [ instance.vrfName ], mode, 
            IsisCliModels.interfaceVRFsModel,
            IsisCliModels.interfaceModel, 
            callback, [ instance ],  *args, **kwargs )

def showIsisIntf( mode, instanceId=None, instanceName=None, intf=None,
                  vrfName=None, detail=None, brief=None, cmdPrefix=None ):

   def genDisabledIntfs():
      for interface in isisConfig.intfConfig:
         if intf is not None and intf.name != interface: 
            continue
         intfConfig = isisConfig.intfConfig.get( interface )
         if instanceName != intfConfig.instanceName:
            continue
         instanceKey = InstanceKey( interface, instanceName )
         intfStatus = isisStatus.intfStatus.get( instanceKey )
         if intfStatus is None or not intfStatus.instanceValid:
            continue
         instance = isisConfig.instanceConfig[ instanceName ]
         if not intfStatus.configValid:
            interfaceInst = IsisCliModels.IsisInterface()
            interfaceInst._interfaceName = interface
            interfaceInst.enabled = False
            disabledReason = IsisCliModels.DisabledReason()
            disabledReason.circuitType = getLevel( intfConfig.circuitType )
            disabledReason.routerCircType = getLevel( instance.level )
            # configValidCheck is to check that the config is considered invalid
            # only because of fitleredRibSatisified and not other factors
            if ( not intfStatus.filteredRibSatisfied and
                 intfStatus.configValidWithoutCond ):
               disabledReason.message = "Disabled due to empty unicast-rib(s)"
            elif intfStatus.activeCircuitType == Level.levelNone:
               disabledReason.message = (
                  f"Disabled because circuit type {disabledReason.circuitType} "
                  "doesn't match with router IS type "
                  f"{disabledReason.routerCircType}" )
            else:
               # if not the above two cases, skip to next loop iteration to avoid
               # returning an unexpected interface or interfaceInst
               continue
            interfaceInst.disabledReason = disabledReason
            interfaceInst.intfLevels = None
            yield interface, interfaceInst

   def updateIntfDataWithSysDb( interfaces ):
      for intfName, intf in interfaces:
         instanceKey = InstanceKey( intfName, instanceName )
         intfStatus = isisStatus.intfStatus.get( instanceKey )
         if not intfStatus:
            continue
         intf.fetchExtraDataFromSysDb( intfStatus )
         yield intfName, intf
         
   command = 'MIO_DGET_ISIS_CIRCUIT_SUMMARY' 
   if detail:
      command = 'MIO_DGET_ISIS_CIRCUIT_DETAIL'
   cliModel = IsisCliModels.IsisInterfaceModel
   isisStatus = isisStatusDir.get( vrfName )
   args = { 'instance': instanceId }
   args[ 'brief' ] = bool( brief )
   if intf is not None:
      #Verify the interface is a valid interface name
      if not intf.lookup():
         return
      name = intf.shortname
      if name != "":
         args[ 'circuit' ] = name

   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if isisStatus is None:
      return
   
   instModel =  showRibCapiCommand( mode, cliModel, command, args, 
         clientName="ISIS", l3Config=l3Config)

   interfaces = updateIntfDataWithSysDb(
           itertools.chain( instModel.interfaces, genDisabledIntfs() ) )
   
   instModel.interfaces = interfaces
   if brief is not None:
      instModel._brief = True   
   return instModel

#-------------------------------------------------------------------------------
# show isis summary
#-------------------------------------------------------------------------------
def showIsisSummary( mode, instanceId=None, instanceName=None, vrfName=None ):
   command = 'MIO_DGET_ISIS_INSTANCE_SUMMARY'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   result = showRibCapiCommand( mode, IsisCliModels.IsisInstanceSummaryModel,
                                command, args, clientName="ISIS", l3Config=l3Config )
   
   if result._duplicateSysIdWarning is not None:
      mode.addWarning( "Duplicate system ID %s detected %s"\
                       % ( result.systemId, result._duplicateSysIdWarning ) )
   return result





#-------------------------------------------------------------------------------
# show isis summary-address [ detail ]
#-------------------------------------------------------------------------------
def showIsisLeakedRouteSumAddr( mode, instanceId=None, instanceName=None,
      vrfName=None, detail=None ):
   command = 'MIO_DGET_ISIS_LEAKED_ROUTE_SUMADDR'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   args[ 'detail' ] = bool( detail )

   try:
      result = showRibCapiCommand( mode,
            IsisCliModels.IsisLeakedRouteSumAddrInstanceModel, command, args,
            clientName='ISIS', l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result and detail:
      result._detailsPresent = True
   return result

#-------------------------------------------------------------------------------
# show isis database [ detail | traffic-engineering ] [ LEVEL ] [lspid]
#-------------------------------------------------------------------------------
def showIsisDatabase( mode, instanceId=None, instanceName=None, vrfName=None,
                      lspid=None, teOrDetail=None, level=None ):
   args = { 'instance': instanceId }
   if teOrDetail:
      command = 'MIO_DGET_ISIS_DATABASE_DETAIL'
   else:
      command = 'MIO_DGET_ISIS_DATABASE_SUMMARY'

   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   if lspid is not None:
      args[ 'lspid' ] = lspid
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if teOrDetail == "traffic-engineering":
      args[ 'te-detail' ] = True
   if teOrDetail:
      result = showRibCapiCommand( mode, IsisCliModels.IsisLsdbDetailModel, command,
                                args, clientName="ISIS", l3Config=l3Config )
   else:
      result = showRibCapiCommand( mode, IsisCliModels.IsisLsdbSummaryModel, command,
                                   args, clientName="ISIS", l3Config=l3Config )
   return result

# -------------------------------------------------------------------------------
# show isis database detail [ LEVEL ] [lspid] tlv
# -------------------------------------------------------------------------------
def showIsisDatabaseDetailTlv( mode, instanceId=None, instanceName=None,
                               vrfName=None, lspid=None, detail=None,
                               level=None, tlv=None ):
   args = { 'instance': instanceId }
   command = 'MIO_DGET_ISIS_DATABASE_DETAIL_TLV'
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   if lspid is not None:
      args[ 'lspid' ] = lspid
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if detail:
      result = showRibCapiCommand( mode, IsisCliModels.IsisLsdbDetailTlvModel,
                                   command, args, clientName="ISIS",
                                   l3Config=l3Config )
   return result


#------------------------------------------------
#  show isis lsp log
#------------------------------------------------
def showIsisLsplog( mode, instanceId=None,
                    instanceName=None , vrfName=None ):

   command = 'MIO_DGET_ISIS_LSP_LOG'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   result = showRibCapiCommand( mode, IsisCliModels.IsisLspLogTableModel, command,
                                args, clientName="ISIS", l3Config=l3Config )
   return result 

#-------------------------------------------------------------------------------
# show isis neighbors [detail] [ LEVEL ]
#-------------------------------------------------------------------------------
def showIsisNeighbors( mode, instanceId=None, instanceName=None, vrfName=None,
                       level=None, detail=None ):
   command = 'MIO_DGET_ISIS_NEIGHBORS'
   args = {}
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = lev

   result = showRibCapiCommand( mode, IsisCliModels.IsisInstanceNeighbors,
                                command, args, clientName='ISIS',
                                l3Config=l3Config )

   if result:
      #Set details attribute of IsisCliModels.IsisInstanceNeighbors
      if detail:
         result._detailsPresent = True
   return result

#-------------------------------------------------------------------------------
# show isis network topology
#
# legacy:
# show isis topology
#-------------------------------------------------------------------------------
def showIsisTopology( mode, instanceId=None, instanceName=None, vrfName=None,
                      level=None, ldpTunneling=None ):
   command = 'MIO_DGET_ISIS_TOPOLOGY'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = int( lev )
   if ldpTunneling:
      args[ 'ldp-shortcut' ] = True
   return showRibCapiCommand( mode, IsisCliModels.IsisTopologyModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis spf log [ LEVEL ]
#-------------------------------------------------------------------------------
def showIsisSpflog( mode, instanceId=None, vrfName=None, instanceName=None,
                    level=None, tunneling=None, ldp=None ):
   command = 'MIO_DGET_ISIS_SPF_LOG'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = int( lev )

   if tunneling and ldp:
      args[ 'ldp-shortcut' ] = True 
   return showRibCapiCommand( mode, IsisCliModels.IsisSpfLogTableModel, command,
                              args, clientName="ISIS", l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis lsp purges [ LEVEL ]
#-------------------------------------------------------------------------------
def showIsisLspPurges( mode, instanceId=None, vrfName=None, instanceName=None,
      level=None ):
   command = 'MIO_DGET_ISIS_LSP_PURGES'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = lev
   return showRibCapiCommand( mode, IsisCliModels.IsisLspPurgesModel, command,
                              args, clientName="ISIS", l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis graceful-restart
#-------------------------------------------------------------------------------
def showIsisGracefulRestart( mode, instanceId=None, vrfName=None, 
                             instanceName=None ):
   command = 'MIO_DGET_ISIS_GRACEFUL_RESTART'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   return showRibCapiCommand( mode, IsisCliModels.IsisGracefulRestartModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis local-convergence-delay [detail]
#-------------------------------------------------------------------------------
def showIsisUloopDelayInfo( mode, instanceId=None, vrfName=None, instanceName=None,
                            detail=None ):
   command = 'MIO_DGET_ISIS_ULOOP'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, IsisCliModels.IsisUloopModel, command,
                                   args, clientName='ISIS', l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result:
      if detail:
         result._printDetails = True
   return result

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] nodes [<node id>]
#-------------------------------------------------------------------------------
def showIsisDynFloodNode( mode, instanceId=None, instanceName=None, vrfName=None,
                          level=None, node=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_NODE'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   if node is not None:
      if re.search( trailingRegEx, node ) is None:
         node = node + '.00'
      args[ 'node' ] = node
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodNodeModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] paths
#-------------------------------------------------------------------------------

def showIsisDynFloodPaths( mode, instanceId=None, instanceName=None, vrfName=None,
                           level=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_PATH'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = int( lev )
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodPathsModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] topology
#-------------------------------------------------------------------------------

def showIsisDynFloodTopology( mode, instanceId=None, instanceName=None, vrfName=None,
                           level=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_TOPOLOGY'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodTopologyModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] interfaces
#-------------------------------------------------------------------------------

def showIsisDynFloodInterfaces( mode, instanceId=None, instanceName=None,
                                vrfName=None, level=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_INTERFACES'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodInterfacesModel,
                              command, args, clientName='ISIS', l3Config=l3Config )

#------------------------------------------------------------------------------
# Area proxy show commands
#
# show isis area proxy
#------------------------------------------------------------------------------

def showIsisAreaProxy( mode, instanceId=None, instanceName=None, vrfName=None,
                       cmdPrefix=None ):
   command = 'MIO_DGET_ISIS_AREA_PROXY_CMD'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      mode.addError( 'Feature only supported for the default VRF' )
      return
   return showRibCapiCommand( mode, IsisCliModels.IsisShowAreaProxyModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# Flex Algo Show commands
#
# show isis flex-algo
#    [ { ( algorithm NAME ) | ( system SYSTEM ) | LEVEL |
#          advertised | routers } ]
#
# Funkiness warning: the algorithm, system, and level can always be specified.
# Routers is both a qualifier and a stand-alone command.
# Ordering is not important.
# If an algorithm name is specified then we return details.
# If 'routers' is specified alone, then we issue that command, otherwise, we
# request a summary.
#-------------------------------------------------------------------------------

# Helper function used by the CLI models
def flexAlgoIdToName( algoId ):
   defin = flexAlgoConfig.definition.get( algoId )
   return defin.name if defin else str( algoId )

def showIsisFlexAlgo( mode, instanceId=None, instanceName=None, vrfName=None,
                      cmdPrefix=None, advertised=None, routers=None,
                      algo=None, system=None, level=None ):
   command = 'MIO_DGET_ISIS_FLEX_ALGO_CMD'
   subcommand = 0
   args = { 'instance' : instanceId }
   if system:
      args[ 'system' ] = system[ 0 ]
   if level:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   if advertised:
      args[ 'advertised' ] = True
   if algo:
      args[ 'algorithm' ] = algo
      subcommand = 1
   if routers:
      subcommand = 3
   args[ 'subcommand' ] = subcommand
   return showRibCapiCommand( mode, IsisCliModels.IsisShowFlexAlgoModel,
                              command, args, clientName='ISIS', l3Config=l3Config )

#-------------------------------------------------------------------------------
# Segment Routing Show Commands
#-------------------------------------------------------------------------------
def isisSrIsActive( mode, instanceName ): 
   instConfig = isisConfig.instanceConfig[ instanceName ]
   srSysdbStatus = srSysdbStatusDir.get( 'default' )
   if not srSysdbStatus:
      return False

   if instConfig.srEnabled == instConfig.srEnabledDefault:
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing has been " \
            "administratively shutdown" % instanceName )
      return False

   if mplsRoutingConfig.mplsRouting == False:
      mode.addWarning( "Mpls routing is not enabled" )
      return False

   if str( srSysdbStatus.igpRtrId ) == str( instConfig.routerIdV4Default ):
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing is disabled because " \
            "the Router ID is not configured" % instanceName )
      return False
   return True

def isisSrv6IsActive( mode, instanceName ): 
   instConfig = isisConfig.instanceConfig[ instanceName ]

   if instConfig.srv6Configured == instConfig.srv6ConfiguredDefault:
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing IPv6 " \
            "not configured" % instanceName )
      return False

   if instConfig.srv6Enabled != instConfig.srv6EnabledDefault:
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing IPv6 has been " \
            "administratively shutdown" % instanceName )
      return False

   return True

#helper function for the below used functions which returns hostnames
#from the isisSystemIdHostname map in srSysdbStatus whenever present 
def getHostName( rId, vrfName="default" ):
   smi = SmashLazyMount.mountInfo( 'reader' )
   path = 'routing/isis/hostnamemap/%s' % vrfName
   if vrfName == 'default':
      isisHostnameMap = isisSystemIdHostnameMap
   else:
      isisHostnameMap = SmashLazyMount.mount(
            entityManager, path, 'IsisExportImpl::IsisHostnameMap', smi,
            autoUnmount=True )
   hostname = None
   hostnameArray = isisHostnameMap.hostnameMap.get( rId )
   if hostnameArray:
      hostname = hostnameArray.smashString()
   return hostname

def getFlexAlgoName( flexAlgoId ):
   if flexAlgoId == 0:
      return IsisCliHelper.SR_ALGO_SPF
   if flexAlgoId == 1:
      return IsisCliHelper.SR_ALGO_SSPF
   if flexAlgoConfig and flexAlgoId:
      name = flexAlgoConfig.flexAlgoName( flexAlgoId )
      if name is not None:
         return name
   if flexAlgoId >= 128 and flexAlgoId <= 255: # pylint: disable=chained-comparison
      return IsisCliHelper.SR_ALGO_USER_DEFINED
   return IsisCliHelper.SR_ALGO_UNDEFINED

#-------------------------------------------------------------------------------
# show mpls segment-routing bindings
#-------------------------------------------------------------------------------
def showMplsSrBindings( mode, args ):
   if 'ipv4' in args or 'ipv6' in args:
      af = args.get( 'ipv4', 'ipv6' ) 
      prefix = None
   elif 'IPADDR' in args:
      prefix = args.get( 'IPADDR' ).stringValue
      af = None
   elif 'IP6ADDR' in args:
      prefix = args.get( 'IP6ADDR' ).stringValue
      af = None
   else:
      prefix = None
      af = None
   if not af:
      source = [ 'isisV4', 'isisV6' ]
   elif af == 'ipv4':
      source = [ 'isisV4' ]
   elif af == 'ipv6':
      source = [ 'isisV6' ]
   else:
      mode.addError( 'Invalid address-family' )
      return MplsModel.MplsBindingsModel( bindings={} )

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   helper = IsisMplsBindingsHelper( mplsStatus.force(), False,
                                    isisSystemIdHostnameMap )
   helper.flexAlgoConfig = flexAlgoConfig.force()
   genPrefix = IpGenPrefix()
   if prefix:
      if isinstance( prefix, str ):
         genPrefix.handleInitialized( prefix ) #ipv4 are obtained as str
      else:
         genPrefix.handleV6Prefix( prefix ) # ipv6 prefix are obtained as Ip6Prefix
   else:
      genPrefix.handleV4Prefix( nullPrefix )
   for src in source:
      fwdEqvClass = FwdEqvClass()
      fwdEqvClass.prefix = genPrefix
      if src == 'isisV4':
         bindingTable = isisSrV4BindingTable
      elif src == 'isisV6':
         bindingTable = isisSrV6BindingTable
      peerLbtColl = LabelBindingTableColl( "plbtc" )
      localLbt = LabelBindingTable( RouterId() )
      commonLibConsumerSm = CommonLibConsumerSm( bindingTable,
                                                 localLbt,
                                                 peerLbtColl,
                                                 True )
      helper.populateFecBindingsFromTablesFiltered(
         commonLibConsumerSm.localLbt,
         commonLibConsumerSm.peerLbtColl,
         'detail' in args,
         fwdEqvClass )
   helper.populateLocalConflicts()

   helper.render( fd, fmt )
   return cliPrinted( MplsModel.MplsBindingsModel )

#-------------------------------------------------------------------------------
# show isis mpls ldp sync [interface]
#-------------------------------------------------------------------------------
#
# ldpConfigColl is mounted at mpls/ldp/ldpConfigColl of Ldp::LdpConfigColl
#
# ldpProtoConfigColl is mounted at mpls/ldp/ldpProtoConfigColl of
# Ldp::LdpProtoConfigColl
#
# linkReadyStatusVrfColl is mounted at mpls/ldp/linkReadyStatus of
# Ldp::LdpLinkReadyStatusVrfColl
#
# instanceLevelConfig is instance-level mpls ldp sync config value (a boolean)
#
def showInterfaceMplsLdpSyncEntry( instanceLevelConfig, vrfName, intf ):
   ldpConfig = ldpConfigColl.config.get( vrfName )
   ldpProtoConfig = ldpProtoConfigColl.protoConfig.get( vrfName )
   readyStatusColl = linkReadyStatusVrfColl.readyStatusVrf.get( vrfName )

   def _syncRequired( intf ):
      if not readyStatusColl:
         return False # LDP not enabled
      if not ldpConfig:
         return instanceLevelConfig
      ldpIntfConfig = ldpConfig.intfConfigColl.get( intf )
      if not ldpIntfConfig:
         return instanceLevelConfig
      if ldpIntfConfig.igpSync == "useGlobal":
         return instanceLevelConfig
      return ldpIntfConfig.igpSync == "on"

   interfaceEntryModel = IsisMplsLdpSyncInterfaceModel()
   required = _syncRequired( intf )
   
   if required:
      readyStatus = readyStatusColl and  readyStatusColl.readyStatus.get( intf )
      interfaceEntryModel.syncAchieved = readyStatus and readyStatus.ready

   interfaceEntryModel.ldpEnabled = bool( readyStatusColl )
   interfaceEntryModel.syncRequired = required
   
   if ldpProtoConfig:
      interfaceEntryModel.igpNotifyDelay = int( ldpProtoConfig.linkReadyDelay )
      holdtime = ldpProtoConfig.linkReadyTimeout
      # For until-established ( holdtime value will be infinite, Tac.endOfTime ),
      # we will use 0 in CAPI model
      if holdtime == Tac.endOfTime:
         holdtime = 0
      interfaceEntryModel.holddownTime = int( holdtime )

   return interfaceEntryModel

def showInterfaceMplsLdpSync(  mode, instanceId=None, instanceName=None, 
                               vrfName=None, intf=None):
   mplsLdpSyncModel = IsisMplsLdpSyncModel() 
   for interface in isisConfig.intfConfig:
      if intf is not None and str( intf.name ) != str( interface ):
         continue
      if instanceName != isisConfig.intfConfig[ interface ].instanceName:
         continue
      instance = isisConfig.instanceConfig[ instanceName ]
      interfaceEntryModel = showInterfaceMplsLdpSyncEntry( instance.mplsLdpSync,
                                                      instance.vrfName,
                                                      interface )
      interfaceEntryModel._instanceName = instanceName
      mplsLdpSyncModel.interfaces[ interface ] = interfaceEntryModel
      mplsLdpSyncModel._instanceName = instanceName
      mplsLdpSyncModel._vrf = instance.vrfName
   return mplsLdpSyncModel

#-------------------------------------------------------------------------------
# show isis hostname
#-------------------------------------------------------------------------------
def showIsisHostname( mode, instanceId=None, instanceName=None, vrfName=None ):
   command = 'MIO_DGET_ISIS_HOSTNAME'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   return showRibCapiCommand( mode, IsisCliModels.IsisHostnameInstanceModel,
                              command, args, clientName='ISIS', l3Config=l3Config )
#-------------------------------------------------------------------------------
# show isis counters [ packet | drop | system ]
# show isis counters [ drop ] details
#-------------------------------------------------------------------------------
def showIsisCounters( mode, instanceId=None, instanceName=None,
                      vrfName=None, drop=None, packet=None, details=None,
                      system=None ): 
   command = 'MIO_DGET_ISIS_COUNTERS'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if details:
      args[ 'details' ] = True
      args[ 'drop' ] = True
      args[ 'packet' ] = not drop
      system = not drop
   else:
      args[ 'details' ] = False
      if drop or packet:
         args[ 'drop' ] = bool( drop )
         args[ 'packet' ] = bool( packet )
         system = False
      elif system:
         args[ 'drop' ] = False
         args[ 'packet' ] = False
      else:
         args[ 'drop' ] = True
         args[ 'packet' ] = True
         system = True

   if ( system and not ( args[ 'packet' ] or args[ 'drop' ] ) ):
      result = IsisCliModels.IsisInstanceCounters()
      result._instanceName = instanceName
      result._vrf = vrfName
      if not verifyShutdown( instanceName ):
         result.systemCounters = IsisCliModels.IsisSystemCounters()
         writeSystemCounters( instanceName, vrfName, result.systemCounters )
      return result
   else:
      result = showRibCapiCommand( mode, IsisCliModels.IsisInstanceCounters,
                                   command, args, clientName='ISIS',
                                   l3Config=l3Config )
      if system:
         if not verifyShutdown( instanceName ):
            result.systemCounters = IsisCliModels.IsisSystemCounters()
            writeSystemCounters( instanceName, vrfName, result.systemCounters )
      return result

def verifyShutdown( instanceName ):
   instConfig = isisConfig.instanceConfig[ instanceName ]
   return instConfig.shutdown

class systemCountersGuardMount:
   def __init__( self, path = 'routing/isis/systemCounters/',
                 smashType = 'IsisExportImpl::IsisSystemCounters', vrfName = None ):
      self.shmemEm = None
      self.readerInfo = None
      self.sysCountersEntity = None
      self.path = path
      self.smashType = smashType
      self.vrfName = vrfName

   def __enter__( self ):
      self.readerInfo = SmashLazyMount.mountInfo( 'reader' ) 
      self.path = self.path + self.vrfName
      self.sysCountersEntity = SmashLazyMount.mount(
            entityManager, self.path, self.smashType, self.readerInfo,
            autoUnmount=True )
      return self.sysCountersEntity
   def __exit__( self, exc_type, exc_val, exc_tb):
      return

def writeSystemCounters( instanceName, vrfName, result ):
   with systemCountersGuardMount( vrfName=vrfName ) as sysCountersEntity:
      # Populating level 1 system counters      
      # Assigning default value {} , in case SystemCounters for l1 and l2
      # is None
      sysCountersL1Entity = sysCountersEntity.systemCountersL1.get(
            instanceName, {} )
      if sysCountersL1Entity:
         result.systemCountersL1 = _populateSystemCountersData( sysCountersL1Entity )
                                                     
      # Populating level 2 system counters           
      sysCountersL2Entity = sysCountersEntity.systemCountersL2.get(
            instanceName, {} )
      if sysCountersL2Entity:
         result.systemCountersL2 = _populateSystemCountersData( sysCountersL2Entity )
                                                     
def _populateSystemCountersData( entity ):
   sysCounters = IsisCliModels.IsisSystemCountersData()
   sysCounters.authTypeFails = entity.authTypeFails
   sysCounters.authKeyFails = entity.authKeyFails 
   sysCounters.corruptedLsps = entity.corruptedLsps
   sysCounters.databaseOverloads = entity.databaseOverloads
   sysCounters.exceededMaxSeqNums = entity.exceededMaxSeqNums
   sysCounters.idLenMismatches = entity.idLenMismatches
   sysCounters.lspErrors = entity.lspErrors 
   sysCounters.maxAreaAddressMismatches = entity.maxAreaAddressMismatches
   sysCounters.ownLspPurges = entity.ownLspPurges
   sysCounters.ownLspPurgesSent = entity.ownLspPurgesSent
   sysCounters.othersLspPurgesSent = entity.othersLspPurgesSent
   sysCounters.seqNumSkips = entity.seqNumSkips
   sysCounters.spfRuns = entity.spfRuns
   return sysCounters

#-------------------------------------------------------------------------------
# show isis ti-lfa tunnel [ TUNNEL_INDEX ]
#-------------------------------------------------------------------------------
def getTilfaTunnelTableEntryModel( tunnelId ):
   tunnelTableEntry = tilfaTunnelTable.entry.get( tunnelId )

   if not tunnelTableEntry:
      return None

   def _getViaModel( tunnelVia, backup=False ):
      labels = []
      labelOp = tunnelVia.labels

      for index in range( labelOp.stackSize ):
         labels.insert( 0, str( labelOp.labelStack( index ) ) )
      return TunnelMplsVia( nexthop=tunnelVia.nexthop,
                            interface=tunnelVia.intfId,
                            type='ip',
                            labels=labels,
                            _isBackupVia=backup )

   vias = [ _getViaModel( tunnelTableEntry.via ) ]
   backupVias = [ _getViaModel( tunnelTableEntry.backupVia, backup=True ) ]
   return TilfaTunnelTableEntry(
      tunnelType='TI-LFA',
      tunnelIndex=getTunnelIndexFromId( tunnelId ),
      vias=vias,
      backupVias=backupVias,
      tunnelId=tunnelId )

def getTilfaTunnelTableModel( tunnelIndex=None ):
   tunnelIds = tilfaTunnelTable.entry
   if tunnelIndex is not None:
      tunnelIds = [ getTunnelIdFromIndex( 'tiLfaTunnel',
                                          tunnelIndex ) ]
   tilfaEntries = {}

   for tunnelId in tunnelIds:
      tilfaEntryModel = getTilfaTunnelTableEntryModel( tunnelId )
      if tilfaEntryModel:
         tilfaEntries[ getTunnelIndexFromId( tunnelId ) ] = tilfaEntryModel
   return TilfaTunnelTable( entries=tilfaEntries )

#-------------------------------------------------------------------------------
# Clear commands
#-------------------------------------------------------------------------------
def clearIsisCommand( *args, **kwargs ):
   '''
   args - Contains CLI mode and callback function
   kwargs - Contains extra arguement for the CLI
   # The format of parser tokens for IS-IS clear commands is like this
   # clear isis [ instance-name ] <command> [ command args ]
   #    or
   # clear isis <command> [ command args ] [ vrf vrf-name ]
   # If instance name is specified we do not show the 'vrf' keyword.
   #
   # If vrf name is specified use the Isis instance in that VRF.
   # If it doesn't exist report error.
   #
   # If neither instance name nor vrf name are specified use the VRF from the
   # routing context if it is set. If routing context is not set and there is an
   # Isis instance configured in the default VRF, then use that instance.
   #
   # Otherwise report error.

   '''
   vrfName = None
   instances = None
   mode = args[ 0 ]
   instanceConfig = isisConfig.instanceConfig

   # callback will be the last entry in *args
   callback = args[ -1 ]
   args = args[ : -1 ]

   for keyword in ( 'clear', 'isis', 'vrf' ):
      kwargs.pop( keyword, None )

   level = kwargs.pop( 'LEVEL', None )
   if level is not None:
      kwargs[ 'level' ] = LVL_KW_TO_TAC_TYPE_MAP[ level ]

   if 'INSTANCE' in kwargs:
      instanceName = kwargs.pop( 'INSTANCE' )
      if instanceName in instanceConfig:
         instances = [ instanceName ]
      else:
         mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
   elif 'ALL_INSTANCE' in kwargs:
      # Clearing all instances in default VRF
      kwargs.pop( 'ALL_INSTANCE' )
      instances = sorted( i.instanceName for i in instanceConfig.values() if
                          i.vrfName == DEFAULT_VRF )
   elif 'VRFNAME' in kwargs:
      # VRF name was specified so find the instance #0 corresponding to that VRF.
      # For non-default VRF, that's the only instance.
      vrfName = kwargs.pop( 'VRFNAME' )
      instanceByVrfName = isisInstanceByVrfName( instanceConfig )
      if vrfName != ALL_VRF_NAME:
         if vrfName in instanceByVrfName:
            instId = 0
            instancesById = isisInstanceNameById( instanceConfig, vrf=vrfName )
            if instId in instancesById:
               instances = [ instancesById[ instId ] ]
         else:
            mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
      else:
         # find all instance #0 of each VRF when VRFNAME is 'all'
         instances = sorted( i.instanceName for i in instanceConfig.values() if
                             i.instanceId == 0 )
   elif 'INTF' in kwargs:
      intf = kwargs[ 'INTF' ]
      kwargs[ 'intf' ] = kwargs.pop( 'INTF' )
      intfConfig = _getIntfConfig( intf.name )
      if ( intfConfig is None or not intfConfig.instanceName ):
         mode.addError( 'No IS-IS instance is enabled on this interface' )
      elif intfConfig.instanceName in instanceConfig:
         instances = [ intfConfig.instanceName ]
      else:
         mode.addError( 'IS-IS instance %s of this interface is not configured'
                        % intfConfig.instanceName )
   else:
      # Neither VRF name nor instance name was specified.
      # Find instance by looking at routing context if that exists.
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      instanceByVrfName = isisInstanceByVrfName( instanceConfig )
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName in instanceByVrfName:
            instances = instanceByVrfName[ vrfName ]
         else:
            mode.addError( "No IS-IS instance is configured in routing-context "
                           "VRF: %s" % vrfName )
      else:
         # Use the default instance in the default VRF (i.e. instance id #0)
         if DEFAULT_VRF in instanceByVrfName:
            instId = 0
            instancesById = isisInstanceNameById( instanceConfig )
            if instId in instancesById:
               instances = [ instancesById[ instId ] ]
   if instances:
      kwargs[ 'instances' ] = instances
      return callback( *args, **kwargs )

CLEAR_NEIGHBOR_TARGET = 'adjacency'
CLEAR_INSTANCE_TARGET = 'instance'
CLEAR_COUNTERS_TARGET = 'counters'
CLEAR_DATABASE_TARGET = 'database'

def clearCommandCallback( mode, instances, target, *args ):
   for instanceName in instances:
      request = ClearRequest( instanceName, target )
      if target == CLEAR_NEIGHBOR_TARGET:
         request.clearAdjRequest = ClearAdjRequest( *args )
      elif target == CLEAR_COUNTERS_TARGET:
         request.clearCountersRequest = ClearCountersRequest( *args )
      elif target == CLEAR_DATABASE_TARGET:
         request.clearDatabaseRequest = ClearDatabaseRequest( *args )
      reqId = "%d.%f" % ( os.getpid(), Tac.now() )
      isisClearReqDir.clearRequest[ reqId ] = request 
      vrfName = isisConfig.instanceConfig[ instanceName ].vrfName
      try:
         Tac.waitFor( lambda: vrfName in isisClearRespDir and
                              reqId in isisClearRespDir[vrfName].clearResponse,
                      description=( "Clear on instance %s" % instanceName ),
                      sleep=True, maxDelay=0.1, timeout=10.0 )
         numCleared = isisClearRespDir[ vrfName ].clearResponse[ reqId ].numCleared
         success = isisClearRespDir[ vrfName ].clearResponse[ reqId ].success
         errorCode = isisClearRespDir[ vrfName ].clearResponse[ reqId ].errorCode
         if success:
            yield numCleared, instanceName, errorCode
         else:
            print( "Failed to clear on instance %s" % instanceName )
      except Tac.Timeout:
         mode.addWarning( "Clearing may not have been successful on instance %s" %
                          ( instanceName ) )

      del isisClearReqDir.clearRequest[ reqId ]


#-------------------------------------------------------------------------------
# "clear isis neighbor
#-------------------------------------------------------------------------------
def doClearIsisNeighbor( mode, instances=None, allAdj=None, adjId=None, intf=None,
      level=None ):
   level = level if level else 'levelNone'
   adjId = adjId if adjId else ''
   if intf:
      if not intf.lookup():
         return
      intf = intf.shortname
   else:
      intf = ''

   if allAdj:
      adjId = intf = ''

   print( "" ) # Newline before output
   # pylint: disable-msg=unused-variable
   for count, inst, errorCode in clearCommandCallback( mode, instances,
                                             CLEAR_NEIGHBOR_TARGET,
                                             intf, adjId, level ):
      print( "%d neighbors cleared on instance %s" % ( count, inst ) )

#-------------------------------------------------------------------------------
# "clear isis instance
#-------------------------------------------------------------------------------
def doClearIsisInstance( mode, instances=None ):
   print( "" )  # Newline before output
   # pylint: disable-msg=unused-variable
   for count, inst, errorCode in clearCommandCallback( mode, instances,
                                             CLEAR_INSTANCE_TARGET ):
      if count > 0:
         print( "IS-IS instance %s cleared." % inst )

#-------------------------------------------------------------------------------
# clear isis counters
#-------------------------------------------------------------------------------
def doClearIsisCounters( mode, instances=None, intf=None ):
   if intf:
      if not intf.lookup():
         return
      intf = intf.shortname
   else:
      intf = ''
   print( "" ) # Newline before output
   # pylint: disable-msg=unused-variable
   for count, inst, errorCode in clearCommandCallback( mode, instances,
                                             CLEAR_COUNTERS_TARGET,
                                             intf ):
      if intf == '':               
         print( "System counters cleared on instance %s." % ( inst ) )
      print( "Counters cleared on %d interfaces of instance %s." % ( count,
         inst ) )

#-------------------------------------------------------------------------------
# clear isis database
#-------------------------------------------------------------------------------
def doClearIsisDatabase( mode, instances=None, allDb=None, LSPID=None, level=None,
                         purge=False ):
   level = level if level else 'levelNone'
   lspid = LSPID if LSPID else ''
   if allDb:
      level = 'levelNone'
      lspid = ''
   purgeWarningPrompt = "Purging LSP(s) will cause all other nodes to remove the "\
                        "LSPs from their database. This is highly disruptive "\
                        "to the network. Do you want to proceed? [y/N]"
   if purge and not BasicCliUtil.confirm( mode, purgeWarningPrompt,
                                          answerForReturn=False ):
      return
   print( "" ) # Newline before output
   for count, inst, errorCode in clearCommandCallback( mode, instances,
                                             CLEAR_DATABASE_TARGET,
                                             lspid, level, purge ):
      dupHnameErrorMsg = \
            f"Hostname conflict with {LSPID}. Use received LSPID instead"
      if ( LSPID and
           errorCode == ClearResponseErrorCode.errorDatabaseConflictHostname ):
         mode.addError( dupHnameErrorMsg )
         return

      clearResponseFmt = "%d LSPs cleared locally on instance %s"
      if purge:
         clearResponseFmt += " and purged from the IS-IS domain"
      clearResponseFmt += "." # Add the fullstop
      print( clearResponseFmt % ( count, inst ) )

# =============== un-staticmethod then moved ==============

def RouterIsisIntfLspTxIntervalCmd_handler( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.lspInterval = args[ 'INTERVAL' ]

def RouterIsisIntfLspTxIntervalCmd_noOrDefaultHandler( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.lspInterval = intfConfig.lspIntervalDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def IsisIntfModeAuthenticationMode_handler( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   authModeVal, keyid, authRxDisabled, authLevel, \
         sharedSecretProfileName = getAuthModeData( args )
   setAuthModeCommon( intfConfig, authLevel, authModeVal, keyid, authRxDisabled,
                      sharedSecretProfileName )

def IsisIntfModeAuthenticationMode_noOrDefaultHandler( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   authModeVal, keyid, _, authLevel, \
         sharedSecretProfileName = getAuthModeData( args )
   noAuthModeCommon( intfConfig, authLevel, authModeVal, keyid,
                     sharedSecretProfileName )
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def RouterIsisIntfAuthenticationKeyCmd_handler( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   keyid, authLevel, algorithm, authKey, shaApadRfc5310 = getAuthKeyData( args )
   setAuthKeyCommon( mode, intfConfig, authLevel, authKey, keyid, algorithm,
                     shaApadRfc5310 )

def RouterIsisIntfAuthenticationKeyCmd_noOrDefaultHandler( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   keyid, authLevel, _, _, _ = getAuthKeyData( args )
   noAuthKeyCommon( intfConfig, authLevel, keyid )
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def IsisIntfFrrProtectionCmd_handler( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   protection = args.get( 'PROTECTION' )
   protectionLevel = args.get( 'LEVEL', 'level-1-2' )
   frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )

   protectionConfig = ProtectionConfig( getProtectionMode( protection ) )
   protectionConfig.level = LVL_KW_TO_TAC_TYPE_MAP[ protectionLevel ]
   if frrAf:
      if frrAf == 'ipv4':
         intfConfig.protectionConfigV4 = protectionConfig
      else:
         intfConfig.protectionConfigV6 = protectionConfig
   else:
      intfConfig.protectionConfigV4V6 = protectionConfig

def IsisIntfFrrProtectionCmd_noOrDefaultHandler( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )
   protectionConfig = ProtectionConfig( ProtectionMode.protectionDefault )
   if frrAf:
      if frrAf == 'ipv4':
         intfConfig.protectionConfigV4 = protectionConfig
      else:
         intfConfig.protectionConfigV6 = protectionConfig
   else:
      intfConfig.protectionConfigV4V6 = protectionConfig

   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def IsisIntfFrrSrlgCmd_handler( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   srlg = 'strict' if 'strict' in args else 'loose'
   srlg = 'disabled' if 'disabled' in args else srlg
   srlgVal = getSrlgProtection( srlg )
   frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )
   if frrAf:
      if frrAf == 'ipv4':
         intfConfig.protectionSrlgV4 = srlgVal
      else:
         intfConfig.protectionSrlgV6 = srlgVal
   else:
      intfConfig.protectionSrlgV4V6 = srlgVal

def IsisIntfFrrSrlgCmd_noOrDefaultHandler( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )
   if frrAf:
      if frrAf == 'ipv4':
         intfConfig.protectionSrlgV4 = intfConfig.protectionSrlgDefault
      else:
         intfConfig.protectionSrlgV6 = intfConfig.protectionSrlgDefault
   else:
      intfConfig.protectionSrlgV4V6 = intfConfig.protectionSrlgDefault

   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def ShowIsisInterfaceCmd_handler( mode, args ):
   del args[ 'interface' ]
   intf = args.get( 'intf', None )
   if not intf:
      args[ 'instDictModelType' ] = IsisCliModels.interfaceModel
      args[ 'cmdVrfModel' ] = IsisCliModels.interfaceVRFsModel
      return showIsisCommand( mode, showIsisIntf, **args )
   else:
      brief = args.get( 'brief' )
      detail = args.get( 'detail', None )
      return showIsisIntfWithName( mode, intf, detail=detail, brief=brief )

def ShowIsisSummaryCmd_handler( mode, args ):
   del args[ 'summary' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisSummaryModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisSummaryVRFsModel
   return showIsisCommand( mode, showIsisSummary, **args )

def ShowIsisLeakedRouteSumAddrCmd_handler( mode, args ):
   del args[ 'summary-address' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisLeakedRouteSumAddrModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisLeakedRouteSumAddrVRFsModel
   return showIsisCommand( mode, showIsisLeakedRouteSumAddr, **args )

def ShowIsisDatabaseCmd_handler( mode, args ):
   del args[ 'database' ]
   args[ 'level' ] = args.pop( 'LEVEL', None )
   args[ 'lspid' ] = args.pop( 'LSPID', None )
   args[ 'instDictModelType' ] = IsisCliModels.lspDbSummaryModel
   args[ 'cmdVrfModel' ] = IsisCliModels.lspDbSummaryVRFModel
   return showIsisCommand( mode, showIsisDatabase, **args )

def ShowIsisDatabaseTeOrDetailCmd_handler( mode, args ):
   del args[ 'database' ]
   args[ 'level' ] = args.pop( 'LEVEL', None )
   args[ 'lspid' ] = args.pop( 'LSPID', None )
   args[ 'teOrDetail' ] = ( args.pop( 'traffic-engineering', None ) or
                            args.pop( 'detail', None ) )
   args[ 'instDictModelType' ] = IsisCliModels.lspDbModel
   args[ 'cmdVrfModel' ] = IsisCliModels.lspDbVRFModel
   return showIsisCommand( mode, showIsisDatabase, **args )

def ShowIsisDatabaseDetailTlvCmd_handler( mode, args ):
   del args[ 'database' ]
   args[ 'level' ] = args.pop( 'LEVEL', None )
   args[ 'lspid' ] = args.pop( 'LSPID', None )
   args[ 'detail' ] = args.pop( 'detail', None )
   args[ 'tlv' ] = args.pop( 'tlv', None )
   args[ 'instDictModelType' ] = IsisCliModels.lspDbDetailTlvModel
   args[ 'cmdVrfModel' ] = IsisCliModels.lspDbDetailTlvVRFModel
   return showIsisCommand( mode, showIsisDatabaseDetailTlv, **args )


def ShowIsisLspLogCmd_handler( mode, args ):
   del args[ 'lsp' ]
   del args[ 'log' ]
   args[ 'instDictModelType' ] = IsisCliModels.lspLogTableModel
   args[ 'cmdVrfModel' ] = IsisCliModels.lspLogTableVRFsModel
   return showIsisCommand( mode, showIsisLsplog, **args )

def ShowIsisNeighborCmd_handler( mode, args ):
   del args[ 'neighbors' ]
   args[ 'level' ] = args.pop( 'LEVEL', None )
   args[ 'instDictModelType' ] = IsisCliModels.showNeighborTableModel
   args[ 'cmdVrfModel' ] = IsisCliModels.showNeighborTableVRFsModel
   return showIsisCommand( mode, showIsisNeighbors, **args )

def ShowIsisNetworkTopologyCmd_handler( mode, args ):
   del args[ 'network' ]
   del args[ 'topology' ]
   if args.pop( 'ldp-tunneling', None ):
      args[ 'ldpTunneling' ] = True
   args[ 'instDictModelType' ] = IsisCliModels.isisTopologyModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisTopologyVRFsModel
   return showIsisCommand( mode, showIsisTopology, **args )

def ShowIsisTopologyCmd_handler( mode, args ):
   del args[ 'topology' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisTopologyModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisTopologyVRFsModel
   return showIsisCommand( mode, showIsisTopology, **args )

def ShowIsisSpfLogCmd_handler( mode, args ):
   del args[ 'spf' ]
   del args[ 'log' ]
   args[ 'instDictModelType' ] = IsisCliModels.spfLogTableModel
   args[ 'cmdVrfModel' ] = IsisCliModels.spfLogTableVRFsModel
   return showIsisCommand( mode, showIsisSpflog, **args )

def ShowIsisLspPurgesCmd_handler( mode, args ):
   del args[ 'lsp' ]
   del args[ 'purges' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisLspPurgesModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisLspPurgesVRFsModel
   return showIsisCommand( mode, showIsisLspPurges, **args )

def ShowIsisGracefulRestartCmd_handler( mode, args ):
   del args[ 'graceful-restart' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisGracefulRestartModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisGracefulRestartVRFsModel
   return showIsisCommand( mode, showIsisGracefulRestart, **args )

def ShowIsisLocalConvergenceDelayCmd_handler( mode, args ):
   del args[ 'local-convergence-delay' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisUloopModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisUloopVRFsModel
   return showIsisCommand( mode, showIsisUloopDelayInfo, **args )

def ShowIsisDynFloodNode_handler( mode, args ):
   for kw in ( 'dynamic', 'flooding', 'nodes' ):
      args.pop( kw, None )
   args[ 'node' ] = args.pop( 'NODENAME', None )
   args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodNodeModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodNodeVRFsModel
   return showIsisCommand( mode, showIsisDynFloodNode, **args )

def ShowIsisDynFloodPaths_handler( mode, args ):
   for kw in ( 'dynamic', 'flooding', 'paths' ):
      args.pop( kw, None )
   args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodPathsModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodPathsVRFsModel
   return showIsisCommand( mode, showIsisDynFloodPaths, **args )

def ShowIsisDynFloodTopology_handler( mode, args ):
   for kw in ( 'dynamic', 'flooding', 'topology' ):
      args.pop( kw, None )
   args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodTopologyModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodTopologyVRFsModel
   return showIsisCommand( mode, showIsisDynFloodTopology, **args )

def ShowIsisDynFloodInterfaces_handler( mode, args ):
   for kw in ( 'dynamic', 'flooding', 'interfaces' ):
      args.pop( kw, None )
   args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodInterfacesModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodInterfacesVRFsModel
   return showIsisCommand( mode, showIsisDynFloodInterfaces, **args )

def ShowIsisAreaProxy_handler( mode, args ):
   for kw in ( 'area', 'proxy' ):
      if kw in args:
         del args[ kw ]
   args[ 'instDictModelType' ] = IsisCliModels.isisShowAreaProxyModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisShowAreaProxyVRFsModel
   return showIsisCommand( mode, showIsisAreaProxy, **args )

def ShowIsisFlexAlgo_handler( mode, args ):
   for kw in ( 'show', 'isis', 'flex-algo', 'algorithm', 'system' ):
      args.pop( kw, None )
   args.update( { 'system' : args.pop( 'SYSTEM', None ) } )
   name = args.pop( 'NAME', None )
   if name:
      algoMap = { flexAlgoConfig.definition[ algoId ].name: algoId
                  for algoId in flexAlgoConfig.definition }
      algo = algoMap.get( name[ 0 ] )
      if algo is None:
         if name[ 0 ].isdigit():
            algo = int( name[ 0 ] )
         else:
            mode.addError( 'Unknown algorithm name: ' + name[ 0 ] )
            return
      args.update( { 'algo': algo } )
   args[ 'instDictModelType' ] = IsisCliModels.isisShowFlexAlgoModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisShowFlexAlgoVRFsModel
   return showIsisCommand( mode, showIsisFlexAlgo, **args )

def ShowIsisMplsLdpSyncCmd_handler( mode, args ):
   del args[ 'mpls' ]
   del args[ 'ldp' ]
   del args[ 'sync' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisMplsLdpSyncModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisMplsLdpSyncVRFsModel
   return showIsisCommand( mode, showInterfaceMplsLdpSync, **args )

def ShowIsisHostnameCmd_handler( mode, args ):
   del args[ 'hostname' ]
   args[ 'instDictModelType' ] = IsisCliModels.isisHostnameModel
   args[ 'cmdVrfModel' ] = IsisCliModels.isisHostnameVRFModel
   return showIsisCommand( mode, showIsisHostname, **args )

def ShowIsisCountersCmd_handler( mode, args ):
   del args[ 'counters' ]
   args[ 'instDictModelType' ] = IsisCliModels.showCountersInstModel
   args[ 'cmdVrfModel' ] = IsisCliModels.showCountersVRFsModel
   return showIsisCommand( mode, showIsisCounters, **args )

def ShowIsisTilfaTunnelTableCmd_handler( mode, args ):
   return getTilfaTunnelTableModel( tunnelIndex=args.get( "TUNNEL_INDEX" ) )

def ClearIsisNeighborCmd_handler( mode, args ):
   del args[ 'neighbor' ]
   if 'interface' in args:
      del args[ 'interface' ]
   args[ 'allAdj' ] = args.pop( 'all', None )
   args[ 'adjId' ] = args.pop( 'Neighbor-ID', None )
   clearIsisCommand( mode, doClearIsisNeighbor, **args )

def ClearIsisInstanceCmd_handler( mode, args ):
   del args[ 'instance' ]
   clearIsisCommand( mode, doClearIsisInstance, **args )

def ClearIsisCountersCmd_handler( mode, args ):
   del args[ 'counters' ]
   if 'interface' in args:
      del args[ 'interface' ]
   clearIsisCommand( mode, doClearIsisCounters, **args )

def ClearIsisDatabaseCmd_handler( mode, args ):
   del args[ 'database' ]
   args[ 'allDb' ] = args.pop( 'all', None )
   args[ 'purge' ] = bool( args.pop( 'domain-wide-purge', False ) )
   clearIsisCommand( mode, doClearIsisDatabase, **args )

def RouterIsisAfIpv6Modelet_multiTopologyIs( mode, args ):
   RouterIsisAfMode_multiTopologyIs( mode )

def RouterIsisAfIpv6Modelet_multiTopologyDel( mode, args ):
   RouterIsisAfMode_multiTopologyDel( mode )
 
def RouterIsisMode_setRedistributeStatic( mode, args ):
   RouterIsisMode_setRedistribute( mode, PROTO_KW_MAP[ 'static' ],
                                   mapName=args.get( 'MAPNAME' ),
                                   rcfName=args.get( 'FUNCTION_NAME' ),
                                   includeLeaked=( 'leaked' in args ) )

def RouterIsisMode_noRedistributeStatic( mode, args ):
   RouterIsisMode_noRedistribute( mode, PROTO_KW_MAP[ 'static' ] )

def RouterIsisMode_setRedistributeCon( mode, args ):
   RouterIsisMode_setRedistribute( mode, PROTO_KW_MAP[ 'connected' ],
                                   mapName=args.get( 'MAPNAME' ),
                                   includeLeaked=( 'leaked' in args ) )

def RouterIsisMode_noRedistributeCon( mode, args ):
   RouterIsisMode_noRedistribute( mode, PROTO_KW_MAP[ 'connected' ] )

def RouterIsisMode_setRedistribute1( mode, args ):
   RouterIsisMode_setRedistribute( mode, PROTO_KW_MAP[ args[ 'PROTOCOL' ] ],
                                   mapName=args.get( 'MAPNAME' ),
                                   includeLeaked=( 'leaked' in args ) )

def RouterIsisAfMode_setRedistributeStaticConnected( mode, args ):
   RouterIsisAfMode_setRedistribute( mode, PROTO_KW_MAP[ args[ 'PROTOCOL' ] ],
                                    mapName=args.get( 'MAPNAME' ),
                                    includeLeaked=( 'leaked' in args ) )

def RouterIsisMode_noRedistribute1( mode, args ):
   RouterIsisMode_noRedistribute( mode, PROTO_KW_MAP[ args[ 'PROTOCOL' ] ] )

def RouterIsisAfMode_noRedistributeStaticConnected( mode, args ):
   RouterIsisAfMode_noRedistribute( mode, PROTO_KW_MAP[ args[ 'PROTOCOL' ] ] )
 
def RouterIsisMode_setRedistribute2( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisMode_setRedistribute( mode, proto, mapName=mapName, routeType=routeType,
                                   includeLeaked='leaked' in args )

def RouterIsisMode_noRedistribute2( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisMode_setRedistribute3( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisMode_setRedistribute( mode, proto, mapName=mapName,
                                   routeType=routeType )

def RouterIsisMode_noRedistribute3( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisAfMode_setRedistribute4( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisAfMode_setRedistribute( mode, proto, mapName=mapName,
                                     routeType=routeType )

def RouterIsisAfMode_noRedistribute4( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisAfMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisMode_setRedistribute5( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisMode_setRedistribute( mode, proto, mapName=mapName,
                                   routeType=routeType )

def RouterIsisMode_noRedistribute5( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisMode_setRedistribute6( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisMode_setRedistribute( mode, proto, mapName=mapName,
                                   routeType=routeType )

def RouterIsisMode_noRedistribute6( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisAfIpv6Modelet_setRedistribute7( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisAfMode_setRedistribute( mode, proto, routeType, mapName )

def RouterIsisAfIpv6Modelet_noRedistribute7( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisAfMode_noRedistribute( mode, proto, routeType )
 
def RouterIsisMode_setRedistribute8( mode, args ):
   RouterIsisMode_setRedistribute( mode, 'protoIsis', mapName=args.get( 'MAPNAME' ) )

def RouterIsisMode_noRedistribute8( mode, args ):
   RouterIsisMode_noRedistribute( mode, 'protoIsis' )

def RouterIsisAfMode_setRedistribute5( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisAfMode_setRedistribute( mode, proto, mapName=mapName,
                                     routeType=routeType )

def RouterIsisAfMode_noRedistribute5( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisAfMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisAfMode_setRedistribute6( mode, args ):
   proto, routeType, mapName = getProtoRouteType( args )
   RouterIsisAfMode_setRedistribute( mode, proto, mapName=mapName,
                                     routeType=routeType )

def RouterIsisAfMode_noRedistribute6( mode, args ):
   proto, routeType, _ = getProtoRouteType( args )
   RouterIsisAfMode_noRedistribute( mode, proto, routeType=routeType )
 
def RouterIsisAfMode_setRedistribute8( mode, args ):
   RouterIsisAfMode_setRedistribute( mode, 'protoIsis', 
                                     mapName=args.get( 'MAPNAME' ) )

def RouterIsisAfMode_noRedistribute8( mode, args ):
   RouterIsisAfMode_noRedistribute( mode, 'protoIsis' )

def RouterIsisAfIpv4Modelet_setLeakSummaryAddress( mode, args ):
   RouterIsisAfMode_setLeakSummaryAddress( mode, args )

def RouterIsisAfIpv4Modelet_setLeakSummaryAddress_no( mode, args ):
   V4Prefix = Arnet.IpGenPrefix( args[ 'V4PREFIX' ].stringValue )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   del instanceConfig.summaryAddressL1ToL2LeakV4[ V4Prefix ]
 
def RouterIsisAfIpv6Modelet_setLeakSummaryAddress( mode, args ):
   RouterIsisAfMode_setLeakSummaryAddress( mode, args )

def RouterIsisAfIpv6Modelet_setLeakSummaryAddress_no( mode, args ):
   V6Prefix = Arnet.IpGenPrefix( args[ 'V6PREFIX' ].stringValue )
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   del instanceConfig.summaryAddressL1ToL2LeakV6[ V6Prefix ]
 
def RouterIsisAfMode_setRouteLeak1( mode, args ):
   RouterIsisAfMode_setRouteLeak( mode, 'level-1', args[ 'MAPNAME' ] )

def RouterIsisAfMode_noRouteLeak1( mode, args ):
   RouterIsisAfMode_noRouteLeak( mode, 'level-1' )
 
def RouterIsisAfMode_setRouteLeak2( mode, args ):
   RouterIsisAfMode_setRouteLeak( mode, 'level-2', args[ 'MAPNAME' ] )

def RouterIsisAfMode_noRouteLeak2( mode, args ):
   RouterIsisAfMode_noRouteLeak( mode, 'level-2' )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entMan ):
   global routingHardwareStatusCommon
   global isisConfig, isisStatusDir, routingHardwareStatus
   global routing6HardwareStatus
   global ipConfig, ip6Config
   global isisClearReqDir, isisClearRespDir
   global vrfInfoDir, vrf6InfoDir, srSysdbStatusDir
   global linkReadyStatusVrfColl, ldpProtoConfigColl, ldpConfigColl
   global entityManager
   global mplsRoutingConfig
   global l3Config
   global mplsStatus
   global isisSystemIdHostnameMap
   global tilfaTunnelTable
   global isisSrV4BindingTable
   global isisSrV6BindingTable
   global srConfig
   global trapConfig
   global flexAlgoConfig

   entityManager = entMan
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )

   l3Config = ConfigMount.mount( entityManager, "l3/config", "L3::Config", 'w' )
   tilfaTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.tiLfaTunnelTable, entityManager )
   isisConfig = ConfigMount.mount( entityManager, 'routing/isis/config',
                                   'Routing::Isis::Config', 'w' )
   isisStatusDir = LazyMount.mount( entityManager, 'routing/isis/status',
                                    'Tac::Dir', 'ri' )
   isisClearReqDir = LazyMount.mount( entityManager, 'routing/isis/clear/request',
                                      'Routing::Isis::ClearRequestNode', 'w' )
   isisClearRespDir = LazyMount.mount( entityManager, 'routing/isis/clear/response',
                                       'Tac::Dir', 'ri' )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager, "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
                                             "routing6/hardware/status",
                                             "Routing6::Hardware::Status", "r" )
   srSysdbStatusDir = LazyMount.mount( entityManager,
                                    "segmentrouting/isis",
                                    "Tac::Dir", "ri" )
   vrfInfoDir = LazyMount.mount( entityManager, "routing/vrf/routingInfo/status",
                                "Tac::Dir", "ri" )
   vrf6InfoDir = LazyMount.mount( entityManager, "routing6/vrf/routingInfo/status",
                                "Tac::Dir", "ri" )
   linkReadyStatusVrfColl = LazyMount.mount( entityManager,
                                  "mpls/ldp/linkReadyStatus",
                                  "Ldp::LdpLinkReadyStatusVrfColl", "r" )
   ldpProtoConfigColl = LazyMount.mount( entityManager,
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpConfigColl = LazyMount.mount( entityManager,
                                    "mpls/ldp/ldpConfigColl",
                                    "Ldp::LdpConfigColl", "r" )
   mplsRoutingConfig = LazyMount.mount( entityManager, 'routing/mpls/config', 
                                                      "Mpls::Config", "r" )
   mplsStatus = LazyMount.mount( entityManager, 'mpls/status',
                                 "Mpls::Api::Status", "r" )
   isisSystemIdHostnameMap = shmemEm.doMount( 'routing/isis/hostnamemap/default',
                                              "IsisExportImpl::IsisHostnameMap",
                                              smi )
   isisSrV4BindingTable = shmemEm.doMount( "mpls/labelBindingTables/isisV4",
                                           "CommonLibSmash::LabelBindingTable",
                                           Smash.mountInfo( 'keyshadow' ) )
   isisSrV6BindingTable = shmemEm.doMount( "mpls/labelBindingTables/isisV6",
                                           "CommonLibSmash::LabelBindingTable",
                                           Smash.mountInfo( 'keyshadow' ) )
   srConfig = ConfigMount.mount( entityManager, "routing/sr/config",
                                 "Routing::SegmentRoutingCli::Config", 'w' )
   trapConfig = ConfigMount.mount( entityManager, "hardware/trap/config/trapConfig",
                                   "Arnet::TrapConfig", 'w' )
   flexAlgoConfig = LazyMount.mount( entityManager, 'te/flexalgo/config',
                                     'FlexAlgo::Config', 'r' )

