#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from BgpLib import NoOrDefault
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   bgpNeighborConfig,
   BgpCmdBaseClass,
   configForVrf,
)
from CliPlugin.RoutingBgpInstanceCli import resetBgpAfModeConfig
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from IpLibTypes import ProtocolAgentModelType

RoutingBgpCliHandler = CliDynamicPlugin( "RoutingBgpCliHandler" )

# class RtMembershipAfModeCmd( BgpCmdBaseClass ):
def RtMembershipAfModeCmd_handleNormal( mode, args ):
   if ( getEffectiveProtocolModel( mode.session.mode ) ==
         ProtocolAgentModelType.ribd ):
      mode.addWarning( 'Routing protocols model multi-agent must be '
                        'configured for the RT membership address-family' )
   addrFamily = 'rt-membership'
   assert addrFamily in afModeExtensionHook.afModeExtensions()

   childMode = mode.childMode(
      afModeExtensionHook.afModeExtension[ addrFamily ],
      addrFamily=addrFamily
   )
   mode.session_.gotoChildMode( childMode )

def RtMembershipAfModeCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      RtMembershipAfModeCmd_handleNormal,
      RtMembershipAfModeCmd_handleNoOrDefault,
      mode,
      args )

def RtMembershipAfModeCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      RtMembershipAfModeCmd_handleNoOrDefault,
      mode,
      args )

def RtMembershipAfModeCmd_handleNoOrDefault( mode, args, noOrDefault ):
   bgpConfig = configForVrf( mode.vrfName )
   resetBgpAfModeConfig( bgpConfig, 'rt-membership', mode.vrfName )

# class RtMembershipNeighborDefaultRt( BgpCmdBaseClass ):
def RtMembershipNeighborDefaultRt_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, mode.vrfName )
   config.rtMembershipDefaultRouteTarget = (
            'rtMembershipDefaultRouteTargetEnabledWithOnly' if ( 'only' in args )
            else 'rtMembershipDefaultRouteTargetEnabled' )

def RtMembershipNeighborDefaultRt_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   if noOrDefault == NoOrDefault.DEFAULT:
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         config.rtMembershipDefaultRouteTarget = \
            'rtMembershipDefaultRouteTargetUnspecified'
         RoutingBgpCliHandler.delNeighborConfigIfDefault( peer,
                                                         vrfName=mode.vrfName )
   else:
      config = bgpNeighborConfig( peer, mode.vrfName )
      config.rtMembershipDefaultRouteTarget = \
            'rtMembershipDefaultRouteTargetDisabled'

def RtMembershipNeighborDefaultRt_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      RtMembershipNeighborDefaultRt_handleNormal,
      RtMembershipNeighborDefaultRt_handleNoOrDefault,
      mode,
      args )

def RtMembershipNeighborDefaultRt_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      RtMembershipNeighborDefaultRt_handleNoOrDefault,
      mode,
      args )

# class RtMembershipNeighborDefaultRtEncoding( BgpCmdBaseClass ):
def RtMembershipNeighborDefaultRtEncoding_handleNormal( mode, args ):
   config = bgpNeighborConfig( args[ 'PEER' ], mode.vrfName )
   config.rtMembershipDefaultRouteTargetEncoding = \
      'rtMembershipDefaultRouteTargetEncodingOriginAsOmit'

def RtMembershipNeighborDefaultRtEncoding_handleNoOrDefault(
      mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   if noOrDefault == NoOrDefault.DEFAULT:
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         config.rtMembershipDefaultRouteTargetEncoding = \
            'rtMembershipDefaultRouteTargetEncodingUnspecified'
         RoutingBgpCliHandler.delNeighborConfigIfDefault( peer,
                                                         vrfName=mode.vrfName )
   else:
      config = bgpNeighborConfig( peer, mode.vrfName )
      config.rtMembershipDefaultRouteTargetEncoding = \
         'rtMembershipDefaultRouteTargetEncodingOriginAsSet'

def RtMembershipNeighborDefaultRtEncoding_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      RtMembershipNeighborDefaultRtEncoding_handleNormal,
      RtMembershipNeighborDefaultRtEncoding_handleNoOrDefault,
      mode,
      args )

def RtMembershipNeighborDefaultRtEncoding_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      RtMembershipNeighborDefaultRtEncoding_handleNoOrDefault,
      mode,
      args )

# class RtMembershipNlriSupport( BgpCmdBaseClass ):
def RtMembershipNlriSupport_handleNormal( mode, args ):
   # NLRI_SUPPORT was already translated by NlriSupportExpr::adapter from the
   # CLI string representation to the internal Enum value.
   nlriSupport = args[ 'NLRI_SUPPORT' ]
   config = configForVrf( mode.vrfName )
   config.rtMembershipNlriSupport = nlriSupport

def RtMembershipNlriSupport_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( vrfName=mode.vrfName )
   config.rtMembershipNlriSupport = 'rtMembershipNlriSupportDisabled'

def RtMembershipNlriSupport_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      RtMembershipNlriSupport_handleNormal,
      RtMembershipNlriSupport_handleNoOrDefault,
      mode,
      args )

def RtMembershipNlriSupport_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      RtMembershipNlriSupport_handleNoOrDefault,
      mode,
      args )

# class RtMembershipNeighborNlriSupport( BgpCmdBaseClass ):
def RtMembershipNeighborNlriSupport_handleNormal( mode, args ):
   peer = args[ 'PEER' ]
   # NLRI_SUPPORT was already translated by NlriSupportExpr::adapter from the
   # CLI string representation to the internal Enum value. 
   nlriSupport = args[ 'NLRI_SUPPORT' ]
   config = bgpNeighborConfig( peer, mode.vrfName )
   config.rtMembershipNlriSupport = nlriSupport

def RtMembershipNeighborNlriSupport_handleNoOrDefault( mode, args, noOrDefault ):
   peer = args[ 'PEER' ]
   if noOrDefault == NoOrDefault.NO:
      # Explicitly disable the feature. Even if it is a 'handlerNoOrDefault' handler,
      # create the config if it does not exist.
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.rtMembershipNlriSupport = 'rtMembershipNlriSupportDisabled'
   else:
      # Default case, the attribute has to be set to 'None'.
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         # Set the default, that is 'unconfigured'. Do not bother doing that
         # if it was already None.
         if config.rtMembershipNlriSupport is not None:
            config.rtMembershipNlriSupport = None
            RoutingBgpCliHandler.delNeighborConfigIfDefault( peer,
                                                            vrfName=mode.vrfName )

def RtMembershipNeighborNlriSupport_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      RtMembershipNeighborNlriSupport_handleNormal,
      RtMembershipNeighborNlriSupport_handleNoOrDefault,
      mode,
      args )

def RtMembershipNeighborNlriSupport_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      RtMembershipNeighborNlriSupport_handleNoOrDefault,
      mode,
      args )
