# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.ScdShowCommands import cliGlobal
from Fru import CliUtilities

from collections import defaultdict

ScdCliModel = CliDynamicPlugin( "ScdCliModel" )

#------------------------------------------------------------------------------------
# show hardware system spi-chips
#------------------------------------------------------------------------------------

# Return a card's slot name according to its status directory name. This function
# should return its input with the exception of the following cases:
#
# "X" (fixed system) -> "Switch"
# "X" (modular system) -> "SupervisorX"
# "LinecardX" (supervisor uplink) -> "SupervisorX"
# "SwitchcardCesX" -> "SwitchcardX"
#
# Performing these transformations ensures the slot name is compatible with the "show
# inventory" command output.
def _getSlotName( statusDirName ):
   for slot in cliGlobal.entityMib.root.cardSlot.values():
      if statusDirName in { slot.tag + slot.label, 'Linecard' + str( slot.relPos ) }:
         return CliUtilities.normalizeTag( slot.tag ) + slot.label

   return 'Switch' if Cell.cellType() == 'fixed' else 'Supervisor' + statusDirName

# This helper function returns a dictionary of Hardware::Spi::ScdFlashStatus models
# keyed by slot name.
def scdFlashChips():
   slots = defaultdict( list )
   for sliceDirName, sliceDir in cliGlobal.spiFlashDir[ 'status' ].items():
      for statusDirName, statusDir in sliceDir.items():
         for statusName, status in statusDir.scdFlashStatus.items():
            try:
               # A ScdFlashStatus is only returned if the chip it represents is also
               # present in the config directory, indicating it is still accessible.
               configDirs = cliGlobal.spiFlashDir[ 'config' ][ sliceDirName ]
               if configDirs[ statusDirName ].scdFlashConfig[ statusName ]:
                  slots[ _getSlotName( statusDirName ) ].append( status )
            except KeyError:
               pass

   return slots

def showHardwareSystemSpiChipsHandler( mode, args ):
   spiChipsModel = ScdCliModel.SpiChips()

   for slotName, chips in scdFlashChips().items():
      for chip in chips:
         spiChip = ScdCliModel.SpiChip( chipName=chip.name,
                                        slot=slotName )

         # We can populate the JEDEC ID for any chip that returns it.
         if chip.chipId:
            spiChip.chipId = chip.chipId

         # A chip returning a known JEDEC ID will populate the following attributes.
         if chip.chipModel:
            spiChip.chipModel = chip.chipModel
            spiChip.chipVendor = chip.chipVendor
            spiChip.chipSize = chip.chipSize

         spiChipsModel.chips[ slotName + '-' + chip.name ] = spiChip

   return spiChipsModel
