#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliDynamicSymbol import CliDynamicPlugin
import SharedMem
import Smash
from TypeFuture import TacLazyType

CmdBatchStatsModel = CliDynamicPlugin( 'SchanAccelCmdBatchStatsModel' )
AgentStat = CmdBatchStatsModel.AgentStat
CmdRingStat = CmdBatchStatsModel.CmdRingStat
TimesliceStat = CmdBatchStatsModel.TimesliceStat

mountConstants = TacLazyType( 'SchanAccel::CmdBatchMountConstants' )
smashBasePath = mountConstants.smashBasePath
sysName = mountConstants.sysName

def readAndStoreFiveMinHistogramBins( cmdBatchDurationStatsSmash,
                                      histogram, agentName ):
   accumulatedBinList = [ 0, 0, 0, 0, 0 ]
   aggregatedTimestamp = None
   key = None
   value = None
   lastTimeSlice = None

   # Group 1 min buckets from the smash collection into 5 min buckets. The timestamp
   # for these grouped 5 min buckets is the timestamp of the latest one in the group
   sortedHistograms = sorted(
         cmdBatchDurationStatsSmash.cmdBatchDurationStatsCollection.items(),
         key=lambda pair: pair[ 1 ].time )
   for index, ( key, value ) in enumerate( sortedHistograms ):
      if index % 5 == 4:
         aggregatedTimestamp = value.time
      elif index % 5 == 0:
         if index:
            # save histograms for previous five minutes
            histogram[ str( key - 1 ) ] = TimesliceStat.buildStat(
               agentName,
               *accumulatedBinList,
               aggregatedTimestamp )
         accumulatedBinList = [ 0, 0, 0, 0, 0 ]

      for i, count in value.histogramInfo.countBin.items():
         accumulatedBinList[ i ] += count

   # Save last timeslice
   if len( sortedHistograms ) > 0:
      key, value = sortedHistograms[ -1 ]
      lastTimeSlice = value.time.nanoseconds()
      aggregatedTimestamp = value.time

   # Group remaining entries under the most recent timestamp
   if any( accumulatedBinList ):
      histogram[ str( key ) ] = TimesliceStat.buildStat(
         agentName,
         *accumulatedBinList,
         aggregatedTimestamp )
   return lastTimeSlice

def showAgentCmdStatsHandler( mode, args ):
   shmemEm = SharedMem.entityManager( sysName, False )
   model = AgentStat( agentName=args.get( 'AGENT_NAME' ) )
   for smashEntityPath in sorted( shmemEm.find( smashBasePath ).path ):
      name = smashEntityPath.split( "/" )[ -1 ]
      if "|" not in name:
         continue

      agentName, cmdRingName = name.split( "|" )
      if agentName != args.get( 'AGENT_NAME' ):
         continue

      mi = Smash.mountInfo( 'reader' )
      cmdBatchDurationStatsSmash = shmemEm.doMount(
         smashEntityPath,
         'SchanAccel::CmdBatchDurationStatsSmash',
         mi )

      if cmdRingName not in model.cmdRings:
         model.cmdRings[ cmdRingName ] = CmdRingStat( cmdRingName=cmdRingName )

      histogram = model.cmdRings[ cmdRingName ].stats

      readAndStoreFiveMinHistogramBins( cmdBatchDurationStatsSmash,
                                        histogram, agentName )

   if not model.cmdRings:
      mode.addWarning( f"No command rings found for {model.agentName}" )

   return model
