#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Ark
from CliModel import Dict, Model, Str, List
import TableOutput

bin0Max = "10us"
bin1Max = "100us"
bin2Max = "10ms"
bin3Max = "100ms"
helpStr = f"""
   Number of command batches that finished categorized by time ranges:
      'Bin 0': Batches finished within the time range <= {bin0Max},
      'Bin 1': Batches finished within the time range {bin0Max} < X <= {bin1Max},
      'Bin 2': Batches finished within the time range {bin1Max} < X <= {bin2Max},
      'Bin 3': Batches finished within the time range {bin2Max} < X <= {bin3Max},
      'Bin 4': Batches finished beyond the time range > {bin3Max}
"""

class TimesliceStat( Model ):
   countbins = List( valueType=int, help=helpStr )
   timeslice = Str( help="Local Time Stamp for histogram" )

   @staticmethod
   def buildStat( agentName, cb1, cb2, cb3, cb4, cb5, timeslice ):
      durationStat = TimesliceStat()
      durationStat.countbins.append( cb1 )
      durationStat.countbins.append( cb2 )
      durationStat.countbins.append( cb3 )
      durationStat.countbins.append( cb4 )
      durationStat.countbins.append( cb5 )
      durationStat.timeslice = Ark.timestampToStr( timeslice, False )

      return durationStat

class CmdRingStat( Model ):
   cmdRingName = Str( help="Command ring for Agent that uses SchanAccel" )
   stats = Dict(
      valueType=TimesliceStat,
      help='Collection of histogram entries' )

   def render( self ):
      print( f"\n{self.cmdRingName}" )
      headers = ( 'Local Time',
                  f'< {bin0Max}',
                  bin1Max,
                  bin2Max,
                  bin3Max,
                  f"> {bin3Max}" )
      table = TableOutput.createTable( headers )

      for fmt in table.columnFormats_:
         fmt.padLimitIs( True )

      totalCounts = [ 0 ] * 5
      for timesliceStat in sorted( self.stats.values(),
                                   key=lambda timeslice: timeslice.timeslice,
                                   reverse=True ):

         table.newRow( timesliceStat.timeslice, *timesliceStat.countbins )
         totalCounts[ : ] = map( lambda x, y: x + y,
                                  totalCounts,
                                  timesliceStat.countbins )

      table.newRow( 'Total', *totalCounts )
      print( table.output() )

class AgentStat( Model ):
   agentName = Str( help="Agent that uses SchanAccel " )
   cmdRings = Dict(
      valueType=CmdRingStat,
      help="Collection of the command rings histograms" )

   def render( self ):
      print( self.agentName )
      for cmdRingStat in self.cmdRings.values():
         cmdRingStat.render()
