# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import array
from CliPlugin.Security import (
   forwardingInfoCallbacks,
   opt,
   PasswordPolicyConfigMode,
   securityCleanupCallbacks,
   securityConfig,
   SecurityConfigMode,
   securityStatus,
   SecurityStatus,
)
import cryptography.exceptions
import fcntl
import FileUrlDefs
import Logging
import os
import re
import ReversibleSecretCli
import termios
import Url

SECURITY_COMMON_ENCRYPTION_KEY_ERROR = Logging.LogHandle(
      "SECURITY_COMMON_ENCRYPTION_KEY_ERROR",
      severity=Logging.logError,
      fmt="Unable to retrieve the customized common encryption key due to: %s",
      explanation="Unable to retrieve the customized common encryption key "
                  "on the device when installing the startup-config. It might "
                  "impact features using type 7 or 8a encrypted secrets "
                  "due to the encryption key missing",
      recommendedAction="Verify the customized common key has been stored "
                        "on the device before load the startup-config" )

def EnterSecurityMode_handler( mode, args ):
   childMode = mode.childMode( SecurityConfigMode )
   mode.session_.gotoChildMode( childMode )

def noEntropySource( self, args=None ):
   if not args or ( "hardware" not in args and "haveged" not in args
                    and "cpu" not in args ):
      self.config_.entropySourceHardware = False
      self.config_.entropySourceHaveged = False
      self.config_.entropySourceJitter = False
   else:
      if "hardware" in args:
         self.config_.entropySourceHardware = False
      if "haveged" in args:
         self.config_.entropySourceHaveged = False
      if "cpu" in args:
         self.config_.entropySourceJitter = False

def EnterSecurityMode_noOrDefaultHandler( mode, args ):
   childMode = mode.childMode( SecurityConfigMode )
   noEntropySource( childMode, args )
   childMode.config_.useEntropyServer = childMode.config_.defaultUseEntropyServer
   childMode.config_.enforceSignature = childMode.config_.defaultEnforceSignature
   childMode.config_.sslProfile = childMode.config_.defaultSslProfile
   childMode.config_.blockedNetworkProtocols.clear()
   childMode.config_.commonKeyEnabled = childMode.config_.commonKeyEnabledDefault
   childMode.config_.commonEncrytionKeyBytes = b''
   childMode.config_.commonEncrytionAESEnabled = \
                             childMode.config_.commonEncrytionAESEnabledDefault
   childMode.config_.minPasswordLength = 0
   childMode.removeComment()
   for callback in securityCleanupCallbacks:
      callback()

def getBytesInFifoPipe( mgmtSecConfig ):
   try:
      fifoFd = os.open( mgmtSecConfig.entropyFifoPath,
                        os.O_RDONLY | os.O_NONBLOCK )
      bytesInPipe = array.array( 'i', [ 0 ] )
      assert fcntl.ioctl( fifoFd, termios.FIONREAD, bytesInPipe, 1 ) == 0
      os.close( fifoFd )
      return bytesInPipe[ 0 ]
   except OSError:
      return 0

def ShowSecurityStatus_handler( mode, args ):
   with open( '/proc/cpuinfo', 'r' ) as file:
      cpuInfo = file.read()
   cpuModelSearch = re.search( re.compile( r'model name\s+: (.*)' ), cpuInfo )
   if cpuModelSearch:
      cpuModel = cpuModelSearch.group( 1 )
   else:
      cpuModel = None
   hardwareInfo = None
   for callback in forwardingInfoCallbacks:
      chipInfo = callback( mode )
      if chipInfo:
         hardwareInfo = chipInfo
         break

   res = SecurityStatus()
   res.cpuModel = opt( cpuModel, None )
   res.switchChip = opt( hardwareInfo, None )
   res.securityChipVersion = opt( securityStatus.securityChipVersion, None )
   res.cryptoModule = securityStatus.cryptoModule
   res.hardwareEntropyEnabled = securityStatus.entropySourceHardwareEnabled
   res.havegedEntropyEnabled = securityStatus.entropySourceHavegedEnabled
   res.jitterEntropyEnabled = securityStatus.entropySourceJitterEnabled
   res.entropyServerEnabled = securityStatus.entropyServerEnabled
   totalBytes = securityStatus.entropyQueueSizeBytes + \
                getBytesInFifoPipe( securityConfig )
   res.hardwareEntropyQueueSize = totalBytes
   res.blockedNetworkProtocols = list( securityConfig.blockedNetworkProtocols )

   return res

def GoToPasswordPolicyMode_handler( mode, args ):
   policyName = args[ 'POLICY' ]
   securityConfig.passwordPolicies.newMember( policyName )
   childMode = mode.childMode( PasswordPolicyConfigMode, policy=policyName )
   mode.session_.gotoChildMode( childMode )

def GoToPasswordPolicyMode_noOrDefaultHandler( mode, args ):
   policyName = args[ 'POLICY' ]
   del securityConfig.passwordPolicies[ policyName ]

def PasswordPolicyMinCmd_handler( mode, args ):
   option = args[ 'OPTION' ]
   value = args[ 'VALUE' ]
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   if option == 'digits':
      policy.minDigits = value
   elif option == 'length':
      policy.minPasswordLength = value
   elif option == 'lower':
      policy.minLower = value
   elif option == 'special':
      policy.minSpecial = value
   else:
      policy.minUpper = value

def PasswordPolicyMinCmd_noOrDefaultHandler( mode, args ):
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   option = args[ 'OPTION' ]
   if option == 'digits':
      policy.minDigits = policy.defaultMinDigits
   elif option == 'length':
      policy.minPasswordLength = policy.defaultMinPasswordLength
   elif option == 'lower':
      policy.minLower = policy.defaultMinLower
   elif option == 'special':
      policy.minSpecial = policy.defaultMinSpecial
   else:
      policy.minUpper = policy.defaultMinUpper

def PasswordPolicyMinChangedCmd_handler( mode, args ):
   value = args[ 'VALUE' ]
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   policy.minChanged = value

def PasswordPolicyMinChangedCmd_noOrDefaultHandler( mode, args ):
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   policy.minChanged = policy.defaultMinChanged

def PasswordPolicyMaxCmd_handler( mode, args ):
   value = args[ 'VALUE' ]
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   if 'repetitive' in args:
      policy.maxRepeated = value
   else:
      policy.maxSequential = value

def PasswordPolicyMaxCmd_noOrDefaultHandler( mode, args ):
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   if 'repetitive' in args:
      policy.maxRepeated = policy.defaultMaxRepeated
   else:
      policy.maxSequential = policy.defaultMaxSequential

def PasswordPolicyDenyCmd_handler( mode, args ):
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   if 'username' in args:
      policy.denyUsername = True
   else:
      count = args[ 'COUNT' ]
      policy.denyLastPasswd = count

def PasswordPolicyDenyCmd_noOrDefaultHandler( mode, args ):
   policy = securityConfig.passwordPolicies[ mode.param_ ]
   if 'username' in args:
      policy.denyUsername = policy.defaultDenyUsername
   else:
      policy.denyLastPasswd = policy.defaultDenyLastPasswd

def EntropySourceHardware_handler( mode, args ):
   if "hardware" in args:
      mode.config_.entropySourceHardware = True
   if "haveged" in args:
      mode.config_.entropySourceHaveged = True
   if "cpu" in args:
      if "jitter" in args:
         mode.config_.entropySourceJitter = True

def EntropySourceHardware_noHandler( mode, args ):
   noEntropySource( mode, args )

def EntropySourceHardware_defaultHandler( mode, args ):
   if "hardware" not in args and "haveged" not in args and "cpu" not in args:
      mode.config_.entropySourceHardware = mode.config_.defaultEntropySourceHardware
      mode.config_.entropySourceHaveged = mode.config_.defaultEntropySourceHaveged
      mode.config_.entropySourceJitter = mode.config_.defaultEntropySourceJitter
   else:
      if "hardware" in args:
         mode.config_.entropySourceHardware = \
               mode.config_.defaultEntropySourceHardware
      if "haveged" in args:
         mode.config_.entropySourceHaveged = mode.config_.defaultEntropySourceHaveged
      if "cpu" in args:
         mode.config_.entropySourceJitter = mode.config_.defaultEntropySourceJitter

def EntropySourceHardwareExclusive_handler( mode, args ):
   mode.config_.useEntropyServer = True

def EntropySourceHardwareExclusive_noOrDefaultHandler( mode, args=None ):
   mode.config_.useEntropyServer = mode.config_.defaultUseEntropyServer

def PasswordMinLengthCmd_handler( mode, args ):
   mode.config_.minPasswordLength = args[ 'LENGTH' ]

def PasswordMinLengthCmd_noOrDefaultHandler( mode, args=None ):
   mode.config_.minPasswordLength = 0

def PasswordEncryptionKeyCommonCmd_handler( mode, args ):
   customized = 'custom' in args
   customizedKey = args.get( 'KEY', '' )

   encryptionKey = ''
   if not customized:
      encryptionKey = mode.config_.commonEncrytionKeyDefault
   elif customizedKey:
      encryptionKey = customizedKey.encode( 'utf-8' )
   else:
      # password encryption-key common custom
      # we need to load the customized common encryption key from somewhere
      if mode.session_.startupConfig():
         # in startup-config, load the key from persistent file on disk
         keyFilePath = Url.parseUrl( f'flash:/{FileUrlDefs.MANAGED_CONFIG_DIR}'
               f'/{securityConfig.commonEncrytionKeyFile}',
               Url.Context( *Url.urlArgsFromMode( mode ) ) ).localFilename()

         systemEncryptionKey = ReversibleSecretCli.getSystemEncryptionKey()
         if not systemEncryptionKey:
            Logging.log( SECURITY_COMMON_ENCRYPTION_KEY_ERROR,
                           'no system encryption key available for decryption' )
            mode.addErrorAndStop( 'No system encryption key available to '
                                  'decrypt customized common key' )
         try:
            with open( keyFilePath, 'r' ) as f:
               encryptedKey = f.read().rstrip( '\n' )
            encryptionKey, _ = ReversibleSecretCli.decodeAES256GCMKey(
                                             encryptedKey, systemEncryptionKey )
         except IOError:
            Logging.log( SECURITY_COMMON_ENCRYPTION_KEY_ERROR,
                           'no customized key file has been stored on the device' )
            mode.addErrorAndStop( 'No customized common encryption key has been '
                                  'stored on the device for startup-config' )
         except ValueError as e:
            errMsg = str( e ).rstrip( '.' )
            Logging.log( SECURITY_COMMON_ENCRYPTION_KEY_ERROR, errMsg )
            mode.addErrorAndStop( 'Failure to decrypt the customized '
                                  'common encryption key stored on the device '
                                  f'({errMsg})' )
         except cryptography.exceptions.InvalidTag:
            Logging.log( SECURITY_COMMON_ENCRYPTION_KEY_ERROR,
                         'decryption error' )
            mode.addErrorAndStop( 'Failure to decrypt the customized '
                                  'common encryption key stored on the device' )
      elif mode.session_.inConfigSession():
         # in session-config, if common encryption key has been configured
         # inside the session config, then use it as encryptionKey.
         # Otherwise we need to copy from global config common encryption key
         sessionCommonKey = mode.config_.commonEncrytionKeyBytes
         globalCommonKey = mode.config_.sysdbObj().commonEncrytionKeyBytes
         encryptionKey = sessionCommonKey or globalCommonKey
      else:
         # in global config, check the Sysdb common encryption key value
         encryptionKey = mode.config_.commonEncrytionKeyBytes

      if ( not encryptionKey or
            encryptionKey == mode.config_.commonEncrytionKeyDefault ):
         # if the encryptionKey is the universal common key,
         # it means user has not configured any customized key, return error
         mode.addErrorAndStop( 'No customized common encryption key has been '
                               'configured on the device' )

   if len( encryptionKey ) != 32:
      mode.addErrorAndStop( 'The length of customized common encryption key '
                            'must be 32-byte for AES-256-GCM encryption' )

   mode.config_.commonKeyEnabled = True
   mode.config_.commonEncrytionKeyBytes = encryptionKey

def PasswordEncryptionKeyCommonCmd_noHandler( mode, args ):
   mode.config_.commonKeyEnabled = False
   mode.config_.commonEncrytionKeyBytes = b''

def PasswordEncryptionKeyCommonCmd_defaultHandler( mode, args ):
   mode.config_.commonKeyEnabled = mode.config_.commonKeyEnabledDefault
   mode.config_.commonEncrytionKeyBytes = b''

def PasswordEncryptionAESEnabledCmd_handler( mode, args ):
   mode.config_.commonEncrytionAESEnabled = True

def PasswordEncryptionAESEnabledCmd_noOrDefaultHandler( mode, args ):
   mode.config_.commonEncrytionAESEnabled = \
                             mode.config_.commonEncrytionAESEnabledDefault

def SignatureVerificationCmd_handler( mode, args ):
   sslProfile = args.get( 'PROFILE', mode.config_.defaultSslProfile )
   mode.config_.enforceSignature = True
   mode.config_.sslProfile = sslProfile

def SignatureVerificationCmd_noHandler( mode, args ):
   mode.config_.enforceSignature = False
   mode.config_.sslProfile = mode.config_.defaultSslProfile

def SignatureVerificationCmd_defaultHandler( mode, args ):
   mode.config_.enforceSignature = mode.config_.defaultEnforceSignature
   mode.config_.sslProfile = mode.config_.defaultSslProfile

def BlockNetworkProtocolCmd_handler( mode, args ):
   protocol = args[ 'PROTO' ]
   mode.config_.blockedNetworkProtocols[ protocol ] = True

def BlockNetworkProtocolCmd_noOrDefaultHandler( mode, args ):
   protocol = args[ 'PROTO' ]
   del mode.config_.blockedNetworkProtocols[ protocol ]

def FipsLogging_handler( mode, args ):
   # FIPS logging is not supported but causes BUG1026094, so make the handler no-op
   pass

def FipsLogging_noOrDefaultHandler( mode, args=None ):
   mode.config_.fipsLogging = False
