#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import calendar
import datetime
import os
import re

import CliDynamicSymbol
from CliPlugin.SnmpCli import snmpEnabled, showSnmpAcrJson
from PyWrappers.NetSnmpUtils import snmpbulkwalk
import Tac
import Tracing

SnmpModel = CliDynamicSymbol.CliDynamicPlugin( "SnmpModel" )

def showSnmpDebugHashedIdent( mode, args ):
   cmd = Tac.Type( "Snmp::SnmpDebugHashedIdentAcr" ).command
   return showSnmpAcrJson( mode, args, SnmpModel.SnmpDebugHashedIdentModel, cmd )

# ------------------------------------------------------------------------------
# The "show snmp mib notifications" command
# ------------------------------------------------------------------------------
def showSnmpMibNotifications( mode, args ):
   notificationList = SnmpModel.SnmpSentNotificationsModel()
   enabled = snmpEnabled( mode )
   notificationList._snmpEnabled = enabled # pylint: disable-msg=protected-access
   if not enabled:
      return notificationList

   # 3 filters: trap-oid, oid, value
   trapOidFilter = args.get( 'trapOID' )
   oidFilter = args.get( 'OID' )
   valueFilter = args.get( 'STRING' )
   noFilter = trapOidFilter is None and oidFilter is None and valueFilter is None
   # We walk the NOTIFICATION-LOG-MIB::nlmLogTable and the
   # NOTIFICATION-LOG-MIB::nlmLogVariableTable and paste them
   # together.  This may appear to be easier with the
   # python bindings, but we don't want to load them into
   # the CLI, especially the MIB modules.

   def walk( walkOid ):
      try:
         objects = Tac.run( [ snmpbulkwalk(), '-Cr30', '-OX', '-m', 'ALL',
                              os.environ.get( 'SNMPHOST', 'localhost' ),
                              walkOid ],
                            stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         mode.addError( str( e ) )
         if e.output:
            for line in e.output.strip().split( "\n" ):
               mode.addError( line )
         return {}

      results = {}
      # objects are obj[idx1][idx2] = TYPE: value
      # we turn this into results[idx1][idx2][obj] = "TYPE: value"
      for line in objects.rstrip().split( "\n" ):
         # Eg: NOTIFICATION-LOG-MIB::nlmLogVariableID[STRING: ][853821][1]
         #                              = OID: SNMPv2-MIB::sysUpTime.0
         m = re.match( r"([^[]+)((\[[^]]+\])+) = (.*)$", line )
         if m is None:
            Tracing.trace2( "Malformed line", line )
            continue
         objName = m.group( 1 )
         idx = m.group( 2 )
         value = m.group( 4 )
         idxs = re.findall( r'\[([^]]+)\]', idx )
         d = results
         for i in idxs:
            d = d.setdefault( i, {} )
         d[ objName ] = value

      return results
   notifications = walk( "NOTIFICATION-LOG-MIB::nlmLogTable" )
   variables = walk( "NOTIFICATION-LOG-MIB::nlmLogVariableTable" )
   # We are dumping the nameless log, so the first-level index will be
   # "STRING: ".
   logName = "STRING: "
   for idx in sorted( notifications.get( logName, [] ), key=int ):
      include = noFilter
      var = notifications[ logName ][ idx ].get(
                           'NOTIFICATION-LOG-MIB::nlmLogNotificationID' )
      if var is None:
         continue
      var = var.replace( 'OID: ', '' )
      if trapOidFilter is not None and any( f in var for f in trapOidFilter ):
         include = True
      when = notifications[ logName ][ idx ].get(
                           'NOTIFICATION-LOG-MIB::nlmLogDateAndTime' )
      if when is None:
         continue
      # The TEXTUAL-CONVENTION formatting results in a fixed format string, so we
      # do not have to worry about parsing this string more carefully
      ( d, t, z ) = when.split( ',' )
      ( zh, zm ) = z.split( ':' )
      tz = datetime.timezone( datetime.timedelta( seconds=(
                     int( zh ) * 60 + int( zm ) ) * 60 ) )
      whendt = datetime.datetime.strptime( d + "," + t,
                                           'STRING: %Y-%m-%d,%H:%M:%S.0' )
      seconds = calendar.timegm( whendt.replace( tzinfo=tz ).utctimetuple() )
      notif = SnmpModel.SnmpSentNotification( oid=var, when=float( seconds ) )
      varList = variables[ logName ].get( idx, [] )
      for varIdx in sorted( varList, key=int ):
         var = variables[ logName ][ idx ][ varIdx ]
         varName = var.get( 'NOTIFICATION-LOG-MIB::nlmLogVariableID' )
         if varName is None:
            continue
         varName = varName.replace( 'OID: ', '' )
         if varName == 'SNMPv2-MIB::snmpTrapEnterprise.0':
            # net-snmp appends this to each trap and it's just noise
            continue
         if varName == 'DISMAN-EVENT-MIB::sysUpTimeInstance':
            # DISMAN-EVENT-MIB redefines sysUpTime.0, so we un-redefine it
            varName = 'SNMPv2-MIB::sysUpTime.0'
         varType = var.get( 'NOTIFICATION-LOG-MIB::nlmLogVariableValueType' )
         if varType is None:
            continue
         m = re.match( r'INTEGER: ([^(]+)', varType )
         if m:
            varType = m.group( 1 ).title()
            varType = { 'Octetstring': 'OctetString',
                        'Timeticks': 'TimeTicks',
                        'Objectid': 'Oid',
                        }.get( varType, varType )
            # pylint: disable-next=consider-using-f-string
            varVal = var.get( 'NOTIFICATION-LOG-MIB::nlmLogVariable%sVal' % varType )
            if varVal is None:
               continue
            if oidFilter is not None and any( f in varName for f in oidFilter ):
               include = True
            if valueFilter is not None and any( f in varVal for f in valueFilter ):
               include = True
            notif.variables.append( SnmpModel.SnmpSentNotificationVariable(
                                                                  oid=varName,
                                                                  value=varVal ) )
      if include:
         notificationList.notifications.append( notif )
   return notificationList
