# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import filecmp
import os
import re
import time

import Cell
import CliDynamicSymbol
from CliPlugin import ConfigConvert
from CliPlugin.VrfCli import DEFAULT_VRF
from CliPlugin import FileCli
import ConfigMount
import LazyMount
from SshAlgorithms import FIPS_HOSTKEYS
import SshCertLib
import SysMgrLib
import Tac
from TypeFuture import TacLazyType

SshConstants = Tac.Type( "Mgmt::Ssh::Constants" )
AuthenMethod = Tac.Type( "Mgmt::Ssh::AuthenMethod" )
SslProfileState = TacLazyType( "Mgmt::Security::Ssl::ProfileState" )

sshConfig = None
sshConfigReq = None
localUserConfig = None
sshStatus = None
sslStatus = None

def getSshModels():
   return CliDynamicSymbol.loadDynamicPlugin( "SshModels" )

sshDynamicSubmodes = CliDynamicSymbol.CliDynamicPlugin( "SshModeCli" )

#-----------------------------------------------------------------------------------
# show management ssh known-hosts [ vrf VRF ] [ HOST ]
#-----------------------------------------------------------------------------------
def knownHostString( knownHostEntry ):
   """
   Returns a string that represents a known-hosts entry for user consumption
   """
   # Translate key type to CLI format
   keyType = SysMgrLib.tacKeyTypeToCliKey[ knownHostEntry.type ]
   str1 = knownHostEntry.host
   str1 += f" {keyType} "
   str1 += knownHostEntry.publicKey
   return str1

def showKnownHosts( mode, args ):
   vrfName = args.get( 'VRF', '' )
   hostFilter = args.get( 'HOST' )
   holdingConfig = sshConfig
   if vrfName and vrfName in sshConfig.vrfConfig:
      holdingConfig = sshConfig.vrfConfig[ vrfName ]
   for host in holdingConfig.knownHost:
      knownHostEntry = holdingConfig.knownHost[ host ]
      if hostFilter and host != hostFilter:
         continue
      print( knownHostString( knownHostEntry ) )

#-------------------------------------------------------------------------------
# show management ssh key ( KEYALGO | FILENAME ) public
#-------------------------------------------------------------------------------
def showPublicHostKey( mode, args ):
   defaultNamedKey = True
   keyPath = ""
   if 'KEYALGO' in args:
      keyAlgo = args[ 'KEYALGO' ]
      if keyAlgo == 'rsa':
         keyPath = SysMgrLib.rsaKeyPath
      elif keyAlgo == 'dsa':
         keyPath = SysMgrLib.dsaKeyPath
      elif keyAlgo == 'ed25519':
         keyPath = SysMgrLib.ed25519KeyPath
      elif keyAlgo == 'ecdsa-nistp521':
         keyPath = SysMgrLib.ecdsa521KeyPath
      elif keyAlgo == 'ecdsa-nistp256':
         keyPath = SysMgrLib.ecdsa256KeyPath
      else:
         assert False, "Illegal key algorithm passed in"
   elif 'FILENAME' in args:
      defaultNamedKey = False
      keyFile = args[ 'FILENAME' ]
      keyPath = keyFile.realFilename_
      # pathname is of the form "/<algorithm>/<file-name>", extract <algorithm>
      keyAlgo = SshCertLib.getAlgoFromKeyPath( keyFile.pathname )
      # If algorithm or filename not specified
      if ( keyAlgo is None or keyAlgo not in SshCertLib.algoDirToSshAlgo ):
         mode.addError( "Invalid key specified, unable to display.")
         return getSshModels().SshHostKey()
   else:
      assert False, "Either algorithm or file name must be provided"

   # Check whether key generation is in progress for specified key
   keyAlgoTac = SysMgrLib.cliKeyTypeToTac[ keyAlgo ]
   keyData = sshConfigReq.sshKeyResetRequest.get( keyAlgoTac )
   keyProcTime = sshStatus.sshKeyResetProcessed.get( keyAlgoTac )
   if keyData and keyProcTime < keyData.timestamp:
      # Key cannot be displayed if key generation is in progress for
      # the requested key. If we are generating the key for default name,
      # requested key is the one being generated hence can't be displayed.
      if ( defaultNamedKey or keyData.path == keyPath ):
         mode.addError( "Key Generation currently in progress. " +
            "Unable to display key." )
         return getSshModels().SshHostKey()

   # Gather public key
   if defaultNamedKey:
      keyPath += '.pub'
      pubKey = ''
      try:
         with open( keyPath ) as f:
            pubKey = f.readline()
      except OSError:
         mode.addError( "Unable to find keyfile" )
         return getSshModels().SshHostKey()
   else:
      # On-demand key generation
      try:
         pubKey = Tac.run( [ "ssh-keygen", "-yf", keyPath ], asRoot=True,
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, ignoreReturnCode=False )
      except Tac.SystemCommandError:
         mode.addError( "Invalid key specified, unable to display." )
         return getSshModels().SshHostKey()

   sshHostKey = getSshModels().SshHostKey( hostKey=pubKey.strip() )

   # Also try to recover the key parameters that were used to create this key
   try:
      if keyAlgo == "rsa":
         pubKeyInfo = Tac.run( [ "ssh-keygen", "-lf", keyPath ],
            asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
            ignoreReturnCode=False )
         # pubKeyInfo is of the form "2048 SHA256:jMapMk9j ... "
         keySize = pubKeyInfo.split()[ 0 ]
         # Validate the output of ssh-keygen, in case an upgrade changes its output
         if keySize not in SysMgrLib.keyTypeToLegalParams[ 'rsa' ][ 'keysize' ]:
            # Do not catch this exception immediately: it's a sign that ssh-keygen
            # formats its output an unexpected way
            raise ValueError(
               f"Unexpected RSA keySize '{keySize}' reported by ssh-keygen" )
         sshHostKey.metadata[ 'keySize' ] = str( keySize )
   except ( Tac.SystemCommandError, AttributeError, IndexError ):
      pass
   return sshHostKey

#-------------------------------------------------------------------------------
# show management ssh trusted-ca key public [ CAKEY ]
#-------------------------------------------------------------------------------
def readNonCommentedLines( mode, filename ):
   try:
      with open( filename ) as fileObj:
         lines = fileObj.read().splitlines()
         return [ line for line in lines if line and not line.startswith( '#' ) ]
   except OSError:
      mode.addError( f"Error displaying contents of file {filename}" )
   return []

def showTrustedCaKeyHandler( mode, args ):
   caKeyFile = args.get( 'CAKEY' )
   filename = SshConstants.caKeyPath( SshConstants.allCaKeysFile )
   if caKeyFile:
      if caKeyFile in sshConfig.caKeyFiles:
         filename = SshConstants.caKeyPath( caKeyFile )
      else:
         return getSshModels().TrustedCaKeys()
   caKeys = readNonCommentedLines( mode, filename )
   return getSshModels().TrustedCaKeys( keys=caKeys )

#-------------------------------------------------------------------------------
# show management ssh key server cert [ HOSTCERT ]
#-------------------------------------------------------------------------------
def showHostkeyCertHandler( mode, args ):
   hostCertFile = args.get( 'HOSTCERT' )
   hostCertFiles = sshConfig.hostCertFiles
   if hostCertFile:
      if hostCertFile in hostCertFiles:
         hostCertFiles = [ hostCertFile ]
      else:
         return getSshModels().HostCertificates()
   hostCerts = {}
   for hostCert in hostCertFiles:
      try:
         hostCertPath = SshConstants.hostCertPath( hostCert )
         # Validate the host cert file
         SshCertLib.validateHostCert( hostCertPath )
         hostCertList = readNonCommentedLines( mode, hostCertPath )
         if hostCertList:
            # Accept only the first host cert because that's the only one
            # that actually takes effect
            hostCerts[ hostCertPath ] = hostCertList[ 0 ]
         if len( hostCertList ) > 1:
            mode.addWarning( f"Multiple host certificates in file {hostCertPath}."
                             " Only the first one is affecting the"
                             " configuration." )
      except SshCertLib.SshHostCertError:
         mode.addError( f"Invalid host certificate {hostCertPath}" )
   return getSshModels().HostCertificates( certificates=hostCerts )

#-------------------------------------------------------------------------------
# show management ssh user-keys revoke-list [ REVOKELIST ]
#-------------------------------------------------------------------------------
def showRevokeListHandler( mode, args ):
   revokeListFile = args.get( 'REVOKELIST' )
   filename = SshConstants.revokeListPath( SshConstants.allRevokeKeysFile )
   if revokeListFile:
      if revokeListFile in sshConfig.revokedUserKeysFiles:
         filename = SshConstants.revokeListPath( revokeListFile )
      else:
         return getSshModels().RevokedKeys()
   revokedKeys = readNonCommentedLines( mode, filename )
   return getSshModels().RevokedKeys( keys=revokedKeys )

#-------------------------------------------------------------------------------
# show management ssh user-keys authorized-keys [ USER ]
#-------------------------------------------------------------------------------
def showAuthorizedKeysHandler( mode, args ):
   user = args.get( 'USER' )
   sshUsers = getSshModels().SshUsers()

   def populateUserConfig( name, userConfig ):
      sshUserConfig = getSshModels().SshUserConfig()
      for keyname, keyConfig in userConfig.sshAuthKeys.items():
         sshKey = getSshModels().SshAuthKey()
         sshKey.keyContents = keyConfig.keyContents
         for optionname, option in keyConfig.sshOptions.items():
            sshOption = getSshModels().SshOption(
                  sshValues=sorted( option.sshValues ) )
            sshKey.sshOptions[ optionname ] = sshOption
         sshUserConfig.sshAuthKeys[ keyname ] = sshKey
      sshUsers.users[ name ] = sshUserConfig

   if user:
      if user in sshConfig.user:
         # Populate configuration for a specific user
         populateUserConfig( user, sshConfig.user[ user ] )
   else:
      # Populate configuration for all users
      for name, userConfig in sshConfig.user.items():
         populateUserConfig( name, userConfig )

   return sshUsers

#-------------------------------------------------------------------------------
# show management ssh authorized-principals [ USER ]
#-------------------------------------------------------------------------------
def showAuthorizedPrincipalsHandler( mode, args ):
   user = args.get( 'USER' )
   sshUsers = getSshModels().SshUsers()

   def populateUserConfig( name, userConfig ):
      sshUserConfig = getSshModels().SshUserConfig()
      for principal, principalConfig in userConfig.sshAuthPrincipals.items():
         sshPrincipal = getSshModels().SshAuthPrincipal()
         for optionname, option in principalConfig.sshOptions.items():
            sshOption = getSshModels().SshOption(
                  sshValues=sorted( option.sshValues ) )
            sshPrincipal.sshOptions[ optionname ] = sshOption
         sshUserConfig.sshAuthPrincipals[ principal ] = sshPrincipal
      sshUsers.users[ name ] = sshUserConfig

   if user:
      if user in sshConfig.user:
         # Populate configuration for a specific user
         populateUserConfig( user, sshConfig.user[ user ] )
   else:
      # Populate configuration for all users
      for name, userConfig in sshConfig.user.items():
         populateUserConfig( name, userConfig )

   return sshUsers

def impossiblePubKeyLoginRequired():
   # Check if public key authentication is required in every authentication path
   for authenticationMethods in sshConfig.authenticationMethodList.values():
      if AuthenMethod.publicKey not in authenticationMethods.method.values():
         return False

   # Check if SSH cert login is possible
   if sshConfig.caKeyFiles:
      return False

   # Check if X509 cert login is possible
   if sshConfig.x509ProfileName:
      x509Profile = sslStatus.profileStatus.get( sshConfig.x509ProfileName )
      if ( x509Profile
           and x509Profile.state == SslProfileState.valid
           and x509Profile.trustedCertsPath ):
         return False

   # Check if public key login is possible by checking at least one user has a
   # configured public key
   for userConfig in sshConfig.user.values():
      for keyConfig in userConfig.sshAuthKeys.values():
         if keyConfig.keyContents != "":
            return False

   return True

#-------------------------------------------------------------------------------
# show management ssh [ vrf VRF ]
#-------------------------------------------------------------------------------
def showSshStatus( mode, args ):
   vrfName = args.get( 'VRF' )
   holdingConfig = sshConfig
   holdingStatus = sshStatus
   vrfFound = False
   if vrfName and vrfName in sshConfig.vrfConfig and vrfName in sshStatus.vrfStatus:
      vrfFound = True
      holdingConfig = sshConfig.vrfConfig[ vrfName ]
      holdingStatus = sshStatus.vrfStatus[ vrfName ]
   elif vrfName and not vrfFound:
      mode.addError( f"VRF {vrfName} not found under SSH management" )
      return
   vrfDisplayStr = "Default VRF"
   if vrfName:
      vrfDisplayStr = f"VRF {vrfName}"
   sshdRunning = holdingStatus.enabled
   if sshdRunning:
      sshdStatus = "enabled"
   else:
      sshdStatus = "disabled"

   if impossiblePubKeyLoginRequired():
      mode.addWarning( "SSH is configured to require keys for authentication,"
                       " but no user has a configured ssh key" )

   userCertAuthMethods = []
   if sshConfig.caKeyFiles:
      userCertAuthMethods.append( "authorized-principals-file" )
      if sshConfig.authPrincipalsCmdFile:
         userCertAuthMethods.append( "authorized-principals-command" )
   if sshConfig.x509ProfileName:
      userCertAuthMethods.append( "x509-certificates" )

   userCertAuthMethodsStr = ", ".join( userCertAuthMethods )
   if not userCertAuthMethodsStr:
      userCertAuthMethodsStr = "none (neither trusted CA nor SSL profile configured)"
   outputStr = f"User certificate authentication methods: {userCertAuthMethodsStr}\n"

   if sshConfig.authPrincipalsCmdFile:
      authCmd = "none (missing)"
      authPrincipalsCmdExecutablePath = "/etc/ssh/auth_principals_cmd"
      authPrincipalsCmdPersistPath = SshConstants.authPrincipalsCmdPath(
            sshConfig.authPrincipalsCmdFile )
      authPrincipalsCmdEosPath = f"/usr/sbin/{sshConfig.authPrincipalsCmdFile}"
      if os.path.isfile( authPrincipalsCmdExecutablePath ):
         if ( os.path.isfile( authPrincipalsCmdEosPath ) and
              filecmp.cmp( authPrincipalsCmdEosPath,
                 authPrincipalsCmdExecutablePath ) ):
            authCmd = f"/usr/sbin/{sshConfig.authPrincipalsCmdFile}"
         elif ( os.path.isfile( authPrincipalsCmdPersistPath ) and
              filecmp.cmp( authPrincipalsCmdPersistPath,
                 authPrincipalsCmdExecutablePath ) ):
            authCmd = f"ssh-auth-principals-cmd:{sshConfig.authPrincipalsCmdFile}"
      outputStr += f"Authorized-principals-command: {authCmd}\n"

   if sshConfig.x509ProfileName:
      x509Profile = sslStatus.profileStatus.get( sshConfig.x509ProfileName )
      if not x509Profile:
         profileStatus = "missing"
      elif x509Profile.state == SslProfileState.valid:
         profileStatus = "valid"
      elif x509Profile.state == SslProfileState.invalid:
         profileStatus = ( "invalid. See \"show management security ssl profile\""
                           " output for details." )
      else:
         profileStatus = "updating"
      outputStr += f"SSL profile: '{sshConfig.x509ProfileName}' {profileStatus}\n"

   outputStr += f"SSHD status for {vrfDisplayStr}: {sshdStatus}\n"
   outputStr += f"SSH connection limit: {sshConfig.connLimit}\n"
   outputStr += f"SSH per host connection limit: {sshConfig.perHostConnLimit}\n"
   sshFipsStatus = "disabled"
   if sshConfig.fipsRestrictions:
      sshFipsStatus = "enabled"
   outputStr += f"FIPS status: {sshFipsStatus}\n"
   if holdingStatus.tunnel:
      outputStr += "SSH Tunnel Information:\n"
   subFmt = '%s: %s\n'
   for tunnelConfig in holdingConfig.tunnel.values():
      tunnelStatus = holdingStatus.tunnel.get( tunnelConfig.name )
      if not tunnelStatus:
         # Tunnel Status should always be created by SuperServer.
         # In Cli --standalone however it will not exist
         continue
      tunnelRunning = ""
      if tunnelStatus.enabled:
         if tunnelStatus.active:
            if tunnelStatus.established:
               tunnelRunning = "established"
            else:
               tunnelRunning = "running"
         else:
            tunnelRunning = "not running"
      else:
         if tunnelConfig.enable and not tunnelStatus.fullyConfigured:
            tunnelRunning = "not fully configured"
         else:
            tunnelRunning = "shutdown"
      tunnelRemote = ( f"{tunnelConfig.sshServerUsername}"
                       f"@{tunnelConfig.sshServerAddress}"
                       f":{tunnelConfig.sshServerPort}" )
      tunnelRemoteBinding = ( f"{tunnelConfig.remoteHost}"
                             f":{tunnelConfig.remotePort}" )
      tunnelMaxDrops = ( f"{tunnelConfig.serverAliveMaxLost} packets"
                         f" / {tunnelConfig.serverAliveInterval} seconds" )
      restartTime = "never"
      if tunnelStatus.lastRestartTime > 0:
         time.tzset()
         restartTime = time.ctime( tunnelStatus.lastRestartTime )
      outputStr += f"SSH Tunnel {tunnelConfig.name}\n"
      outputStr += subFmt % ( "Status", tunnelRunning )
      outputStr += subFmt % ( "Local Port", tunnelConfig.localPort )
      outputStr += subFmt % ( "SSH Server Port", tunnelRemote )
      outputStr += subFmt % ( "Remote Host Port", tunnelRemoteBinding )
      outputStr += subFmt % ( "Max Packet Drops", tunnelMaxDrops )
      outputStr += subFmt % ( "Restart Count", tunnelStatus.restartCount )
      outputStr += subFmt % ( "Last Restart Time", restartTime )
      outputStr += subFmt % ( "Last Restart Cause", tunnelStatus.lastRestartCause )

   keyNameFmt = 'SSH Key: %s\n'
   principalFmt = 'SSH Principal: %s\n'
   keyContentsFmt = 'Public Key: %s\n'
   if sshConfig.user:
      outputStr += "\nSSH User-specific Configuration:\n"
   for user in sorted( sshConfig.user ):
      outputStr += f"User: {user}\n"
      # Output only if we have a non-default value
      if sshConfig.user[ user ].userTcpForwarding != \
            sshConfig.user[ user ].userTcpForwardingDefault:
         outputStr += subFmt % ( "Allow TCP forwarding",
                                 sshConfig.user[ user ].userTcpForwarding )
      keyConfig = sshConfig.user[ user ].sshAuthKeys
      principalConfig = sshConfig.user[ user ].sshAuthPrincipals
      for key in sorted( keyConfig ):
         outputStr += keyNameFmt % key
         if keyConfig[ key ].keyContents:
            outputStr += keyContentsFmt % keyConfig[ key ].keyContents
         outputStr = SysMgrLib.outputOptions( key, keyConfig, outputStr )
      for principal in sorted( principalConfig ):
         outputStr += principalFmt % principal
         outputStr = SysMgrLib.outputOptions( principal, principalConfig, outputStr )
      outputStr += "\n"
   if len( sshConfig.user ) > 0:
      # Avoid the additional new line added to separate users
      outputStr = outputStr[ : -1 ]
   print( outputStr )

#----------------------------------------------------------------
# SshConfigMode
#----------------------------------------------------------------
def gotoSshConfigMode( mode, args ):
   childMode = mode.childMode( sshDynamicSubmodes.SshConfigMode )
   mode.session_.gotoChildMode( childMode )

def defaultSshConfig( mode, args ):
   childMode = mode.childMode( sshDynamicSubmodes.SshConfigMode )
   childMode.noIdleTimeout()
   if childMode.config().legacyAuthenticationModeSet:
      childMode.defaultAuthenticationMode()
   else:
      childMode.defaultAuthenticationProtocol()
   childMode.noServerPort()
   childMode.defaultCiphers()
   childMode.noFipsRestrictions()
   childMode.defaultMac()
   childMode.defaultKeyExchange()
   childMode.defaultHostKeyAlgorithms()
   childMode.noCheckHostKeys()
   childMode.setPermitEmptyPasswordsDefault()
   childMode.setClientAliveIntervalDefault()
   childMode.setClientAliveCountMaxDefault()
   for tunnelName in sshConfig.tunnel:
      childMode.noMainSshTunnel( { 'TUNNELNAME': tunnelName } )
   for vrf in childMode.aclCpConfig_.cpConfig[ 'ip' ].serviceAcl:
      childMode.noIpAcl( { 'VRF': vrf } )
   for vrf in childMode.aclCpConfig_.cpConfig[ 'ipv6' ].serviceAcl:
      childMode.noIp6Acl( { 'VRF': vrf } )
   childMode.defaultRekeyData()
   childMode.defaultRekeyTime()
   childMode.noShutdown()
   childMode.noDscp()
   childMode.noVerifyDns()
   childMode.defaultLoginGraceTime( {} )
   childMode.setConnectionLimit( {} )
   childMode.setPerHostConnectionLimit( {} )
   childMode.removeComment()
   childMode.setAuthPrincipalsCmd( {} )
   childMode.noCaKeyFile( {} )
   childMode.noHostCertFile( {} )
   childMode.noRevokedUserKeysFile( {} )
   childMode.noLogLevel()
   childMode.disableLoggingTarget()
   sshConfig.vrfConfig.clear()
   sshConfig.knownHost.clear()
   sshConfig.x509ProfileName = ''
   sshConfig.x509UsernameDomainOmit = False
   # Default all SSH users
   for user in list( sshConfig.user ):
      SysMgrLib.defaultSshUserConfig( user, sshConfig )

def doResetHostKey( mode, args ):
   keyAlgo = args.get( 'KEYALGO', None )
   keyIdentity = args.get( 'FILENAME', None )
   cliKeyParamNames = args.get( 'KEY_PARAM_NAME', [] )
   cliKeyParamValeus = args.get( 'KEY_PARAM_VALUE', [] )
   cliKeyParams = dict( zip( cliKeyParamNames, cliKeyParamValeus ) )
   keyPath = ""
   # Handle legacy aliases - this will be useful when we rewrite
   # ecdsa-nistp256 to ecdsa with parameters curve=nist-p256
   if keyAlgo in SysMgrLib.legacyKeyTypeToAliasedParams:
      cliKeyParams.update( SysMgrLib.legacyKeyTypeToAliasedParams[ keyAlgo ][ 1 ] )
      keyAlgo = SysMgrLib.legacyKeyTypeToAliasedParams[ keyAlgo ][ 0 ]
   if sshConfig.fipsRestrictions and \
      ( keyAlgo not in FIPS_HOSTKEYS ):
      mode.addError( "Can only regenerate FIPS keys when FIPS mode is on" )
      return
   if keyIdentity is not None:
      # Enure the file name is valid. Same rules for URL files names,
      # ability to specify a "/" used to seperate file name from
      # algorithm sub-directory.
      # Eg: ssh-key:<algo>/<file-name>
      filePattern = re.compile( r'([A-Za-z0-9_:{}\.\[\]\/-]+)$' )
      if not filePattern.match( keyIdentity.pathname ):
         mode.addError( "Invalid SSH private key file name." )
         return
      # if keyIdentity is not none then keyIdentity.pathname will be of the
      # form /<algo>/<file-name>, extract keyAlgo
      keyAlgo = SshCertLib.getAlgoFromKeyPath( keyIdentity.pathname )
      # Ensure valid key algorithm is specified
      if ( keyAlgo is None or keyAlgo not in SshCertLib.algoDirToSshAlgo ):
         mode.addError( "Invalid key algorithm specified." )
         return
      keyPath = keyIdentity.realFilename_
   # the key parameters are pre-validated by the dynamic keyword matchers, so
   # we might get way without this separate validation below?
   mappedKeyAlgo = SysMgrLib.cliKeyTypeToTac[ keyAlgo ]
   legalKeyParams = SysMgrLib.keyTypeToLegalParams.get( mappedKeyAlgo, {} )
   # Handle legacy aliases - this will be useful when we rewrite
   # ecdsa-nistp256 to ecdsa with parameters curve=nist-p256
   if keyAlgo in SysMgrLib.legacyKeyTypeToAliasedParams:
      cliKeyParams.update( SysMgrLib.legacyKeyTypeToAliasedParams[ keyAlgo ] )
   for paramName, paramValue in cliKeyParams.items():
      if paramName not in legalKeyParams:
         mode.addError(
            f"Invalid key parameter {paramName} for {mappedKeyAlgo} key. "
            f"Expecting one of: {', '.join( legalKeyParams.keys() )}" )
         return
      if paramValue not in legalKeyParams[ paramName ]:
         mode.addError(
            f"Invalid {paramName} key parameter value {paramValue} for "
            f"{mappedKeyAlgo} key. Expecting one of: "
            f"{', '.join( legalKeyParams[ paramName ] )}" )
         return
   sshKeyResetRequest = Tac.Value( "Mgmt::Ssh::KeyData", Tac.now(), keyPath )
   for paramName, paramValue in cliKeyParams.items():
      sshKeyResetRequest.keyParams[ paramName ] = paramValue
   sshConfigReq.sshKeyResetRequest[ mappedKeyAlgo ] = sshKeyResetRequest

#-------------------------------------------------------------------------------
# Register convertAuthenticationConfigurationSyntax via
# "config convert new-syntax"
#-------------------------------------------------------------------------------
def convertAuthenticationConfigurationSyntax( mode ):
   sshConfig.legacyAuthenticationModeSet = False

ConfigConvert.registerConfigConvertCallback(
   convertAuthenticationConfigurationSyntax )

def gotoSshVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( sshDynamicSubmodes.SshVrfConfigMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )
   mode.updateDscpRules()

def noSshVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( sshDynamicSubmodes.SshVrfConfigMode, vrfName=vrfName )
   for tunnelName in childMode.config().tunnel:
      # pylint: disable-msg=protected-access
      childMode._noSshVrfTunnel( tunnelName )
      # pylint: enable-msg=protected-access
   del sshConfig.vrfConfig[ vrfName ]

   # if vrf config is default, delete the known host in the default config
   if vrfName == DEFAULT_VRF:
      sshConfig.knownHost.clear()

   mode.updateDscpRules()

def verifyHook( mode, hashName ):
   if hashName == 'md5' and sshConfig.fipsRestrictions:
      mode.addError( "MD5 is not allowed when using FIPS algorithms." )
      return False
   return True

def Plugin( entityManager ):
   global localUserConfig, sshConfig, sshConfigReq, sshStatus, sslStatus

   sshConfig = ConfigMount.mount( entityManager, "mgmt/ssh/config",
                                  "Mgmt::Ssh::Config", "w" )
   localUserConfig = ConfigMount.mount( entityManager, "security/aaa/local/config",
                                    "LocalUser::Config", "w" )

   sshConfigReq = LazyMount.mount( entityManager, "mgmt/ssh/configReq",
                                  "Mgmt::Ssh::ConfigReq", "w" )
   sshStatus = LazyMount.mount( entityManager, Cell.path( "mgmt/ssh/status" ),
                                "Mgmt::Ssh::Status", "r" )
   sslStatus = LazyMount.mount( entityManager, "mgmt/security/ssl/status",
                                "Mgmt::Security::Ssl::Status", "r" )
   FileCli.verifyHook.addExtension( verifyHook )
