#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from socket import AF_INET, AF_INET6
import Arnet
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.IraIp6RouteCliLib import isValidIpv6PrefixWithError
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError
from CliPlugin.IraServiceCli import getEffectiveProtocolModel 
from CliPlugin import StaticMribShowCli
import FibUtils
import LazyMount
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import Tac
from Toggles.RoutingLibToggleLib import toggleFibGenMountPathEnabled

McastCommonModel = CliDynamicPlugin( "McastCommonModel" )

mrib4StatusColl = None
_mfib4VrfConfig = None
mrib6StatusColl = None
_mfib6VrfConfig = None
mribGenForwardingStatus = None

mrib4RouteStatus = {}
mrib6RouteStatus = {}
mrib4ForwardingStatus = {}
mrib6ForwardingStatus = {}

class MribTrieBase:
   def __init__( self, mribStatus, af, protocolModel, mribFwdStatus=None,
                 mribGenFwdStatus=None ):
      self.mribStatus = mribStatus
      self.mribFwdStatus = mribFwdStatus
      self.mribGenFwdStatus = mribGenFwdStatus
      self.trie = Tac.newInstance( "Routing::TrieGen", "MribTrie", af )
      routing = 'Routing' if af == 'ipv4' else 'Routing6'
      if protocolModel != ProtoAgentModel.multiAgent:
         self.mribShadow = Tac.newInstance( 
               "%s::Multicast::Shadow::Status" % routing, "mribShadow" )
         self.mribTrieBuilder = Tac.newInstance( 
            "%s::Multicast::Rib::MribTrieBuilderSm" % routing, self.mribStatus, 
            self.mribShadow )
      else:
         self.mribShadow = Tac.newInstance( "%s::Shadow::Status" % routing,
                                            "mribShadow" )
         self.mribTrieBuilder = Tac.newInstance( 
            "%s::Shadow::UribRouteShadowSm" % routing, self.mribStatus,
            self.mribFwdStatus, self.mribGenFwdStatus, self.trie, self.mribShadow )

   def longestMatch( self, routePrefix ):
      return self.mribShadow.trie.longestMatch( routePrefix )

class MribTrie( MribTrieBase ):
   def __init__( self, mribStatus, protocolModel, mribFwdStatus=None,
                 mribGenFwdStatus=None ):
      MribTrieBase.__init__( self, mribStatus, 'ipv4', protocolModel,
                             mribFwdStatus=mribFwdStatus,
                             mribGenFwdStatus=mribGenFwdStatus)

class Mrib6Trie( MribTrieBase ):
   def __init__( self, mribStatus, protocolModel, mribFwdStatus=None,
                 mribGenFwdStatus=None ):
      MribTrieBase.__init__( self, mribStatus, 'ipv6', protocolModel,
                             mribFwdStatus=mribFwdStatus,
                             mribGenFwdStatus=mribGenFwdStatus)

def timeFormat( t ):
   t = int( Tac.now() - t )
   s = t % 60
   t = t // 60
   m = t % 60
   t = t // 60
   h = t % 24
   t = t // 24
   d = t

   if d:
      return '%dd %02dh' % ( d, h )
   else:
      return '%02d:%02d:%02d' % ( h, m, s )

def v4RouteProtocolToString( route ):
   routeProtocolStr = ''
   routeProtocol = route.routeType

   # pylint: disable-next=consider-using-in
   if ( routeProtocol == 'connected' or routeProtocol == 'receive' 
         or routeProtocol == 'receiveBcast' ):
      routeProtocolStr = 'C'
   elif routeProtocol == 'staticConfig':
      routeProtocolStr = 'S'
   elif routeProtocol == 'rip':
      routeProtocolStr = 'R'
   elif routeProtocol == 'ebgp':
      routeProtocolStr = 'B E'
   elif routeProtocol == 'ibgp':
      routeProtocolStr = 'B I'
   elif routeProtocol == 'ospfIntraArea':
      routeProtocolStr = 'O'
   elif routeProtocol == 'ospfInterArea':
      routeProtocolStr = 'O IA'
   elif routeProtocol == 'ospfAseE1':
      routeProtocolStr = 'O E1'
   elif routeProtocol == 'ospfAseE2':
      routeProtocolStr = 'O E2'
   elif routeProtocol == 'ospfNssaE1':
      routeProtocolStr = 'O N1'
   elif routeProtocol == 'ospfNssaE2':
      routeProtocolStr = 'O N2'
   elif routeProtocol == 'nexthopGroup':
      routeProtocolStr = 'NG'
   elif routeProtocol == 'isis':
      routeProtocolStr = 'I'
   elif routeProtocol == 'kernel':
      routeProtocolStr = 'K'
   elif routeProtocol == 'bgpAggregate':
      routeProtocolStr = 'A B'
   elif routeProtocol == 'ospfAggregate':
      routeProtocolStr = 'A O'
   elif routeProtocol == 'martian':
      routeProtocolStr = 'M'

   return routeProtocolStr

def v6RouteProtocolToString( route ):
   routeProtocolStr = ''
   routeProtocol = route.routeType

   # pylint: disable-next=consider-using-in
   if ( routeProtocol == 'connected' or routeProtocol == 'receive' 
         or routeProtocol == 'receiveBcast' ):
      routeProtocolStr = 'C'
   elif routeProtocol == 'staticConfig':
      routeProtocolStr = 'S'
   elif routeProtocol == 'rip':
      routeProtocolStr = 'R'
   elif routeProtocol == 'bgp':
      routeProtocolStr = 'B'
   elif routeProtocol == 'ibgp':
      routeProtocolStr = 'B I'
   elif routeProtocol == 'ebgp':
      routeProtocolStr = 'B E'
   elif routeProtocol == 'ospf':
      routeProtocolStr = 'O3'
   elif routeProtocol == 'nexthopGroup':
      routeProtocolStr = 'NG'
   elif routeProtocol == 'isisL1':
      routeProtocolStr = 'I L1'
   elif routeProtocol == 'isisL2':
      routeProtocolStr = 'I L2'
   elif routeProtocol == 'kernel':
      routeProtocolStr = 'K'
   elif routeProtocol == 'bgpAggregate':
      routeProtocolStr = 'A B'

   return routeProtocolStr

def showIpRouteMulticast( mode, prefix=None, vrfName=None, af=None ):

   model = ( McastCommonModel.MulticastRibRoutes() if af == AF_INET else
         McastCommonModel.MulticastRib6Routes() )

   _mfibVrfConfig = _mfib4VrfConfig if af == AF_INET else _mfib6VrfConfig
   if not vrfName :
      vrfName = DEFAULT_VRF
   elif vrfName not in _mfibVrfConfig.config: 
      mode.addError( "Invalid vrf name %s " % vrfName )
      return None

   mribStatus = None
   mribFwdStatus = None
   nRoutes = None
   nFec = None

   mribStatusColl = mrib4StatusColl if af == AF_INET else mrib6StatusColl
   mribRouteStatus = mrib4RouteStatus if af == AF_INET else mrib6RouteStatus
   mribForwardingStatus = ( mrib4ForwardingStatus if af == AF_INET 
         else mrib6ForwardingStatus )

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      if af == AF_INET6:
         # BUG 373633 : IPv6 multicast routes not supported in single agent mode 
         mode.addError( "Not supported" )
         return None
      if vrfName not in mribStatusColl.vrfStatus:
         return model
      mribStatus = mribStatusColl.vrfStatus[ vrfName ]
      nRoutes = mribStatus.route
   else:
      if vrfName not in mribRouteStatus:
         routeInfo = FibUtils.routeStatusInfo( 'reader' )
         mountPath = '/routing/mrib/routeStatus/ipv{}/{}'.format(
               '4' if af == AF_INET else '6', vrfName )
         mribRouteStatus[ vrfName ] = StaticMribShowCli.smashMount.doMount(
            mountPath, 
            'Smash::Fib%s::RouteStatus' % ( '' if af == AF_INET else '6' ), 
            routeInfo )

      if toggleFibGenMountPathEnabled():
         nFec = mribGenForwardingStatus.fec
      else:
         if vrfName not in mribForwardingStatus:
            fwdInfo = FibUtils.forwardingStatusInfo( 'reader' )
            mountPath = '/routing/mrib/forwardingStatus/ipv{}/{}'.format(
                  '4' if af == AF_INET else '6', vrfName )
            mribForwardingStatus[ vrfName ] = StaticMribShowCli.smashMount.doMount(
               mountPath,
               'Smash::Fib%s::ForwardingStatus' % ( '' if af == AF_INET else '6' ),
               fwdInfo )
         mribFwdStatus = mribForwardingStatus[ vrfName ]
         nFec = mribFwdStatus.fec

      if vrfName not in mribRouteStatus:
         return model

      mribStatus = mribRouteStatus[ vrfName ]
      nRoutes = mribStatus.route

   defaultIntfId = Tac.newInstance( "Arnet::IntfId" )

   if nRoutes is None:
      return model

   def renderRoute( route_, fec_=None ):
      submodel = McastCommonModel.MulticastRibRoute()
      submodel.routeProtocol = ( v4RouteProtocolToString( route_ ) if af == AF_INET 
            else v6RouteProtocolToString( route_ ) )
      submodel.routePreference = route_.preference
      submodel.routeMetric = route_.metric
      if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
         submodel.routeCreationTime = timeFormat( route_.creationTime )
         vias = route_.via
         for via in vias:
            intfId = via.intfId if via.intfId == defaultIntfId else "None"
            submodel.vias[ str( via.hop ) ] = str( intfId )
      else:
         if fec_:
            vias = fec_.via
            for via in vias.values():
               submodel.vias[ str( via.hop ) ] = str( via.intfId )
      return submodel

   def updateModel( prefix ):
      route_ = nRoutes.get( prefix )
      if not route_:
         return

      m = Arnet.IpGenPrefix( str( prefix ) )
      if not hasattr( route_, 'fecId' ):
         model.routes[ m ] = renderRoute( route_ )
      else:
         fec_ = nFec.get( route_.fecId )
         model.routes[ m ] = renderRoute( route_, fec_ )

   if not prefix:
      for p in nRoutes:
         updateModel( p )
   else:
      if ( ( af == AF_INET and not isValidPrefixWithError( mode, prefix ) ) or 
           ( af == AF_INET6 and not isValidIpv6PrefixWithError( mode, prefix ) ) ):
         return model
      protocolModel = getEffectiveProtocolModel( mode )
      Trie = MribTrie if af == AF_INET else Mrib6Trie
      if protocolModel == ProtoAgentModel.ribd:
         mribTrie = Trie( mribStatus, protocolModel )
      else:
         mribTrie = Trie( mribStatus, protocolModel, mribFwdStatus,
                          mribGenForwardingStatus )
      p = mribTrie.longestMatch( Arnet.IpGenPrefix( str( prefix ) ) )
      updateModel( p.v4Prefix if af == AF_INET else p.v6Prefix )

   return model

def handlerShowIpv4RouteMulticast( mode, args ):
   prefix = args.get( 'PREFIX' )
   vrfName = args.get( 'VRF' )
   return showIpRouteMulticast( mode, prefix=prefix, vrfName=vrfName, af=AF_INET )

def handlerShowIpv6RouteMulticast( mode, args ):
   prefix = args.get( 'PREFIX' )
   vrfName = args.get( 'VRF' )
   return showIpRouteMulticast( mode, prefix=prefix, vrfName=vrfName, af=AF_INET6 )

def Plugin( entityManager ):
   global mrib4StatusColl, mrib6StatusColl
   global _mfib4VrfConfig, _mfib6VrfConfig
   global mribGenForwardingStatus
   mrib4StatusColl = LazyMount.mount(
         entityManager, 'routing/multicast/rib/status',
         'Routing::Multicast::Rib::StatusColl', 'r' )   
   _mfib4VrfConfig = LazyMount.mount( entityManager, 
         'routing/multicast/vrf/config', 
         'Routing::Multicast::Fib::VrfConfig', 'r' )
   mrib6StatusColl = LazyMount.mount(
         entityManager, 'routing6/multicast/rib/status',
         'Routing6::Multicast::Rib::StatusColl', 'r' )   
   _mfib6VrfConfig = LazyMount.mount( entityManager, 
         'routing6/multicast/vrf/config', 
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   if toggleFibGenMountPathEnabled():
      mribGenForwardingStatus = StaticMribShowCli.smashMount.doMount(
         '/routing/mrib/unifiedForwardingStatus/ipGen',
         'Smash::FibGen::ForwardingStatus',
         FibUtils.forwardingStatusInfo( 'reader' ) )
