# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliGlobal
from CliPlugin.VlanCli import (
      Vlan,
      getSwitchInputConfig,
      vlanInUseErrStr,
)
import LazyMount
import Tac
from TypeFuture import TacLazyType

gv = CliGlobal.CliGlobal( bridgingConfig=None )

Dot1qEncapMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )
SplitHorizonGroupName = TacLazyType( "Bridging::SplitHorizonGroupName" )

# The "encapsulation dot1q vlan <vlan-id> [inner <vlan-id>]" command handlers
def setVlanEncap( mode, args ):
   vlanId = args.get( 'VLAN' )
   innerVlanId = args.get( 'INNER_VLAN' )
   if innerVlanId:
      innerVlanId = Vlan( innerVlanId )
   dot1qEncapModeLegacy = Dot1qEncapMode.subIntfDot1qEncapConfigLegacy
   mode.intf.setVlanEncap( mode, dot1qEncapModeLegacy, vlanId, innerVlanId )

def noVlanEncap( mode, args ):
   dot1qEncapModeLegacy = Dot1qEncapMode.subIntfDot1qEncapConfigLegacy
   mode.intf.setVlanEncap( mode, dot1qEncapModeLegacy )

# "[ no | default ] mac address learning disabled" command handlers
def disableMacLearning( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   if not sic:
      return
   sic.macLearningEnabled = False

def defaultMacLearning( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   if not sic:
      return
   sic.macLearningEnabled = True

# " [no] vlan id <vlan-id>" command handlers
def noL2SubIntfVlan( sic ):
   sic.l2SubIntfVlan = Tac.Value( 'Bridging::VlanNameOrId', 'none', '', 0 )
   sic.enabled = False # Disable bridging

def setL2SubIntfVlan( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   sic = getSwitchInputConfig( mode.intf.name )
   vname = args.get( 'VLAN_NAME' )
   vlanId = args.get( 'VLAN_ID' )

   if not sic:
      return

   if no:
      if ( args.get( 'name' ) and sic.l2SubIntfVlan.nameOrId == 'name' ) or \
         ( args.get( 'id' ) and sic.l2SubIntfVlan.nameOrId == 'id' ):
         noL2SubIntfVlan( sic )
      return
   elif vname:
      # Check if VLAN with given name exists. If not, show a warning but still
      # accept it.
      vlans = Vlan.getAll( mode )
      vlans = [ vlan for vlan in vlans if vlan.name( mode ) == vname ]
      if not vlans:
         mode.addWarning( f'VLAN with name { vname } does not exist' )

      vlanNameOrId = Tac.Value( 'Bridging::VlanNameOrId', 'name', vname, 0 )
   elif vlanId:
      # Is this VLAN ID an internal VLAN? If so, don't accept in interactive
      # session only.
      vc = gv.bridgingConfig.vlanConfig.get( vlanId.id )
      if vc and vc.internal and mode.session_.interactive_:
         mode.addError( vlanInUseErrStr( vlanId.id ) )
         return

      # Does this VLAN ID already exist? If not, create it now.
      if not vlanId.lookupConfig( mode ) and mode.session_.interactive_:
         mode.addWarning( f'VLAN does not exist. Creating vlan { vlanId.id }' )
         vlanId.create( mode )

      vlanNameOrId = Tac.Value( 'Bridging::VlanNameOrId', 'id', '', vlanId.id )
   else:
      assert 0, 'nameOrId error'

   sic.l2SubIntfVlan = vlanNameOrId
   sic.enabled = True # Enable bridging

# "[no] split-horizon group <shg-name>" command handlers
def noL2SubIntfSplitHorizonGroup( sic ):
   sic.splitHorizonGroup = SplitHorizonGroupName.defaultSplitHorizonGroup

def setL2SubIntfSplitHorizonGroup( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   sic = getSwitchInputConfig( mode.intf.name )
   shgName = args.get( 'GROUP' )

   if not sic:
      return

   if no:
      noL2SubIntfSplitHorizonGroup( sic )
      return

   assert shgName, 'No name provided'
   if shgName == SplitHorizonGroupName.defaultSplitHorizonGroup:
      mode.addError( 'Split horizon group name cannot be "default"' )
      return

   sic.splitHorizonGroup = shgName

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
