#
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Float, Model, Str, Submodel
from CliPlugin import SystemdModels

SystemdUnitState = SystemdModels.SystemdUnitState

class SystemdUnitStatus( Model ):
   unitState = SystemdUnitState.unitActiveState
   unitDesc = Str( help='Brief description of unit' )

class SystemdStatusBrief( Model ):

   unitsStatus = Dict( keyType=str, valueType=SystemdUnitStatus,
                       help='Mapping of a systemd unit to its status' )

   def render( self ):
      for k in self.unitsStatus:
         v = self.unitsStatus[ k ]
         print( f'{k:<34.34s} {v.unitState:<10s} {v.unitDesc:s}' )

class SystemdMountpointInfo( Model ):
   mountDesc = Str( help='Mount point description' )
   mountState = SystemdUnitState.unitActiveState
   mountPath = Str( help='Where is the mountpoint' )
   fsType = Str( help='Filesystem type' )
   mountTime = Float( help="Timestamp of when unit reached current state",
                      optional=True )

class SystemdMountpointsDetail( Model ):
   systemdMountpointsInfo = Dict( keyType=str, valueType=SystemdMountpointInfo,
                                  help='Mapping of a systemd mount to detailed'
                                       'description of its state' )

   def render( self ):
      keys = sorted( self.systemdMountpointsInfo.keys() )
      for m in keys:
         v = self.systemdMountpointsInfo[ m ]
         print( "       Name:", m )
         print( "Description:", v.mountDesc )
         print( "     Active:", end=' ' )
         timeStr = SystemdModels.timeToStr( v.mountTime )
         print( f"{v.mountState:<15s} {timeStr:s}" )
         print( "      Where:", v.mountPath )
         print( "       What:", v.fsType )
         print()

class SystemdAllUnitsDetail( Model ):
   systemdServices = Submodel( valueType=SystemdModels.SystemdServicesDetail,
                           help='Systemd managed services information' )
   systemdSockets = Submodel( valueType=SystemdModels.SystemdSocketsDetail,
                           help='Systemd managed sockets information' )
   systemdMounts = Submodel( valueType=SystemdMountpointsDetail,
                           help='Systemd aware mount points' )

   def render( self ):
      self.systemdServices.render()
      self.systemdSockets.render()
      self.systemdMounts.render()
