# Copyright (c) 2008-2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# -------------------------------------------------------------------------------
# This module implements TACACS+ configuration.
#
# In enable mode:
#
#     show tacacs
#     clear aaa counters tacacs
#
# In config mode:
#
#     [no] tacacs-server key [0] <key-text>
#     [no] tacacs-server timeout <1-1000>
#     [no] tacacs-server policy unknown-mandatory-attribute ignore
#     [no] tacacs-server host <ip-addr-or-hostname> [single-connection]
#           [port <1-65535>] [timeout <1-1000>] [key [0 | 7] <key-text>]
#     [no] tacacs-server qos dscp <0-63>
#     [no] tacacs-server username max-length <1-255>
#
# Child mode of config mode:
#
#     aaa group server tacacs+ <server-group-name>
#        [no] server <ip-addr-or-hostname> [port <1-65535>]
#
#
# In global IP config mode:
#     [no] ip tacacs source-interace <interface-name>
# -------------------------------------------------------------------------------
import Cell
import CliDynamicSymbol
import CliGlobal
from CliPlugin.VrfCli import DEFAULT_VRF, DEFAULT_VRF_OLD
import AaaCliLib
from AaaPluginLib import hostProtocol
import Ark
import ConfigMount
import DscpCliLib
import HostnameCli
import Intf.Log
import LazyMount
import ReversibleSecretCli
import Tac
import Tacacs
import TacacsGroup

TacacsModel = CliDynamicSymbol.CliDynamicPlugin( "TacacsModel" )

gv = CliGlobal.CliGlobal( tacacsConfig=None,
                          tacacsCounterConfig=None,
                          tacacsStatus=None,
                          aaaConfig=None,
                          dscpConfig=None )

def tacacsHost( mode, hostname, port, vrf, create=False ):
   hosts = gv.tacacsConfig.host
   assert vrf and vrf != ''
   spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                     acctPort=0, vrf=vrf, protocol=hostProtocol.protoTacacs )
   if spec in hosts:
      host = hosts[ spec ]
   elif create:
      host = hosts.newMember( spec )
      host.index = AaaCliLib.getHostIndex( hosts )
   else:
      host = None
   if host is not None:
      assert host.hostname == hostname
      assert host.port == port
   return host

# -------------------------------------------------------------------------------
# "show tacacs" in enable mode
# -------------------------------------------------------------------------------
def showTacacsHost( host, counters ):
   assert host.vrf != ''
   if not counters:
      counters = Tac.Value( "Tacacs::Counters" )
   ret1 = TacacsModel.TacacsStats()
   ret1.serverInfo = TacacsModel.ServerInfo( hostname=host.hostname,
                                             authport=host.port )
   if host.vrf != DEFAULT_VRF:
      ret1.serverInfo.vrf = host.vrf
   ret1.connectionOpens = counters.connOpens
   ret1.connectionCloses = counters.connCloses
   ret1.connectionDisconnects = counters.connDisconnects
   ret1.connectionFailures = counters.connFails
   ret1.connectionTimeouts = counters.connTimeouts
   ret1.messagesSent = counters.messagesSent
   ret1.messagesReceived = counters.messagesReceived
   ret1.receiveErrors = counters.receiveErrors
   ret1.receiveTimeouts = counters.receiveTimeouts
   ret1.sendTimeouts = counters.sendTimeouts
   ret1.dnsErrors = counters.dnsErrors
   ret1.unknownMandatoryAttrIgnored = counters.unknownMandatoryAttrIgnored
   ret1.unknownMandatoryAttrFailures = counters.unknownMandatoryAttrFailures
   return ret1

def showTacacs( mode, args ):
   ret = TacacsModel.ShowTacacs()
   for h in sorted( gv.tacacsConfig.host.values(), key=lambda host: host.index ):
      ret.tacacsServers.append(
         showTacacsHost( h, gv.tacacsStatus.counter.get( h.spec ) ) )
   ret.srcIntf = dict( gv.tacacsConfig.srcIntfName )
   for k in sorted( gv.aaaConfig.hostgroup.keys() ):
      g = gv.aaaConfig.hostgroup[ k ]
      if g.groupType == 'tacacs':
         serverGroupDisplay = AaaCliLib.getCliDisplayFromGroup( g.groupType )
         serverGroupName = g.name
         ret.groups[ serverGroupName ] = TacacsModel.ServerGroup()
         ret.groups[ serverGroupName ].serverGroup = serverGroupDisplay
         for m in g.member.values():
            ret2 = TacacsModel.ServerInfo()
            ret2.hostname = m.spec.hostname
            ret2.authport = m.spec.port
            if m.spec.vrf != DEFAULT_VRF:
               ret2.vrf = m.spec.vrf
            ret.groups[ serverGroupName ].members.append( ret2 )
   ret.lastCounterClearTime = Ark.switchTimeToUtc( gv.tacacsStatus.lastClearTime )
   return ret

# -------------------------------------------------------------------------------
# "clear aaa counters tacacs" in enable mode
# -------------------------------------------------------------------------------
def clearCounters( mode, args ):
   Intf.Log.logClearCounters( "tacacs" )
   gv.tacacsCounterConfig.clearCounterRequestTime = Tac.now()
   try:
      Tac.waitFor( lambda: gv.tacacsStatus.lastClearTime >=
                   gv.tacacsCounterConfig.clearCounterRequestTime,
                   description='Tacacs clear counter request to complete',
                   warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning(
         "TACACS counters may not have been reset yet" )

# -------------------------------------------------------------------------------
# "[no] tacacs-server host <ip-addr-or-hostname> [single-connection]
#     [port <1-65535>] [timeout <1-1000>] [key [0 | 7] <key-text>]"
# in config mode
# -------------------------------------------------------------------------------
def _getArg( args, name, defaultValue ):
   # turn list into single-elements for args in a set
   arg = args.get( name, defaultValue )
   if arg and isinstance( arg, list ):
      assert len( arg ) == 1
      return arg[ 0 ]
   return arg

def setTacacsServerHost( mode, args ):
   hostname = args[ '<HOSTNAME>' ]
   singleConn = _getArg( args, 'single-connection', False )
   vrf = _getArg( args, 'VRF', DEFAULT_VRF )
   port = _getArg( args, '<PORT>', TacacsGroup.defaultPort )
   timeout = _getArg( args, '<TIMEOUT>', None )
   key = args.get( '<KEY>' )
   HostnameCli.resolveHostname( mode, hostname, doWarn=True )
   if timeout is None:
      timeoutVal = 1
   else:
      timeoutVal = int( timeout )
   assert vrf != ''
   host = tacacsHost( mode, hostname, port, vrf, create=True )
   host.useKey = ( key is not None )
   host.key = key or ReversibleSecretCli.getDefaultSecret()
   host.useTimeout = ( timeout is not None )
   host.timeout = timeoutVal
   if singleConn:
      host.connType = 'singleConnection'
   else:
      host.connType = 'multiConnection'
   if mode.session_.interactive_:
      vrfString = f" in vrf {vrf}" if vrf != DEFAULT_VRF else ""
      mode.addMessage( f"TACACS+ host {hostname} with port {port} created"
                       f"{vrfString}" )
   updateDscpRules()

def noTacacsServerHost( mode, args ):
   hostname = args.get( '<HOSTNAME>' )
   hosts = gv.tacacsConfig.host
   if hostname:
      vrf = args.get( 'VRF', DEFAULT_VRF )
      port = args.get( '<PORT>', TacacsGroup.defaultPort )
      spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                        acctPort=0, vrf=vrf, protocol=hostProtocol.protoTacacs )
      if spec in hosts:
         del hosts[ spec ]
      else:
         if mode.session_.interactive_:
            warningMessage = f"TACACS+ host {hostname} with port {port} not found"
            mode.addWarning( warningMessage )
   else:
      # Delete all hosts since no hostname was specified
      hosts.clear()
   updateDscpRules()

# -------------------------------------------------------------------------------
# "[no] ip tacacs source-interface <interface-name>"
# global command
# -------------------------------------------------------------------------------
def setTacacsSrcIntf( mode, args ):
   intf = args[ "INTF" ]
   vrf = args.get( "VRF" )
   if not vrf or vrf == DEFAULT_VRF_OLD:
      vrf = DEFAULT_VRF
   assert vrf
   gv.tacacsConfig.srcIntfName[ vrf ] = intf.name

def noTacacsSrvIntf( mode, args ):
   vrf = args.get( "VRF" )
   if not vrf or vrf == DEFAULT_VRF_OLD:
      vrf = DEFAULT_VRF
   assert vrf
   del gv.tacacsConfig.srcIntfName[ vrf ]

# -------------------------------------------------------------------------------
# "[no] tacacs-server qos dscp <0-63>" in config mode
# -------------------------------------------------------------------------------

def updateDscpRules():
   dscpValue = gv.tacacsConfig.dscpValue

   if not dscpValue:
      del gv.dscpConfig.protoConfig[ 'tacacs' ]
      return

   protoConfig = gv.dscpConfig.newProtoConfig( 'tacacs' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for spec in gv.tacacsConfig.host:
      # Traffic connecting to external tacacs server auth.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'tcp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'tcp', dscpValue, v6=True )

      # Traffic connecting to internal tacacs server acct.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.acctPort, False, spec.vrf,
                              'tcp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.acctPort, False, spec.vrf,
                              'tcp', dscpValue, v6=True )

def setDscp( mode, args ):
   gv.tacacsConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   gv.tacacsConfig.dscpValue = gv.tacacsConfig.dscpValueDefault
   updateDscpRules()

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.aaaConfig = LazyMount.mount( entityManager, "security/aaa/config",
                                   "Aaa::Config", "r" )
   gv.tacacsConfig = ConfigMount.mount( entityManager, "security/aaa/tacacs/config",
                                        "Tacacs::Config", "w" )
   gv.tacacsCounterConfig = LazyMount.mount( entityManager,
                                             "security/aaa/tacacs/counterConfig",
                                             "AaaPlugin::CounterConfig", "w" )
   gv.tacacsStatus = LazyMount.mount( entityManager,
                                      Cell.path( "security/aaa/tacacs/status" ),
                                      "Tacacs::Status", "r" )
   gv.dscpConfig = ConfigMount.mount( entityManager, "mgmt/dscp/config",
                                      "Mgmt::Dscp::Config", "w" )
