# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
from CliModel import Dict
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import List
from CliModel import Float
from CliModel import Submodel
from IntfModels import Interface
import Tac

class ServerInfo( Model ):
   hostname = Str( help="IP address or hostname of the TACACS+ server" )
   authport = Int( help="Specifies server authorization port number" )
   vrf = Str( help="The VRF over which this server is reachable", optional=True )

class TacacsStats( Model ):
   serverInfo = Submodel( valueType=ServerInfo, help="Configured server "
                                                     "information" )
   connectionOpens = Int( help="Number of successful TCP connections to the "
                               "TACACS+ server" )
   connectionCloses = Int( help="Number of successfully closed TCP connections" )
   connectionDisconnects = Int( help="Number of disconnected TCP connections "
                                     "to the TACACS+ server" )
   connectionFailures = Int( help="Number of failed TCP connections to the "
                                  "TACACS+ server" )
   connectionTimeouts = Int( help="Number of timedout TCP connections to "
                                  "the TACACS+ server" )
   messagesSent = Int( help="Number of packets sent to the TACACS+ server" )
   messagesReceived = Int( help="Number of packets received from the TACACS+ "
                                "server" )
   receiveErrors = Int( help="Number of errors received from the TACACS+ server" )
   receiveTimeouts = Int( help="Number of timeouts received from the "
                               "TACACS+ server" )
   sendTimeouts = Int( help="Number of timeouts sent to the TACACS+ server" )
   dnsErrors = Int( help="Number of DNS errors by the TACACS+ server" )
   unknownMandatoryAttrIgnored = Int( help="Number of unknown mandatory attributes "
                                           "received from the TACACS+ server which "
                                           "got ignored" )
   unknownMandatoryAttrFailures = Int( help="Number of unknown mandatory "
                                            "attributes received from the TACACS+ "
                                            "server which failed the "
                                            "authorizations" )

class ServerGroup( Model ):
   serverGroup = Str( help="Configured server-group" )
   members = List( valueType=ServerInfo,
         help="Mapping of server IP address or hostname to group information" )

class ShowTacacs( Model ):
   tacacsServers = List( valueType=TacacsStats, help="Mapping of server "
         "IP address or hostname to server information" )
   groups = Dict( keyType=str, valueType=ServerGroup, help="Mapping of group name "
         "to group information", optional=True )
   srcIntf = Dict( valueType=Interface, help="Configured source-interface name" )
   lastCounterClearTime = Float( help="UTC time when the counters were "
                                      "last cleared" )

   def _getVrfString( self, vrf ):
      return f" (vrf {vrf})" if vrf else ""

   def render( self ):
      template = "%26s:%11d"
      for value in self.tacacsServers:
         print( f"TACACS+ server            : {value.serverInfo.hostname}/"
                f"{value.serverInfo.authport}"
                f"{self._getVrfString( value.serverInfo.vrf )}" )
         print( template % ( "Connection opens", value.connectionOpens ) )
         print( template % ( "Connection closes", value.connectionCloses ) )
         print( template % ( "Connection disconnects",
                             value.connectionDisconnects ) )
         print( template % ( "Connection failures", value.connectionFailures ) )
         print( template % ( "Connection timeouts", value.connectionTimeouts ) )
         print( template % ( "Messages sent", value.messagesSent ) )
         print( template % ( "Messages received", value.messagesReceived ) )
         print( template % ( "Receive errors", value.receiveErrors ) )
         print( template % ( "Receive timeouts", value.receiveTimeouts ) )
         print( template % ( "Send timeouts", value.sendTimeouts ) )
         print( template % ( "DNS errors", value.dnsErrors ) )
         print( template % ( "Unknown attribute ignored",
                             value.unknownMandatoryAttrIgnored ) )
         print( template % ( "Unknown attribute failures",
                             value.unknownMandatoryAttrFailures ) )
         print()

      for serverGroupName, tacacsGroup in self.groups.items():
         print( f"{tacacsGroup.serverGroup} server-group: {serverGroupName}" )
         for sequence, groupDetails in enumerate( tacacsGroup.members ):
            print( f"    {sequence:2d}: {groupDetails.hostname}/"
                   f"{groupDetails.authport} "
                   f"{self._getVrfString( groupDetails.vrf )}" )
         if not tacacsGroup.members:
            print( "    (server-group is empty)" )
         print()
      print( "Last time counters were cleared:",
             Ark.utcTimeRelativeToNowStr( self.lastCounterClearTime ) )

      if self.srcIntf:
         for vrf in sorted( self.srcIntf.keys() ):
            print( "\nTACACS+ source-interface: outgoing packets will be "
                   "sourced with an IP address "
                   "associated with interface as follows:\n" )
            print( "\tVRF \tsource-interface" )
            print( "\t--- \t----------------" )
            print( f"\t{vrf}\t{self.srcIntf[ vrf ]}" )
