#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

import Cell
import CliCommand
import CliParser
import ConfigMount
from CliPlugin.IpGenAddrMatcher import isValidTunnelEndpointAddress
from CliPlugin.TapAggIntfCli import (
      addTapGroups, removeTapGroups, addTapRawIntf, removeTapRawIntf,
      addTunnelTapIntf, removeTunnelTapIntf
)
from CliPlugin.TunnelIntfModel import L3IntfTunnelTableEntry, GreIntfTunnelTable
from CliPlugin.TunnelModels import IpVia
from CliPlugin.VrfCli import DEFAULT_VRF, vrfExists
from CliPlugin.TunnelCli import (
      TunnelTableIdentifier,
      getTunnelIndexFromId,
      readMountTunnelTable,
)
from CliPlugin.TunnelCli import getTunnelIdFromIndex
from CliPlugin.TunnelIntfCli import ipsecHook
import LazyMount
import SharedMem
import Smash
import Tac
import Tracing
import TunnelIntfUtil
from Arnet import IpGenAddr
from Arnet import IntfId
from Arnet.NsLib import DEFAULT_NS

em = None

tunModeEnum = Tac.Type( 'Arnet::TunnelIntfMode' )
afEnum = Tac.Type( 'Arnet::AddressFamily' )

# Maps Eos tunnel (mode, address-family) to Linux tunnel (mode, device)
kernelTunnelAttributesMap = {
   ( tunModeEnum.tunnelIntfModeGre, afEnum.ipv4 )  : ( 'gre', 'gre0' ),
   ( tunModeEnum.tunnelIntfModeGre, afEnum.ipv6 )  : ( 'ip6gre', 'ip6gre0' ),
   ( tunModeEnum.tunnelIntfModeIpip, afEnum.ipv4 ) : ( 'ipip', 'tunl0' ),
   ( tunModeEnum.tunnelIntfModeIpip, afEnum.ipv6 ) : ( 'ipip6', 'ip6tnl0' ),
   ( tunModeEnum.tunnelIntfModeIpsec, afEnum.ipv4 ) : ( 'vti', 'ip_vti0' ) }

ip4ConfigDir = None
ip6ConfigDir = None
tunIntfConfigDir = None
tunIntfStatusDir = None
allIntfNsConfigDir = None
hwCapabilitiesCommon = None
hwCapabilities = None
greIntfTunnelTable = None
hw6Capabilities = None
tunnelCountersDir = None
tunnelCountersCheckpointDir = None
aleCliConfig = None
tunnelIntfCliConfig = None
bridgingHwCapabilities = None
internalVlanStatus = None

U32_MAX_VALUE =  0xFFFFFFFF

# Add trace filter class to the Cli agent for tunnel interface commands
__defaultTraceHandle__ = Tracing.Handle( 'TunnelIntfCli' )
t0 = Tracing.trace0

# Returns true if tunnel interface can handle a 'vrf' change
def intfCanSetVrfHookForTunnel( intfName, oldVrf, newVrf, vrfDelete ):
   t0( 'CanSetVrfHook callback for interface', intfName, 'vrf', newVrf )
   cfg = tunIntfConfigDir.intfConfig.get( intfName )
   if cfg is None:
      return ( True, None )
   ( alias, errMsg ) = findTunnelConfig( cfg.mode, cfg.srcAddr, cfg.srcIntf,
                                         cfg.dstAddr, cfg, cfg.tunnelNs,
                                         cfg.key )
   if alias:
      return ( True, errMsg )

   return ( True, None )

def getHardwareSupportedModes():
   encapModes = []
   if hwCapabilities and hwCapabilities.staticInterfaceTunnelSupport:
      encapModes = list( hwCapabilities.staticInterfaceTunnelSupport )
   if hw6Capabilities and hw6Capabilities.staticInterfaceTunnelSupport:
      encapModes += list( hw6Capabilities.staticInterfaceTunnelSupport )
   return encapModes

def getDefaultEncapMode():
   encapModes = getHardwareSupportedModes()
   if encapModes and tunModeEnum.tunnelIntfModeGre not in encapModes:
      return encapModes[0]

   return tunModeEnum.tunnelIntfModeGre

def checkTunnelMode( tunMode ):
   if not tunMode or tunMode == tunModeEnum.tunnelIntfModeUnspecified:
      tunMode = getDefaultEncapMode()
   return tunMode

# Indicates if a mode and endpoint address are compatible.
def isValidTunnelModeContext( cfg, tunMode=None, tunAddr=None, tunKey=None ):
   assert cfg and ( tunMode or tunAddr or tunKey )
   if not tunAddr:
      if not cfg.srcAddr.isUnspecified:
         tunAddr = cfg.srcAddr
      elif not cfg.dstAddr.isUnspecified:
         tunAddr = cfg.dstAddr
   if not tunMode:
      tunMode = cfg.mode
   tunMode = checkTunnelMode( tunMode )

   # check for trying to use mode supported only over IPv4
   if tunMode == tunModeEnum.tunnelIntfModeIpsec:
      if tunAddr and ( not tunAddr.isUnspecified ) and \
         ( tunAddr.af != afEnum.ipv4 ):
         return ( False, 'IPsec over' + tunAddr.af + ' is unsupported' )

   if tunMode == tunModeEnum.tunnelIntfModeIpip:
      if tunKey or cfg.key:
         return ( False, "IPIP does not support keying" )

   return ( True, '' )

# Returns true if srcAddr/srcIntf matches srcAddr/srcIntf in cfg, else False.
# When comparing source address with source interface, returns False.
def srcCmp( cfg, srcAddr, srcIntf, dstAddr ):
   if ( srcAddr and not srcAddr.isUnspecified and cfg.srcAddr and
        not cfg.srcAddr.isUnspecified ):
      return ( srcAddr == cfg.srcAddr ) # pylint: disable=superfluous-parens
   if ( srcIntf and cfg.srcIntf ):
      return ( srcIntf == cfg.srcIntf ) # pylint: disable=superfluous-parens
   return False

# Returns the tunnel config that matches the supplied keys
def findTunnelConfig( mode, srcAddr, srcIntf, dstAddr, cfgToSkip,
                      tunnelNs=DEFAULT_NS, key=TunnelIntfUtil.tunnelDefaultKey,
                      ipsecProfile='' ):
   if not ( mode and ( srcAddr or srcIntf ) and dstAddr ):
      return ( None, '' )
   if ( not srcIntf and srcAddr.isUnspecified ) or dstAddr.isUnspecified:
      return ( None, '' )
   for cfg in tunIntfConfigDir.intfConfig.values():
      if cfgToSkip and ( cfg == cfgToSkip ):
         continue
      cfgMode = checkTunnelMode( cfg.mode )
      cmpMode = checkTunnelMode( mode )
      srcMatch = srcCmp( cfg, srcAddr, srcIntf, dstAddr )

      #Disallow config if:
      # 1. While configuring Ipsec tunnel, there exists a GRE tunnel with the
      #    same src/dst/tunnelNs
      # 2. While configuring a GRE tunnel, there exists an Ipsec tunnel with the
      #    same src/dst/tunnelNs
      # 3. While configuring an Ipsec tunnel, there exists an Ipsec tunnel with the
      #    same src/dst/tunnelNs
      # Note: This only applies to Ipsec tunnels. Multiple GRE tunnels with the same
      #       same src/dst/tunnelNs are allowed, if the key is different.
      if cmpMode == tunModeEnum.tunnelIntfModeIpsec and \
         ( cfgMode == tunModeEnum.tunnelIntfModeGre or \
         cfgMode == tunModeEnum.tunnelIntfModeIpsec ) and \
         dstAddr == cfg.dstAddr and srcMatch and \
         cfg.tunnelNs == tunnelNs:
         errString = "Conflicts with tunnel %s with same source and destination." \
                     % ( cfg.intfId )
         return ( cfg, errString )

      if cmpMode == tunModeEnum.tunnelIntfModeGre and \
         cfgMode == tunModeEnum.tunnelIntfModeIpsec and \
         dstAddr == cfg.dstAddr and srcMatch and \
         cfg.tunnelNs == tunnelNs:
         errString = "Conflicts with tunnel %s with same source and destination." \
                     % ( cfg.intfId )
         return ( cfg, errString )

      #If for 2 tunnels, the src, dest tunnelNs match; the ipsec profiles
      #have to match too.
      if ipsecProfile != '' and \
         dstAddr == cfg.dstAddr and srcMatch and \
         cfg.tunnelNs == tunnelNs and cfg.ipsecProfile != '' and \
         cfg.ipsecProfile != ipsecProfile:
         errString = "Ipsec profile mismatch between %s and %s" \
                     % ( cfg.intfId, cfgToSkip.intfId )
         return ( cfg, errString )
      # If for 2 tunnels, all 4 of ( src, dest, key, mode ) match
      # but they use different tunnelNs then it's considred unique.
      # BUT if they use same tunnelNs and even if they are in different
      # nsName (VRFs), then this is a DUPLICATE config because the kernel
      # needs to use the same tunnelNs for packet I/O in/out of the box
      if cfgMode != cmpMode or cfg.dstAddr != dstAddr or ( not srcMatch ) \
         or cfg.key != key or cfg.tunnelNs != tunnelNs:
         continue
      return ( cfg, 'Conflicts with ' + cfg.intfId )

   return ( None, '' )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global em
   em = entityManager

   global tunIntfConfigDir
   tunIntfConfigDir = ConfigMount.mount(
         entityManager, 'interface/config/tunnel/intf',
         'Interface::TunnelIntfConfigDir', 'w' )

   global ip4ConfigDir
   ip4ConfigDir = LazyMount.mount(
         entityManager, "ip/config", "Ip::Config", "r" )

   global ip6ConfigDir
   ip6ConfigDir = LazyMount.mount(
         entityManager, "ip6/config", "Ip6::Config", "r" )

   global tunIntfStatusDir
   tunIntfStatusDir = LazyMount.mount(
         entityManager, 'interface/status/tunnel/intf',
         'Interface::TunnelIntfStatusDir', 'r' )

   global allIntfNsConfigDir
   allIntfNsConfigDir = LazyMount.mount(
         entityManager, Cell.path( 'interface/nsconfig' ),
         'Interface::AllIntfNamespaceConfigDir', 'r' )

   global hwCapabilities
   hwCapabilities = LazyMount.mount(
         entityManager, 'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )

   global hwCapabilitiesCommon
   hwCapabilitiesCommon = LazyMount.mount(
         entityManager, 'routing/hardware/statuscommon',
         'Routing::Hardware::StatusCommon', 'r' )

   global internalVlanStatus
   internalVlanStatus = LazyMount.mount(
         entityManager, 'bridging/internalvlan/status',
         'Bridging::InternalVlanStatusDir', 'r' )

   global greIntfTunnelTable
   greIntfTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.greTunnelInterfaceTunnelTable, entityManager )

   global hw6Capabilities
   hw6Capabilities = LazyMount.mount(
         entityManager, 'routing6/hardware/status',
         'Routing6::Hardware::Status', 'r' )

   sMountGroup = SharedMem.entityManager( sysdbEm=em )
   sMountInfo = Smash.mountInfo( 'reader' )
   global tunnelCountersDir
   tunnelCountersDir = sMountGroup.doMount(
      'interface/counter/tunnel/fastpath/current',
      'Smash::Interface::AllIntfCounterDir', sMountInfo )

   global tunnelCountersCheckpointDir
   tunnelCountersCheckpointDir = sMountGroup.doMount(
      'interface/counter/tunnel/fastpath/snapshot',
      'Smash::Interface::AllIntfCounterSnapshotDir', sMountInfo )

   global aleCliConfig
   aleCliConfig = LazyMount.mount( entityManager,
         'hardware/ale/cliconfig',
         'Ale::HwCliConfig', 'r' )

   global tunnelIntfCliConfig
   tunnelIntfCliConfig = LazyMount.mount( entityManager,
         'tunnelintf/cliconfig',
         'TunnelIntf::TunnelIntfCliConfig', 'w' )

   global bridgingHwCapabilities
   bridgingHwCapabilities = LazyMount.mount(
         entityManager, 'bridging/hwcapabilities',
         'Bridging::HwCapabilities', 'r' )
   
#-------------------------------------------------------------------------------
# Cli keyword handler functions
#-------------------------------------------------------------------------------
def setSrcAddr( cliMode, args ):
   ipGenAddr = args[ 'IPGENADDR' ]
   errString = isValidTunnelEndpointAddress( ipGenAddr )
   if errString:
      cliMode.addError( errString )
      return
   tun = cliMode.intf
   cfg = tun.config()
   if not cfg.dstAddr.isUnspecified:
      if ipGenAddr.af != cfg.dstAddr.af:
         cliMode.addError( 'Must be same IP version as destination' )
         return
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunAddr=ipGenAddr )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.srcIntf = ''
   cfg.srcAddr = ipGenAddr
   t0( tun.name, 'source =', ipGenAddr )

def unsetSrc( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.srcIntf = ''
   cfg.srcAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
   t0( tun.name, 'source unset' )

def setSrcIntf( cliMode, args ):
   srcIntf = args[ 'INTF' ]
   tun = cliMode.intf
   cfg = tun.config()
   cfg.srcAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
   cfg.srcIntf = srcIntf.name
   t0( tun.name, 'source intf =', srcIntf.name )

def setDst( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()

   hostOrAddr = args[ 'HOST_OR_ADDR' ]
   if isinstance( hostOrAddr, str ):
      cfg.dstHost = hostOrAddr
      cfg.dstAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
      t0( tun.name, 'destination =', hostOrAddr )
      return
   ipGenAddr = hostOrAddr

   errString = isValidTunnelEndpointAddress( ipGenAddr )
   if errString:
      cliMode.addError( errString )
      return
   if not cfg.srcAddr.isUnspecified:
      if ipGenAddr.af != cfg.srcAddr.af:
         cliMode.addError( 'Must be same IP version as source' )
         return
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunAddr=ipGenAddr )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.dstAddr = ipGenAddr
   cfg.dstHost = ''
   t0( tun.name, 'destination =', ipGenAddr )

def unsetDst( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.dstAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
   cfg.dstHost = ''
   t0( tun.name, 'destination unset' )

def setMode( cliMode, args ):

   keyToTunModeMap = {
      'gre': tunModeEnum.tunnelIntfModeGre,
      'ipip': tunModeEnum.tunnelIntfModeIpip,
      'ipsec': tunModeEnum.tunnelIntfModeIpsec
   }

   tunnelType = keyToTunModeMap[ args[ 'TUNNEL_MODE' ] ]
   tun = cliMode.intf
   cfg = tun.config()
   if tunnelType == tunModeEnum.tunnelIntfModeIpip:
      cliMode.addWarning( "IP in IP tunnel encapsulation is unsupported. "
         "Please use GRE encapsulation instead." )
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunMode=tunnelType )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.mode = tunnelType
   t0( tun.name, 'mode set to', tunnelType )

def unsetMode( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   newTunMode = tunModeEnum.tunnelIntfModeUnspecified
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunMode=newTunMode )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.mode = newTunMode
   t0( tun.name, 'mode set to default' )

def setTtl( cliMode, args ):
   tunnelTTL = args[ 'TUNNEL_TTL' ]
   tun = cliMode.intf
   cfg = tun.config()
   if tunnelTTL and not cfg.pathMtuDiscovery:
      if hwCapabilities and hwCapabilities.kernelGreEncapForSlowPath:
         cliMode.addError( 'Path MTU Discovery must be enabled first' )
      return
   cfg.ttl = tunnelTTL
   t0( tun.name, 'ttl set to', tunnelTTL )

def unsetTtl( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.ttl = TunnelIntfUtil.tunnelDefaultTtl
   t0( tun.name, 'ttl unset' )

def setTos( cliMode, args ):
   tunnelTos = args[ 'TUNNEL_TOS' ]
   tun = cliMode.intf
   cfg = tun.config()
   cfg.tos = tunnelTos
   t0( tun.name, 'tos set to', tunnelTos )

def unsetTos( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.tos = TunnelIntfUtil.tunnelDefaultTos
   t0( tun.name, 'tos unset' )

def setPathMtuDiscovery( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.pathMtuDiscovery = True
   t0( tun.name, 'path MTU discovery enabled' )

def unsetPathMtuDiscovery( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   if cfg.ttl and hwCapabilities and \
           hwCapabilities.kernelGreEncapForSlowPath:
      cliMode.addError( 'Incompatible with non-zero TTL' )
      return
   cfg.pathMtuDiscovery = False
   t0( tun.name, 'path MTU discovery disabled' )

def setTunnelDecapDipOnlyMatch( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   if cfg.mode == tunModeEnum.tunnelIntfModeGre or \
         cfg.mode == tunModeEnum.tunnelIntfModeUnspecified:
      cfg.decapDipOnlyMatch = True
      t0( tun.name, 'DIP only match on termination' )
   else:
      cliMode.addError( "DIP only match on termination "
            "is not supported for any mode other than GRE." )
      cfg.decapDipOnlyMatch = False

def unsetTunnelDecapDipOnlyMatch( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.decapDipOnlyMatch = False
   t0( tun.name, 'SIP + DIP match on termination' )

def setIpsec( cliMode, args ):

   profile = args[ 'PROFILE' ]
   tun = cliMode.intf
   cfg = tun.config()
   if not ipsecHook.extensions():
      return
   for hook in ipsecHook.extensions():
      [ val, message ] = hook( profile )
      if not val:
         cliMode.addError( message )
         return

   if cfg.mode == tunModeEnum.tunnelIntfModeGre or \
      tunModeEnum.tunnelIntfModeUnspecified:
      cliMode.addWarning( "IPsec adds an overhead of up to 82 bytes. "
         "Example: A GRE tunnel with an MTU=1476 should be changed to 1394 "
         "when using IPsec." )

   if cfg.ipsecProfile != profile:
      cfg.ipsec = True
      cfg.ipsecProfile = profile

def noSetIpsec( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.ipsec = False
   cfg.ipsecProfile = ""

def setMss( cliMode, args ):
   tunnelMss = args[ 'TUNNEL_MSS' ]
   tun = cliMode.intf
   cfg = tun.config()
   cfg.maxMss = tunnelMss
   t0( tun.name, "MSS set to", cfg.maxMss )

def setKey( cliMode, args ):
   tunnelKey = args[ 'TUNNEL_KEY' ]
   tun = cliMode.intf
   cfg = tun.config()
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunKey = tunnelKey )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.key = tunnelKey
   t0( tun.name, "key set to", tunnelKey )
   if hwCapabilities and not hwCapabilities.greTunnelKeysSupported:
      cliMode.addWarning( "Tunnel key is not supported on this hardware platform" )
   cliMode.addWarning( "Configuring key adds an overhead of 4 bytes. "
         "Example: A GRE tunnel with an MTU=1476 should be changed to 1472 "
         "when key is configured on it." )

def unsetKey( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.key = TunnelIntfUtil.tunnelDefaultKey
   t0( tun.name, "key unset " )

def unsetMss( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.maxMss = TunnelIntfUtil.tunnelDefaultMaxMss
   t0( tun.name, "MSS set to", cfg.maxMss )

def setUnderlayVrf( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()

   underlayVrfName = args.get( 'VRF', DEFAULT_VRF )

   if cliMode.session.skipConfigCheck():
      cfg.underlayVrfName = underlayVrfName
      t0( tun.name, "vrf set to", underlayVrfName )
      return

   if not vrfExists( underlayVrfName ):
      cliMode.addWarning( "VRF %s doesn't exist" % underlayVrfName )
      return

   cfg.underlayVrfName = underlayVrfName
   t0( tun.name, "vrf set to", underlayVrfName )

   if not ( hwCapabilities and
            hwCapabilities.staticTunIntfUnderlayVrfSupported ):
      cliMode.addWarning( "not supported on this hardware platform" )

#-------------------------------------------------------------------------
# The "show gre tunnel static [ TUNNEL-INDEX ]" command
#-------------------------------------------------------------------------
def getL3IntfTunnelEntryModel( tunnelId, tunnelTable ):
   tunnelEntry = tunnelTable.entry.get( tunnelId, None )
   if not tunnelEntry:
      return None
   tunnelIndex = int( getTunnelIndexFromId( tunnelId ) )
   tunnelIntfId = "Tunnel%d" % tunnelIndex
   viaList = list() # pylint: disable=use-list-literal
   if hwCapabilities and \
      hwCapabilities.staticTunIntfInternalVlanNeeded:
      ksp = Tac.Value( 'Bridging::InternalVlanKeySourcePair',
            'staticTunnelIntf',
            IntfId( tunnelIntfId ).intfId,
            tunnelIntfId )
      if internalVlanStatus.extendedStatus.get( ksp, None ):
         internalVlanId = internalVlanStatus.extendedStatus[ ksp ]
         viaModel = IpVia( nexthop=IpGenAddr(),
               interface="Vlan%d" % internalVlanId,
               type='ip' )
         viaList.append( viaModel )
   else:
      for via in tunnelEntry.via.values():
         viaModel = IpVia( nexthop=via.nexthop, interface=via.intfId, type='ip' )
         viaList.append( viaModel )

   return L3IntfTunnelTableEntry( name=tunnelIntfId,
                                  index=tunnelIndex,
                                  srcAddr=tunnelEntry.tunnelInfo.src,
                                  dstAddr=tunnelEntry.tunnelInfo.dst,
                                  vias=viaList )

#--------------------------------------------------------------------------------
# tunnel tos TUNNEL_TOS
#--------------------------------------------------------------------------------
def tunnelIntfTosSupportedGuard( mode, token ):
   if hwCapabilities and not hwCapabilities.staticTunIntfTosSupported:
      return CliParser.guardNotThisPlatform
   return None

tosGuardedKw = CliCommand.guardedKeyword( 'tos',
      helpdesc='Set IP type of service',
      guard=tunnelIntfTosSupportedGuard )

#--------------------------------------------------------------------------------
# tunnel ttl TUNNEL_TTL
#--------------------------------------------------------------------------------
def tunnelIntfTtlSupportedGuard( mode, token ):
   if hwCapabilities and not hwCapabilities.staticTunIntfTtlSupported:
      return CliParser.guardNotThisPlatform
   return None

ttlGuardedKw = CliCommand.guardedKeyword( 'ttl',
      helpdesc='Set time to live',
      guard=tunnelIntfTtlSupportedGuard )

#--------------------------------------------------------------------------------
# tunnel decap source any
#--------------------------------------------------------------------------------
def tunnelDecapDipOnlyMatchSupportedGuard( mode, token ):
   if hwCapabilities and not hwCapabilities.tunnelIntfDecapDipOnlyMatch:
      return CliParser.guardNotThisPlatform
   return None

decapGuardedKw = CliCommand.guardedKeyword( 'decap',
      helpdesc='Configure decapsulation parameter',
      guard=tunnelDecapDipOnlyMatchSupportedGuard )

## Commands for TapAgg GRE tunnel ##
def handleTunnelTapRawIntf( mode, args ):
   t0( "handleTunnelTapRawIntf" )
   tun = mode.intf
   cfg = tun.config()

   addTapRawIntf( mode, args, cfg )

def handleNoTunnelTapRawIntf( mode, args ):
   t0( "handleNoTunnelTapRawIntf" )
   tun = mode.intf
   cfg = tun.config()

   removeTapRawIntf( mode, args, cfg )

def handleTunnelTapGroups( mode, args ):
   t0( "handleTunnelTapGroups" )
   tun = mode.intf
   cfg = tun.config()

   addTapGroups( mode, args, cfg )

def handleNoTunnelTapGroups( mode, args ):
   t0( "handleNoTunnelTapGroups" )
   tun = mode.intf
   cfg = tun.config()

   removeTapGroups( mode, args, cfg )

def handleTapEncapGrePreserve( mode, args ):
   t0( "handleTapEncapGrePreserve" )
   tun = mode.intf
   cfg = tun.config()

   tapEncapGrePreserve = not CliCommand.isNoOrDefaultCmd( args )
   cfg.tapEncapGrePreserve = tapEncapGrePreserve

def handleTapEncapGreProtocol( mode, args ):
   t0( "handleTapEncapGreProtocol" )
   TapGreProtocolTunnel = Tac.Type( 'Interface::TapGreProtocolTunnel' )
   tun = mode.intf
   cfg = tun.config()

   tapGreProtocolTunnel = TapGreProtocolTunnel( args[ 'PROTOCOL' ],
                                featureHeaderLength=args.get( 'LENGTH', 0 ),
                                ethEncapEnabled='ethernet' in args )
   cfg.tapGreProtocolTunnel.addMember( tapGreProtocolTunnel )

def handleNoTapEncapGreProtocol( mode, args ):
   t0( "handleNoTapEncapGreProtocol" )
   tun = mode.intf
   cfg = tun.config()

   if 'PROTOCOL' in args:
      del cfg.tapGreProtocolTunnel[ args[ 'PROTOCOL' ] ]
   else:
      cfg.tapGreProtocolTunnel.clear()

def handleTunnelTapIntf( mode, args ):
   t0( "handleTunnelTapIntf" )
   tun = mode.intf
   cfg = tun.config()
   addTunnelTapIntf( mode, args, cfg )

def handleNoTunnelTapIntf( mode, args ):
   t0( "handleNoTunnelTapIntf" )
   tun = mode.intf
   cfg = tun.config()
   removeTunnelTapIntf( mode, args, cfg )

def ShowGreTunnelStaticCmdHandler( mode, args ):
   if 'TUNNEL-INDEX' in args:
      tunnelIds = [ getTunnelIdFromIndex( 'staticInterfaceTunnel',
                                             args[ 'TUNNEL-INDEX' ] ) ]
   else:
      tunnelIds = greIntfTunnelTable.entry

   greTunnels = {}
   for tunnelId in tunnelIds:
      tunnelEntryModel = getL3IntfTunnelEntryModel( tunnelId, greIntfTunnelTable )
      if tunnelEntryModel:
         greTunnels[ tunnelEntryModel.index ] = tunnelEntryModel
   return GreIntfTunnelTable( greTunnels=greTunnels )
