# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.TwampLight import (
    MonitorTwampMode,
    TwampLightMode,
    TwampLightReflectorProfileMode,
    TwampLightSenderProfileMode,
    TwampLightSenderDefaultsMode,
    TwampLightReflectorDefaultsMode
)
from CliParser import InvalidInputError
from CliPlugin.TwampLightCli import twampLightConfig
from CliPlugin.TwampLightClearCommands import (
   twampLightStatus,
   ClearRequest,
   getIpAddresses )
import Tac
import UtmpDump

TWAMP_LIGHT_PROFILE_REFLECTOR = 0
TWAMP_LIGHT_PROFILE_SENDER = 1

# -----------------------------------------------------------------------------------
# (config)# monitor twamp
# -----------------------------------------------------------------------------------
def handlerMonitorTwampModeCmd( mode, args ):
   childMode = mode.childMode( MonitorTwampMode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerMonitorTwampModeCmd( mode, args ):
   noOrDefaultHandlerTwampLightModeCmd( mode, args )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] twamp-light
# -----------------------------------------------------------------------------------
def handlerTwampLightModeCmd( mode, args ):
   twampLightProfileCmd( mode,
                         { 'NAME': twampLightConfig.defaultReflectorProfileName },
                         TWAMP_LIGHT_PROFILE_REFLECTOR, changeMode=False )
   twampLightConfig.reflectorActive = True
   childMode = mode.childMode( TwampLightMode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerTwampLightModeCmd( mode, args ):
   twampLightConfig.reflectorActive = False
   twampLightConfig.senderProfile.clear()
   twampLightConfig.senderIdMap.clear()
   twampLightConfig.reflectorProfile.clear()
   twampLightConfig.reflectorIdMap.clear()
   noOrDefaultHandlerSenderSourcePortCmd( mode, args )
   noOrDefaultHandlerSenderDestinationPortCmd( mode, args )
   noOrDefaultHandlerReflectorListenPortCmd( mode, args )

# -----------------------------------------------------------------------------------
# (config-twamp-light)# [ no | default ] ( reflector | sender ) profile NAME
# -----------------------------------------------------------------------------------
def getOrCreateProfileConfig( mode, profileName, profileType, profileId ):
   if profileType == TWAMP_LIGHT_PROFILE_REFLECTOR:
      profileConfig = twampLightConfig.reflectorProfile.get( profileName )
      if profileConfig:
         return profileConfig
      return twampLightConfig.reflectorProfile.newMember( profileName, profileId )
   elif profileType == TWAMP_LIGHT_PROFILE_SENDER:
      profileConfig = twampLightConfig.senderProfile.get( profileName )
      if profileConfig:
         return profileConfig
      return twampLightConfig.senderProfile.newMember( profileName, profileId )
   return None

def twampLightProfileCmd( mode, args, profileType, changeMode=True ):
   assert profileType in [ TWAMP_LIGHT_PROFILE_REFLECTOR,
                           TWAMP_LIGHT_PROFILE_SENDER ]
   profileName = args[ 'NAME' ]

   profileStr = ""
   if profileType == TWAMP_LIGHT_PROFILE_REFLECTOR:
      profiles = twampLightConfig.reflectorProfile
      idMap = twampLightConfig.reflectorIdMap
      newMode = TwampLightReflectorProfileMode
      profileStr = "reflector"
   elif profileType == TWAMP_LIGHT_PROFILE_SENDER:
      profiles = twampLightConfig.senderProfile
      idMap = twampLightConfig.senderIdMap
      newMode = TwampLightSenderProfileMode
      profileStr = "sender"

   profileId = idMap.get( profileName )
   if not profileId:
      if len( profiles ) >= twampLightConfig.maxProfilesCount:
         error = f"Maximum number of {profileStr} profiles reached. " +\
            "Delete existing profile to create a new one"
         mode.addError( error )
         return
      for i in range( 1, twampLightConfig.maxProfilesCount + 1 ):
         if i not in idMap.values():
            newId = i
            break
      idMap[ profileName ] = newId
      profileId = idMap[ profileName ]

   profileConfig = getOrCreateProfileConfig( mode, profileName, profileType,
                                             profileId )
   if profileType == TWAMP_LIGHT_PROFILE_SENDER:
      profileConfig.sourcePort = twampLightConfig.senderSourcePort
      profileConfig.reflectorPort = twampLightConfig.senderDestinationPort
   elif profileType == TWAMP_LIGHT_PROFILE_REFLECTOR:
      profileConfig.listenPort = twampLightConfig.reflectorListenPort

   if changeMode:
      childMode = mode.childMode( newMode, instanceConfig=profileConfig )
      mode.session_.gotoChildMode( childMode )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] reflector profile NAME
# -----------------------------------------------------------------------------------
def handlerTwampLightReflectorProfileModeCmd( mode, args ):
   twampLightProfileCmd( mode, args, TWAMP_LIGHT_PROFILE_REFLECTOR )

def noOrDefaultHandlerTwampLightReflectorProfileModeCmd( mode, args ):
   if args[ 'NAME' ] in twampLightConfig.reflectorProfile:
      del twampLightConfig.reflectorIdMap[ args[ 'NAME' ] ]
      del twampLightConfig.reflectorProfile[ args[ 'NAME' ] ]

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] sender profile NAME
# -----------------------------------------------------------------------------------
def handlerTwampLightSenderProfileModeCmd( mode, args ):
   twampLightProfileCmd( mode, args, TWAMP_LIGHT_PROFILE_SENDER )

def noOrDefaultHandlerTwampLightSenderProfileModeCmd( mode, args ):
   if args[ 'NAME' ] in twampLightConfig.senderProfile:
      del twampLightConfig.senderIdMap[ args[ 'NAME' ] ]
      del twampLightConfig.senderProfile[ args[ 'NAME' ] ]

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] sender defaults
# -----------------------------------------------------------------------------------
def handlerTwampLightSenderDefaultsModeCmd( mode, args ):
   childMode = mode.childMode( TwampLightSenderDefaultsMode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerTwampLightSenderDefaultsModeCmd( mode, args ):
   noOrDefaultHandlerSenderSourcePortCmd( mode, args )
   noOrDefaultHandlerSenderDestinationPortCmd( mode, args )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] reflector defaults
# -----------------------------------------------------------------------------------
def handlerTwampLightReflectorDefaultsModeCmd( mode, args ):
   childMode = mode.childMode( TwampLightReflectorDefaultsMode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerTwampLightReflectorDefaultsModeCmd( mode, args ):
   noOrDefaultHandlerReflectorListenPortCmd( mode, args )

# -----------------------------------------------------------------------------------
# (twamp-light-reflector-defaults)# listen port PORT
# -----------------------------------------------------------------------------------
def handlerReflectorListenPortCmd( mode, args ):
   twampLightConfig.reflectorListenPort = args[ 'PORT' ]
   for reflectorProfile in twampLightConfig.reflectorProfile.values():
      reflectorProfile.listenPort = twampLightConfig.reflectorListenPort

def noOrDefaultHandlerReflectorListenPortCmd( mode, args ):
   twampLightConfig.reflectorListenPort = twampLightConfig.reflectorListenPortDefault
   for reflectorProfile in twampLightConfig.reflectorProfile.values():
      reflectorProfile.listenPort = twampLightConfig.reflectorListenPort

# -----------------------------------------------------------------------------------
# (twamp-light-sender-defaults)# source port PORT
# -----------------------------------------------------------------------------------
def handlerSenderSourcePortCmd( mode, args ):
   twampLightConfig.senderSourcePort = args[ 'PORT' ]
   for senderProfile in twampLightConfig.senderProfile.values():
      senderProfile.sourcePort = twampLightConfig.senderSourcePort

def noOrDefaultHandlerSenderSourcePortCmd( mode, args ):
   twampLightConfig.senderSourcePort = twampLightConfig.senderSourcePortDefault
   for senderProfile in twampLightConfig.senderProfile.values():
      senderProfile.sourcePort = twampLightConfig.senderSourcePort

# -----------------------------------------------------------------------------------
# (twamp-light-sender-defaults)# destination port PORT
# -----------------------------------------------------------------------------------
def handlerSenderDestinationPortCmd( mode, args ):
   twampLightConfig.senderDestinationPort = args[ 'PORT' ]
   for senderProfile in twampLightConfig.senderProfile.values():
      senderProfile.reflectorPort = twampLightConfig.senderDestinationPort

def noOrDefaultHandlerSenderDestinationPortCmd( mode, args ):
   twampLightConfig.senderDestinationPort =\
      twampLightConfig.senderDestinationPortDefault
   for senderProfile in twampLightConfig.senderProfile.values():
      senderProfile.reflectorPort = twampLightConfig.senderDestinationPort

# -----------------------------------------------------------------------------------
# (config-twamp-light-sender-profile-NAME)# measurement interval INTERVAL
# -----------------------------------------------------------------------------------
def handlerSenderProfileIntervalCmd( mode, args ):
   senderProfile = twampLightConfig.senderProfile.get( mode.instanceName )
   senderProfile.interval = args[ 'INTERVAL' ]

def noOrDefaultHandlerSenderProfileIntervalCmd( mode, args ):
   senderProfile = twampLightConfig.senderProfile.get( mode.instanceName )
   senderProfile.interval = senderProfile.intervalDefault

# -----------------------------------------------------------------------------------
# (config-twamp-light-sender-profile-NAME)# measurement samples NUMBER
# -----------------------------------------------------------------------------------
def handlerSenderProfileSamplesCmd( mode, args ):
   senderProfile = twampLightConfig.senderProfile.get( mode.instanceName )
   senderProfile.samples = args[ 'NUMBER' ]

def noOrDefaultHandlerSenderProfileSamplesCmd( mode, args ):
   senderProfile = twampLightConfig.senderProfile.get( mode.instanceName )
   senderProfile.samples = senderProfile.samplesDefault

# -----------------------------------------------------------------------------------
# (config-twamp-light-sender-profile-NAME)# significance SIGNIFICANCE offset OFFSET
# -----------------------------------------------------------------------------------
def handlerSenderProfileSignificanceOffsetCmd( mode, args ):
   if args[ 'SIGNIFICANCE' ] <= args[ 'OFFSET' ]:
      raise InvalidInputError

   senderProfile = twampLightConfig.senderProfile.get( mode.instanceName )
   senderProfile.significance = args[ 'SIGNIFICANCE' ]
   senderProfile.offset = args[ 'OFFSET' ]

def noOrDefaultHandlerSenderProfileSignificanceOffsetCmd( mode, args ):
   senderProfile = twampLightConfig.senderProfile.get( mode.instanceName )
   senderProfile.significance = senderProfile.significanceDefault
   senderProfile.offset = senderProfile.offsetDefault

def handlerTwampLightCountersClearCmd( mode, args ):
   for k in twampLightStatus.clearCountersRequestTime:
      del twampLightConfig.clearCountersRequest[ k ]
   info = UtmpDump.getUserInfo()
   who = info[ 'user' ]
   where = info[ 'tty' ]
   timestamp = Tac.now()

   targetType = 'allCounterTypes'
   if 'reflector' in args:
      targetType = 'reflectorCounters'
   elif 'sender' in args:
      targetType = 'senderCounters'

   target = 'all'
   if 'ip' in args:
      target = args[ 'IP' ]

   ipSet = getIpAddresses( mode )
   if target != 'all' and target not in ipSet:
      mode.addError( "Invalid input" )
      return

   reqId = f"{timestamp:f}.{target}"
   twampLightConfig.clearCountersRequest[ reqId ] =\
      ClearRequest( timestamp, who, where, targetType, target )

def handlerTwampLightResultsClearCmd( mode, args ):
   for k in twampLightStatus.clearResultsRequestTime:
      del twampLightConfig.clearResultsRequest[ k ]
   info = UtmpDump.getUserInfo()
   who = info[ 'user' ]
   where = info[ 'tty' ]
   timestamp = Tac.now()

   targetType = 'results'
   target = 'all'
   if 'ip' in args:
      target = args[ 'IP' ].stringValue

   reqId = f"{timestamp:f}.{target}"
   twampLightConfig.clearResultsRequest[ reqId ] =\
      ClearRequest( timestamp, who, where, targetType, target )
