# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin
from ArnetModel import IpAddrAndPort
import LazyMount
import subprocess
import os
from IpLibConsts import DEFAULT_VRF

UpnpCliModels = CliDynamicPlugin( 'UpnpShowCliModels' )

upnpPortMapConfig = None
upnpPortMapStatus = None
upnpConfig = None
upnpStatus = None
ssdpConfig = None
ssdpStatus = None

# Create a dict for enums representing port map state
portMapState = {
      'portMapInitAdd': 'adding',
      'portMapInitDel': 'deleting',
      'portMapCreated': 'created',
      'portMapRefused': 'refused',
      'portMapCleared': 'cleared'
   }

def showUpnpPortMap( mode, args ):
   portMapInfo = UpnpCliModels.UpnpPortMapInfo()

   # Iterate over each vrf and populate the config
   for vrfName, vrfCfg in upnpPortMapConfig.upnpPortMapVrfConfig.items():
      # Create a new collection for this VRF
      portMapList = UpnpCliModels.VrfUpnpPortMap()
      for portMapKey, portMap in vrfCfg.portMapConfig.items():
         portMapEntry = UpnpCliModels.UpnpPortMapEntry()
         portMapEntry.internalIpAndPort = IpAddrAndPort(
               ip=portMap.internalIp, port=portMap.internalPort )
         portMapEntry.externalIpAndPort = IpAddrAndPort(
               ip=portMapKey.externalIp, port=portMapKey.externalPort )
         portMapEntry.protocol = portMapKey.protocol
         portMapEntry.leaseTime = portMap.leaseTime

         # The default state is NA.  But if the status is present,
         # update it with current state.
         if upnpPortMapStatus and vrfName in upnpPortMapStatus.upnpPortMapVrfStatus:
            vrfPortMapStatus = upnpPortMapStatus.upnpPortMapVrfStatus[ vrfName ]
            if portMapKey in vrfPortMapStatus.portMapStatus:
               # Use the string mapped the state enum
               portMapEntry.state = portMapState[
                     vrfPortMapStatus.portMapStatus[ portMapKey ] ]

            # Add the entry to the list
            portMapList.portMaps.append( portMapEntry )

      portMapInfo.vrfs[ vrfName ] = portMapList

   return portMapInfo

# Handler for show upnp service status
def showUpnpService( mode, args ):
   serviceStatus = UpnpCliModels.UpnpServiceInfo()

   testSim = False

   # Is this in a test env
   if os.getenv( 'IGD_SIMULATE_UP' ):
      testSim = True

   def getServiceStatus( vrfName, serviceName ):
      if vrfName != 'default':
         serviceName = serviceName + "-" + vrfName
      try:
         output = subprocess.check_output( [ 'service',
             serviceName, 'status' ] )
         return 'is running' in str( output )
      except subprocess.CalledProcessError:
         return False

   for vrfName, igdVrfConfig in sorted( upnpConfig.igdVrfConfig.items() ):
      entry = UpnpCliModels.UpnpServiceEntry()

      entry.upnpStatus = 'down'

      # If config is disabled, set admin down
      if igdVrfConfig.ipv4State == 'disabled':
         entry.upnpStatus = 'admin down'

      igdVrfStatus = upnpStatus.igdVrfStatus.get( vrfName )
      if igdVrfStatus:
         # If the status is not admin down, update it now
         if entry.upnpStatus != 'admin down':
            # Fill UPnP status
            entry.upnpStatus = 'down'
            if igdVrfStatus.ipv4State == 'enabled':
               # When running in test, the service state won't work.
               if testSim:
                  entry.upnpStatus = 'up'
               elif getServiceStatus( vrfName, 'miniupnpd' ):
                  # sysdb status is up.  Lets confirm service is indeed running
                  entry.upnpStatus = 'up'

      # Similar handling of SSDP status
      vrfSsdpStatus = ssdpConfig.vrfConfig.get( vrfName )
      if vrfSsdpStatus and vrfSsdpStatus.serverState == 'disabled':
         entry.ssdpStatus = 'admin down'
      else:
         entry.ssdpStatus = 'down'
         if vrfName in ssdpStatus.vrfStatus:
            if testSim:
               entry.ssdpStatus = 'up'
            elif getServiceStatus( vrfName, 'minissdpd' ):
               # sysdb status is up.  Lets confirm service is indeed running
               entry.ssdpStatus = 'up'

      serviceStatus.vrfs[ vrfName ] = entry
   return serviceStatus

def showUpnpCounters( mode, args ):
   vrf = args.get( 'VRF' )

   countersModel = UpnpCliModels.UpnpCountersModel()

   if not vrf:
      vrfNames = \
         [ vrfName for vrfName in upnpStatus.igdVrfStatus if vrfName != DEFAULT_VRF ]
      vrfNames.sort()
      if upnpStatus.igdVrfStatus.get( DEFAULT_VRF ):
         vrfNames.insert( 0, DEFAULT_VRF )
   else:
      vrfNames = [ vrf ]

   for vrfName in vrfNames:
      status = upnpStatus.igdVrfStatus.get( vrfName )
      vrfCounters = UpnpCliModels.UpnpCountersModel.VrfCounters()
      vrfCounters.addRequestsTotal = status.counters.addRequestsTotal
      vrfCounters.delRequestsTotal = status.counters.delRequestsTotal
      vrfCounters.getSpecificMappingRequestsTotal = \
            status.counters.getSpecificMappingRequestsTotal
      vrfCounters.addSuccess = status.counters.addSuccess
      vrfCounters.delSuccess = status.counters.delSuccess
      vrfCounters.getSpecificMappingRequestsSuccess = \
            status.counters.getSpecificMappingRequestsSuccess
      vrfCounters.addRefused = status.counters.addRefused
      vrfCounters.addConflicts = status.counters.addConflicts
      vrfCounters.addPrematureSessionClosure = \
            status.counters.addPrematureSessionClosure
      vrfCounters.portMappingClearedLocally = \
            status.counters.portMappingClearedLocally
      vrfCounters.delFailed = status.counters.delFailed
      vrfCounters.getSpecificMappingFailed = \
            status.counters.getSpecificMappingFailed
      countersModel.vrfs[ vrfName ] = vrfCounters
   return countersModel

def Plugin( entityManager ):
   global upnpPortMapConfig, upnpPortMapStatus
   global upnpConfig, upnpStatus, ssdpConfig, ssdpStatus

   upnpPortMapConfig = LazyMount.mount( entityManager,
         'upnp/igd/portMap/config', 'IgdUpnpShared::Config', 'r' )
   upnpPortMapStatus = LazyMount.mount( entityManager,
         'upnp/igd/portMap/status', 'IgdUpnpShared::Status', 'r' )
   upnpConfig = LazyMount.mount( entityManager,
         'upnp/igd/config', 'IgdUpnp::Config', 'r' )
   upnpStatus = LazyMount.mount( entityManager,
         'upnp/igd/status', 'IgdUpnp::Status', 'r' )
   ssdpConfig = LazyMount.mount( entityManager,
         'upnp/ssdp/config', 'Ssdp::Config', 'r' )
   ssdpStatus = LazyMount.mount( entityManager,
         'upnp/ssdp/status', 'Ssdp::Status', 'r' )
