# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Int, List, Submodel, Dict, Float, Enum
from ArnetModel import IpAddrAndPort
import TableOutput
from IpLibConsts import DEFAULT_VRF

def formatRow( minWidth, maxWidth, wrap=False, justify='left', dotAlign=False ):
   fmt = TableOutput.Format( justify=justify, minWidth=minWidth,
                             maxWidth=maxWidth, wrap=wrap, dotAlign=dotAlign )
   fmt.noPadLeftIs( True )
   fmt.padLimitIs( True )
   return fmt

class UpnpPortMapEntry( Model ):
   internalIpAndPort = Submodel( valueType=IpAddrAndPort,
         help='IP address of the client and port' )
   externalIpAndPort = Submodel( valueType=IpAddrAndPort,
         help='Global IP address and port' )
   protocol = Int( help='IP protocol' )
   leaseTime = Float( help='Lease time in seconds' )
   state = Enum( values=( 'adding', 'deleting', 'created',
         'refused', 'cleared', 'NA' ),
         help="Port map state", optional=True, default="NA" )

class VrfUpnpPortMap( Model ):
   portMaps = List( valueType=UpnpPortMapEntry,
         help='UPnP port maps' )

class UpnpPortMapInfo( Model ):
   vrfs = Dict( keyType=str, valueType=VrfUpnpPortMap,
         help='Map of VRF Names to the UPnP port map' )

   def render( self ):
      headings = ( 'VRF', 'Inner IP/Port',
            'Global IP/Port', 'Protocol', 'Lease Time', 'State' )
      table = TableOutput.createTable( headings )
      fmtStr = formatRow( minWidth=16, maxWidth=16, wrap=True )
      fmtProto = formatRow( minWidth=8, maxWidth=8, wrap=True, justify='right' )
      fmtLease = formatRow( minWidth=10, maxWidth=10, wrap=True, justify='right',
            dotAlign=True )
      fmtIpAndPort = formatRow( minWidth=21, maxWidth=21 )
      fmtState = formatRow( minWidth=9, maxWidth=9, wrap=True )
      table.formatColumns( fmtStr, fmtIpAndPort, fmtIpAndPort,
            fmtProto, fmtLease, fmtState )

      # Sort the VRF names and display default in the beginning
      vrfNames = \
            [ vrfName for vrfName in self.vrfs if vrfName != DEFAULT_VRF ]
      vrfNames.sort()
      vrfNames.insert( 0, DEFAULT_VRF )

      for vrfName in vrfNames:
         portMapList = self.vrfs.get( vrfName )

         # If the vrf does not exist, continue
         # Can happen for default VRF
         if not portMapList:
            continue

         # Create table rows using port maps sortred on extIp and port
         for entry in sorted( portMapList.portMaps,
               key=lambda entry: ( entry.externalIpAndPort.ip,
               entry.externalIpAndPort.port ) ):
            table.newRow( vrfName,
                  entry.internalIpAndPort.formatStr(),
                  entry.externalIpAndPort.formatStr(),
                  entry.protocol, entry.leaseTime,
                  entry.state )
      print( table.output() )

serviceStatus = ( 'up', 'down', 'admin down' )
class UpnpServiceEntry( Model ):
   upnpStatus = Enum( values=serviceStatus, help='UPnP service status' )
   ssdpStatus = Enum( values=serviceStatus, help='UPnP service status' )

class UpnpServiceInfo( Model ):
   vrfs = Dict( keyType=str, valueType=UpnpServiceEntry,
         help='Map of VRF names to UPnP service status' )

   def render( self ):
      headings = ( 'VRF', 'IGD Status', 'SSDP Status' )
      table = TableOutput.createTable( headings )
      fmtStr = formatRow( minWidth=16, maxWidth=16, wrap=True )
      fmtStatus = formatRow( minWidth=11, maxWidth=11, wrap=True )
      table.formatColumns( fmtStr, fmtStatus, fmtStatus )

      # Sort the vrf names with default in the beginning
      vrfNames = [ vrfName for vrfName in self.vrfs
            if vrfName != DEFAULT_VRF ]
      vrfNames.sort()
      vrfNames.insert( 0, DEFAULT_VRF )

      # Iterate over the vrfNames and create rows
      for vrfName in vrfNames:
         entry = self.vrfs.get( vrfName )
         # Skip non existing vrfs
         if not entry:
            continue

         table.newRow( vrfName, entry.upnpStatus, entry.ssdpStatus )

      print( table.output() )

class UpnpCountersModel( Model ):

   class VrfCounters( Model ):
      addRequestsTotal = Int( help="Total AddPortMapping requests" )
      addSuccess = Int( help="Successful AddPortMapping requests" )
      addRefused = Int( help="NAT flow conflicts" )
      addConflicts = Int( help="UPNP flow conflicts" )
      addPrematureSessionClosure = Int( help="Premature session closures" )
      portMappingClearedLocally = Int( help="Port mappings cleared locally" )

      delRequestsTotal = Int( help="Total DeletePortMapping requests" )
      delSuccess = Int( help="Successful DeletePortMapping requests" )
      delFailed = Int( help="DeletePortMapping failures" )

      getSpecificMappingRequestsTotal = \
            Int( help="Total GetSpecificMapping requests" )
      getSpecificMappingRequestsSuccess = \
         Int( help="Successful GetSpecificMapping requests" )
      getSpecificMappingFailed = Int( help="GetSpecificMapping failures" )

      def render( self ):
         headings = ( 'Counter', 'Value' )
         table = TableOutput.createTable( headings )
         fmtCnt = formatRow( minWidth=16, maxWidth=40, justify='left' )
         fmtValue = formatRow( minWidth=12, maxWidth=12, wrap=True, justify='right' )
         table.formatColumns( fmtCnt, fmtValue )

         table.newRow( "AddPortMapping", '' )
         table.newRow( "Total", self.addRequestsTotal )
         table.newRow( "Successful", self.addSuccess )
         table.newRow( "UPNP flow conflicts", self.addConflicts )
         table.newRow( "NAT flow conflicts", self.addRefused )
         table.newRow( "Premature session closures",
               self.addPrematureSessionClosure )
         table.newRow( "Port mappings cleared locally",
               self.portMappingClearedLocally )
         table.newRow( "", '' )

         table.newRow( "DeletePortMapping", '' )
         table.newRow( "Totatl", self.delRequestsTotal )
         table.newRow( "Successful", self.delSuccess )
         table.newRow( "Failed", self.delFailed )
         table.newRow( "", '' )

         table.newRow( "GetSpecificMapping", '' )
         table.newRow( "Total",
               self.getSpecificMappingRequestsTotal )
         table.newRow( "Successful",
               self.getSpecificMappingRequestsSuccess )
         table.newRow( "Failed",
               self.getSpecificMappingFailed )

         print( table.output() )

   vrfs = Dict( keyType=str, valueType=VrfCounters,
                help="UPNP counters keyed by VRF name" )

   def render( self ):
      for vrf, counters in self.vrfs.items():
         print( "VRF:", vrf )
         counters.render()
