# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from BasicCli import ConfigModeBase
from CliCommand import CliCommandClass
from CliMode.Vpls import (
   RouterVplsMode,
   VplsInstanceMode,
   VplsVlanOrBundleMode,
   VplsGroupMode,
)
from CliMatcher import (
   EnumMatcher,
   KeywordMatcher,
   DynamicNameMatcher,
   IntegerMatcher,
)
from MultiRangeRule import (
   DoParseNoMerge,
   MultiRangeMatcher,
)
from CliPlugin.PwaCli import (
   ldpPwNameMatcher,
   ldpPwProfileMatcher,
   pwIdMatcher,
)
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher
from CliPlugin.VplsCli import getVplsConfigInstanceCollection
from PseudowireLib import vplsNameRegex
from RouteMapLib import getExtCommValue
from Toggles.PseudowireToggleLib import (
   toggleVplsBgpSignalingEnabled )
from Toggles.EvpnLibToggleLib import toggleEvpnVplsInteropEnabled
from TypeFuture import TacLazyType

VplsLdpPseudowireProfileConfig = TacLazyType(
   'Pseudowire::VplsLdpPseudowireProfileConfig' )

vplsMatcherForConfig = KeywordMatcher( 'vpls',
      helpdesc='Virtual Private LAN Service' )
pseudowireMatcherForConfig = KeywordMatcher( 'pseudowire',
      helpdesc='LDP pseudowire' )
macMatcherForConfig = KeywordMatcher( 'mac',
      helpdesc='Ethernet address' )
ldpHelpDesc = 'Label Distribution Protocol'
ldpMatcherForConfig = KeywordMatcher( 'ldp',
      helpdesc=ldpHelpDesc )
withdrawalMatcherForConfig = KeywordMatcher( 'withdrawal',
      helpdesc='Withdraw action' )
triggerMatcherForConfig = KeywordMatcher( 'trigger',
      helpdesc='Trigger reason' )
encapsulationMatcherForConfig = KeywordMatcher( 'encapsulation',
      helpdesc='Encapsulation information' )
dot1qMatcherForConfig = KeywordMatcher( 'dot1q',
      helpdesc='Encapsulation type' )

# ------------------------------------------------------------------------------
# Config Mode for:
#    config
#      router vpls
# ------------------------------------------------------------------------------
class RouterVplsConfigMode( RouterVplsMode, ConfigModeBase ):
   name = 'VPLS Configuration'

   def __init__( self, parent, session ):
      RouterVplsMode.__init__( self )
      ConfigModeBase.__init__( self, parent, session )

# ------------------------------------------------------------------------------
# Command handlers for:
#    config
#      [ no | default ] router vpls
# ------------------------------------------------------------------------------

def routerVplsCmdHandler( mode, args ):
   childMode = mode.childMode( RouterVplsConfigMode )
   mode.session_.gotoChildMode( childMode )

def routerVplsCmdNoHandler( mode, args ):
   getVplsConfigInstanceCollection().clear()

# ------------------------------------------------------------------------------
# Config Mode for:
#    config
#      router vpls
#        vpls <instance-name>
# ------------------------------------------------------------------------------
class VplsInstanceConfigMode( VplsInstanceMode, ConfigModeBase ):
   name = 'VPLS Instance Configuration'

   def __init__( self, parent, session, vplsInstanceName, vplsInstance ):
      VplsInstanceMode.__init__( self, vplsInstanceName )
      ConfigModeBase.__init__( self, parent, session )
      self.vplsInstance = vplsInstance

# ------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        [ no | default ] vpls <instance-name>
# ------------------------------------------------------------------------------
class RouterVplsInstanceCmd( CliCommandClass ):
   syntax = 'vpls VPLS_INSTANCE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vpls': vplsMatcherForConfig,
      'VPLS_INSTANCE_NAME': DynamicNameMatcher(
                                  lambda mode: getVplsConfigInstanceCollection(),
                                  helpdesc='VPLS Instance Name',
                                  pattern=vplsNameRegex )
   }

   @staticmethod
   def handler( mode, args ):
      vplsInstanceName = args[ 'VPLS_INSTANCE_NAME' ]
      vplsInstance = getVplsConfigInstanceCollection().get( vplsInstanceName )
      if not vplsInstance:
         vplsInstance = getVplsConfigInstanceCollection().newMember(
            vplsInstanceName )
      childMode = mode.childMode( VplsInstanceConfigMode,
                                  vplsInstanceName=vplsInstanceName,
                                  vplsInstance=vplsInstance )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vplsInstanceName = args[ 'VPLS_INSTANCE_NAME' ]
      del getVplsConfigInstanceCollection()[ vplsInstanceName ]

RouterVplsConfigMode.addCommandClass( RouterVplsInstanceCmd )

# ------------------------------------------------------------------------------
# Config Mode for:
#    config
#      router vpls
#        vpls <instance-name>
#           vlan <id>
#
# Note: This Mode has been defined differently from the "vlan bundle <id-list>"
#       below because we want to load different Command class under each of these
#       modes. The plain vlan mode here would only take dot1q encapsulation
#       for a single vlan. While the bundle mode would take a range of ids
#       for dot1q encapsulation. The VlanOrBundleCmd below would either load
#       this plain vlan mode or the bundle mode depending on what is specified
# ------------------------------------------------------------------------------
class VplsVlanConfigMode( VplsVlanOrBundleMode, ConfigModeBase ):
   name = 'VPLS VLAN Configuration'

   def __init__( self, parent, session, vlanId ):
      VplsVlanOrBundleMode.__init__( self, parent.vplsName, False, vlanId )
      ConfigModeBase.__init__( self, parent, session )

# ------------------------------------------------------------------------------
# Config Mode for:
#    config
#      router vpls
#        vpls <instance-name>
#           vlan bundle <id-list>
# ------------------------------------------------------------------------------
class VplsBundleConfigMode( VplsVlanOrBundleMode, ConfigModeBase ):
   name = 'VPLS VLAN Configuration'

   def __init__( self, parent, session, vlanId ):
      VplsVlanOrBundleMode.__init__( self, parent.vplsName, True, vlanId )
      ConfigModeBase.__init__( self, parent, session )

def vlanIdIsAlreadyInUse( vlanId ):
   for inst in getVplsConfigInstanceCollection().values():
      if vlanId in inst.vlan:
         return True
   return False

def addVlanConfig( mode, vlanId, translation=0, translationValid=False ):
   vlanColl = mode.vplsInstance.vlan
   tri = Tac.Value( 'Ark::TristateU16', translation, translationValid )
   vlanConfig = Tac.Value( 'Pseudowire::VplsVlanConfig', vlanId, tri )
   vlanColl.addMember( vlanConfig )

def maybeAddVlanConfig( mode, vlanId, translation=0, translationValid=False ):
   if vlanIdIsAlreadyInUse( vlanId ):
      clearAllVlanConfigs( mode.vplsInstance )
      mode.addErrorAndStop( f'VLAN ID {vlanId} is already in use.' +
                            ' Clearing VLAN configuration' )
   addVlanConfig( mode, vlanId, translation, translationValid )

def clearAllVlanConfigs( vplsInstanceConfig ):
   vplsInstanceConfig.vlanBundleStrForCliSave = ''
   vplsInstanceConfig.normalizedTagStrForCliSave = ''
   vplsInstanceConfig.vlan.clear()

def psuedowireNameAlreadyInUse( name, excludeInstanceName, excludeGroupName ):
   for instName, inst in getVplsConfigInstanceCollection().items():
      for groupName in inst.ldpPseudowireGroup:
         if excludeGroupName and \
            excludeInstanceName and \
            groupName == excludeGroupName and \
            instName == excludeInstanceName:
            continue
         group = inst.ldpPseudowireGroup[ groupName ]
         if name in group.pseudowire:
            return True
   return False

# ------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           [ no | default ] vlan ( <id> | ( bundle <id-list> ) )
# ------------------------------------------------------------------------------
class VlanOrBundleCmd( CliCommandClass ):
   _vlanSyntax = 'vlan VLAN_ID'
   _vlanOrBundleSyntax = 'vlan ( VLAN_ID | ( bundle VLAN_ID_LIST ) )'

   syntax = _vlanOrBundleSyntax
   noOrDefaultSyntax = 'vlan ...'
   data = {
      'vlan': 'VLANs to associate with the VPLS instance',
      'VLAN_ID': IntegerMatcher( 1, 4094, helpdesc='VLAN Identifier' ),
      'bundle': 'Associate multiple VLANs with the VPLS instance',
      'VLAN_ID_LIST': MultiRangeMatcher( rangeFn=lambda: ( 1, 4094 ),
                       noSingletons=False,
                       helpdesc='VLAN Identifier',
                       # No sorting and merging to keep the original order
                       DoParseClass=DoParseNoMerge ),
   }

   @staticmethod
   def handler( mode, args ):
      isBundle = 'bundle' in args
      if isBundle:
         vlanId = args[ 'VLAN_ID_LIST' ]
         if mode.vplsInstance.vlanBundleStrForCliSave != str( vlanId ):
            clearAllVlanConfigs( mode.vplsInstance )
            for v in vlanId.values():
               # For a bundle configuration, we set VplsVlanConfig::normalizedTag
               # with the same vlanId to begin with.
               # If "encapsulation dot1q <tag-list>" config is issued later, the new
               # translation tag would be overwritten
               # This is to indicate that we need to tag the frame going out of this
               # PW always.
               maybeAddVlanConfig( mode, v, v, True )

            mode.vplsInstance.vlanBundleStrForCliSave = str( vlanId )
      else:
         vlanId = args[ 'VLAN_ID' ]
         if ( mode.vplsInstance.vlanBundleStrForCliSave or
              vlanId not in mode.vplsInstance.vlan ):
            clearAllVlanConfigs( mode.vplsInstance )
            maybeAddVlanConfig( mode, vlanId )

      cls = VplsBundleConfigMode if isBundle else VplsVlanConfigMode
      childMode = mode.childMode( cls, vlanId=vlanId )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearAllVlanConfigs( mode.vplsInstance )

VplsInstanceConfigMode.addCommandClass( VlanOrBundleCmd )

# ------------------------------------------------------------------------------
# Command class for:
#    config
#      router vpls
#        vpls <instance-name>
#           vlan <id>
#              encapsulation dot1q <tag>
# ------------------------------------------------------------------------------
class VplsVlanEncapsulationCmd( CliCommandClass ):
   syntax = 'encapsulation dot1q VLAN_ID'
   noOrDefaultSyntax = 'encapsulation dot1q ...'
   data = {
      'encapsulation': encapsulationMatcherForConfig,
      'dot1q': dot1qMatcherForConfig,
      # Note: We allow encapsulation tag of 0. Hence range starts from 0
      'VLAN_ID': IntegerMatcher( 0, 4094, helpdesc='Encapsulation tag value' ),
   }

   @staticmethod
   def handler( mode, args ):
      addVlanConfig( mode.parent_,
                     mode.vlanId,
                     args[ 'VLAN_ID' ],
                     True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addVlanConfig( mode.parent_, mode.vlanId )

VplsVlanConfigMode.addCommandClass( VplsVlanEncapsulationCmd )

# ------------------------------------------------------------------------------
# Command class for:
#    config
#      router vpls
#        vpls <instance-name>
#           vlan bundle <id-list>
#              encapsulation dot1q <tag-list>
# ------------------------------------------------------------------------------
class VplsVlanBundleEncapsulationCmd( CliCommandClass ):
   syntax = 'encapsulation dot1q VLAN_ID_LIST'
   noOrDefaultSyntax = 'encapsulation dot1q ...'
   data = {
      'encapsulation': encapsulationMatcherForConfig,
      'dot1q': dot1qMatcherForConfig,
      # Note: We allow encapsulation tag of 0. Hence range starts from 0
      'VLAN_ID_LIST': MultiRangeMatcher( rangeFn=lambda: ( 0, 4094 ),
                       noSingletons=False,
                       helpdesc='Encapsulation tag list',
                       # No sorting and merging to keep the original order
                       DoParseClass=DoParseNoMerge ),
   }

   @staticmethod
   def handler( mode, args ):
      tagList = args[ 'VLAN_ID_LIST' ]
      tags = tagList.values()
      if len( set( tags ) ) < len( tags ):
         mode.addErrorAndStop(
            'Encapsulation tags have duplicates. Ignoring configuration' )

      vlans = mode.vlanId.values()
      if len( tags ) != len( vlans ):
         mode.addErrorAndStop( 'Number of encapsulation tags do not match with ' +
                               'vlans in bundle. Ignoring configuration' )

      mode.parent_.vplsInstance.normalizedTagStrForCliSave = str( tagList )
      for vlan, tag in zip( vlans, tags ):
         addVlanConfig( mode.parent_,
                        vlan,
                        tag,
                        True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.parent_.vplsInstance.normalizedTagStrForCliSave = ''
      for v in mode.vlanId.values():
          # Note: For a bundle configuration, we always set
          # VplsVlanConfig::normalizedTag. This is to indicate that we need to tag
          # the frame going out of this PW always.
          # Since the encapsulation tag is being unconfigured here, we need to set
          # it back to the original vlan ID
         addVlanConfig( mode.parent_, v, v, True )

VplsBundleConfigMode.addCommandClass( VplsVlanBundleEncapsulationCmd )

# ------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           [ no | default ] ldp mac withdrawal trigger interface
# ------------------------------------------------------------------------------
class MacWithdrawalInterfaceCmd( CliCommandClass ):
   syntax = 'ldp mac withdrawal trigger interface'
   noOrDefaultSyntax = syntax
   data = {
      'ldp': ldpMatcherForConfig,
      'mac': macMatcherForConfig,
      'withdrawal': withdrawalMatcherForConfig,
      'trigger': triggerMatcherForConfig,
      'interface': 'Interface trigger',
   }

   @staticmethod
   def handler( mode, args ):
      mode.vplsInstance.macWithdrawalTriggerInterface = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vplsInstance.macWithdrawalTriggerInterface = False

VplsInstanceConfigMode.addCommandClass( MacWithdrawalInterfaceCmd )

# ------------------------------------------------------------------------------
# Config Mode for:
#    config
#      router vpls
#        vpls <instance-name>
#           pseudowire ldp <group-name> ( split-horizon | spokes )
#
# Two following 2 config mode classes have been defined so that they can have
# different configs for pseudowires. The SplitHorizon config class does not
# support secondary PW config, while the Spoke config class does. Also, the
# SplitHorizon config class supports forwarding preference evpn (a.k.a. evpn/vpls
# interop), while the Spoke config class does not.
#  - VplsSplitHorizonGroupConfigMode
#  - VplsSpokeGroupConfigMode
# ------------------------------------------------------------------------------
class VplsLdpSplitHorizonGroupConfigMode( VplsGroupMode, ConfigModeBase ):
   name = 'VPLS LDP split-horizon group configuration'

   def __init__( self, parent, session, groupName, vplsGroup, groupColl ):
      self.vplsGroup = vplsGroup
      self.groupColl = groupColl
      VplsGroupMode.__init__( self, parent.vplsName, groupName,
                              signalingProtocol='ldp', splitHorizon=True )
      ConfigModeBase.__init__( self, parent, session )

class VplsLdpSpokeGroupConfigMode( VplsGroupMode, ConfigModeBase ):
   name = 'VPLS LDP spoke group configuration'

   def __init__( self, parent, session, groupName, vplsGroup, groupColl ):
      self.vplsGroup = vplsGroup
      self.groupColl = groupColl
      VplsGroupMode.__init__( self, parent.vplsName, groupName,
                              signalingProtocol='ldp', splitHorizon=False )
      ConfigModeBase.__init__( self, parent, session )

class VplsBgpSplitHorizonGroupConfigMode( VplsGroupMode, ConfigModeBase ):
   name = 'VPLS BGP split-horizon group configuration'

   def __init__( self, parent, session, groupName, vplsGroup, groupColl ):
      self.vplsGroup = vplsGroup
      self.groupColl = groupColl
      VplsGroupMode.__init__( self, parent.vplsName, groupName,
                              signalingProtocol='bgp', splitHorizon=True )
      ConfigModeBase.__init__( self, parent, session )

class VplsBgpSpokeGroupConfigMode( VplsGroupMode, ConfigModeBase ):
   name = 'VPLS BGP spoke group configuration'

   def __init__( self, parent, session, groupName, vplsGroup, groupColl ):
      self.vplsGroup = vplsGroup
      self.groupColl = groupColl
      VplsGroupMode.__init__( self, parent.vplsName, groupName,
                              signalingProtocol='bgp', splitHorizon=False )
      ConfigModeBase.__init__( self, parent, session )

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           [ no | default ] pseudowire ( bgp |ldp ) <group-name>
#                                                      ( split-horizon | spokes )
# ----------------------------------------------------------------------------------
if toggleVplsBgpSignalingEnabled():
   signalingProtocolMatcher = EnumMatcher( {
      'bgp': 'Border Gateway Protocol',
      'ldp': ldpHelpDesc,
   } )
else:
   signalingProtocolMatcher = EnumMatcher( {
      'ldp': ldpHelpDesc,
   } )

class VplsGroupCmd( CliCommandClass ):
   syntax = 'pseudowire SIGNALING_PROTOCOL GROUP_NAME ( split-horizon | spokes )'
   noOrDefaultSyntax = 'pseudowire SIGNALING_PROTOCOL GROUP_NAME ...'

   data = {
      'pseudowire': pseudowireMatcherForConfig,
      'SIGNALING_PROTOCOL': signalingProtocolMatcher,
      'GROUP_NAME': DynamicNameMatcher(
                          lambda mode: (
                             list( mode.vplsInstance.ldpPseudowireGroup )
                             + list( mode.vplsInstance.bgpPseudowireGroup ) ),
                          helpdesc='VPLS Group Name',
                          pattern=vplsNameRegex ),
      'split-horizon': 'Group enforcing split-horizon',
      'spokes': 'Group not enforcing split-horizon',
   }

   @staticmethod
   def _getGroupColl( mode, isLdp ):
      return ( mode.vplsInstance.ldpPseudowireGroup if isLdp else
               mode.vplsInstance.bgpPseudowireGroup )

   @staticmethod
   def handler( mode, args ):
      signalingProtocol = args[ 'SIGNALING_PROTOCOL' ]
      isLdp = signalingProtocol == 'ldp'
      assert toggleVplsBgpSignalingEnabled() or isLdp
      isSplitHorizon = 'split-horizon' in args
      groupName = args[ 'GROUP_NAME' ]
      groupColl = VplsGroupCmd._getGroupColl( mode, isLdp )
      vplsGroup = groupColl.get( groupName )

      # If different signaling types are enabled, check the group hasn't
      # already been configured with a different signaling type
      if toggleVplsBgpSignalingEnabled():
         otherGroupColl = VplsGroupCmd._getGroupColl( mode, not isLdp )
         if groupName in otherGroupColl:
            errStr = 'bgp' if isLdp else 'ldp'
            mode.addErrorAndStop(
               f'Group already exists with {errStr} signaling configuration.' +
               ' Ignoring configuration' )

      # Check the group hasn't already been configured with a different
      # split-horizon type
      if vplsGroup and vplsGroup.splitHorizon != isSplitHorizon:
         errStr = 'split-horizon' if not isSplitHorizon else 'spokes'
         mode.addErrorAndStop( f'Group already exists with {errStr} configuration.' +
                               ' Ignoring configuration' )
      if not vplsGroup:
         vplsGroup = groupColl.newMember( groupName, isSplitHorizon )

      # Go into the correct group config mode from the four possible choices
      cls = None
      if isLdp:
         if isSplitHorizon:
            cls = VplsLdpSplitHorizonGroupConfigMode
         else:
            cls = VplsLdpSpokeGroupConfigMode
      else:
         if isSplitHorizon:
            cls = VplsBgpSplitHorizonGroupConfigMode
         else:
            cls = VplsBgpSpokeGroupConfigMode
      childMode = mode.childMode( cls, groupName=groupName,
                                  vplsGroup=vplsGroup, groupColl=groupColl )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      signalingProtocol = args[ 'SIGNALING_PROTOCOL' ]
      isLdp = signalingProtocol == 'ldp'
      assert toggleVplsBgpSignalingEnabled() or isLdp
      groupName = args[ 'GROUP_NAME' ]
      groupColl = VplsGroupCmd._getGroupColl( mode, isLdp )
      del groupColl[ groupName ]

VplsInstanceConfigMode.addCommandClass( VplsGroupCmd )

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           pseudowire ldp <group-name> spokes
#              [ no | default ] pseudowire <name> [secondary [<precedence>]]
# ----------------------------------------------------------------------------------
class VplsLdpSpokeGroupPseudowireNameCmd( CliCommandClass ):
   syntax = 'pseudowire LDP_PW_NAME [ secondary [ PRECEDENCE ] ]'
   noOrDefaultSyntax = 'pseudowire LDP_PW_NAME ...'
   data = {
      'pseudowire': pseudowireMatcherForConfig,
      'LDP_PW_NAME': ldpPwNameMatcher,
      'secondary': 'Secondary pseudowire',
      'PRECEDENCE': IntegerMatcher( 1, 255,
                      helpdesc='Precedence value for secondary pseudowire' ),
   }

   @staticmethod
   def handler( mode, args ):
      pwName = args[ 'LDP_PW_NAME' ]
      secondary = 'secondary' in args
      precedence = args.get( 'PRECEDENCE', 0 )
      if psuedowireNameAlreadyInUse( pwName, mode.vplsName, mode.groupName ):
         mode.addErrorAndStop(
               f'Pseudowire name {pwName} is already in use.' +
               ' Ignoring configuration' )

      numPrimary = 0
      numSecondary = 0
      for n in mode.vplsGroup.pseudowire:
         # Exclude the pw name that is undergoing the change
         if n == pwName:
            continue
         pw = mode.vplsGroup.pseudowire[ n ]
         if pw.secondary:
            numSecondary += 1
         else:
            numPrimary += 1
      if secondary:
         # A secondary pseudowire cannot be configured in a spoke-group with more
         # than one primary pseudowire.
         if numPrimary > 1:
            mode.addErrorAndStop(
                   'A secondary pseudowire cannot be configured' +
                   f' with {numPrimary} primary pseudowires' )
      else:
         # A second primary pseudowire cannot be configured if spoke-group already
         # has a secondary pseudowire
         if numPrimary > 0 and numSecondary > 0:
            errStr = 'A primary pseudowire cannot be configured' + \
                  ' with {p} primary and {s} secondary pseudowires.' + \
                  ' Only a single primary pseudowire is allowed while other' + \
                  ' secondary pseudowires are configured.'
            mode.addErrorAndStop( errStr.format( p=numPrimary, s=numSecondary ) )

      mode.vplsGroup.pseudowire.addMember(
               Tac.Value( 'Pseudowire::VplsLdpPseudowireConfig',
                          pwName, secondary, precedence ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwName = args[ 'LDP_PW_NAME' ]
      del mode.vplsGroup.pseudowire[ pwName ]

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           pseudowire ldp <group-name> split-horizon
#              [ no | default ] pseudowire <name>
# ----------------------------------------------------------------------------------
class VplsLdpSplitHorizonGroupPseudowireNameCmd( CliCommandClass ):
   syntax = 'pseudowire LDP_PW_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowire': pseudowireMatcherForConfig,
      'LDP_PW_NAME': ldpPwNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      pwName = args[ 'LDP_PW_NAME' ]
      if psuedowireNameAlreadyInUse( pwName, mode.vplsName, mode.groupName ):
         mode.addErrorAndStop(
                f'Pseudowire name {pwName} is already in use.' +
                ' Ignoring configuration' )
      mode.vplsGroup.pseudowire.addMember(
               Tac.Value( 'Pseudowire::VplsLdpPseudowireConfig', pwName, False, 0 ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwName = args[ 'LDP_PW_NAME' ]
      del mode.vplsGroup.pseudowire[ pwName ]

VplsLdpSplitHorizonGroupConfigMode.addCommandClass(
             VplsLdpSplitHorizonGroupPseudowireNameCmd )
VplsLdpSpokeGroupConfigMode.addCommandClass( VplsLdpSpokeGroupPseudowireNameCmd )

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls < instance >
#           pseudowire ldp <group-name> ( split-horizon | spokes )
#              [ no | default ] bgp auto-discovery profile < name >
#                                  id ( < 4-byte-id > | < l2vpn-identifier > )
# ----------------------------------------------------------------------------------
class VplsLdpGroupPseudowireProfileCmd( CliCommandClass ):
   _syntaxPrefix = 'bgp auto-discovery profile'
   syntax = _syntaxPrefix + ' LDP_PW_PROFILE id ( PWID | L2VPN-ID )'
   noOrDefaultSyntax = _syntaxPrefix + ' ...'
   data = {
      'bgp': 'BGP config',
      'auto-discovery': 'BGP auto-discovery config',
      'profile': 'LDP pseudowire profile',
      'LDP_PW_PROFILE': ldpPwProfileMatcher,
      'id': 'BGP auto-discovery ID',
      'PWID': pwIdMatcher,
      'L2VPN-ID': RtSooExtCommCliMatcher( "L2VPN Identifier", acceptLongAsn=False ),
   }

   @staticmethod
   def handler( mode, args ):
      profile = args[ 'LDP_PW_PROFILE' ]
      profileConfig = VplsLdpPseudowireProfileConfig( profile )
      if "PWID" in args:
         profileConfig.rawId = args[ 'PWID' ]
      else:
         extCommVal = getExtCommValue( "l2VpnId " + args[ "L2VPN-ID" ] )
         profileConfig.l2VpnId = extCommVal
      mode.vplsGroup.pseudowireProfile = profileConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vplsGroup.pseudowireProfile = VplsLdpPseudowireProfileConfig()

VplsLdpSplitHorizonGroupConfigMode.addCommandClass(
   VplsLdpGroupPseudowireProfileCmd )
VplsLdpSpokeGroupConfigMode.addCommandClass( VplsLdpGroupPseudowireProfileCmd )

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           pseudowire ldp <group-name> ( split-horizon | spokes )
#              [ no | default ] mac withdrawal trigger pseudowire
# ----------------------------------------------------------------------------------
class VplsLdpGroupMacWithdrawalPseudowireCmd( CliCommandClass ):
   syntax = 'mac withdrawal trigger pseudowire'
   noOrDefaultSyntax = syntax
   data = {
      'mac': macMatcherForConfig,
      'withdrawal': withdrawalMatcherForConfig,
      'trigger': triggerMatcherForConfig,
      'pseudowire': pseudowireMatcherForConfig,
   }

   @staticmethod
   def handler( mode, args ):
      mode.vplsGroup.macWithdrawalTriggerPseudowire = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vplsGroup.macWithdrawalTriggerPseudowire = False

VplsLdpSplitHorizonGroupConfigMode.addCommandClass(
        VplsLdpGroupMacWithdrawalPseudowireCmd )
VplsLdpSpokeGroupConfigMode.addCommandClass( VplsLdpGroupMacWithdrawalPseudowireCmd )

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           pseudowire ldp <group-name> ( split-horizon | spokes )
#              [ no | default ] mac withdrawal propagation (received | suppressed)
# ----------------------------------------------------------------------------------
class VplsLdpGroupMacWithdrawalControlCmd( CliCommandClass ):
   syntax = 'mac withdrawal propagation ( received | suppressed )'
   noOrDefaultSyntax = 'mac withdrawal propagation ...'
   data = {
      'mac': macMatcherForConfig,
      'withdrawal': withdrawalMatcherForConfig,
      'propagation': 'Control propagation of withdrawal',
      'received': 'Allow propagation of received and local withdrawal',
      'suppressed': 'suppress propagation of both received and local withdrawal',
   }

   @staticmethod
   def handler( mode, args ):
      propagationConfig = None
      if 'received' in args:
         propagationConfig = 'locallyTriggeredAndReceived'
      else:
         propagationConfig = 'suppress'
      mode.vplsGroup.macWithdrawalPropagation = propagationConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vplsGroup.macWithdrawalPropagation = 'locallyTriggered'

VplsLdpSplitHorizonGroupConfigMode.addCommandClass(
   VplsLdpGroupMacWithdrawalControlCmd )
VplsLdpSpokeGroupConfigMode.addCommandClass( VplsLdpGroupMacWithdrawalControlCmd )

# ----------------------------------------------------------------------------------
# Command Class for:
#    config
#      router vpls
#        vpls <instance-name>
#           pseudowire ldp <group-name> split-horizon
#              [ no | default ] forwarding preference evpn
#
# Note: This command is used for EVPN/VPLS interop
# ----------------------------------------------------------------------------------

class VplsLdpGroupForwardingPreferenceEvpnCmd( CliCommandClass ):
   syntax = 'forwarding preference evpn'
   noOrDefaultSyntax = syntax
   data = {
      'forwarding': 'Forwarding configuration',
      'preference': 'Preferred control plane for data forwarding',
      'evpn': 'Seamless integration of EVPN with VPLS as described in RFC8560'
   }

   @staticmethod
   def handler( mode, args ):
      evpnInteropedGroupName = None
      for name, group in mode.groupColl.items():
         if group.forwardingPreferenceEvpn and group != mode.vplsGroup:
            evpnInteropedGroupName = name
            break
      if evpnInteropedGroupName:
         errStr = ( f'Group {evpnInteropedGroupName} is already configured '
                    'with "forwarding preference evpn" within the current '
                    'VPLS instance. Ignoring configuration' )
         mode.addErrorAndStop( errStr )
      mode.vplsGroup.forwardingPreferenceEvpn = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vplsGroup.forwardingPreferenceEvpn = False

if toggleEvpnVplsInteropEnabled():
   VplsLdpSplitHorizonGroupConfigMode.addCommandClass(
         VplsLdpGroupForwardingPreferenceEvpnCmd )
