# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from CliModel import (
   Bool,
   Dict,
   DeferredModel,
   Enum,
   Float,
   Int,
   List,
   Model,
   Submodel,
   Str,
)
from CliDynamicSymbol import CliDynamicPlugin
from IntfModels import Interface
from PseudowireLib import FlowLabelMode
from TunnelTypeLib import tunnelTypeStrDict

PwaModel = CliDynamicPlugin( 'PwaModel' )

# The PseudowireAttributeStatusColl provides the status for all
# pseudowires and uses the PseudowireConnectorStatus type. List the
# status values that are applicable to VPLS.
pseudowireStatusValues = [
   'cliConflict',
   'cliIncomplete',
   'noLocalMapping',
   'noRemoteEnd',
   'noPeerLabel',
   'pwTypeMismatch',
   'controlWordMismatch',
   'mtuMismatch',
   'pwNotForwardingRemote',
   'noTunnel',
   'noTunnelRib',
   'unsupportedTunnelMode',
   'down',
   'up',
   'standby',
   'neighborEvpn'
]

pseudowireLocalStatusValues = [
   'up',
   'standby',
]

pseudowireNeighborStatusValues = [
   'noLocalMapping',
   'noPeerLabel',
   'noRemoteEnd',
   'pwTypeMismatch',
   'controlWordMismatch',
   'mtuMismatch',
   'pwNotForwardingRemote',
   'down',
   'up',
   'standby',
]

class VplsPseudowire( Model ):
   status = Enum( values=pseudowireStatusValues,
                  help='Pseudowire status' )

   pseudowireType = Enum( values=[ 'pwType4', 'pwType5' ],
                          help='Pseudowire type',
                          optional=True )
   interface = Interface( help='Pseudowire interface',
                          optional=True )

   controlWord = Bool( help='Control word used',
                       optional=True )
   flowLabelUsed = Enum( values=FlowLabelMode.attributes,
                         help='Flow label used',
                         optional=True )

   tunnelType = Enum( values=list( tunnelTypeStrDict.values() ),
                      help='Tunnel type',
                      optional=True )
   tunnelIndexes = List(
      valueType=int,
      help='List of tunnel indexes ending at this tunnel endpoint',
      optional=True )

   vccvCvTypes = Int( optional=True, help="VCCV CV types used" )
   vccvCcTypes = Int( optional=True, help="VCCV CC types used" )

   # Includes pseudowire label
   localStatus = Enum( values=pseudowireLocalStatusValues,
                       help='Local pseudowire status',
                       optional=True )
   localConfig = Submodel( valueType=PwaModel.LdpPseudowirePeerConfig,
                           help='Local pseudowire configuration',
                           optional=True )
   neighborStatus = Enum( values=pseudowireNeighborStatusValues,
                          help='Neighbor pseudowire status',
                          optional=True )
   neighborConfig = Submodel( valueType=PwaModel.LdpPseudowirePeerConfig,
                              help='Neighbor pseudowire configuration',
                              optional=True )

   negativeMacWithdrawalReceived = Int(
      help='Number of negative MAC withdrawal all-from-me messages received',
      optional=True )
   lastNegativeMacWithdrawalReceivedTime = Float(
      help='Time when last negative MAC withdrawal all-from-me message was received',
      optional=True )
   negativeMacWithdrawalSent = Int(
      help='Number of negative MAC withdrawal all-from-me messages sent',
      optional=True )
   lastNegativeMacWithdrawalSentTime = Float(
      help='Time when last negative MAC withdrawal all-from-me message was sent',
      optional=True )

class VplsPseudowireGroupNeighbor( Model ):
   pseudowires = Dict( keyType=str,
                       valueType=VplsPseudowire,
                       help='VPLS pseudowire information keyed by pseudowire ID' )

class VplsPseudowireGroup( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # macWithdrawalTriggerOnPseudowireFailure and macWithdrawalPropagation
         # were mandatory when we only supported VPLS-LDP. However these fields
         # do not apply for VPLS-BGP groups, so they will not be present
         triggerKey = 'macWithdrawalTriggerOnPseudowireFailure'
         dictRepr[ triggerKey ] = dictRepr.get( triggerKey, False )
         propagationKey = 'macWithdrawalPropagation'
         dictRepr[ propagationKey ] = dictRepr.get( propagationKey, 'suppress' )
      return dictRepr

   splitHorizon = Bool(
      help='VPLS split-horizon rule enforced across group pseudowires' )
   signalingProtocol = Enum( values=[ 'ldp', 'bgp' ],
                             help='The signaling protocol used to negotiate the '
                            'pseudowires in the group' )
   profile = Str( help='VPLS auto-discovery LDP pseudowire profile configuration',
                  optional=True )
   macWithdrawalTriggerOnPseudowireFailure = Bool(
      help=( 'Trigger MAC withdrawal transmission '
             'when a group pseudowire state transitions away from up' ),
      optional=True )
   macWithdrawalPropagation = Enum(
      values=[ 'locallyTriggered',
               'locallyTriggeredAndReceived',
               'suppress' ],
      help='Propagate locally triggered and received MAC withdrawal messages',
      optional=True )
   forwardingPreferenceEvpn = Bool(
      help='Prefer EVPN control plane for data forwarding as per RFC8560',
      optional=True )
   neighbors = Dict(
      keyType=IpGenericAddress,
      valueType=VplsPseudowireGroupNeighbor,
      help='VPLS neighbor pseudowire information keyed by neighbor address' )

class VplsVlan( Model ):
   normalizedTag = Int( help='Pseudowire VPLS 802.1Q tag value',
                        optional=True )

class VplsInstance( Model ):
   __revision__ = 2
   vlans = Dict( keyType=int,
                 valueType=VplsVlan,
                 help='VPLS VLAN information keyed by VLAN ID' )
   macWithdrawalTriggerInterface = Bool(
      help=( 'Send MAC withdrawal messages on instance pseudowires '
             'when VPLS local interface goes down' ) )
   pseudowireGroups = Dict(
      keyType=str,
      valueType=VplsPseudowireGroup,
      help='VPLS pseudowire group information keyed by group name' )

class Vpls( DeferredModel ):
   '''Model for show vpls and show vpls detail'''
   __revision__ = 2
   instances = Dict( keyType=str,
                     valueType=VplsInstance,
                     help='VPLS instance information keyed by instance name' )
