# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Model, Dict, Float, Int, Bool, Enum, Str, Submodel
import TableOutput

# -----------------------------------------------------------------------------------
# Models for "show transceiver cdb capabilities [ slot <slotNum> ]
# -----------------------------------------------------------------------------------

# Dictionaries that associate a rendered string to each Enum value used in CAPI
cdbCmdExecTriggerStr = {
   "commandIdWrite": "command ID write",
   "fullCommandWrite": "full command write"
}

fwPasswordTypeStr = {
   "notApplicable": "not applicable",
   "vendorPassword": "vendor password",
   "vendorPasswordSequence": " vendor password sequence",
   "msaPassword": "MSA password"
}

fwAccessMethodStr = {
   "none": "none",
   "lpl": "LPL",
   "epl": "EPL",
   "lplAndEpl": "LPL,EPL"
}

class TransceiverCdbCapabilitiesModuleInfo( Model ):
   __revision__ = 2
   cdbInstancesSupported = Int( help="The number of CDB instances the inserted "
                                     "transceiver supports",
                                optional=True )
   backgroundModeSupport = Bool( help="Indicates whether CDB commands can be "
                                      "executed in the background",
                                 optional=True )
   autoPagingSupport = Bool( help="Indicates whether EPL pages can be automatically "
                                  "selected",
                             optional=True )
   maxEplPagesSupported = Int( help="The number of supported EPL pages",
                               optional=True )
   maxEplTransactionSize = Int( help="The maximum number of bytes that can be "
                                     "written to the EPL pages in a single SMBus "
                                     "transaction",
                                optional=True )
   maxLplTransactionSize = Int( help="The maximum number of bytes that can be "
                                     "written to the LPL page in a single SMBus "
                                     "transaction",
                                optional=True )
   cdbCommandExecutionTrigger = Enum( values=cdbCmdExecTriggerStr,
                                      help="Action that triggers the start of a CDB "
                                           "command execution",
                                      optional=True )
   maxCompletionTime = Float( help="Maximum time CDB commands require to complete",
                              optional=True )
   cdbMaxBusyTime = Float( help="Maximum time CDB commands require before "
                                "reporting execution status",
                           optional=True )
   cdbModuleCommandsSupported = Dict( keyType=str, valueType=bool,
                                      help="Mapping of supported general CDB "
                                           "commands",
                                      optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # In rev2 of this model, all attributes became optional, so we need a
         # degrade function to ensure scripts processing CAPI models don't break
         # after an EOS upgrade
         dictRepr[ 'cdbInstancesSupported' ] = dictRepr.get(
            'cdbInstancesSupported', 0 )
         dictRepr[ 'backgroundModeSupport' ] = dictRepr.get(
            'backgroundModeSupport', False )
         dictRepr[ 'autoPagingSupport' ] = dictRepr.get(
            'autoPagingSupport', False )
         dictRepr[ 'maxEplPagesSupported' ] = dictRepr.get(
            'maxEplPagesSupported', 0 )
         dictRepr[ 'maxEplTransactionSize' ] = dictRepr.get(
            'maxEplTransactionSize', 0 )
         dictRepr[ 'maxLplTransactionSize' ] = dictRepr.get(
            'maxLplTransactionSize', 0 )
         dictRepr[ 'cdbCommandExecutionTrigger' ] = dictRepr.get(
            'cdbCommandExecutionTrigger', 'commandIdWrite' )
         dictRepr[ 'maxCompletionTime' ] = dictRepr.get(
            'maxCompletionTime', 0.0 )
         dictRepr[ 'cdbMaxBusyTime' ] = dictRepr.get(
            'cdbMaxBusyTime', 0.0 )
      return dictRepr

class TransceiverCdbCapabilitiesFirmwareManagement( Model ):
   firmwarePasswordType = Enum( values=fwPasswordTypeStr,
                                help="Password type required to unlock CDB firmware "
                                     "management commands",
                                optional=True )
   firmwareImageReadbackSupport = Bool( help="Firmware image readback is supported",
                                        optional=True )
   skipFirmwareErasedBlock = Bool( help="Skip firmware erased block", optional=True )
   firmwareCopyCommandSupport = Bool( help="Firmware copy command is supported",
                                      optional=True )
   firmwareAbortCommandSupport = Bool( help="Firmware update abort command is "
                                            "supported",
                                       optional=True )
   firmwareStartCommandPayloadSize = Int( help="Expected size in bytes of the "
                                               "firmware start payload",
                                          optional=True )
   firmwareErasedByteMarker = Int( help="Byte value used to mark erased blocks",
                                   optional=True )
   firmwareMaxEplTransactionSize = Int(
      help="The maximum number of bytes that can be written to the EPL pages in a "
           "single SMBus transaction during a firmware CDB command",
      optional=True )
   firmwareMaxLplTransactionSize = Int(
      help="The maximum number of bytes that can be written to the LPL page in a "
           "single SMBus transaction during a firmware CDB command",
      optional=True )
   firmwareWriteMethod = Enum( values=fwAccessMethodStr,
                               help="Method used for writing a new firmware binary "
                                    "to the transceiver",
                               optional=True )
   firmwareReadMethod = Enum( values=fwAccessMethodStr,
                              help="Method used for reading a firmware binary from "
                                   "the transceiver",
                              optional=True )
   firmwareHitlessRestartSupported = Bool(
      help="Indicates whether the transceiver can restart its firmware hitlessly",
      optional=True )
   firmwareStartCommandMaxDuration = Float(
      help="Maximum time in seconds the 'start' firmware update command requires to "
           "execute and complete",
      optional=True )
   firmwareAbortCommandMaxDuration = Float(
      help="Maximum time in seconds the 'abort' firmware update command requires to "
           "execute and complete",
      optional=True )
   firmwareWriteCommandMaxDuration = Float(
      help="Maximum time in seconds the 'download' firmware update command requires "
           "to execute and complete",
      optional=True )
   firmwareCompleteCommandMaxDuration = Float(
      help="Maximum time in seconds the 'complete' firmware update command requires "
           "to execute and complete",
      optional=True )
   firmwareCopyCommandMaxDuration = Float(
      help="Maximum time in seconds the 'copy' firmware update command requires to "
           "execute and complete",
      optional=True )

class TransceiverCdbCapabilitiesSlotDetail( Model ):
   __revision__ = 2
   serialNumber = Str( help="Transceiver serial number" )
   moduleCdbCapabilitiesAdvertised = Submodel(
      valueType=TransceiverCdbCapabilitiesModuleInfo,
      help="Generic CDB capabilities advertisements",
      optional=True )
   moduleCdbCapabilitiesOverride = Submodel(
      valueType=TransceiverCdbCapabilitiesModuleInfo,
      help="Generic CDB capabilities overrides",
      optional=True )
   firmwareManagementCdbCapabilitiesAdvertised = Submodel(
      valueType=TransceiverCdbCapabilitiesFirmwareManagement,
      help="Firmware management CDB advertisements",
      optional=True )
   firmwareManagementCdbCapabilitiesOverride = Submodel(
      valueType=TransceiverCdbCapabilitiesFirmwareManagement,
      help="Firmware management CDB overrides",
      optional=True )

   def renderDuration( self, duration, noValueStr="unknown" ):
      '''
      Helper function used to render the duration strings. Values under 1 second are
      rendered in integer units of milliseconds, values over 1 second, are rendered
      as float units of seconds.

      When the value provided is None, the noValueStr will be rendered.

      Parameters
      ----------
      duration : Tac::Seconds
      noValueStr : str

      Return
      ------
      str : rendered output
      '''
      if duration is None:
         return noValueStr
      if duration < 1:
         return f"{duration * 1000:.0f} milliseconds"
      return f"{duration:.2f} seconds"

   def renderBool( self, boolValue, noValueStr="unknown" ):
      '''
      Helper function used to render boolean values as yes (true)/no (false) strings.

      When the value provided is None, the noValueStr will be rendered.

      Parameters
      ----------
      boolValue : bool
      noValueStr : str

      Return
      ------
      str : rendered output
      '''
      if boolValue is None:
         return noValueStr
      return "yes" if boolValue else "no"

   def renderNumeric( self, value, noValueStr="unknown" ):
      '''
      Helper function used to convert numerical values to str.
      When the value provided is None, the noValueStr will be rendered.

      Parameters
      ----------
      value : int/float
      noValueStr : str

      Return
      ------
      str : rendered output
      '''
      if value is None:
         return noValueStr
      return str( value )

   def renderBytes( self, value, noValueStr ):
      '''
      Helper function used to render a numerical value representing a byte counter to
      str.
      When the value provided is None, the noValueStr will be rendered.

      Parameters
      ----------
      value : int/float
      noValueStr : str

      Return
      ------
      str : rendered output
      '''
      return ( self.renderNumeric( value, noValueStr ) + " bytes" ) \
             if value is not None else noValueStr

   def getRenderedCdbData( self, model, noValueStr="" ):
      '''
      Helper function used to compute all the rendered attributes of the
      TransceiverCdbCapabilitiesModuleInfo model (module/generic CDB capabilities.

      The noValueStr is used to indicate what should be rendered when an attribute's
      value is None.

      Parameters
      ----------
      model : TransceiverCdbCapabilitiesModuleInfo
      noValueStr : str

      Returns
      -------
      tuple of strings of all rendered attributes stored by the given model
      '''
      cdbInstanceSup = noValueStr
      backgroundModeSup = noValueStr
      autoPagingSup = noValueStr
      maxEplPages = noValueStr
      maxEplTransactionSize = noValueStr
      maxLplTransactionSize = noValueStr
      cdbCmdExecTrigger = noValueStr
      maxCompletionTime = noValueStr
      cdbMaxBusyTime = noValueStr
      if model:
         cdbInstanceSup = self.renderNumeric( model.cdbInstancesSupported,
                                              noValueStr )
         backgroundModeSup = self.renderBool( model.backgroundModeSupport,
                                              noValueStr )
         autoPagingSup = self.renderBool( model.autoPagingSupport, noValueStr )
         maxEplPages = self.renderNumeric( model.maxEplPagesSupported, noValueStr )
         maxEplTransactionSize = self.renderBytes( model.maxEplTransactionSize,
                                                   noValueStr )
         maxLplTransactionSize = self.renderBytes( model.maxLplTransactionSize,
                                                   noValueStr )
         cdbCmdExecTrigger = cdbCmdExecTriggerStr.get(
            model.cdbCommandExecutionTrigger, noValueStr )
         maxCompletionTime = self.renderDuration( model.maxCompletionTime,
                                                  noValueStr )
         cdbMaxBusyTime = self.renderDuration( model.cdbMaxBusyTime, noValueStr )
      return ( cdbInstanceSup, backgroundModeSup, autoPagingSup, maxEplPages,
               maxEplTransactionSize, maxLplTransactionSize, cdbCmdExecTrigger,
               maxCompletionTime, cdbMaxBusyTime )

   def getRenderedCdbCmds( self ):
      '''
      Helper function used to retrieve the rendered strings for CDB command support.
      Because there are up to 255 command entries, we only want to render the
      relevant ones to save screen space. An entry is considered relevant if the
      command is supported, or if there is an override specified for the same
      command.

      This function takes into account both the advertised and overriden CDB command
      support.

      Returns
      -------
      { cmd : ( advertisedCmdSupport, overridenCmdSupport ) }
      '''
      adv = self.moduleCdbCapabilitiesAdvertised
      overrides = self.moduleCdbCapabilitiesOverride
      cmds = {}
      for cmd in adv.cdbModuleCommandsSupported:
         advSup = self.renderBool( adv.cdbModuleCommandsSupported[ cmd ], "" )
         ovSup = ""
         if overrides and cmd in overrides.cdbModuleCommandsSupported:
            ovSup = self.renderBool( overrides.cdbModuleCommandsSupported[ cmd ],
                                     "" )
         if not adv.cdbModuleCommandsSupported[ cmd ] and not ovSup:
            # If the module doesn't support the command, and we haven't specified any
            # overrides for it, we won't display it. We do this to save on screen
            # space. There are up to 256 commands, but really only a handful are
            # usually supported.
            continue
         cmds[ cmd ] = ( advSup, ovSup )

      if overrides is None:
         return cmds

      for cmd in overrides.cdbModuleCommandsSupported:
         if cmd in cmds:
            # If we have an override set, and the command is already in the rendered
            # list, we can skip it now
            continue
         cmds[ cmd ] = (
            "no",
            self.renderBool( overrides.cdbModuleCommandsSupported[ cmd ], "" ) )

      return cmds

   def updateTableWithModuleCdbCaps( self, table ):
      '''
      This function populates the output table with all the module-wide generic CDB
      capabilities and overrides.
      '''
      adv = self.moduleCdbCapabilitiesAdvertised
      overrides = self.moduleCdbCapabilitiesOverride
      table.newRow( "Module CDB capabilities", "", "" )
      ( advCdbInstanceSup, advBackgroundModeSup, advAutoPagingSup, advMaxEplPages,
        advMaxEplTransactionSize, advMaxLplTransactionSize, advCdbCmdExecTrigger,
        advMaxCompletionTime, advCdbMaxBusyTime ) = self.getRenderedCdbData( adv )

      ( ovCdbInstanceSup, ovBackgroundModeSup, ovAutoPagingSup, ovMaxEplPages,
        ovMaxEplTransactionSize, ovMaxLplTransactionSize, ovCdbCmdExecTrigger,
        ovMaxCompletionTime, ovCdbMaxBusyTime ) = self.getRenderedCdbData(
            overrides )

      cdbCmds = self.getRenderedCdbCmds()

      # Add the output table rows for all this data
      table.newRow( "  CDB instances supported", advCdbInstanceSup,
                    ovCdbInstanceSup )
      table.newRow( "  Background mode supported", advBackgroundModeSup,
                    ovBackgroundModeSup )
      table.newRow( "  Auto paging supported", advAutoPagingSup, ovAutoPagingSup )
      table.newRow( "  Maximum EPL pages supported", advMaxEplPages, ovMaxEplPages )
      table.newRow( "  Maximum EPL transaction size", advMaxEplTransactionSize,
                    ovMaxEplTransactionSize )
      table.newRow( "  Maximum LPL transaction size", advMaxLplTransactionSize,
                    ovMaxLplTransactionSize )
      table.newRow( "  CDB command execution trigger", advCdbCmdExecTrigger,
                    ovCdbCmdExecTrigger )
      table.newRow( "  Maximum completion time", advMaxCompletionTime,
                    ovMaxCompletionTime )
      table.newRow( "  CDB maximum busy time", advCdbMaxBusyTime, ovCdbMaxBusyTime )

      # Render the list of supported commands (and commands with an explicit support
      # override)
      table.newRow( "  CDB module commands supported", "", "" )
      for cdbCmd, ( advSupport, overridenSupport ) in sorted( cdbCmds.items() ):
         table.newRow( f"    {cdbCmd}", advSupport, overridenSupport )

   def getRenderedFwData( self, model, noValueStr ):
      '''
      Helper function used to retrieve the rendered strings for firmware management
      CDB support from the provided TransceiverCdbCapabilitiesFirmwareManagement
      model.

      The noValueStr is used to indicate what should be rendered when an attribute's
      value is None.

      Parameters
      ----------
      model : TransceiverCdbCapabilitiesFirmwareManagement
      noValueStr : str

      Returns
      -------
      tuple of strings associated with all attributes of the model
      '''
      pswdType = noValueStr
      imageReadback = noValueStr
      skipFwErasedBlock = noValueStr
      copyCmdSup = noValueStr
      abortCmdSup = noValueStr
      startCmdPayload = noValueStr
      erasedByte = noValueStr
      maxEplTransaction = noValueStr
      maxLplTransaction = noValueStr
      writeMethod = noValueStr
      readMethod = noValueStr
      hitlessRestart = noValueStr
      startCmdMaxDuration = noValueStr
      abortCmdMaxDuration = noValueStr
      writeCmdMaxDuration = noValueStr
      completeCmdMaxDuration = noValueStr
      copyCmdMaxDuration = noValueStr
      if model:
         pswdType = fwPasswordTypeStr.get( model.firmwarePasswordType, noValueStr )
         imageReadback = self.renderBool( model.firmwareImageReadbackSupport,
                                          noValueStr )
         skipFwErasedBlock = self.renderBool( model.skipFirmwareErasedBlock,
                                              noValueStr )
         copyCmdSup = self.renderBool( model.firmwareCopyCommandSupport, noValueStr )
         abortCmdSup = self.renderBool( model.firmwareAbortCommandSupport,
                                        noValueStr )
         startCmdPayload = self.renderBytes( model.firmwareStartCommandPayloadSize,
                                             noValueStr )
         erasedByte = self.renderNumeric( model.firmwareErasedByteMarker,
                                          noValueStr )
         maxEplTransaction = self.renderBytes( model.firmwareMaxEplTransactionSize,
                                               noValueStr )
         maxLplTransaction = self.renderBytes( model.firmwareMaxLplTransactionSize,
                                               noValueStr )
         writeMethod = fwAccessMethodStr.get( model.firmwareWriteMethod, noValueStr )
         readMethod = fwAccessMethodStr.get( model.firmwareReadMethod, noValueStr )
         hitlessRestart = self.renderBool( model.firmwareHitlessRestartSupported,
                                           noValueStr )
         startCmdMaxDuration = self.renderDuration(
            model.firmwareStartCommandMaxDuration, noValueStr )
         abortCmdMaxDuration = self.renderDuration(
            model.firmwareAbortCommandMaxDuration, noValueStr )
         writeCmdMaxDuration = self.renderDuration(
            model.firmwareWriteCommandMaxDuration, noValueStr )
         completeCmdMaxDuration = self.renderDuration(
            model.firmwareCompleteCommandMaxDuration, noValueStr )
         copyCmdMaxDuration = self.renderDuration(
            model.firmwareCopyCommandMaxDuration, noValueStr )
      return ( pswdType, imageReadback, skipFwErasedBlock, copyCmdSup, abortCmdSup,
               startCmdPayload, erasedByte, maxEplTransaction, maxLplTransaction,
               writeMethod, readMethod, hitlessRestart, startCmdMaxDuration,
               abortCmdMaxDuration, writeCmdMaxDuration, completeCmdMaxDuration,
               copyCmdMaxDuration )

   def updateTableWithFwManagementCaps( self, table ):
      '''
      Function used to populate the output table with all the firmware management
      capabilities and overrides.
      '''
      # If no fw capabilities are advertised, and there are no overrides associated
      # with them, we won't display anything
      if ( self.firmwareManagementCdbCapabilitiesAdvertised is None and
           self.firmwareManagementCdbCapabilitiesOverride is None ):
         return

      adv = self.firmwareManagementCdbCapabilitiesAdvertised
      overrides = self.firmwareManagementCdbCapabilitiesOverride
      table.newRow( "", "", "" )
      table.newRow( "Firmware management CDB capabilities (CMD 0041h)", "", "" )

      # Grab the strings for all advertisements
      ( advPswdType, advImageReadback, advSkipFwErasedBlock, advCopyCmdSup,
        advAbortCmdSup, advStartCmdPaylod, advErasedByte, advMaxEplTransaction,
        advMaxLplTransaction, advWriteMethod, advReadMethod, advHitlessRestart,
        advStartCmdMaxDuration, advAbortCmdMaxDuration, advWriteCmdMaxDuration,
        advCompleteCmdMaxDuration, advCopyCmdMaxDuration ) = \
            self.getRenderedFwData( adv, noValueStr="unknown" )

      # Grab the strings for all overrides
      ( ovPswdType, ovImageReadback, ovSkipFwErasedBlock, ovCopyCmdSup,
        ovAbortCmdSup, ovStartCmdPaylod, ovErasedByte, ovMaxEplTransaction,
        ovMaxLplTransaction, ovWriteMethod, ovReadMethod, ovHitlessRestart,
        ovStartCmdMaxDuration, ovAbortCmdMaxDuration, ovWriteCmdMaxDuration,
        ovCompleteCmdMaxDuration, ovCopyCmdMaxDuration ) = \
            self.getRenderedFwData( overrides, noValueStr="" )

      # Add all the rows to the output table
      table.newRow( "  Firmware password type", advPswdType, ovPswdType )
      table.newRow( "  Firmware image readback supported", advImageReadback,
                    ovImageReadback )
      table.newRow( "  Skip firmware erased block", advSkipFwErasedBlock,
                    ovSkipFwErasedBlock )
      table.newRow( "  Firmware copy command supported", advCopyCmdSup,
                    ovCopyCmdSup )
      table.newRow( "  Firmware abort command supported", advAbortCmdSup,
                    ovAbortCmdSup )
      table.newRow( "  Firmware start command payload size", advStartCmdPaylod,
                    ovStartCmdPaylod )
      table.newRow( "  Firmware erased byte marker", advErasedByte, ovErasedByte )
      table.newRow( "  Firmware maximum EPL transaction size", advMaxEplTransaction,
                    ovMaxEplTransaction )
      table.newRow( "  Firmware maximum LPL transaction size", advMaxLplTransaction,
                    ovMaxLplTransaction )
      table.newRow( "  Firmware write method", advWriteMethod, ovWriteMethod )
      table.newRow( "  Firmware read method", advReadMethod, ovReadMethod )
      table.newRow( "  Firmware hitless restart supported", advHitlessRestart,
                    ovHitlessRestart )
      table.newRow( "  Firmware start command maximum duration",
                    advStartCmdMaxDuration, ovStartCmdMaxDuration )
      table.newRow( "  Firmware abort command maximum duration",
                    advAbortCmdMaxDuration, ovAbortCmdMaxDuration )
      table.newRow( "  Firmware write command maximum duration",
                    advWriteCmdMaxDuration, ovWriteCmdMaxDuration )
      table.newRow( "  Firmware complete command maximum duration",
                    advCompleteCmdMaxDuration, ovCompleteCmdMaxDuration )
      table.newRow( "  Firmware copy command maximum duration",
                    advCopyCmdMaxDuration, ovCopyCmdMaxDuration )

   def updateTableWithSlotDetails( self, slotNum, table ):
      '''
      This is the main function of this model, it handler populating all the rows in
      the output table related to specified slot.
      '''
      # If the module-wise CDB capabilities are not available yet, we have nothing to
      # display
      if self.moduleCdbCapabilitiesAdvertised is None:
         return

      table.newRow( f"Slot {slotNum}", "", "" )
      table.newRow( f"Transceiver SN: {self.serialNumber}", "", "" )
      table.newRow( "", "Support", "Override" )
      table.newRow( "", "-------", "--------" )

      self.updateTableWithModuleCdbCaps( table )
      self.updateTableWithFwManagementCaps( table )

class TransceiverCdbCapabilitiesSlotCollection( Model ):
   __revision__ = 2
   ports = Dict( keyType=str, valueType=TransceiverCdbCapabilitiesSlotDetail,
                 help="Mapping between the slot name and the CDB capabilities "
                      "information of the inserted transceiver" )

   def render( self ):
      if self.ports is None:
         return

      # Create the table
      table = TableOutput.TableFormatter( 0, tableWidth=130 )
      columnFormat = TableOutput.Format( justify="left", minWidth=30 )
      columnFormat.padLimitIs( True )
      table.formatColumns( columnFormat, columnFormat, columnFormat )

      slotRange = Arnet.sortIntf( self.ports )
      for slot in slotRange:
         slotDetail = self.ports[ slot ]
         slotDetail.updateTableWithSlotDetails( slot, table )
         table.newRow( "", "", "" )

      # If nothing was added to the table, don't print anything
      if not table.entries_:
         return

      print( table.output(), end='' )
