# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Model, Dict, Submodel, Enum, Float
from IntfModels import Interface
import TableOutput
from TypeFuture import TacLazyType

AlarmStatus = TacLazyType( "Xcvr::XcvrAlarm::AlarmStatus" )

# --------------------------------------------------------------------------------
# Models for show transceiver error-correction coherent degrade [ interface INTFS ]
# --------------------------------------------------------------------------------
class TransceiverCoherentFecDegradeThresholds( Model ):
   tRaise = Float( help="Value of error-correction degrade raise threshold" )
   tClear = Float( help="Value of error-correction degrade clear threshold" )

class TransceiverCoherentFecDegradeParameter( Model ):
   status = Enum( values=AlarmStatus.attributes,
                  help='Error-correction degrade alarm status' )
   configured = Submodel( valueType=TransceiverCoherentFecDegradeThresholds,
                          help="Values of configured degrade thresholds",
                          optional=True )
   operational = Submodel( valueType=TransceiverCoherentFecDegradeThresholds,
                           help="Values of operational degrade thresholds ",
                           optional=True )

   @staticmethod
   def printableAlarmStatus( alarmStatus ):
      return "not present" if alarmStatus == "notPresent" else alarmStatus

   def renderParam( self, intf, degradeType, table ):
      configured = self.configured
      # If degrade configuration has not been accepted, print 'none'
      operational = self.operational
      operationalRaiseStr = f"{operational.tRaise:6.2e}" if operational else "none"
      operationalClearStr = f"{operational.tClear:6.2e}" if operational else "none"
      table.newRow( intf, degradeType,
                    f"{configured.tRaise:6.2e}",
                    f"{configured.tClear:6.2e}",
                    operationalRaiseStr,
                    operationalClearStr,
                    self.printableAlarmStatus( self.status ) )

class TransceiverCoherentFecDegradeSlot( Model ):
   detected = Submodel( valueType=TransceiverCoherentFecDegradeParameter,
                        help="Detected degrade status", optional=True )
   excessive = Submodel( valueType=TransceiverCoherentFecDegradeParameter,
                              help="Excessive degrade status", optional=True )

   def renderSlot( self, intf, table ):
      # Print status of detected and excessive degrade monitoring per interface
      if self.detected is not None:
         self.detected.renderParam( intf, "detected", table )
      if self.excessive is not None:
         self.excessive.renderParam( intf, "excessive", table )

class TransceiverCoherentFecDegradeCollection( Model ):
   # Interface keyType is still valid for slots
   slots = Dict( keyType=Interface,
                 valueType=TransceiverCoherentFecDegradeSlot,
                 help="Mapping between slot name and "
                 "coherent error-correction degrade info" )

   def _createOutputTable( self ):
      formatIntfColumn = TableOutput.Format( justify="left", minWidth=5 )
      formatDegradeColumn = TableOutput.Format( justify="left",
                                                minWidth=7, maxWidth=9, wrap=True )
      formatOperationalColumn = TableOutput.Format( justify="left",
                                                    minWidth=8, maxWidth=9,
                                                    wrap=True )
      formatStatusColumn = TableOutput.Format( justify="left", minWidth=6 )

      # Remove additional column paddings
      formatIntfColumn.padLimitIs( True )
      formatDegradeColumn.padLimitIs( True )
      formatOperationalColumn.padLimitIs( True )
      formatStatusColumn.padLimitIs( True )

      headers = ( "Slot",
                  "Degrade  Type",
                  ( "Configured", ( "Raise   Threshold", "Clear   Threshold" ) ),
                  ( "Operational", ( "Raise   Threshold", "Clear   Threshold" ) ),
                  "Status" )
      table = TableOutput.createTable( headers )
      table.formatColumns( formatIntfColumn, formatDegradeColumn,
                           formatOperationalColumn, formatOperationalColumn,
                           formatOperationalColumn, formatOperationalColumn,
                           formatStatusColumn )
      return table

   def render( self ):
      if not self.slots:
         return
      outputTable = self._createOutputTable()
      for slot in Arnet.sortIntf( self.slots ):
         self.slots[ slot ].renderSlot( slot, outputTable )
      print( outputTable.output() )
