# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from itertools import chain
import Arnet
import Ark
import XcvrErrorCliLib as xecl
from CliModel import Model, Dict, Enum, Int, Float
from TypeFuture import TacLazyType

# --------------------------------------------------------------------------------
#
# Models for "show transceiver error"
#
# --------------------------------------------------------------------------------
#
# Example Output:
#
# Ethernet 10
# Error                       State                     Count       Last Error
# -----------------------     ---------------------     -------     -----------
# Controller fault state      unsupported power         1           0:10:07 ago
# Mixed application state     accepted                  0           never
# Smbus failures              eeprom read               3           0:10:08 ago
# Config state                accepted                  0           never
# Module fault                data memory corrupted     1           0:10:03 ago
# Datapath firmware error                               0           never
# Module firmware error                                 0           never
# VDM freeze timeout                                    1           0:09:55 ago
# VDM unfreeze timeout                                  0           never
# Module power up timeout                               3           0:09:59 ago
# Module power down timeout                             1           0:10:02 ago
# Datapath deinit timeout                               1           0:10:05 ago

# Last reason error has occurred
FaultyStateReason = TacLazyType( "Xcvr::FaultyStateReason" )
CmisModuleFaultCause = TacLazyType( "Xcvr::CmisModuleFaultCause" )
SmbusErrorLoc = TacLazyType( "Xcvr::SmbusErrorLoc" )
CmisConfigurationStatus = TacLazyType( "Xcvr::CmisConfigurationStatus" )
MixedApplicationStatus = TacLazyType( "Xcvr::MixedApplicationStatus" )

class TransceiverErrorValueFormat( Model ):
   count = Int( help="Number of failures since system boot" )
   lastError = Float( help="Last error update time in UTC" )
   lastReason = Enum( values=chain( FaultyStateReason.attributes,
                                    CmisModuleFaultCause.attributes,
                                    SmbusErrorLoc.attributes,
                                    CmisConfigurationStatus.attributes,
                                    MixedApplicationStatus.attributes ),
                      help="Last reason for error", optional=True )

   def renderRow( self, errorKey ):
      if self.lastReason in FaultyStateReason.attributes:
         lastReason = xecl.getFaultyStateReason( self.lastReason )
      elif self.lastReason in CmisModuleFaultCause.attributes:
         lastReason = xecl.getModuleFaultCause( self.lastReason )
      elif self.lastReason in SmbusErrorLoc.attributes:
         lastReason = xecl.getSmbusErrorReason( self.lastReason )
      elif self.lastReason in CmisConfigurationStatus.attributes:
         lastReason = xecl.getCmisConfigStatus( self.lastReason )
      elif self.lastReason in MixedApplicationStatus.attributes:
         lastReason = xecl.getMixedApplicationStatus( self.lastReason )
      else:
         lastReason = ""

      lastErrorDelta = Ark.timestampToStr( self.lastError )
      errorValues = [ errorKey, lastReason, str( self.count ), lastErrorDelta ]
      errorString = xecl.columnFormat.format( *errorValues )
      print( errorString )

class TransceiverErrorValues( Model ):
   transceiverErrors = Dict( keyType=str, valueType=TransceiverErrorValueFormat,
                      help='A mapping of error to its details' )

   def render( self ):
      # Need to render rows in the same order as they appear in xcvrErrorKeys().
      for key in xecl.errorKeys:
         modelKey = xecl.lowerCamelCase( key )
         # If the error key is not in the dict, than the error is not supported by
         # the module and should not be rendered.
         if modelKey in self.transceiverErrors:
            self.transceiverErrors[ modelKey ].renderRow( key )

class TransceiverErrorInterfaces( Model ):
   ports = Dict( keyType=str, valueType=TransceiverErrorValues,
                 help="Mapping between port name and transceiver errors" )

   def render( self ):
      if not self.ports:
         return

      headers = [ "Error", "State", "Count", "Last Error" ]
      headersStr = xecl.columnFormat.format( *headers )
      headerLinesLengthList = [ "-" * headerLineLength for headerLineLength
                                in xecl.columnWidths ]
      headerLines = xecl.columnFormat.format( *headerLinesLengthList )

      for port in Arnet.sortIntf( self.ports ):
         print( port )
         print( headersStr )
         print( headerLines )
         self.ports[ port ].render()
         print()
