#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import Arnet
import HumanReadable
import Tac
from IntfModels import Interface
from CliModel import Dict, Model, List, Submodel, Bool, Enum, Int, Float
from XcvrPrbsLib import ( fecEnableKw, prbsCliHostKw, prbsCliMediaKw,
                          prbsCliGeneratorKw, prbsCliCheckerKw, configuredKw,
                          operationalKw, noneKw )
from XcvrPrbsLib import CliSideEnum, aristaSupportedPatterns

# ------------------------------------------------------------------------------
#
# Models for show interfaces transceiver diag test pattern
#
# ------------------------------------------------------------------------------

headerFormatConfig = '{:15s} {:15s} {:15s} {:15s} {:15s}'

# This is a list of column sizes for the 'show test pattern counters' command,
# Will result in a list of values/headers/spacings looking like this:
#
#  <value 1>   <value 2>   ...   <value n>
#  spacing 1   spacing 2   ...   spacing n
# <---------> <--------->  ...  <--------->
#
countersColumnSizes = [ 15, 5, 14, 21, 14, 12, 16 ]
columnFormatCounters = " ".join( f'{{:{c}s}}'
                                 for c in countersColumnSizes )

# This is a list of column sizes for the 'show test pattern counters detail'
# command. Will result in a list of values/headers/spacings looking like this:
#
# <Value Name> <Current Value> <Number of Changes> <Last Change>
#                 spacing 1         spacing 2        spacing 3
#              <-------------> <-----------------> <----------->
# i.e
# Lock State      locked                1            0:00:00 ago
#
countersDetailColumnSizes = [ 22, 12, 15 ]
countersDetailColumnFormat = "".join( f"{{:<{c}}}"
                                      for c in countersDetailColumnSizes )

class InterfacesTransceiverTestPatternSideConfig( Model ):
   """
   This class holds the options for a given side and type.
   Thus, for each linear combination of:
   [ intf ] [ line | system ] [ transmitter | receiver ] [ configured | operational ]
   there is one of these objects, if that particular side has an active
   config
   """
   pattern = Enum( values=aristaSupportedPatterns,
                   help="The pattern present in this config" )
   fec = Bool( help="Forward error correction enabled in this config" )

   def renderModel( self ):
      configList = [ self.pattern.upper() ]
      if self.fec:
         configList.append( fecEnableKw )
      configStr = ",".join( configList )
      return configStr

class InterfacesTransceiverTestPatternSideConfigs( Model ):
   """
   This class holds both the configured and operational options for a given
   side's transmitter or receiver.
   Thus, for each linear combination of:
   [ intf ] [ line | system ] [ transmitter | receiver ]
   There is one of these objects. However, whether or not these objects hold
   anything depends on the presence of active config in the tac models.
   """
   configuredOptions = Submodel(
      help="Test Pattern options selected by the user", optional=True,
      valueType=InterfacesTransceiverTestPatternSideConfig )
   operationalOptions = Submodel(
      help="Test Pattern options currently operational in the module",
      valueType=InterfacesTransceiverTestPatternSideConfig, optional=True )

   def renderModel( self ):
      # Mapping the lists of supported patterns/options to a desired format
      configuredOptionsStr = noneKw
      operationalOptionsStr = noneKw
      if self.configuredOptions:
         configuredOptionsStr = self.configuredOptions.renderModel()
      if self.operationalOptions:
         operationalOptionsStr = self.operationalOptions.renderModel()
      return { configuredKw: configuredOptionsStr,
               operationalKw: operationalOptionsStr }

class InterfacesTransceiverTestPatternConfiguration( Model ):
   """
   This class holds all of the options for a given side.
   Thus for each linear combination of [ intf ] [ line | system ] there exists
   one of these objects.
   """
   transmitter = Submodel(
      help="Configuration for transmitter / generator",
      valueType=InterfacesTransceiverTestPatternSideConfigs )
   receiver = Submodel(
      help="Configuration for receiver / checker",
      valueType=InterfacesTransceiverTestPatternSideConfigs )

   def renderModel( self, intf ):
      if self.transmitter or self.receiver:
         transmitterOptions = self.transmitter.renderModel()
         receiverOptions = self.receiver.renderModel()
         print( headerFormatConfig.format( intf, transmitterOptions[ configuredKw ],
                                           receiverOptions[ configuredKw ],
                                           transmitterOptions[ operationalKw ],
                                           receiverOptions[ operationalKw ] ) )

class InterfacesTransceiverTestPatternConfigurationCollection( Model ):
   """
   This class holds two collections: One for the line-side test pattern options,
   and one for the system-side options. There is an entry in both collections for
   each interface.
   """
   lineInterfaces = Dict( keyType=Interface,
                          valueType=InterfacesTransceiverTestPatternConfiguration,
                          help="Mapping from interface name to line/media side "
                               "transceiver Test Pattern config information" )
   systemInterfaces = Dict( keyType=Interface,
                            valueType=InterfacesTransceiverTestPatternConfiguration,
                            help="Mapping from interface name to system/host side "
                                 "transceiver Test Pattern config information" )

   def render( self ):
      if not self.lineInterfaces or not self.systemInterfaces:
         return
      # pylint: disable-next=consider-using-f-string
      optionTypeStr = "{:>36s} {:>31s}".format( configuredKw.capitalize(),
                                                operationalKw.capitalize() )
      headersStr = headerFormatConfig.format( 'Interface',
                                              prbsCliGeneratorKw.capitalize(),
                                              prbsCliCheckerKw.capitalize(),
                                              prbsCliGeneratorKw.capitalize(),
                                              prbsCliCheckerKw.capitalize() )
      headerLines = headerFormatConfig.format( '-' * 15, '-' * 15, '-' * 15,
                                               '-' * 15, '-' * 15 )
      # Line/Media side options
      print( prbsCliMediaKw.capitalize() )
      print( optionTypeStr )
      print( headersStr )
      print( headerLines )
      for intf in Arnet.sortIntf( self.lineInterfaces ):
         self.lineInterfaces[ intf ].renderModel( intf )
      print()
      # System/Host side options
      print( prbsCliHostKw.capitalize() )
      print( optionTypeStr )
      print( headersStr )
      print( headerLines )
      for intf in Arnet.sortIntf( self.systemInterfaces ):
         self.systemInterfaces[ intf ].renderModel( intf )
      print()

# ------------------------------------------------------------------------------
#
# Models for show interfaces transceiver diag test pattern capabilities
#
# ------------------------------------------------------------------------------
#
# Example Output:
#
# Ethernet13/1
# Line transmitter patterns: PRBS7Q,9Q,13Q,15Q,23Q,31Q
# Line transmitter options: fec
# Line receiver patterns: PRBS7Q,9Q,13Q,15Q,23Q,31Q
# Line receiver options: fec
# System transmitter patterns: PRBS7Q,9Q,13Q,15Q,23Q,31Q
# System transmitter options: none
# System receiver patterns: PRBS7Q,9Q,13Q,15Q,23Q,31Q
# System receiver options: none
#
# Ethernet16/1
# Transceiver does not support test pattern diagnostics
#
# Ethernet17/1
# Transceiver does not support test pattern diagnostics
#

class InterfacesTransceiverTestPatternSideCapabilities( Model ):
   supportedPatterns = List( valueType=str, optional=True,
                             help="The possible patterns that the "
                                  "module's diagnostics can generate "
                                  "and do error counting for" )
   fecSupported = Bool( help="Does this module support forward "
                             "error correction during test pattern testing" )

   @staticmethod
   def patternSortingFunction( pattern ):
      return ( len( pattern ) - pattern.count( 'Q' ), pattern )

   def renderModel( self, sideName ):
      sideName = sideName.capitalize()
      PatternEnum = Tac.Type( "Xcvr::TestPatternType" )
      # Mapping the lists of supported patterns/options to a desired format
      patternsList = [ pattern.replace( "prbs", "" ).upper()
                       for pattern in self.supportedPatterns ]
      patternsList = sorted( patternsList, key=self.patternSortingFunction )

      supportedOptions = []
      if self.fecSupported:
         supportedOptions.append( fecEnableKw )

      patternsString = ",".join( patternsList )
      # If there are only non-prbs patterns in the list of patterns we
      # want to omit the 'PRBS' identifier
      patternId = '' if patternsString == PatternEnum.ssprq.upper() else 'PRBS'
      if patternsString:
         patternsString = f'{patternId}{patternsString}'
      else:
         patternsString = noneKw
      print( f"{sideName} patterns: {patternsString}" )

      optionsString = ",".join( supportedOptions )
      if patternsList:
         # Only print the options line if at least one pattern is available
         print( f"{sideName} options: {optionsString or noneKw}" )
      else:
         print()

class InterfacesTransceiverTestPatternCapabilities( Model ):
   systemTransmitter = Submodel(
      help="Capabilities for system transmitter / host generator",
      valueType=InterfacesTransceiverTestPatternSideCapabilities, optional=True )
   systemReceiver = Submodel(
      help="Capabilities for system receiver / host checker",
      valueType=InterfacesTransceiverTestPatternSideCapabilities, optional=True )
   lineTransmitter = Submodel(
      help="Capabilities for line transmitter / media generator",
      valueType=InterfacesTransceiverTestPatternSideCapabilities, optional=True )
   lineReceiver = Submodel(
      help="Capabilities for line receiver / media checker",
      valueType=InterfacesTransceiverTestPatternSideCapabilities, optional=True )

   def renderModel( self ):
      if any( [ self.systemTransmitter, self.systemReceiver,
                self.lineTransmitter, self.lineReceiver ] ):
         for side in CliSideEnum:
            getattr( self, side.name ).renderModel( side.value )
      else:
         print( "Transceiver does not support test pattern diagnostics" )

class InterfacesTransceiverTestPatternCapabilitiesCollection( Model ):
   # Attribute used for test pattern capabilities only, since
   # capabilities are grouped by interface and not side
   interfaces = Dict( keyType=Interface,
                      valueType=InterfacesTransceiverTestPatternCapabilities,
                      help="Mapping between interface name "
                           "and the transceiver info" )

   def render( self ):
      if not self.interfaces:
         return
      for intf in Arnet.sortIntf( self.interfaces ):
         print( intf )
         self.interfaces[ intf ].renderModel()
         print()

# ------------------------------------------------------------------------------
#
# Models for show interfaces transceiver diag test pattern counters [ detail ]
#
# ------------------------------------------------------------------------------
#
# Example Output:
#
# System
#
# Interface       Lane  ...  Largest Burst  Burst Count  Last Error Time
# --------------- ----- ...  -------------- ------------ ----------------
# Ethernet9/1     0     ...  1              8            0:00:24 ago
# Ethernet9/1     1     ...  1              27           0:00:01 ago
# Ethernet9/1     2     ...  3              27           0:00:01 ago
# Ethernet9/1     3     ...  1              27           0:00:01 ago
# Ethernet9/1     4     ...  13             19           0:00:01 ago
# Ethernet9/1     5     ...  7              27           0:00:01 ago
# Ethernet9/1     6     ...  3              27           0:00:01 ago
# Ethernet9/1     7     ...  22             27           0:00:01 ago
#
# Example Detailed Output:

# Line
#
#  Ethernet11/1
#
#     Last Clear:                0:04:41 ago
#
#     Operational Test Pattern:  PRBS7Q
#
#
#                                Current State         Changes     Last Change
#
#                                --------------------- ----------- --------------
#
#     Lane 0
#
#      Lock State                locked                1           never
#
#      Largest Burst             2
#
#      Bit Errors                12                    9           0:00:00 ago
#
#      Total Bits                29.7029012 Tb
#
#      Bit Error Rate            4.36e-13
#
#      .
#      .
#      .
#
#     Lane n
#
#      Lock State                locked                1           never
#
#      Largest Burst             5
#
#      Bit Errors                61                    28          0:00:08 ago
#
#      Total Bits                29.7034926 Tb
#
#      Bit Error Rate            2.04e-12
#

class InterfacesTransceiverTestPatternLaneCountersDetail( Model ):
   """
   This class holds all the 'detailed' test pattern results and link status
   for a given side and lane.
   """
   totalBits = Int( help="The total number of bits sent through the link" )
   ber = Float( help="The bit error rate of the link" )
   linkStateChanges = Int( help="The number of link state transitions "
                                "since test pattern was enabled" )
   lastLinkStateChange = Float( help="The time when the link state last changed",
                                optional=True )

class InterfacesTransceiverTestPatternLaneCounters( Model ):
   """
   This class holds the test pattern results and link status for a given side
   and lane. Since this class holds the non-detailed version of the command,
   some attributes from TestPatternLaneStatus are missing.
   """
   linkState = Enum( values=Tac.Type( "Hardware::Phy::TestPatternLock" ).attributes,
                     help="The lock state of the lane link under test" )
   bitErrors = Int( help="The number of bit errors since test pattern was enabled" )
   largestBurst = Int( help="The largest amount of bit errors in a polling period "
                            "since test pattern was enabled" )
   burstCount = Int( help="The number of polling periods where counters changed "
                          "since test pattern was enabled" )
   lastErrorTime = Float( help="The last time an error was reported since "
                               "test pattern was enabled", optional=True )
   detailed = Submodel( valueType=InterfacesTransceiverTestPatternLaneCountersDetail,
                        help="Detailed test pattern lane counters", optional=True )

   def renderModel( self, intfName, lane ):
      lastErrorTimeDelta = Ark.timestampToStr( self.lastErrorTime, now=Tac.utcNow() )
      laneStatusValues = [ intfName, str( lane ), self.linkState,
                           str( self.bitErrors ), str( self.largestBurst ),
                           str( self.burstCount ), lastErrorTimeDelta ]
      laneStatusString = columnFormatCounters.format( *laneStatusValues )
      print( laneStatusString )

   def renderModelDetailed( self ):
      rowHeaderLeftOffset = 6 # spaces before the row titles/headers
      rowHeaderRightOffset = 25 # offset to fit the headers + spacing
      # Format for the row titles i.e "Lock State"
      rowHeaderFormat = " " * rowHeaderLeftOffset + \
                        f"{{:{rowHeaderRightOffset}s}} "
      # Combine the row title fmt with the format for column values
      # to create a "row" of values
      rowFormat = rowHeaderFormat + countersDetailColumnFormat

      lockStateValues = [ "Lock State", self.linkState,
                          self.detailed.linkStateChanges,
                          Ark.timestampToStr( self.detailed.lastLinkStateChange,
                                              now=Tac.utcNow() ) ]
      # Some results don't have a "change" or "last change" attribute so those
      # columns are left blank, hence the "".
      largestBurstValues = [ "Largest Burst", self.largestBurst, "", "" ]
      bitErrorValues = [ "Bit Errors", self.bitErrors, self.burstCount,
                         Ark.timestampToStr( self.lastErrorTime, now=Tac.utcNow() ) ]
      # Convert the total bits value to a human readable format
      # ( in (Kilo|Mega|Giga|Tera)bytes ) with 9 significant digits
      totalBits = HumanReadable.formatValueSi( self.detailed.totalBits,
                                               precision=9, unit="b" )
      totalBitsValues = [ "Total Bits", totalBits, "", "" ]
      berValues = [ "Bit Error Rate", self.detailed.ber, "", "" ]

      for line in [ lockStateValues, largestBurstValues, bitErrorValues,
                    totalBitsValues, berValues ]:
         print( rowFormat.format( *line ) )

class InterfacesTransceiverTestPatternCounters( Model ):
   """
   This class holds a collection of test pattern counters for all lanes associated
   with a given side (system or line).
   """
   operationalPattern = Enum( values=aristaSupportedPatterns, optional=True,
                              help="The active test pattern" )
   lastClear = Float( help="The time when the pattern was last cleared",
                      optional=True )
   lanes = List( valueType=InterfacesTransceiverTestPatternLaneCounters,
                 optional=True, help="Mapping from lane number to "
                                     "test pattern result lane counters" )

   def renderModel( self, intfName ):
      for lane, laneStatus in enumerate( self.lanes ):
         if laneStatus:
            laneStatus.renderModel( intfName, lane )

   def renderModelDetailed( self ):
      # Formatting for the "last clear" and "operational test pattern" attributes,
      # defined here as side-specific values ( as opposed to lane-specific )
      sideValuesOffset = 5
      sideValuesFormat = ( " " * sideValuesOffset ) + "{:26s} {:20}"
      # Formatting for the column headers/titles
      countersHeadersOffset = 32
      countersDetailHeaderFormat = ( " " * countersHeadersOffset +
                                     countersDetailColumnFormat )
      lines = [ "-" * ( headerLineLength - 1 ) for headerLineLength
                in countersDetailColumnSizes ]
      # Formatting for the lane numbers, using the same spacing as the "side values"
      laneFormat = ( " " * sideValuesOffset ) + "{:4} {}"

      lastClearDelta = Ark.timestampToStr( self.lastClear, now=Tac.utcNow() )
      print( sideValuesFormat.format( "Last Clear:", lastClearDelta ) )

      if self.operationalPattern:
         patternVal = self.operationalPattern.upper()
      else:
         patternVal = "none"
      print( sideValuesFormat.format( "Operational Test Pattern:", patternVal ) )

      print()
      print( countersDetailHeaderFormat.format( "Current State",
                                                "Changes", "Last Change" ) )
      print( countersDetailHeaderFormat.format( *lines ) )

      for lane, laneStatus in enumerate( self.lanes ):
         print( laneFormat.format( "Lane", lane ) )
         laneStatus.renderModelDetailed()

class InterfacesTransceiverTestPatternCountersCollection( Model ):
   """
   This class holds two collections: One for the line-side test pattern counters,
   and one for the system-side counters. There is an entry in both collections for
   each interface.
   """
   _detailed = Bool( help="Whether we're using the detailed "
                          "version of the command." )
   lineInterfaces = Dict( keyType=Interface,
                          valueType=InterfacesTransceiverTestPatternCounters,
                          help="List of line/media side transceiver test pattern "
                               "result counters indexed by lane number" )
   systemInterfaces = Dict( keyType=Interface,
                            valueType=InterfacesTransceiverTestPatternCounters,
                            help="List of system/host side transceiver test pattern "
                                 "result counters indexed by lane number" )

   def detailedIs( self, detailed ):
      self._detailed = detailed

   def render( self ):
      if self._detailed:
         self.renderModelDetailed()
      else:
         self.renderModel()

   def renderModel( self ):
      headers = [ "Interface", "Lane", "Link State", "Bit Errors",
                  "Largest Burst", "Burst Count", "Last Error Time" ]

      headersStr = columnFormatCounters.format( *headers )
      headerLinesLengthList = [ "-" * headerLineLength for headerLineLength
                                in countersColumnSizes ]
      headerLines = columnFormatCounters.format( *headerLinesLengthList )

      for sideName, intfCollection in [ ( prbsCliMediaKw, self.lineInterfaces ),
                                        ( prbsCliHostKw, self.systemInterfaces ) ]:
         if intfCollection:
            print( sideName.capitalize() )
            print()
            print( headersStr )
            print( headerLines )
            for intf in Arnet.sortIntf( intfCollection ):
               intfCollection[ intf ].renderModel( intf )
            print()

   def renderModelDetailed( self ):
      # Formatting for the interface number displayed, at a slight offset
      # from the side name
      intfOffset = 2
      intfFormat = ( " " * intfOffset ) + "{:5s}"

      for sideName, intfCollection in [ ( prbsCliMediaKw, self.lineInterfaces ),
                                        ( prbsCliHostKw, self.systemInterfaces ) ]:
         if intfCollection:
            print( sideName.capitalize() )
            for intf in Arnet.sortIntf( intfCollection ):
               print( intfFormat.format( intf ) )
               intfCollection[ intf ].renderModelDetailed()
            print()
