# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import math
from typing import Optional

from TypeFuture import TacLazyType

import CliPlugin.XcvrAllStatusDir
from CliDynamicPlugin.XcvrShowChannelsModel import (
    InterfacesTransceiverChannelRow, InterfacesTransceiverChannelsBase,
    InterfaceTransceiverChannels )
from CliPlugin.IntfCli import Intf
from CliPlugin.XcvrCli import computeWavelength, getAllIntfsWrapper
from CliPlugin.XcvrConfigCli import ( getTuningChannel, ghzFreqToMhz,
                                     mhzFreqToGhz )
from XcvrLib import getXcvrSlotName, getXcvrStatus

XcvrType = TacLazyType( 'Xcvr::XcvrType' )
xcvrStatusDir = None

# ----------------------------------------------------------------------------
#
# "show interfaces [<name>] transceiver channels"
#
# ----------------------------------------------------------------------------

def _createXcvrChannel( baseFreq: float, freq: float,
                       gridSpacingList: list[ float ] ) -> \
                        InterfacesTransceiverChannelRow:
   """
   Populates a channel model with frequency, wavelength, and grid information.
   A channel exists for every multiple of the frequency spacing(s) within the c-band.

   Parameters
   -----------------
   freq : float
      Float representing the lower frequency boundary of a possible channel
   wavelength : float
      Wavelength equivalent of the lower boundary frequency ( wavelength = c/f)]
   gridSpacingList : List[ float ]
      List of integer grid spacings, measured in MHz

   Returns
   -----------------
   A InterfacesTransceiverChannel model with attributes populated
   """
   model = InterfacesTransceiverChannelRow()
   model.wavelength = computeWavelength( freq )
   model.frequency = mhzFreqToGhz( freq )
   for gridSpacing in gridSpacingList:
      # Model wants a dictionary mapping of grid spacing (str format) to channel no.
      gridSpacingStr = str( mhzFreqToGhz( gridSpacing ) )
      if getTuningChannel( baseFreq, freq, gridSpacing ) is not None:
         model.addChannelNumber( gridSpacingStr,
                                 getTuningChannel( baseFreq, freq, gridSpacing ) )
   return model

def _populateInterfacesXcvrChannels( mode, xcvrStatus, shortName: str,
                                     userMinFreq: float, userMaxFreq: float,
                                     gridSpacingList: list[ float ] ) -> \
                                       Optional[ InterfaceTransceiverChannels ]:
   """
   Creats a channel model for each possible spacing division within (minFreq,
   maxFreq).

   Parameters
   ----------
   xcvrStatus : Xcvr::XcvrNewStatus derived type
   shortName : str
      intf name
   userMinFreq : float
      Minimum frequency requested for output in MHz
   userMaxFreq : float
      Maximum frequency requested for output in MHz
   gridSpacingList : List[ float ]
      A list of the requested grid spacings in MHz. E.g [100000, 50000, 25000]

   Returns InterfaceTransceiverChannels model.
   """
   model = InterfaceTransceiverChannels()
   tuningCap = xcvrStatus.tuningCapabilities
   firstFreq = tuningCap.firstFrequency
   baseFreq = tuningCap.baseFrequency
   lastFreq = tuningCap.lastFrequency

   minimumFreq = max( firstFreq, baseFreq )
   maximumFreq = lastFreq
   minimumFreq = max( minimumFreq, userMinFreq ) if userMinFreq is not None else \
      minimumFreq
   maximumFreq = min( maximumFreq, userMaxFreq, ) if userMaxFreq is not None else \
      maximumFreq

   if minimumFreq > maximumFreq:
      return None

   # Smaller grid spacings are factors of the larger grid spacings
   # All options are: 100, 50, 25, 12.5, 6.25 GHz
   # This lets us iterate with just the smallest grid spacing as an incrementor when
   # creating each possible channel.
   smallestGridSpacing = min( gridSpacingList )
   model.gridSpacings = [ mhzFreqToGhz( gridSpacing ) for gridSpacing in
                          gridSpacingList ]
   # Round to the closest multiple of smallestGridSpacing. We use this freq to
   # create each frequency row.
   freq = smallestGridSpacing * math.ceil( minimumFreq / smallestGridSpacing )
   while freq <= maximumFreq:
      model.appendChannel( _createXcvrChannel( baseFreq, freq, gridSpacingList ) )
      freq += smallestGridSpacing
   return model

def spacingChannelStringToMHzList( channelString: str ) -> list[ float ]:
   """
   Helper to convert a string of comma-seperated integers (representing frequency
   spacing channels) into an array in MHz.
   If nothing is supplied, we return [100000, 50000] MHz by default
   Parameters
   ---------------
   channelString : str
      A string of comma-seperated integers e.g "100,50"
   """
   if channelString is None:
      return [ 100000, 50000 ]
   # Retrieve a unique collection of grid-spacings.
   gridSpacingListStr = channelString.split( "," )
   gridSpacingListMhz = [ ghzFreqToMhz( float( gridSpacing ) ) for
                                       gridSpacing in gridSpacingListStr ]
   return gridSpacingListMhz

def supportsShowXcvrChannels( status ) -> bool:
   # This command is only for tunable CFP2 and SFP transceivers. Other
   # transceivers also can have tunableWavelength capabilities!
   return status.capabilities.tunableWavelength and status.typeInUse in \
      ( XcvrType.sfpPlus, XcvrType.cfp2 )

def showInterfacesXcvrChannels( mode, args ):
   """
   Cli handler for 'show interface transceiver channels'
   """
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   minFreq = args.get( 'MIN_FREQ' )
   maxFreq = args.get( 'MAX_FREQ' )
   grid = args.get( 'GRID' )
   gridSpacingListMhz = spacingChannelStringToMHzList( grid )

   model = InterfacesTransceiverChannelsBase()
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model

   minFreqArgsMHz = ghzFreqToMhz( minFreq ) if minFreq is not None else None
   maxFreqArgsMHz = ghzFreqToMhz( maxFreq ) if maxFreq is not None else None

   singleIntf = len( intfs ) == 1
   xcvrStatus = xcvrStatusDir.xcvrStatus
   for intfName in intfNames:
      slotName = getXcvrSlotName( intfName )
      status = getXcvrStatus( xcvrStatus.get( slotName ) )
      if not status:
         continue
      for laneId in range( status.capabilities.lineSideChannels ):
         # `lineSideChannels` represents the number of *media* lanes on a
         # transceiver, but `xcvrConfig.intfName` is a mapping which is keyed
         # by *host* lane.  This just works out because the only transceivers
         # which support this command, SFP and CFP2, have a one-to-one media/host
         # lane mapping.

         name = status.xcvrConfig.intfName.get( laneId )
         if name not in intfNames:
            continue
         shortName = Intf.getShortname( name )
         if supportsShowXcvrChannels( status ):
            if intfModel:= _populateInterfacesXcvrChannels( mode, status,
               shortName, minFreqArgsMHz, maxFreqArgsMHz,
               gridSpacingListMhz ):
               model.interfaces[ name ] = intfModel
         else:
            if singleIntf:
               mode.addError( f"{shortName} - does not apply" )
   return model

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   global xcvrStatusDir
   xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
