# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Model, Dict, Float, List
from IntfModels import Interface
from CliPlugin import IntfCli

import TableOutput

# --------------------------------------------------------------------------------
#
# Models for "show interfaces [<name>] transceiver channels"
#
# --------------------------------------------------------------------------------

class InterfacesTransceiverChannelRow( Model ):
   # Represents a row in output. Conveys wavelength and frequency characteristics of
   # a channel, with the corresponding channel number for each spacing
   wavelength = Float( help="Tuning channel wavelength (nm)" )
   frequency = Float( help="Tuning channel frequency (GHz)" )
   gridSpacingToChannelNumber = Dict( keyType=str, valueType=int,
                                      help="Mapping of the channel number for each "
                                            "grid spacing frequency requested" )

   def addChannelNumber( self, gridSpacing, channel ):
      """
      Adds a channel number for the corresponding gridSpacing
      Parameters
      ---------
      gridSpacing : str
         gridSpacing header of this column (measured in GHz)
      channel : int
         The corresponding channel number of this row and gridspacing
      """
      self.gridSpacingToChannelNumber[ gridSpacing ] = channel

   def getFormattedWavelength( self ):
      return f"{self.wavelength:.3f}"

   def getFormattedFrequency( self ):
      return f"{self.frequency :.2f}"

   def getFormattedChannel( self, gridSpacing ):
      """
      Returns the formatted channel number for the requested gridSpacing.
      If a corresponding channel number does not exist, an empty cell is returned.
      -------
         gridSpacing : str
      """
      return str( self.gridSpacingToChannelNumber.get( gridSpacing, "" ) )

class InterfaceTransceiverChannels( Model ):
   # Model containing header information and holds a list of the channel rows
   channels = List( valueType=InterfacesTransceiverChannelRow,
                    help="Rows in output, specifiy a channel and its corresponding "
                         "channel number for each spacing" )
   gridSpacings = List( valueType=float,
                        help="Channel spacings (in GHz). Each channel spacing "
                             "demands a column filled with channel numbers "
                             "for each frequency" )

   def appendChannel( self, channel ):
      self.channels.append( channel )

   def renderModel( self ):
      # We can have between 3 and 7 columns, so we must dynamically create
      # the column headers based the no. of grid-spacings requested.
      headerRow1 = [ "", "" ]
      headerRow2 = [ "Wavelength", "Frequency" ]
      headerRow3 = [ "(nm)", "(GHz)" ]
      formatWavelength = TableOutput.Format( justify="right", minWidth=9 )
      formatFreq = TableOutput.Format( justify="right", minWidth=10 )
      formatSpacing = TableOutput.Format( justify="right", minWidth=8 )
      columnFormatList = [ formatWavelength, formatFreq ]
      for gridSpacing in self.gridSpacings:
         headerRow1.append( f"{gridSpacing}GHz-" )
         headerRow2.append( "spacing" )
         headerRow3.append( "Channel" )
         columnFormatList.append( formatSpacing )

      # TableOutput module only supports headings to a depth of 2, but we require
      # 3 lines.
      # Because of this we create our own header rows.
      table = TableOutput.TableFormatter()
      table.formatColumns( *columnFormatList )
      for headingRow in [ headerRow1, headerRow2, headerRow3 ]:
         table.startRow()
         fmt = TableOutput.Format( justify="center" )
         for cell in headingRow:
            table.newFormattedCell( cell, format=fmt )

      for channel in self.channels:
         formattedRow = [
            channel.getFormattedWavelength(),
            channel.getFormattedFrequency(),
         ]
         for gridSpacing in self.gridSpacings:
            gridSpacingStr = str( gridSpacing )
            # Grid spacing (str format) key:val map to corresponding channel number
            formattedRow.append( channel.getFormattedChannel( gridSpacingStr ) )
         table.newRow( *formattedRow )

      print( table.output() )

class InterfacesTransceiverChannelsBase( Model ):
   # list of channel transceiver channels
   interfaces = Dict( keyType=Interface, valueType=InterfaceTransceiverChannels,
                      help="Mapping between interface name and the transceiver "
                           "channel info" )

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         print( f"Name: {IntfCli.Intf.getShortname( intf )}" )
         self.interfaces[ intf ].renderModel( )
