# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Model
from CliModel import Str
from CliModel import Int
from CliModel import List
from CliModel import Submodel
from IntfModels import Interface
import Tac
from TypeFuture import TacLazyType

SUPPORTED_FORMATS = frozenset( [ "default", "csv" ] )

CmisModuleState = TacLazyType( "Xcvr::CmisModuleState" )
DataPathState = TacLazyType( "Xcvr::DataPathState" )
MediaType = TacLazyType( "Xcvr::MediaType" )
TributaryId = TacLazyType( 'Xcvr::TributaryId' )
XcvrPresence = TacLazyType( "Xcvr::XcvrPresence" )

# --------------------------------------------------------------------------------
#
# Models for "show interface [<interfaces>] transceiver eeprom"
#
# --------------------------------------------------------------------------------

def spacedHex( val, n, minWidth=1 ):
   """Converts an integer into space-separated hex form with a space every n numbers
      and left-padded with 0, e.g. (0x123456789, 2) => '0x01 23 45 67 89'"""
   # pylint: disable-next=consider-using-f-string
   hexed = "{0:0>{1}X}".format( val, minWidth )
   hexed = "0" * ( len( hexed ) % n ) + hexed
   return "0x" + " ".join( hexed[ i : i + n ] for i in range( 0, len( hexed ), n ) )

def collapseRange( ints ):
   """Collapses a list of ints into its shortened string form (e.g. '132-135, 172').
      Assumes input is sorted; will return descending ranges if reverse sorted"""
   ranges = []
   for i in ints:
      if ranges and abs( i - ranges[ -1 ][ -1 ] ) <= 1:
         ranges[ -1 ].append( i )
      else:
         ranges.append( [ i ] )
   # pylint: disable-next=consider-using-f-string
   return ",".join( "%s-%s" % ( r[ 0 ], r[ -1 ] ) if r[ 0 ] != r[ -1 ] else
                    # pylint: disable-next=consider-using-f-string
                    "%s" % r[ 0 ] for r in ranges )

# pylint: disable-msg=attribute-defined-outside-init,dangerous-default-value
def eepromFieldBuilder( fields, name, pages, raw=None, decode=None, valueAttrs={} ):
   """Creates a field with the given name in the current field's subfields.
      Creates it with the given raw value, and a decoded value by passing
      raw to the decode function if it exists."""
   field = InterfacesTransceiverEepromField( _name=name )
   fields[ name ] = field
   for page, byteList in pages:
      field.addBytes( page, *byteList )
   if raw is not None:
      decoded = decode( raw ) if callable( decode ) else None
      if isinstance( decoded, bool ):
         field.value = InterfacesTransceiverEepromBoolValue()
      elif isinstance( decoded, str ):
         field.value = InterfacesTransceiverEepromStrValue()
      elif isinstance( decoded, int ):
         field.value = InterfacesTransceiverEepromIntValue()
      elif isinstance( decoded, float ):
         field.value = InterfacesTransceiverEepromFloatValue()
      else:
         # Either there wasn't a decoding function or we didn't understand its
         # output, so don't include the decode value
         field.value = InterfacesTransceiverEepromValue()
      field.value.raw = int( raw )
      if decoded is not None and hasattr( field.value, "decoded" ):
         field.value.decoded = decoded
      for key, val in valueAttrs.items():
         if val is not None and hasattr( field.value, key ):
            setattr( field.value, key, val )
   return field

# TODO: update decode functions to use these constants
class eepromUnits:
   @staticmethod
   def enums():
      return [ v for a, v in vars( eepromUnits ).items()
               if not a.startswith( "_" ) and a != "enums" ]

   # Output values for the different units we support in the output
   megabits = "Mbps"
   megabaud = "MBd"
   decibels = "dB"
   kilometers = "km"
   meters = "m"
   nanometers = "nm"
   celsius = "C"
   watts = "W"
   megahertz = "MHz"
   second = "s"
   millisecond = "ms"

class eepromPrefixes:
   @staticmethod
   def enums():
      return [ v for a, v in vars( eepromPrefixes ).items()
               if not a.startswith( "_" ) and a != "enums" ]

   # Output values for the different prefixes we support in the output
   minus = "-"
   plus = "+"
   plusOrMinus = "+/-"

# Builds the recursive printing and representation based on sub, sortVal, and text
class EepromBaseModel( Model ):
   def renderIndented( self, indentLevel, indentStr ):
      print( indentStr * indentLevel + self.text() )
      if self.sub():
         for item in sorted( self.sub().values(), key=lambda s: s.sortVal() ):
            item.renderIndented( indentLevel + 1, indentStr )

   # Override this with the sub dictionary to iterate over for printing
   def sub( self ):
      return {}

   # Override this with the field to compare when sorting
   def sortVal( self ):
      return self.sub()

   # Override this with the text output for this line (don't include indent)
   def text( self ):
      raise NotImplementedError

class InterfacesTransceiverEepromValue( EepromBaseModel ):
   raw = Int( help="The raw value stored in EEPROM" )
   # This value is needed to consistently print the full width of a raw value as we
   # lose the width information about the int when going from c++ to python
   _numBytes = Int( help="The number of bytes in this raw value.", default=1 )

   def sortVal( self ):
      return int( self.raw )

   def text( self ):
      return spacedHex( self.raw, 2, minWidth=self._numBytes * 2 )

class InterfacesTransceiverEepromBoolValue( InterfacesTransceiverEepromValue ):
   decoded = Bool( help="The decoded value from the EEPROM" )

   def text( self ):
      return str( self.decoded ).lower()

class InterfacesTransceiverEepromStrValue( InterfacesTransceiverEepromValue ):
   decoded = Str( help="The decoded value from the EEPROM" )
   possibleValues = List( help="The possible values that decoded can take on",
                    valueType=str, optional=True )

   def text( self ):
      return self.decoded

class InterfacesTransceiverEepromIntValue( InterfacesTransceiverEepromValue ):
   decoded = Int( help="The decoded value from the EEPROM" )
   units = Enum( help="The units for the decoded value", values=eepromUnits.enums(),
                 optional=True )
   prefix = Enum( help="Prefix such as \"+/-\" for the decoded value",
                  values=eepromPrefixes.enums(), optional=True )

   def text( self ):
      out = self.prefix or ""
      out += str( self.decoded )
      if self.units:
         out += " " + self.units
      return out

class InterfacesTransceiverEepromFloatValue( InterfacesTransceiverEepromValue ):
   decoded = Float( help="The decoded value from the EEPROM" )
   units = Enum( help="The units for the decoded value", values=eepromUnits.enums(),
                 optional=True )
   prefix = Enum( help="Prefix such as \"+/-\" for the decoded value",
                  values=eepromPrefixes.enums(), optional=True )

   def text( self ):
      out = self.prefix or ""
      out += str( self.decoded )
      if self.units:
         out += " " + self.units
      return out

class InterfacesTransceiverEepromBytes( EepromBaseModel ):
   _page = Int( help="The page the byte offsets are for." )
   byteOffsets = List( help="The byte offsets for the field.", valueType=int )

   def sortVal( self ):
      # Have to re-pack the list because CliModel.List doesn't compare properly
      # pylint: disable-next=unnecessary-comprehension
      return ( self._page, [ b for b in self.byteOffsets ] )

   def text( self ):
      return f"{self._page:0>2X}h:{collapseRange( self.byteOffsets )!s}"

# Create a dummy parent class for the recursive fields
# pylint: disable-msg=abstract-method
class InterfacesTransceiverEepromFieldBase( EepromBaseModel ):
   pass

class InterfacesTransceiverEepromField( InterfacesTransceiverEepromFieldBase ):
   _name = Str( help="The name of this field" )
   pages = Dict( help="Mapping between a page and the bytes on the page the field "
                      "is defined over.",
                 keyType=int, valueType=InterfacesTransceiverEepromBytes )
   value = Submodel( help="The value of this field",
                     valueType=InterfacesTransceiverEepromValue, optional=True )
   subfields = Dict( help="Mapping between a subfield name and its data",
                     keyType=str, valueType=InterfacesTransceiverEepromFieldBase,
                     optional=True )
   note = Str( help="A note about this field's interpretation",
               optional=True )

   def sub( self ):
      return self.subfields

   def sortVal( self ):
      # Have to re-pack the list to grab the correct sort value for the pages
      return ( [ pg.sortVal() for pg in self.pages.values() ], self._name )

   def text( self ):
      out = [ self._name ]
      byteStr = ";".join( sorted( pg.text()
                          for pg in self.pages.values() ) )
      out.append( "(" + byteStr + "):" )
      if self.value:
         out.append( self.value.text() )
      if self.note:
         out.append( "(%s)" % self.note ) # pylint: disable=consider-using-f-string
      return " ".join( out )

   def addBytes( self, page, *byteList ):
      byteListLocal = []
      for byte in byteList:
         if isinstance( byte, list ):
            byteListLocal += byte[ : ]
         else:
            byteListLocal.append( byte )
      if page not in self.pages:
         self.pages[ page ] = InterfacesTransceiverEepromBytes( _page=page )
      self.pages[ page ].byteOffsets += byteListLocal

   def addField( self, name, byteList, raw=None, decode=None, **valueAttrs ):
      return eepromFieldBuilder( self.subfields, name, byteList, raw=raw,
                                 decode=decode, valueAttrs=valueAttrs )

class InterfacesTransceiverEepromIntf( EepromBaseModel ):
   _name = Str( help="The name of the interface this EEPROM is from" )
   fields = Dict( help="Mapping between a field name and its data",
                  keyType=str, valueType=InterfacesTransceiverEepromField )

   def sub( self ):
      return self.fields

   def text( self ):
      return "%s EEPROM:" % self._name # pylint: disable=consider-using-f-string

   def addField( self, name, byteList, raw=None, decode=None, **valueAttrs ):
      return eepromFieldBuilder( self.fields, name, byteList, raw=raw,
                                 decode=decode, valueAttrs=valueAttrs )

class InterfacesTransceiverEepromBase( Model ):
   interfaces = Dict( help="Mapping between interface name and the EEPROM",
                      keyType=Interface, valueType=InterfacesTransceiverEepromIntf )

   def render( self ):
      if self.interfaces:
         for intf in Arnet.sortIntf( self.interfaces ):
            self.interfaces[ intf ].renderIndented( 0, "  " )
            print()
