# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
from CliModel import ( Bool, Dict, Enum, Float, Model, Str, Int, Submodel )
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfModel as EthIntfModel
from IntfModels import Interface

# --------------------------------------------------------------------------------
#
# Models for "show interfaces [ <interface> ] transceiver hardware"
#
# --------------------------------------------------------------------------------

class InterfacesTransceiverHardwarePower( Model ):
   configuredTxPower = Float( help="TX power in dBm", optional=True )
   _configuredTxPowerDefault = Bool( help="Indicates if the TX power is "
                                         "configured to its default value",
                                    optional=True )
   configuredRxPower = Float( help="RX power in dBm", optional=True )
   _configuredRxPowerDefault = Bool( help="Indicates if the RX power is "
                                         "configured to its default value",
                                    optional=True )
   operationalRxAttenuation = Float( help="RX attenuation in dB", optional=True )
   _tunableLaserEnabled = Bool( help="Indicates if (laser disabled) should be "
                                     "displayed for coherent optics" )
   _dualLaserModulePresent = Bool( help="Indicates that the receiver tuning"
                                        "should be displayed for coherent optics" )
   operationalTxPower = Float( help="TX power in dBm", optional=True )

   def configuredTxPowerDefaultIs( self, val ):
      self._configuredTxPowerDefault = val

   def configuredRxPowerDefaultIs( self, val ):
      self._configuredRxPowerDefault = val

   def renderModel( self ):
      printFmt = "{0}: {1}{2}"
      laserDisabledMsg = ""
      if not self._tunableLaserEnabled and not self._dualLaserModulePresent:
         laserDisabledMsg = " (laser disabled)"
      if self.configuredTxPower is not None:
         if self._tunableLaserEnabled:
            print( printFmt.format( "Configured TX power (dBm)",
                                    self.configuredTxPower, " (default)" if
                                    self._configuredTxPowerDefault else "" ) )
         else:
            print( printFmt.format( "Configured TX power (dBm)",
                                    str( self.configuredTxPower ) +
                                    laserDisabledMsg, "" ) )
      if self.configuredRxPower is not None:
         print( printFmt.format( "Configured Rx Power (dBm)",
               self.configuredRxPower,
                                 "(Default)" if self._configuredRxPowerDefault else
                                 "" ) )
      if self.operationalRxAttenuation is not None:
         print( printFmt.format( "Operational Rx Attenuation (dB)",
                                 self.operationalRxAttenuation if
                                 self._tunableLaserEnabled else
                                 str( self.operationalRxAttenuation ) +
                                 laserDisabledMsg, "" ) )
      if self.operationalTxPower is not None:
         print( printFmt.format( "Operational TX power (dBm)",
                                 self.operationalTxPower, "" ) )

class InterfacesTransceiverHardwareTuning( Model ):
   # Attributes for tuning
   _outputSelect = Bool( help="Selects tuning data set to display" )
   _tunableLaserEnabled = Bool( help="Indicates which display format to use for "
                                     "coherent optics tuning data" )
   _dualLaserModulePresent = Bool( help="Indicates that the receiver tuning"
                                        "should be displayed for coherent optics" )
   configuredFrequency = Float( help="Configured TX frequency in GHz",
                                optional=True )
   configuredRxFrequency = Float( help="Configured RX frequency in GHz",
                                  optional=True )
   configuredRxFineFrequency = Float( help="Configured RX fine frequency in GHz",
                                      optional=True )
   _configuredFrequencyUnsupported = Bool( help="Indicates if the configured "
                                                "frequency is unsupported",
                                           optional=True )
   computedWavelength = Float( help="Computed wavelength in nm", optional=True )
   computedRxWavelength = Float( help="Computed RX wavelength in nm",
                                 optional=True )
   operationalFrequency = Float( help="Operational TX frequency in GHz",
                                 optional=True )
   operationalRxFrequency = Float( help="Operational RX frequency in GHz",
                                   optional=True )
   _operationalFrequencyDefault = Bool( help="Indicates if the operational "
                                             "frequency is configured to its "
                                             "default value",
                                        optional=True )
   operationalWavelength = Float( help="Operational TX wavelength in nm",
                                  optional=True )
   operationalRxWavelength = Float( help="Operational RX wavelength in nm",
                                    optional=True )
   configuredChannel = Int( help="Configured channel", optional=True )
   _configuredChannelUnsupported = Bool( help="Indicates if the configured "
                                             "channel is unsupported",
                                         optional=True )
   configuredGrid = Float( help="Configured grid in GHz", optional=True )
   computedFrequency = Float( help="Computed frequency in GHz", optional=True )
   operationalChannel = Int( help="Operational channel", optional=True )
   _operationalChannelDefault = Bool( help="Indicates if the operational "
                                          "channel is configured to its "
                                          "default value",
                                     optional=True )
   operationalGrid = Float( help="Operational grid in GHz", optional=True )
   _operationalGridDefault = Bool( help="Indicates if the operational grid is "
                                       "configured to its default value",
                                  optional=True )
   gridSpacingCapabilities = Str( help="Grid spacing capabilities in GHz",
                                  optional=True )
   _gridSpacingCapabilitiesUnknown = Bool( help="Indicates if the grid-spacing "
                                               "capabilities are unknown",
                                          optional=True )

   def configuredFrequencyUnsupportedIs( self, val ):
      self._configuredFrequencyUnsupported = val

   def operationalFrequencyDefaultIs( self, val ):
      self._operationalFrequencyDefault = val

   def configuredChannelUnsupportedIs( self, val ):
      self._configuredChannelUnsupported = val

   def operationalChannelDefaultIs( self, val ):
      self._operationalChannelDefault = val

   def operationalGridDefaultIs( self, val ):
      self._operationalGridDefault = val

   def gridSpacingCapabilitiesUnknownIs( self, val ):
      self._gridSpacingCapabilitiesUnknown = val

   def renderModel( self ):
      printFmt = "{0}: {1}{2}"
      if self._outputSelect:
         if self._dualLaserModulePresent:
            # dual laser module must also display output for RX laser
            print( printFmt.format( "Configured TX frequency (GHz)",
                                    f"{self.configuredFrequency:,}" if
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Configured RX frequency (GHz)",
                                    f"{self.configuredRxFrequency:,}",
                                    "" ) )
            print( printFmt.format( "Configured RX fine frequency (GHz)",
                                    self.configuredRxFineFrequency, "" ) )
            print( printFmt.format( "Computed TX wavelength (nm)",
                                    "%.2f" % self.computedWavelength if
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Computed RX wavelength (nm)",
                                    "%.2f" % self.computedRxWavelength, "" ) )
            print( printFmt.format( "Operational TX frequency (GHz)",
                                    f"{self.operationalFrequency:,}" if
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Operational RX frequency (GHz)",
                                    f"{self.operationalRxFrequency:,}",
                                    "" ) )
            print( printFmt.format( "Operational TX wavelength (nm)",
                                    "%.2f" % self.operationalWavelength if
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Operational RX wavelength (nm)",
                                    "%.2f" % self.operationalRxWavelength, "" ) )
         else:
            print( printFmt.format( "Configured frequency (GHz)",
                                    f"{self.configuredFrequency:,}",
                                    "(unsupported)" if
                                    self._configuredFrequencyUnsupported else "" ) )
            print( printFmt.format( "Computed wavelength (nm)",
                                    "%.2f" % self.computedWavelength, "" ) )
            print( printFmt.format( "Operational frequency (GHz)",
                                    f"{self.operationalFrequency:,}",
                                    "(default)" if self._operationalFrequencyDefault
                                       else "" ) )
            print( printFmt.format( "Operational wavelength (nm)",
                                    "%.2f" % self.operationalWavelength, "" ) )
      else:
         if self._tunableLaserEnabled:
            print( printFmt.format( "Configured channel", self.configuredChannel,
                                    "(unsupported)" if
                                    self._configuredChannelUnsupported else "" ) )
            print( printFmt.format( "Configured grid (GHz)", self.configuredGrid,
                                    "" ) )
            print( printFmt.format( "Computed frequency (GHz)",
                                   f"{self.computedFrequency:,}", "" ) )
            print( printFmt.format( "Computed wavelength (nm)",
                                   "%.2f" % self.computedWavelength, "" ) )
            print( printFmt.format( "Operational channel", self.operationalChannel,
                                   "(default)" if self._operationalChannelDefault
                                         else "" ) )
            print( printFmt.format( "Operational grid (GHz)", self.operationalGrid,
                                   "(default)" if self._operationalGridDefault
                                         else "" ) )
            print( printFmt.format( "Operational frequency (GHz)",
                                   f"{self.operationalFrequency:,}",
                                   "(default)" if self._operationalFrequencyDefault
                                         else "" ) )
            print( printFmt.format( "Operational wavelength (nm)",
                                   "%.2f" % self.operationalWavelength, "" ) )
         else:
            print( printFmt.format( "Configured channel", "none (default)", "" ) )
            print( printFmt.format( "Configured grid (GHz)", "none (default)",
                                    "" ) )
            print( printFmt.format( "Computed frequency (GHz)", "none (default)",
                                   "" ) )
            print( printFmt.format( "Computed wavelength (nm)", "none (default)",
                                   "" ) )
            print( printFmt.format( "Operational channel", "none (laser disabled)",
                                   "" ) )
            print( printFmt.format( "Operational grid (GHz)",
                                    "none (laser disabled)", "" ) )
            print( printFmt.format( "Operational frequency (GHz)",
                                    "none (laser disabled)", "" ) )
            print( printFmt.format( "Operational wavelength (nm)",
                                    "none (laser disabled)", "" ) )
         if self.gridSpacingCapabilities or self._gridSpacingCapabilitiesUnknown:
            print( printFmt.format( "Grid spacing capabilities (GHz)",
                                    self.gridSpacingCapabilities, "(unknown)" if
                                         self._gridSpacingCapabilitiesUnknown else
                                         "" ) )

class InterfacesTransceiverHardwareXcvrCapabilities( Model ):
   speed = Enum( values=EthIntfModel.CapabilitiesMixIn.speedValues,
                 help='Transceiver speed' )
   duplex = Enum( values=( 'full', 'half', 'unknown' ),
                  help='Duplex communication capabilities with this speed' )
   laneCount = Int( optional=True, help='Number of lanes on the interface' )
   _showLanes = Bool( optional=True, help='Lane count is set' )

   def showLanesIs( self, showLanes ):
      self._showLanes = showLanes

   def showLanes( self ):
      return self._showLanes

class InterfacesTransceiverHardware( Model ):
   __revision__ = 2

   def modulePowerIgnored( self ):
      return self._modulePowerIgnored

   def modulePowerIgnoredIs( self, value ):
      self._modulePowerIgnored = value

   _modulePowerIgnored = Bool( help=( "Slot power requirements ignored" ) )
   mediaType = Str( help="The effective media type, normally the"
            "detectedMediaType, but can be overridden by cli config",
                    optional=True )
   detectedMediaType = Str( help="The media type set in device's IDPROM",
                            optional=True )
   cableType = Enum( values=[ 'CA-N', 'CA-S', 'CA-L' ],
                     help="Type of copper cable", optional=True )
   wavelength = Float( help="Wavelength of fiber (nm)", optional=True )
   _wavelengthPrecNm = Bool( help="Determines the formatting of the printed "
                                  "wavelength", optional=True )
   tuning = Submodel( valueType=InterfacesTransceiverHardwareTuning,
                      help="Tuning information", optional=True )
   txDisabled = Bool( help="Optical transmitter disabled", optional=True )
   power = Submodel( valueType=InterfacesTransceiverHardwarePower,
                     help="Power information", optional=True )
   powerCharacteristic = Float( help="Maximum module power in watts",
                                optional=True )
   maxPowerSupportedCharacteristic = Float( help="Maximum slot power in watts",
                                            optional=True )
   modulePresent = Bool( help="Transceiver is present", optional=True )

   def degrade( self, dictRepr: dict, revision: int ):
      """
      Parameters
      ----------
      dictRepr : dict()
         The dictionary representation of the current model which we need to
         massage to fit the passed in revision number


      Previous Model Attributes
      -------------------------
      mediaType = Str( help="The effective media type, normally the "
                    " detectedMediaType, but can be overridden by cli config" )
      detectedMediaType = Str( help="The media type set in device's IDPROM" )
      powerCharacteristic = Float( help="Maximum module power in watts" )
      maxPowerSupportedCharacteristic = Float(
        help="Maximum slot power in watts" )

      The attributes mediaType, detectedMediaType, powerCharacteristic, and
      maxPowerSupportedCharacteristic were non-optional in revision 1.  They
      are now optional.  In order to maintain backwards compatibility, we use
      placeholder values.
      """

      if revision == 1:
         dictRepr[ "mediaType" ] = dictRepr.get( "mediaType", "" )
         dictRepr[ "detectedMediaType" ] = dictRepr.get( "detectedMediaType", "" )
         dictRepr[ "powerCharacteristic" ] = dictRepr.get( "powerCharacteristic",
                                                           0.0 )
         dictRepr[ "maxPowerSupportedCharacteristic" ] = dictRepr.get(
               "maxPowerSupportedCharacteristic", 0.0 )
      return dictRepr

   def wavelengthPrecNmIs( self, val ):
      self._wavelengthPrecNm = val

   def renderModel( self, intfName ):
      print( f"Name: {intfName}" )
      if self.mediaType is not None:
         print( f"Media type: {self.mediaType}" )
      if self.modulePresent is not None:
         print( "Module presence: {}".format( "detected" if self.modulePresent
                                              else "not detected" ) )
      if self.powerCharacteristic is not None:
         print( f"Maximum module power (W): {self.powerCharacteristic}" )
      if self.maxPowerSupportedCharacteristic is not None:
         maxSlotPowerStr = "Maximum slot power (W): {}".format(
            self.maxPowerSupportedCharacteristic if
            self.maxPowerSupportedCharacteristic != 0 else 'N/A' )
         if self.modulePowerIgnored():
            maxSlotPowerStr += " (power limit check disabled)"
         print( maxSlotPowerStr )
      if self.cableType is not None:
         print( f"Cable type: {self.cableType}" )
      if self.wavelength is not None:
         # Casting to provide the same output as the unconverted command.
         print( "Wavelength (nm): {}".format( int( self.wavelength ) if
                self._wavelengthPrecNm else float( self.wavelength ) ) )
      if self.tuning:
         self.tuning.renderModel()
      if self.txDisabled is not None:
         print( "TX laser status: {}".format( "disabled" if self.txDisabled
                                              else "enabled" ) )
      if self.power:
         self.power.renderModel()
      if self.detectedMediaType is not None and \
         self.detectedMediaType != self.mediaType:
         print( f"Detected media type: {self.detectedMediaType}" )
      print( "" )

class InterfacesTransceiverHardwareBase( Model ):
   __revision__ = 2
   interfaces = Dict( keyType=Interface, valueType=InterfacesTransceiverHardware,
                      help="Mapping between interface name and the transceiver info"
                      )
   _xcvrNotPresent = Str( help="Print xcvr presence error" )

   def xcvrNotPresentIs( self, intf ):
      self._xcvrNotPresent = intf

   def render( self ):
      if not self.interfaces:
         if self._xcvrNotPresent is not None:
            print( "%s - transceiver not present" % self._xcvrNotPresent )
         return

      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )
