# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Model, Dict, Submodel, Bool, Enum

class TransceiverLaneSquelchSetting( Model ):
   default = Bool( 'Module default setting', optional=True )
   operational = Bool( 'Module operational setting' )

class TransceiverSquelchType( Model ):
   lanes = Dict( keyType=int, valueType=TransceiverLaneSquelchSetting,
                 help='Mapping from lane to transceiver squelch setting' )
   configured = Enum( values=( 'moduleDefault', 'enabled', 'disabled' ),
                      help='The user-configured squelch setting', optional=True )
   supported = Bool( help='Configuration supported on this module' )

   def renderSquelch( self, slot ):
      fmt = '%-15s%4s  %-16s%-16s%-13s'
      if self.configured == 'moduleDefault':
         config = 'module default'
      else:
         config = self.configured
      star = '' if self.supported else '*'
      for lane, setting in sorted( self.lanes.items() ):
         laneDefault = 'enabled' if setting.default else 'disabled'
         laneOper = 'enabled' if setting.operational else 'disabled'

         print( fmt % ( slot, lane, laneDefault + star, config, laneOper + star ) )

class TransceiverSquelchTypes( Model ):
   # Add additional squelch types here (e.g. Tx Auto, Tx Forced)
   rxAutoSquelch = Submodel( valueType=TransceiverSquelchType,
                             help='Transceiver RX auto squelch' )

class TransceiverSquelchSlots( Model ):
   slots = Dict( keyType=str, valueType=TransceiverSquelchTypes,
                 help='Mapping between slot name and transceiver squelch types' )

   def _printStarHelp( self ):
      print( '* = Unsupported by module, value may be inaccurate' )

   def _printSquelchHeaderDefault( self, squelchType ):
      headerFormat = '%-15s%-6s%-16s%-16s%-13s'
      print( squelchType )
      print( headerFormat %
            ( 'Slot', 'Lane  ', 'Module Default', 'Configured', 'Operational' ) )
      print( headerFormat % ( '-' * 13, '-' * 4, '-' * 14, '-' * 14, '-' * 11 ) )

   def render( self ):
      if not self.slots:
         return

      self._printStarHelp()
      self._printSquelchHeaderDefault( 'Rx Auto Squelch' )
      for slot in Arnet.sortIntf( self.slots ):
         squelchTypes = self.slots[ slot ]
         squelchTypes.rxAutoSquelch.renderSquelch( slot )
