#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

__writeEnabled__ = True

def enableWrite( enabled ):
   global __writeEnabled__
   __writeEnabled__ = enabled

class CliGlobal:
   """This class allows defining global CLI variables that are initialized
   at the beginning (before plugins are fully loaded). Instead of

      var = None

      def foo(): # typically called by Plugin entry function
         global var
         var = ...

      # and have to add to the allowed list in CliGlobalChecker.py

   you do

      gv = CliGlobal.CliGlobal( var=None )

      def foo():
         gv.var = ...

   There are two benefits:
   1. CliGlobal can only be set once. This minimizes race conditions caused by
      multithreading in ConfigAgent.
   2. Avoid having to add to the allowed list it in EosImage/test/CliGlobalChecker.py

   """

   def __init__( self, __dictionaryVars__=None, **attrs ):
      # Note, the first param is for backward compatibility as originally
      # we allowed users to pass a dictionary.
      attrs = attrs or __dictionaryVars__ or {}
      for k in attrs:
         assert not k.startswith( '__' ), "name cannot start with '__'"
      self.__globalAttrs__ = attrs

   def __getattr__( self, attr ):
      if attr != '__globalAttrs__' and attr in self.__globalAttrs__:
         return self.__globalAttrs__[ attr ]
      raise AttributeError( attr )

   def __setattr__( self, attr, val ):
      if attr.startswith( '__' ) or attr in self.__dict__:
         object.__setattr__( self, attr, val )
      elif attr in self.__globalAttrs__:
         # Allow the first write, but the default might not be None,
         # so this isn't a perfect check. Should work in practice.
         if self.__globalAttrs__.get( attr ) is not None:
            assert __writeEnabled__, \
               "CliGlobal variables can only be written once"
         self.__globalAttrs__[ attr ] = val
      else:
         raise AttributeError( val )
