# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
try:
   from inspect import getfullargspec
except ImportError:
   from inspect import getargspec as getfullargspec

import CliParserCommon
from CliParserCommon import MatchResult, noMatch, partialMatch, debugFuncName
from CliParserCommon import Completion

# TODO :
#    - BUG558638 : Refactor derived class and move some methods to the parent class.
###

def _checkDeprecatedHelpDesc( helpdesc ):
   if helpdesc and helpdesc.lower() in CliParserCommon.DEPRECATED_HELP_STRINGS:
      newHelpMessage = CliParserCommon.DEPRECATED_HELP_STRINGS[ helpdesc.lower() ]
      msg = ( 'This helpdesc %r is deprecated. Use helpdesc=%r' %
              ( helpdesc, newHelpMessage ) )
      raise ValueError( msg )

class Matcher:
   __slots__ = ( 'helpname_', 'helpdesc_', 'common_', 'priority_',
                 'value_' )

   def __init__( self, helpdesc=None, helpname=None, common=False,
                 priority=CliParserCommon.PRIO_NORMAL, value=None ):
      err = '`%s` must be a string. Got %s instead.'
      if not isinstance( helpdesc, ( str, type( None ) ) ):
         raise TypeError( err % ( 'helpdesc', type( helpdesc ) ) )
      if not isinstance( helpname, ( str, type( None ) ) ):
         raise TypeError( err % ( 'helpname', type( helpdesc ) ) )

      self.helpname_ = helpname
      self.helpdesc_ = helpdesc
      _checkDeprecatedHelpDesc( helpdesc )
      _checkDeprecatedHelpDesc( helpname )
      self.common_ = common
      self.priority_ = priority
      assert priority < CliParserCommon.PRIO_LOWEST, \
         "priority has to be less than % d" % CliParserCommon.PRIO_LOWEST
      self.value_ = value

   def match( self, mode, context, token ):
      # match one token, and return MatchResult.
      #
      # There are two special result values: noMatch meaning this token cannot
      # match; partialMatch meaning we need more tokens.
      #
      # For single-token, return noMatch or MatchResult( result, aaaToken ).
      #
      # For multi-token, we can set context.state (which is initially None)
      # to whatever state necessary to help matching additional tokens or
      # getting completions.
      raise NotImplementedError

   def completions( self, mode, context, token ):
      raise NotImplementedError

   def valueFunction( self, context ):
      return self.value_

   def canMerge( self, otherMatcher ):
      return self.__eq__( otherMatcher )

   def __eq__( self, otherMatcher ):
      return id( self ) == id( otherMatcher )

   def __hash__( self ):
      return hash(
         (
            self.helpname_,
            self.helpdesc_,
            self.common_,
            self.priority_,
            self.value_
         )
      )

class KeywordMatcher( Matcher ):
   __slots__ = ( 'keyword_', 'alternates_', 'autoCompleteMinChars_' )

   def __init__( self, keyword, helpdesc, alternates=None,
                 autoCompleteMinChars=0, **kargs ):
      kargs.setdefault( 'priority', CliParserCommon.PRIO_KEYWORD )
      super().__init__( helpname=keyword,
                        helpdesc=helpdesc,
                        **kargs )
      self.keyword_ = keyword
      self.alternates_ = tuple( alternates ) if alternates else None
      self.autoCompleteMinChars_ = autoCompleteMinChars
      assert self.helpdesc_ is not None, 'KeywordMatcher must have a helpdesc'

   def match( self, mode, context, token ):
      token = token.lower()
      if token == self.keyword_.lower():
         return MatchResult( self.keyword_, self.keyword_ )

      if self.alternates_:
         for k in self.alternates_: # pylint: disable=not-an-iterable
            if token == k.lower():
               return MatchResult( self.keyword_, self.keyword_ )
      return noMatch

   def completions( self, mode, context, token ):
      token = token.lower()
      if self.keyword_.lower().startswith( token ):
         # make sure autocomplete only happens if the token has enough length
         literal = len( token ) >= self.autoCompleteMinChars_
         return [ CliParserCommon.Completion( self.keyword_, self.helpdesc_,
                                              literal=literal,
                                              common=self.common_ ) ]
      return []

   def __str__( self ):
      return self.keyword_

   def isSimpleKeyword( self ):
      # A simple matcher only has a description
      return ( not self.alternates_ and
               not self.autoCompleteMinChars_ and
               self.priority_ == CliParserCommon.PRIO_KEYWORD and
               not self.common_ and not self.value_ and
               # subclasses might not be simple
               self.__class__ is KeywordMatcher )

   def canMerge( self, otherMatcher ):
      result = self.__eq__( otherMatcher )
      if result:
         helpdesc = otherMatcher.helpdesc_
         common = otherMatcher.common_
      elif isinstance( otherMatcher, str ):
         result = self.isSimpleKeyword()
         helpdesc = otherMatcher
         common = False

      if result:
         assert self.helpdesc_ == helpdesc, \
               ( 'Keyword \'%s\' has 2 different helpdesc: \'%s\' and \'%s\'' %
                     ( self.keyword_, self.helpdesc_, helpdesc ) )
         assert self.common_ == common, \
               ( 'Keyword \'%s\' has 2 different common flags: \'%s\' and \'%s\'' %
                     ( self.keyword_, self.common_, common ) )

      return result

   def __eq__( self, otherMatcher ):
      if id( self ) == id( otherMatcher ):
         return True
      return ( isinstance( otherMatcher, KeywordMatcher ) and
               self.keyword_ == otherMatcher.keyword_ and
               self.alternates_ == otherMatcher.alternates_ and
               self.autoCompleteMinChars_ == otherMatcher.autoCompleteMinChars_ and
               self.priority_ == otherMatcher.priority_ and
               self.value_ == otherMatcher.value_ and
               self.common_ == otherMatcher.common_ )

   def __ne__( self, otherMatcher ):
      return not self.__eq__( otherMatcher )

   def __hash__( self ):
      return hash(
         (
            self.helpname_,
            self.helpdesc_,
            self.common_,
            self.priority_,
            self.value_,
            self.alternates_,
            self.keyword_,
            self.autoCompleteMinChars_
                        )
                )

# This is used internally by the keyword matchers defined as a string (the helpdesc)
# in the command definition, this is not a classical Matcher. This is for plain
# keywords that are uniquely defined by just their name and helpdesc. Special parser
# treatment of such simple keywords (we have many of them) saves memory.
class SimpleKeywordMatcher:

   @staticmethod
   def match( keyword, token ):
      if token.lower() == keyword.lower():
         return CliParserCommon.MatchResult( keyword, keyword )
      return CliParserCommon.noMatch

   @staticmethod
   def completions( keyword, token, helpdesc ):
      if keyword.lower().startswith( token.lower() ):
         return [ CliParserCommon.Completion( keyword, helpdesc, common=False ) ]
      return []

class DynamicKeywordMatcher( Matcher ):
   __slots__ = ( 'keywordsFn_', 'emptyTokenCompletion_',
                 'alwaysMatchInStartupConfig_', 'passContext_' )

   def __init__( self, keywordsFn, emptyTokenCompletion=None,
                 alwaysMatchInStartupConfig=False,
                 passContext=False,
                 **kargs ):
      kargs.setdefault( 'priority', CliParserCommon.PRIO_KEYWORD )
      super().__init__(
         helpdesc='', **kargs )
      self.keywordsFn_ = keywordsFn
      self.passContext_ = passContext
      # Note that emptyTokenCompletion sets literal to False, so that cli doesn't
      # allow the generic token as a valid tab-completion when user hasn't started
      # entering the word yet ( the empty-token case )
      if emptyTokenCompletion is not None:
         for x in emptyTokenCompletion:
            assert x.literal is False
      self.emptyTokenCompletion_ = emptyTokenCompletion
      self.alwaysMatchInStartupConfig_ = alwaysMatchInStartupConfig

   def _getKeywords( self, mode, context ):
      if self.passContext_:
         return self.keywordsFn_( mode, context )
      else:
         return self.keywordsFn_( mode )

   def match( self, mode, context, token ):
      if ( self.alwaysMatchInStartupConfig_ and mode and mode.session and
           mode.session.startupConfig() ):
         return MatchResult( token, token )
      keywords = self._getKeywords( mode, context )
      if token in keywords:
         return MatchResult( token, token )
      else:
         # We can't find an exact match, but it may be due to different cases
         # so iterate through all the keys and match again. It could be expensive
         # if there are a lot of keywords.
         token = token.lower()
         for k in keywords:
            if token == k.lower():
               return MatchResult( k, k )
      return noMatch

   def completions( self, mode, context, token ):
      if self.emptyTokenCompletion_ is None:
         completions = []
         # keywordsFn must return a dictionary with help descriptions as value
         for ( k, v ) in self._getKeywords( mode, context ).items():
            _checkDeprecatedHelpDesc( v )
            if k.lower().startswith( token.lower() ):
               completions.append( CliParserCommon.Completion( k, v,
                  common=self.common_ ) )
         return completions
      else: # only care about keywords, not help descriptions
         if token == '':
            return self.emptyTokenCompletion_
         return [ CliParserCommon.Completion( k, k, common=self.common_ )
                  for k in self._getKeywords( mode, context )
                  if k.lower().startswith( token.lower() ) ]

   def __str__( self ):
      return '<DynamicKeywordMatcher(%s)>' % debugFuncName( self.keywordsFn_ )

# This is a multi-token matcher. A sort of dynamic keyword matcher that accepts
# keywords that can contain spaces. It matches a list of keyword list. Thus the
# name of KeywordLists (note the plural on lists). We call that list of list
# "paths" below, and must be an iterable that returns a list of keyword/tokens.
# So paths could be [ ["one", "two"], ["one", "two", "three"] ] if both "one two"
# as well as "one two three" can match. In that case, paths is the unrolling of the
# syntax "one two [three]"...
# For making those keywordlists dynamic, since creating an iterator is more annoying
# that creating a function, "paths" can also be a function that returns the lists
class KeywordListsMatcher( Matcher ):

   def __init__( self, paths, **kargs ):
      if "helpdesc" not in kargs:
         kargs[ 'helpdesc' ] = '%s'
      super().__init__( **kargs )
      self.allPaths = paths

   def __str__( self ):
      return '(%s)' % "|".join( [ " ".join( p ) for p in self.allPaths ] )

   def getState( self, mode, context ):
      if not context.state: # first token (of potentially multiple): init state
         index = 0 # iteration of the matcher call (current parse index in path)
         paths = self.allPaths # initial paths, will shrink at each iteration
         if callable( paths ):
            paths = paths( mode )
         tokens = [] # holds the full resulting match, grows at each iteration
         return ( index, paths, tokens )
      else: # multi-token, second round: pick up the last state from the context
         return context.state

   def match( self, mode, context, token ):
      ( index, paths, tokens ) = self.getState( mode, context )
      # now look for matches in our remaining paths at the current index
      matches = []
      canEatMore = False # when at least 1 matching path can consume even more tokens
      terminal = False # when at least 1 matched token is terminal in its path
      toRm = [] # paths with terminal matches will be removed from next iteration
      for path in paths:
         if path[ index ] == token:
            matches.append( path )
            if len( path ) == index + 1:
               terminal = True
               toRm.append( path ) # no longer needed in next round
            else:
               canEatMore = True
      if not matches:
         return CliParserCommon.noMatch
      # Hurray, we got a match: figure out what to return and store state
      # grow our tokens by the extra token of this iteration
      tokens.append( token )
      for path in toRm: # remove all terminal matches (consumed now)
         matches.remove( path )
      # remember our state
      context.state = ( index + 1, matches, tokens )
      # now return results
      if not canEatMore: # we consumed last token in all remaining paths
         # the "False" below means "cannot accept more tokens"
         return CliParserCommon.MatchResult( tokens, tokens, False )
      else: # we can accept more tokens beyond this one
         # Unless one of our matches is terminal, we must say "partialMatch",
         # (if there are no further tokens, that's a syntax error)
         # otherwise we can return a match with a qualifier of "can eat more"
         if not terminal: # more tokens are mandatory
            return CliParserCommon.partialMatch
         else: # that could be it, but a longer match is also still possible
            # say "one two" and "one two three" are registered and user input is
            # "one two" -> we have a match AND we can take more tokens ("three")
            return CliParserCommon.MatchResult( tokens, tokens, True )

   def completions( self, mode, context, token ):
      cs = []
      ( index, paths, _ ) = self.getState( mode, context )
      for p in paths:
         if p[ index ].startswith( token ):
            c = CliParserCommon.Completion( p[ index ],
                                            self.helpdesc_ % p[ index ] )
            if c not in cs:
               cs.append( c )
      return cs

class EnumMatcher( Matcher ):
   __slots__ = (
      'completions_',
      'keywords_',
      'hiddenHelpDescMapping_',
      'keywordsHelpDescMapping_',
   )

   def __init__( self, keywordsHelpDescMapping, hiddenHelpDescMapping=None,
         **kargs ):
      super().__init__( priority=CliParserCommon.PRIO_HIGH,
                        helpdesc='', **kargs )

      if hiddenHelpDescMapping is None:
         hiddenHelpDescMapping = {}
      self.hiddenHelpDescMapping_ = hiddenHelpDescMapping

      self.keywords_ = []
      self.completions_ = []
      self.keywordsHelpDescMapping_ = keywordsHelpDescMapping

      if not callable( keywordsHelpDescMapping ):
         self.keywords_ = list( keywordsHelpDescMapping )
         for name, helpdesc in keywordsHelpDescMapping.items():
            _checkDeprecatedHelpDesc( helpdesc )
            self.completions_.append( CliParserCommon.Completion( name, helpdesc ) )

      for name, helpdesc in hiddenHelpDescMapping.items():
         _checkDeprecatedHelpDesc( helpdesc )

   def _keywords( self, mode, context ):
      if self.keywords_ or not callable( self.keywordsHelpDescMapping_ ):
         return self.keywords_

      return list( self.keywordsHelpDescMapping_( mode, context ) )

   def _completions( self, mode, context ):
      if self.completions_ or not callable( self.keywordsHelpDescMapping_ ):
         return self.completions_

      completions = []
      dynamicMapping = self.keywordsHelpDescMapping_( mode, context )
      for name, helpdesc in dynamicMapping.items():
         _checkDeprecatedHelpDesc( helpdesc )
         completions.append( CliParserCommon.Completion( name, helpdesc ) )

      return completions

   def getValidKeywords( self, mode, context ):
      return self._keywords( mode, context ) + list( self.hiddenHelpDescMapping_ )

   def getValidCompletions( self, mode, context ):
      return self._completions( mode, context )

   def match( self, mode, context, token ):
      keywords = self.getValidKeywords( mode, context )
      if token in keywords:
         return MatchResult( token, token )
      else:
         # We can't find an exact match, but it may be due to different cases
         # so iterate through all the keys and match again. It could be expensive
         # if there are a lot of keywords.
         token = token.lower()
         for k in keywords:
            if token == k.lower():
               return MatchResult( k, k )
      return noMatch

   def completions( self, mode, context, token ):
      token = token.lower() # TODO: we should remove this call
      completions = self.getValidCompletions( mode, context )
      return [ c for c in completions if c.name.lower().startswith( token ) ]

   def __str__( self ):
      return '|'.join( self.keywords_ )

class PatternMatcher( Matcher ):
   __slots__ = ( 're_', 'rawResult_', 'partialRe_', 'completion_' )

   def __init__( self, pattern, partialPattern=None,
                 rawResult=False, **kargs ):
      super().__init__( **kargs )
      assert 'helpname' in kargs, 'PatternMatcher requires a helpname'
      # Note that it is important to append '$' to these patterns rather than simply
      # testing whether a returned match object matches the entire token, as the
      # latter doesn't work correctly when the pattern contains '|' characters (since
      # '|' characters are not greedy).
      self.re_ = re.compile( '(?:%s)$' % pattern )
      self.rawResult_ = rawResult
      partialPattern = ( partialPattern or
                         CliParserCommon.makePartialPattern( pattern ) )
      self.partialRe_ = re.compile( '(%s)$' % partialPattern )

   def match( self, mode, context, token ):
      m = self.re_.match( token )
      if m:
         return MatchResult( m if self.rawResult_ else token, token )
      return noMatch

   def completions( self, mode, context, token ):
      m = self.partialRe_.match( token )
      if m:
         return [ CliParserCommon.Completion( self.helpname_, self.helpdesc_,
                                              False, common=self.common_ ) ]
      return []

   def __str__( self ):
      return self.re_.pattern

class DynamicNameMatcher( PatternMatcher ):
   __slots__ = ( 'namesFn_', 'passContext_', 'extraEmptyTokenCompletionFn_' )
   """Accept a name, and do autocompletion for existing names.
      - extraEmptyTokenCompletionFn: This match when showing completion options
         for an empty input will by default only show the wildcard match. It will
         not call the namesFn to get possible completions because that list is
         potentially quite large. However this behavior can be customized and
         additional completions can be added.
   """
   def __init__( self, namesFn, helpdesc, pattern=r'[A-Za-z0-9_.:{}\[\]-]+',
                 partialPattern=None, helpname='WORD', passContext=False,
                 extraEmptyTokenCompletionFn=None,
                 **kargs ):
      super().__init__( pattern=pattern,
                        partialPattern=partialPattern,
                        helpdesc=helpdesc,
                        helpname=helpname,
                        **kargs )
      if not callable( namesFn ):
         raise TypeError( 'Expected a function that takes `mode` as first argument' )
      self.namesFn_ = namesFn
      self.extraEmptyTokenCompletionFn_ = extraEmptyTokenCompletionFn
      self.passContext_ = passContext

   def completions( self, mode, context, token ):
      patternCompletions = PatternMatcher.completions( self, mode, context, token )
      if not patternCompletions:
         # no completion for '?'
         return []

      if not token:
         if self.extraEmptyTokenCompletionFn_:
            return ( self.extraEmptyTokenCompletionFn_( mode, context ) +
                        patternCompletions )
         else:
            return patternCompletions

      # Add existing matching names to completions (case-sensitive)
      return [ CliParserCommon.Completion( k, k ) for k in
               ( self.namesFn_( mode, context ) if self.passContext_ else
                 self.namesFn_( mode ) )
               if k.startswith( token ) ] + patternCompletions

   def __str__( self ):
      return '<DynamicNameMatcher(%s)>' % debugFuncName( self.namesFn_ )

class StringMatcher( Matcher ):
   __slots__ = ( 're_', 'partialRe_', 'completion_' )

   def __init__( self, pattern='.+', helpname='LINE', helpdesc=None, **kargs ):
      assert helpdesc is not None, "helpdesc required for StringMatcher"
      kargs[ 'priority' ] = kargs.get( 'priority', CliParserCommon.PRIO_LOW )
      Matcher.__init__( self, helpname=helpname, helpdesc=helpdesc, **kargs )
      self.re_ = re.compile( r'(?:%s)$' % pattern ) # "(?:...)" means non-grouping.
      partialPattern = CliParserCommon.makePartialPattern( pattern )
      self.partialRe_ = re.compile( '(%s)$' % partialPattern )

   def match( self, mode, context, token ):
      # with a string rule we always consume all of the tokens.
      if token:
         # token can be empty if passed in from QuotedStringMatcher
         m = self.re_.match( token )
         if not m:
            return noMatch
      if not context.state:
         context.state = []
      context.state.append( token )
      return MatchResult( ' '.join( context.state ), context.state,
                          more=True )

   def completions( self, mode, context, token ):
      if self.partialRe_.match( token ):
         return [ CliParserCommon.Completion( self.helpname_, self.helpdesc_,
                                              False, common=self.common_ ) ]
      return []

   def __str__( self ):
      return '<string>'

class TrailingGarbageMatcher( StringMatcher ):
   def __init__( self ):
      super().__init__( helpdesc='LINE' )

   def match( self, mode, context, token ):
      matchResult = super().match( mode, context, token )
      matchResult.aaaTokens.clear()
      return matchResult

class QuotedStringMatcher( StringMatcher ):
   __slots__ = ( 'requireQuote_', )
   # Matches one word or quoted multi-word

   def __init__( self, pattern='.+', helpname='QUOTED LINE',
                 helpdesc='A quoted string or a single unquoted word',
                 requireQuote=False, **kargs ):
      StringMatcher.__init__( self, pattern=pattern,
                              helpname=helpname, helpdesc=helpdesc,
                              **kargs )
      self.requireQuote_ = requireQuote

   def _parseToken( self, context, token ):
      # return ( token, finished ). token can be noMatch.
      finished = False
      if context.state is None:
         # first token, check if we have quotes
         if token.startswith( '"' ):
            if token.endswith( '"' ) and len( token ) > 1:
               token = token[ 1 : -1 ]
               finished = True
            else:
               token = token[ 1 : ]
         elif self.requireQuote_:
            return noMatch, True
         else:
            # unquoted token, we are done
            finished = True
      else:
         if token.endswith( '"' ):
            token = token[ : -1 ]
            finished = True

      return token, finished

   def match( self, mode, context, token ):
      token, finished = self._parseToken( context, token )
      if token is noMatch:
         return noMatch

      # call StringMatcher
      r = StringMatcher.match( self, mode, context, token )
      if r is noMatch:
         return r

      assert r.more

      if finished:
         r.more = False
         # AAA tokens are quoted
         r.aaaTokens[ 0 ] = '"' + r.aaaTokens[ 0 ]
         r.aaaTokens[ -1 ] = r.aaaTokens[ -1 ] + '"'
         return r
      else:
         return partialMatch

   def completions( self, mode, context, token ):
      if token:
         token, _ = self._parseToken( context, token )
      if token is not noMatch:
         return StringMatcher.completions( self, mode, context, token )
      return []

   def __str__( self ):
      return '<QuotedString>'

class _RangeMatcherBase( Matcher ):
   __slots__ = ( 'dynamic_', )

   def __init__( self, dynamic, **kwargs ):
      Matcher.__init__( self, **kwargs )
      self.dynamic_ = dynamic

   def completions( self, mode, context, token ):
      lbound, ubound = self._computeRange( mode, context )
      if lbound > ubound:
         return []

      if token == '' or ( self.match( mode, context, token ) is not noMatch ):
         helpname = self.helpname_ or self._getFmtStr() % ( lbound, ubound )
         return [ CliParserCommon.Completion( helpname, self.helpdesc_,
                                              False, common=self.common_ ) ]
      return []

   def _computeRange( self, mode, context ):
      raise NotImplementedError

   def _getFmtStr( self ):
      raise NotImplementedError

   def toValue( self, token ):
      raise NotImplementedError

   def match( self, mode, context, token ):
      val = self.toValue( token )
      if val is not None:
         lbound, ubound = self._computeRange( mode, context )
         if ( lbound <= val <= ubound or
              # for dynamic matchers, allow it in startup-config
              ( self.dynamic_ and mode and mode.session and
                mode.session.startupConfig() ) ):
            return MatchResult( val, str( val ) )
      return noMatch

class _IntegerMatcherBase( _RangeMatcherBase ):
   # Regular expressions for the formats of integers we accept.
   decRegEx = re.compile( r'^-?[0-9]+$' )
   hexRegEx = re.compile( r'^0[xX][0-9a-fA-F]+$' )

   def toValue( self, token ):
      # Convert our token to base-10 or base-16 integer.
      if self.decRegEx.match( token ):
         return int( token )

      if self.hexRegEx.match( token ):
         return int( token, 16 )

      return None

   def _getFmtStr( self ):
      return '<%d-%d>'

   # Note that we assume that calling self._computeRange() may be an expensive
   # operation, and therefore we go to some effort within this class to minimize the
   # number of times it is called.
   def _computeRange( self, mode, context ):
      raise NotImplementedError

class _FloatMatcherBase( _RangeMatcherBase ):
   def __init__( self, *args, **kargs ):
      self.precisionString_ = kargs.pop( 'precisionString' )
      err = "Expected format string such as '%%.1f', but got %r instead."
      if not isinstance( self.precisionString_, str ):
         raise TypeError( err % type( self.precisionString_ ) )
      try:
         self.precisionString_ % 3.1415
      except TypeError: # Not all arguments converted or not enough arguments.
         # pylint: disable-next=raise-missing-from
         raise ValueError( err % self.precisionString_ )
      super().__init__( *args, **kargs )

   def _getFmtStr( self ):
      return '<%s-%s>' % ( self.precisionString_, self.precisionString_ )

   def toValue( self, token ):
      try:
         return float( token )
      except ValueError:
         return None

   def _computeRange( self, mode, context ):
      raise NotImplementedError

def NumberMatcherFactory( baseClass ):
   class StaticMatcher( baseClass ):
      def __init__( self, lbound, ubound, **kargs ):
         if lbound > ubound:
            raise ValueError( 'Empty range. %s > %s.' % ( lbound, ubound ) )
         super().__init__( False, **kargs )
         self.lbound_ = lbound
         self.ubound_ = ubound

      def _computeRange( self, mode, context ):
         return self.lbound_, self.ubound_

      def __str__( self ):
         return self._getFmtStr() % ( self.lbound_, self.ubound_ )

   class DynamicMatcher( baseClass ):
      def __init__( self, rangeFn, **kargs ):
         if not callable( rangeFn ):
            raise TypeError( '`rangeFn` must be callable' )

         super().__init__( True, **kargs )

         self._computeRange = rangeFn
         # pylint: disable-next=deprecated-method
         assert len( getfullargspec( rangeFn ).args ) == 2, debugFuncName( rangeFn )

         self.template_ = '<Dynamic%sMatcher>(%%s)'
         self.template_ %= 'Float' if 'precisionString' in kargs else 'Integer'

      def __str__( self ):
         return self.template_ % debugFuncName( self._computeRange )

   return StaticMatcher, DynamicMatcher

IntegerMatcher, DynamicIntegerMatcher = NumberMatcherFactory( _IntegerMatcherBase )
FloatMatcher, DynamicFloatMatcher = NumberMatcherFactory( _FloatMatcherBase )

class WrapperMatcher( Matcher ):
   # matcher wrapping another matcher with a different priority or value function
   __slots__ = ( 'matcher_', )

   def __init__( self, matcher, priority=None, value=None ):
      # by default use the underlying matcher's priority and value function
      self.matcher_ = matcher
      if priority is None:
         priority = matcher.priority_
      Matcher.__init__( self, priority=priority, value=value )

   def match( self, mode, context, token ):
      return self.matcher_.match( mode, context, token )

   def completions( self, mode, context, token ):
      return self.matcher_.completions( mode, context, token )

   def valueFunction( self, context ):
      return self.value_ or self.matcher_.valueFunction( context )

class DelimitedIntegerMatcherBase_( Matcher ):
   """Type of matcher that matches an integer-with-sub format, e.g.,
      x.y or x/y based on delimiter, where x is between 'lbound' and 'ubound',
      and y is between 'subLbound' and 'subUbound'. x.y is not a float number, but
      a number format for subinterfaces for example. All ranges are inclusive.
      Likewise x/y format used to represent the tunnel counter keys.
   """
   __slots__ = ( 'lbound_', 'ubound_', 'subLbound_', 'subUbound_', 'rangeFn_' )
   # Up to the subclasses.
   delimiter_ = None
   fmtStr_ = None

   def __init__( self, lbound=None, ubound=None, rangeFn=None,
                 subLbound=CliParserCommon.subLowerBound,
                 subUbound=CliParserCommon.subUpperBound,
                 **kargs ):
      super().__init__( **kargs )
      msg = 'Must specify either `rangeFn` or both `lbound` and `ubound`'
      dynamic = rangeFn is not None
      static = lbound is not None
      assert dynamic ^ static, msg
      assert ( lbound is None ) == ( ubound is None ), msg

      if rangeFn:
         # pylint: disable-next=deprecated-method
         assert len( getfullargspec( rangeFn ).args ) == 2, debugFuncName( rangeFn )

      self.lbound_ = lbound
      self.ubound_ = ubound
      self.subLbound_ = subLbound
      self.subUbound_ = subUbound
      self.rangeFn_ = rangeFn

   def _computeRange( self, mode, context ):
      if self.rangeFn_:
         return self.rangeFn_( mode, context )
      return self.lbound_, self.ubound_

   def match( self, mode, context, token ):
      try: # Trailing comments note possible exceptions.
         value, subValue = token.split( self.delimiter_ ) # `ValueError`.
         # Assume `_computeRange` is expensive, so check lower bounds first.
         if self.subLbound_ <= int( subValue ) <= self.subUbound_: # `ValueError`.
            lbound, ubound = self._computeRange( mode, context )
            if lbound <= int( value ) <= ubound: # `ValueError`.
               return MatchResult( token, token )

      except ValueError:
         pass
      return noMatch

   def completions( self, mode, context, token ):
      try:
         val = int( token.split( self.delimiter_ )[ 0 ] ) if token != '' else 0
      except ValueError:
         return []
      lbound, ubound = self._computeRange( mode, context )
      if token == '' or lbound <= val <= ubound:
         if self.delimiter_ not in token:
            helpname = self.helpname_ or self.fmtStr_ % ( lbound, ubound,
               self.subLbound_, self.subUbound_ )
         elif ( token[ -1 ] == self.delimiter_ or
                self.match( mode, context, token ) is not noMatch ):
            helpname = self.helpname_ or self.subRangeStr()
         else:
            return []
         return [ Completion( helpname, self.helpdesc_, False ) ]

      return []

   def __str__( self ):
      if self.rangeFn_:
         funcStr = debugFuncName( self.rangeFn_ )
         return '<%s(%s)>' % ( self.__class__.__name__, funcStr )
      return self.fmtStr_ % ( self.lbound_, self.ubound_,
                              self.subLbound_, self.subUbound_ )

   def subRangeStr( self ):
      return '<%d-%d>' % ( self.subLbound_, self.subUbound_ )

class DottedIntegerMatcher( DelimitedIntegerMatcherBase_ ):
   """Type of matcher that matches an integer-with-sub format, e.g.,
      x.y, where x is between 'lbound' and 'ubound', and y is between 'subLbound'
      and 'subUbound'. x.y is not a float number, but a number format for
      subinterfaces for example. All ranges are inclusive.
      DottedIntegerMatcher derived from DelimitedIntegerMatcherBase_.
   """
   delimiter_ = '.'
   fmtStr_ = '<%d-%d>' + delimiter_ + '<%d-%d>'

class SlashedIntegerMatcher( DelimitedIntegerMatcherBase_ ):
   '''Type of matcher that matches an integer-with-sub format, e.g.,
      x/y, where x is between 'lbound' and 'ubound', and y is between 'subLbound'
      and 'subUbound'. Currently the x/y format used in the tunnel platform
      dependent/independent counter keys representation.
      SlashedIntegerMatcher derived from DelimitedIntegerMatcherBase_.
   '''
   delimiter_ = '/'
   fmtStr_ = '<%d-%d>' + delimiter_ + '<%d-%d>'
