#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
import CliMode

class RoutingBgpBaseMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'router bgp %s' % self.param_

   def __init__( self, param ):
      # The industry standard cli has "config-router" as the prompt
      # for this mode.
      self.modeKey = 'router'
      self.longModeKey = 'router-bgp'
      self.addrFamily = 'all'
      CliMode.ConfigMode.__init__( self, param )

class RoutingBgpVrfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'vrf %s' % self.param_

   def __init__( self, param ):
      # The industry standard cli has "config-router" as the prompt
      # for this mode.
      self.modeKey = 'router'
      self.longModeKey = 'router-bgp-vrf-%s' % param
      self.addrFamily = 'all'
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpBaseMode )

class BgpFieldSetMappingsVrfMode( CliMode.ConfigMode ):
   def __init__( self, mapCtx, vrfName ):
      assert mapCtx in [ 'tp', 'vsp' ]
      self.mapCtx_ = mapCtx
      self.vrfName_ = vrfName
      self.cmdPrefix = 'traffic-policy' if mapCtx == 'tp' else 'vrf selection policy'
      self.modeKey = '%s-field-set-mappings' % mapCtx
      self.longModeKey = 'vrf-%s-%s-field-set-mappings' % ( vrfName, mapCtx )
      CliMode.ConfigMode.__init__( self, vrfName, parentMode=RoutingBgpVrfMode )

   def enterCmd( self ):
      return self.cmdPrefix + ' field-set mappings'

class FieldSetSharedMode( CliMode.ConfigMode ):
   def __init__( self, param, fsName, mapCtx, vrfName ):
      assert mapCtx in [ 'tp', 'vsp' ]
      self.ipv6 = param
      self.ipStr = 'ipv%s' % ( 4 if not self.ipv6 else 6 )
      self.fsName = fsName
      self.mapCtx = mapCtx
      self.vrfName = vrfName
      self.modeKey = '%s-field-set' % mapCtx
      self.longModeKey = '%s-field-set-%s-%s' % ( mapCtx, self.ipStr, self.fsName )
      CliMode.ConfigMode.__init__(
         self, param, parentMode=BgpFieldSetMappingsVrfMode )

   def getContext( self ):
      return ( self.ipv6, self.fsName )

   def enterCmd( self ):
      return 'field-set %s %s' % ( self.ipStr, self.fsName )

class FieldSetIpv4BaseMode( FieldSetSharedMode ):
   def __init__( self, fsName, mapCtx, vrfName ):
      FieldSetSharedMode.__init__( self, False, fsName, mapCtx, vrfName )

class FieldSetIpv6BaseMode( FieldSetSharedMode ):
   def __init__( self, fsName, mapCtx, vrfName ):
      FieldSetSharedMode.__init__( self, True, fsName, mapCtx, vrfName )

class RoutingBgpBaseAfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'address-family %s' % self.param_

   def __init__( self, param, modeKey=None ):
      # The industry standard cli has "config-router-af" as the prompt
      # for this mode (i.e. no '-v4'/'-v6' suffix).
      self.modeKey = 'router'
      if modeKey:
         self.longModeKey = modeKey
      else:
         self.longModeKey = 'router-bgp-af'
      self.addrFamily = param
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpBaseMode )

   def commentKey( self ):
      # For labeled-unicast or srte mode we want to append the 'ipv4' or 'ipv6',
      # which is the first part of the addrFamily.
      if 'labeled-unicast' in self.addrFamily:
         return "router-bgp-af-label-%s" % self.addrFamily.split()[ 0 ]
      elif 'sr-te' in self.addrFamily:
         return "router-bgp-af-srte-%s" % self.addrFamily.split()[ 0 ]
      return "router-bgp-af-%s" % self.addrFamily.replace( ' ', '-' )

class RoutingBgpVrfAfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'address-family %s' % self.param_

   def __init__( self, param, vrfName ):
      self.vrfName_ = vrfName
      # The industry standard cli has "config-router-af" as the prompt
      # for this mode (i.e. no '-v4'/'-v6' suffix).
      self.modeKey = 'router'
      self.longModeKey = 'router-bgp-vrf-%s-af' % vrfName
      self.addrFamily = param
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpVrfMode )

   def commentKey( self ):
      return "router-bgp-vrf-%s-af-%s" % ( self.vrfName_, self.addrFamily )

class BgpOrrPositionMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'optimal-route-reflection position %s' % ( self.param_ )

   def __init__( self, param ):
      self.modeKey = 'router-bgp-orr-position'
      self.longModeKey = 'router-bgp-orr-position-%s' % ( param )
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpBaseMode )

   @property
   def orrPositionName( self ):
      return self.param_

class BgpRfdPolicyMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'route-flap-damping %s' % ( self.param_ )

   def __init__( self, param ):
      self.modeKey = 'router-bgp-rfd'
      self.longModeKey = 'router-bgp-rfd-%s' % ( param )
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpBaseMode )

   @property
   def rfdPolicyName( self ):
      return self.param_

class BgpRouteDistinguisherMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'route-distinguisher'

   def __init__( self ):
      self.modeKey = 'rd'
      self.longModeKey = 'route-distinguisher'
      CliMode.ConfigMode.__init__( self, None, parentMode=RoutingBgpBaseMode )
