#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliMode
import BasicCli
from EosDhcpServerLib import reservationsAristaSwitchCmd
from EosDhcpServerLib import formatInfoOptionHexOrStrCmd
from EosDhcpServerLib import vrfStr
import Tac

# maximum number of dns servers we allow
_maxDnsServers = 2

# pylint: disable-msg=W0105
"""
Mode Structure
   dhcp server mode
      client class mode (IPv4)
         assignment mode
         match mode
            nested match mode
      client class mode (IPv6)
         assignment mode
         match mode
            nested match mode
      subnet mode (IPv4)
         range mode
            client class mode
               assignment mode
               match mode
         client class mode
            assignment mode
            match mode
         reservation mode
            mac-address mode
      client class mode (IPv6)
      subnet mode (IPv6)
         range mode
            client class mode
               assignment mode
               match mode
         client class mode
            assignment mode
            match mode
         reservation mode
            mac-address mode
      vendor-option mode
"""

class DhcpServerBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      dhcp server mode
   """

   def enterCmd( self ):
      if self.vrf == 'default':
         return "dhcp server"
      else:
         return f"dhcp server vrf {self.vrf}"

   def __init__( self, param ):
      self.vrf = param
      self.modeKey = f"dhcp-{self.vrf}"
      self.vrfStr = vrfStr( self.vrf )
      self.longModeKey = f"dhcp-server{self.vrfStr}"
      CliMode.ConfigMode.__init__( self, None )

class DhcpServerMode( DhcpServerBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server"

   def __init__( self, parent, session, vrf ):
      DhcpServerBaseMode.__init__( self, vrf )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = None

class DhcpServerAssignBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      client class assignments mode
   """

   # param will be ( clientClassConfig, clientClassName, vrf, af )
   def __init__( self, param ):
      self.clientClassConfig = param[ 0 ]
      self.clientClassName, self.vrf, self.af = param[ 1 : ]
      self.vrfStr = vrfStr( self.vrf )
      self.modeKey = f"{self.modeKeyStr()}-{self.vrf}"
      afStr = 'v4' if self.af == 'ipv4' else 'v6'
      self.longModeKey = self.longModeKeyFormatStr().format( self.vrfStr, afStr )
      CliMode.ConfigMode.__init__( self, param[ 1 : ] )

   def modeKeyStr( self ):
      raise NotImplementedError

   def longModeKeyFormatStr( self ):
      raise NotImplementedError

   def enterCmd( self ):
      return "assignments"

class DhcpServerClientClassAssignBaseMode( DhcpServerAssignBaseMode ):
   """
   CLI mode base class for:
      client class assignments mode
   """

   # param will be ( clientClassConfig, clientClassName, vrf, af )
   def __init__( self, param ):
      DhcpServerAssignBaseMode.__init__( self, param )

   def modeKeyStr( self ):
      return "client class assignments vrf {}"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-cls-{}-assignments"

   def commentKey( self ):
      return "dhcp-client-class-assignments-{}-{}{}".format(
            self.af, self.clientClassName, self.vrfStr )

   def dnsServersIs( self, args ):
      servers = args[ 'SERVERS' ]
      self.clientClassConfig.dnsServers.clear()

      ipAddrFn = Arnet.IpAddr if args[ 'af' ] == 'ipv4' else Arnet.Ip6Addr
      for i in range( min( _maxDnsServers, len( servers ) ) ):
         self.clientClassConfig.dnsServers[ i ] = ipAddrFn( servers[ i ] )

   def dnsServersDel( self, args ):
      self.clientClassConfig.dnsServers.clear()

class DhcpServerSubnetClientClassAssignBaseMode( DhcpServerAssignBaseMode ):
   """
   CLI mode base class for:
      client class assignments mode
   """

   # param will be ( subnet, clientClassConfig, clientClassName, af )
   def __init__( self, param ):
      DhcpServerAssignBaseMode.__init__( self, param[ 1 : ] )
      self.subnet = param[ 0 ]

   def modeKeyStr( self ):
      return "subnet client class assignments"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-sub-{}-cls-assignments"

   def commentKey( self ):
      return "dhcp-subnet-{}-client-class-{}-assignments{}".format(
               self.subnet, self.clientClassName, self.vrfStr )

class DhcpServerRangeClientClassAssignBaseMode( DhcpServerAssignBaseMode ):
   """
   CLI mode base class for:
      client class assignments mode
   """

   # param will be ( subnet, range, clientClassConfig, clientClassName, vrf, af )
   def __init__( self, param ):
      DhcpServerAssignBaseMode.__init__( self, param[ 2 : ] )
      self.subnet = param[ 0 ]
      self.range = param[ 1 ]

   def modeKeyStr( self ):
      return "range client class assignments"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-sub-{}-rng-cls-assignments"

   def commentKey( self ):
      return "dhcp-subnet-{}-range-{}-{}-client-class-{}-assignments{}".format(
                self.subnet, self.range.start, self.range.end, self.clientClassName,
                self.vrfStr )

class DhcpServerClientClassBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      client class mode
      subnet client class mode
      range client class mode
   """
   def enterCmd( self ):
      raise NotImplementedError()

   def __init__( self, param, parentMode=None ):
      self.clientClassName, self.vrf, self.af = param
      self.vrfStr = vrfStr( self.vrf )
      self.modeKey = "client class"
      self.longModeKey = "stub; to be updated in parent class"
      CliMode.ConfigMode.__init__( self, param, parentMode=parentMode )

class DhcpServerGlobalClientClassBaseMode( DhcpServerClientClassBaseMode ):
   """
   CLI mode base class for:
      client class mode
   """
   def enterCmd( self ):
      return f"client class {self.af} definition {self.clientClassName}"

   # param will be ( clientClassName, af, vrf )
   def __init__( self, param ):
      DhcpServerClientClassBaseMode.__init__( self, param )
      self.longModeKey = f"dhcp{self.vrfStr}-client-class-{self.af}"

   def commentKey( self ):
      return "dhcp-client-class-{}-{}{}".format(
            self.af, self.clientClassName, self.vrfStr )

class DhcpServerMatchBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      all match modes
   """

   def __init__( self, param ):
      self.clientClassName, self.matchAny, self.vrf, self.af = param
      self.modeKey = f"{self.modeKeyStr()}-vrf-{self.vrf}"
      self.vrfStr = vrfStr( self.vrf )
      afStr = 'v4' if self.af == 'ipv4' else 'v6'
      self.longModeKey = self.longModeKeyFormatStr().format( self.vrfStr, afStr )
      CliMode.ConfigMode.__init__( self, param )

   def modeKeyStr( self ):
      raise NotImplementedError

   def longModeKeyFormatStr( self ):
      raise NotImplementedError

   def enterCmd( self ):
      return "match any" if self.matchAny else "match"

class DhcpServerClientClassMatchBaseMode( DhcpServerMatchBaseMode ):
   """
   CLI mode base class for:
      global client class match mode
   """

   def __init__( self, param ):
      DhcpServerMatchBaseMode.__init__( self, param )

   def modeKeyStr( self ):
      return "client class match"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-cls-{}-match"

   def commentKey( self ):
      anyStr = '-any' if self.matchAny else ''
      return "dhcp-client-class-{}-{}-match{}{}".format(
            self.af, self.clientClassName, anyStr, self.vrfStr )

class DhcpServerSubnetClientClassMatchBaseMode( DhcpServerMatchBaseMode ):
   """
   CLI mode base class for:
      subnet client class match mode
   """

   def __init__( self, param ):
      DhcpServerMatchBaseMode.__init__( self, param[ 1 : ] )
      self.subnetPrefix = param[ 0 ]

   def modeKeyStr( self ):
      return "subnet client class match"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-sub-cls-{}-match"

   def commentKey( self ):
      anyStr = '-any' if self.matchAny else ''
      return "dhcp-subnet-{}-client-class-{}-match{}{}".format( self.subnetPrefix,
                                                                self.clientClassName,
                                                                anyStr, self.vrfStr )

class DhcpServerRangeClientClassMatchBaseMode( DhcpServerMatchBaseMode ):
   """
   CLI mode base class for:
      range client class match mode
   """

   # param will be ( subnetConfig, rangeConfig, clientClassName, af )
   def __init__( self, param ):
      DhcpServerMatchBaseMode.__init__( self, param[ 2 : ] )
      self.rangeConfig = param[ 1 ]
      self.subnetConfig = param[ 0 ]

   def modeKeyStr( self ):
      return "range client class match"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-sub-{}-rng-cls-match"

   def commentKey( self ):
      anyStr = '-any' if self.matchAny else ''
      return "dhcp-subnet-{}-range-{} {}-client-class-{}-match{}{}".format(
                                                      self.subnetConfig.subnetId,
                                                      self.rangeConfig.range.start,
                                                      self.rangeConfig.range.end,
                                                      self.clientClassName,
                                                      anyStr, self.vrfStr )

class DhcpServerNestedMatchBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      all nested match mode
   """

   def __init__( self, param ):
      self.clientClassName, self.matchAny, self.matchName, self.vrf, self.af = param
      self.vrfStr = vrfStr( self.vrf )
      self.modeKey = self.modeKeyStr()
      afStr = 'v4' if self.af == 'ipv4' else 'v6'
      self.longModeKey = self.longModeKeyFormatStr().format(
            self.vrfStr, afStr, self.matchName )
      CliMode.ConfigMode.__init__( self, param )

   def modeKeyStr( self ):
      raise NotImplementedError

   def longModeKeyFormatStr( self ):
      raise NotImplementedError

   def enterCmd( self ):
      anyStr = ' any' if self.matchAny else ''
      return f"match{anyStr} {self.matchName}"

class DhcpServerClientClassNestedMatchBaseMode( DhcpServerNestedMatchBaseMode ):
   """
   CLI mode base class for:
      nested match mode
   """

   def __init__( self, param ):
      DhcpServerNestedMatchBaseMode.__init__( self, param )

   def modeKeyStr( self ):
      return "client class match match"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-cls-{}-match-{}"

   def commentKey( self ):
      anyStr = '-any' if self.matchAny else ''
      return "dhcp-client-class-{}-{}-match-{}{}{}".format( self.af,
                                                            self.clientClassName,
                                                            self.matchName,
                                                            anyStr, self.vrfStr )

class DhcpServerSubnetClientClassNestedMatchBaseMode(
      DhcpServerNestedMatchBaseMode ):
   """
   CLI mode base class for:
      subnet nested match mode
   """

   def __init__( self, param ):
      DhcpServerNestedMatchBaseMode.__init__( self, param[ 1 : ] )
      self.subnetPrefix = param[ 0 ]

   def modeKeyStr( self ):
      return "subnet client class match match"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-sub-cls-{}-match-{}"

   def commentKey( self ):
      anyStr = '-any' if self.matchAny else ''
      return "dhcp-subnet-{}-client-class-{}-{}-match-{}{}{}".format(
                   self.subnetPrefix, self.af, self.clientClassName, self.matchName,
                   anyStr, self.vrfStr )

class DhcpServerRangeClientClassNestedMatchBaseMode(
      DhcpServerNestedMatchBaseMode ):
   """
   CLI mode base class for:
      range nested match mode
   """

   """
   param will be ( subnetConfig, rangeConfig,
                   clientClassName, matchAny, matchName, af )
   """
   def __init__( self, param ):
      DhcpServerNestedMatchBaseMode.__init__( self, param[ 2 : ] )
      self.subnetConfig = param[ 0 ]
      self.rangeConfig = param[ 1 ]

   def modeKeyStr( self ):
      return "range client class match match"

   def longModeKeyFormatStr( self ):
      return "dhcp{}-sub-{}-rng-cls-match-{}"

   def commentKey( self ):
      anyStr = '-any' if self.matchAny else ''
      return "dhcp-subnet-{}-range-{}-{}-client-class-{}-match-{}{}{}".format(
                                                   self.subnetConfig.subnetId,
                                                   self.rangeConfig.range.start,
                                                   self.rangeConfig.range.end,
                                                   self.clientClassName,
                                                   self.matchName,
                                                   anyStr, self.vrfStr )

class DhcpServerSubnetBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      subnet mode
   """

   def enterCmd( self ):
      return "subnet %s" % self.subnet

   # param will be ( subnetConfig, subnet, af, vrf )
   def __init__( self, param ):
      self.subnetConfig = param[ 0 ]
      self.subnet, self.vrf, self.af = param[ 1 : ]
      self.modeKey = "subnet"
      self.vrfStr = vrfStr( self.vrf )
      self.longModeKey = f"dhcp{self.vrfStr}-subnet-{self.af}"
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return f"dhcp-subnet-{self.subnet}{self.vrfStr}"

   def dnsServersIs( self, args ):
      servers = args[ 'SERVERS' ]
      self.subnetConfig.dnsServers.clear()
      ipAddrFn = Arnet.IpAddr if args[ 'af' ] == 'ipv4' else Arnet.Ip6Addr
      for i in range( min( _maxDnsServers, len( servers ) ) ):
         self.subnetConfig.dnsServers[ i ] = ipAddrFn( servers[ i ] )

   def dnsServersDel( self, args ):
      self.subnetConfig.dnsServers.clear()

class DhcpServerSubnetV4Mode( DhcpServerSubnetBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Subnet"

   def __init__( self, parent, session, param ):
      prefix = Arnet.Prefix( param )
      self.subnetConfig = (
            parent.dhcpServerConfig.subnetConfigIpv4.newMember( prefix ) )
      self.vrf = parent.vrf
      param = ( self.subnetConfig, param, self.vrf, 'ipv4' )
      DhcpServerSubnetBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class DhcpServerRangeBaseMode( CliMode.ConfigMode ):
   """
   CLI mode and CLI save mode base class for:
      range mode
   """

   def enterCmd( self ):
      return f"range {self.range.start} {self.range.end}"

   # param will be set to ( subnetConfig, range, vrf,af )
   def __init__( self, param ):
      self.subnetConfig, self.range, self.vrf, self.af = param
      self.rangeConfig = None
      self.modeKey = "range"
      self.vrfStr = vrfStr( self.vrf )
      self.longModeKey = f"dhcp{self.vrfStr}-subnet-{self.af}-range"
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=DhcpServerSubnetBaseMode )

   def commentKey( self ):
      return "dhcp-subnet-{}-range-{}-{}{}".format( self.subnetConfig.subnetId,
                                                     self.range.start,
                                                     self.range.end, self.vrfStr )

class DhcpServerSubnetClientClassBaseMode( DhcpServerClientClassBaseMode ):
   """
   CLI mode base class for:
      subnet's client class mode
   """

   def enterCmd( self ):
      return f"client class definition {self.clientClassName}"

   # param will be ( subnetConfig, clientClassName, af )
   def __init__( self, param ):
      self.subnetConfig = param[ 0 ]
      DhcpServerClientClassBaseMode.__init__( self, param[ 1 : ],
                                              parentMode=DhcpServerSubnetBaseMode )
      clientClassConfig = self.subnetConfig.clientClassConfig
      self.subnetClientClassConfig = clientClassConfig[ self.clientClassName ]
      self.longModeKey = "dhcp{}-subnet-client-class-{}".format(
            self.vrfStr, self.af )

   def commentKey( self ):
      subnetPrefix = self.subnetConfig.subnetId
      return "dhcp-subnet-{}-client-class-{}{}".format( subnetPrefix,
                                                        self.clientClassName,
                                                        self.vrfStr )

class DhcpServerRangeClientClassBaseMode( DhcpServerClientClassBaseMode ):
   """
   CLI mode base class for:
      range's client class mode
   """

   def enterCmd( self ):
      return f"client class definition {self.clientClassName}"

   # param will be ( subnetConfig, rangeConfig, clientClassName, af )
   def __init__( self, param ):
      self.rangeConfig = param[ 1 ]
      self.subnetConfig = param[ 0 ]
      self.range = self.rangeConfig.range
      DhcpServerClientClassBaseMode.__init__( self, param[ 2 : ],
                                              parentMode=DhcpServerRangeBaseMode )
      clientClassConfig = self.rangeConfig.clientClassConfig
      self.rangeClientClassConfig = clientClassConfig[ self.clientClassName ]
      self.longModeKey = "dhcp{}-sub-{}-rng-class".format(
            self.vrfStr, self.af[ 2 : ] )

   def commentKey( self ):
      commentKeyString = "dhcp-subnet-{}-range-{}-{}-client-class-{}{}"
      return commentKeyString.format( self.subnetConfig.subnetId,
                                      self.range.start,
                                      self.range.end,
                                      self.clientClassName,
                                      self.vrfStr )

class DhcpServerVendorOptionBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      vendor-option mode
   """

   def enterCmd( self ):
      return f"vendor-option {self.af} {self.vendorId}"

   def __init__( self, param ):
      self.vendorId, self.vrf, self.af = param
      self.vrfStr = vrfStr( self.vrf )
      self.modeKey = "vendor"
      self.longModeKey = f"dhcp{self.vrfStr}-vendor-{self.af}"
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return f"dhcp-vendor-{self.af}-{self.vendorId}{self.vrfStr}"

class DhcpServerReservationsBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      reservations mode

   This is an umbrella mode for:
      mac-address mode
   """

   def enterCmd( self ):
      return "reservations"

   # param will be ( subnetConfig, vrf, af )
   def __init__( self, param ):
      self.subnetConfig, self.vrf, self.af = param
      self.vrfStr = vrfStr( self.vrf )
      self.modeKey = "reservations"
      afStr = 'v4' if self.af == 'ipv4' else 'v6'
      self.longModeKey = f"dhcp{self.vrfStr}-sub-{afStr}-reserve"
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=DhcpServerSubnetBaseMode )

   def commentKey( self ):
      subnetPrefix = self.subnetConfig.subnetId
      return f"dhcp-subnet-{subnetPrefix}-reservations{self.vrfStr}"

class DhcpServerReservationsMacAddressBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      mac-address mode
   """

   def enterCmd( self ):
      return f"mac-address {self.macAddr}"

   # param will be ( subnetConfig, macAddr, vrf, af )
   def __init__( self, param ):
      self.subnetConfig, self.macAddr, self.vrf, self.af = param
      self.vrfStr = vrfStr( self.vrf )
      self.modeKey = "mac-address"
      afStr = 'v4' if self.af == 'ipv4' else 'v6'
      self.longModeKey = "dhcp{}-sub-{}-rsrv-mac-address".format(
            self.vrfStr, afStr )
      self.reservationConfig = self.subnetConfig.reservationsMacAddr[ self.macAddr ]
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=DhcpServerReservationsBaseMode )

   def commentKey( self ):
      subnetPrefix = self.subnetConfig.subnetId
      return "dhcp-subnet-{}-reservations-mac-address-{}{}".format(
            subnetPrefix, self.macAddr, self.vrfStr )

class DhcpServerReservationsAristaSwitchBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      { header } arista-switch ( ( Ethernet <ETH_PORT> |
         Port-Channel <PORT-CHANNEL> ) vlan <VLAN> [ switch-mac <MAC> ] |
         switch-mac <MAC> )
   where the header can be either information-option or remote-id,
   depending on whether we're in v4 or v6
   """
   def enterCmd( self ):
      return reservationsAristaSwitchCmd( self.port, self.vlan, self.macAddr,
                                         self.af )

   def __init__( self, param ):
      ( subnetConfig, port, vlan, macAddr, vrf ) = param
      self.macAddr = macAddr
      self.vlan = vlan
      self.port = port
      self.subnetConfig = subnetConfig
      self.vrf = vrf
      self.vrfStr = vrfStr( vrf )
      self.header = "information-option" if self.af == "ipv4" else "remote-id"
      self.modeKey = f"arista-{self.header}"
      self.longModeKey = "dhcp{}-sub-v4-rsrv-arista-{}".format(
            self.vrfStr, self.header )

      tacType = 'DhcpServer::Layer2Info'
      self.layer2Info = Tac.Value( tacType, self.port, self.vlan, self.macAddr )
      self.reservationConfig = self.subnetConfig.reservationLayer2Intf[
            self.layer2Info ]
      CliMode.ConfigMode.__init__(
            self, param, parentMode=DhcpServerReservationsAristaSwitchBaseMode )

   @property
   def af( self ):
      raise NotImplementedError

   def commentKey( self ):
      subnetPrefix = self.subnetConfig.subnetId
      commentKey = "dhcp-subnet-{}-reservations-arista-{}".format(
            subnetPrefix, self.header )
      if self.port == '':
         commentKey += f"-{self.macAddr}"
      else:
         commentKey += "-{}-vlan-{}".format( self.port.strip( "'" ), self.vlan )
         if self.macAddr == '0000.0000.0000':
            commentKey += f"-{self.macAddr}"
      commentKey += self.vrfStr
      return commentKey

class DhcpServerReservationsInfoOptionBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      information option ( ( CIRCUIT-ID | REMOTE-ID ) ) | REMOTE-ID )
   """
   def enterCmd( self ):
      return formatInfoOptionHexOrStrCmd( self.infoOpt )

   def __init__( self, param ):
      ( subnetConfig, infoOpt, vrf ) = param
      self.subnetConfig = subnetConfig
      self.infoOpt = infoOpt
      self.vrf = vrf
      self.vrfStr = vrfStr( vrf )
      self.modeKey = "information-option"
      self.longModeKey = "dhcp{}-sub-v4-rsrv-information-option".format(
            self.vrfStr )
      self.reservationConfig = self.subnetConfig.reservationCircuitIdRemoteId[
            self.infoOpt ]
      CliMode.ConfigMode.__init__(
            self, param, parentMode=DhcpServerReservationsInfoOptionBaseMode )

   def commentKey( self ):
      subnetPrefix = self.subnetConfig.subnetId
      commentKey = "dhcp-subnet-{}-reservations-information-option".format(
            subnetPrefix )
      if self.infoOpt.circuitIdHex:
         commentKey += f"-circuit-id-{self.infoOpt.circuitIdHex}"
      elif self.infoOpt.circuitIdStr:
         commentKey += f"-circuit-id-{self.infoOpt.circuitIdStr}"
      if self.infoOpt.remoteIdHex:
         commentKey += f"-remote-id-{self.infoOpt.remoteIdHex}"
      elif self.infoOpt.remoteIdStr:
         commentKey += f"-remote-id-{self.infoOpt.remoteIdStr}"
      commentKey += self.vrfStr
      return commentKey

class DhcpServerReservationsInfoOptionMode(
                           DhcpServerReservationsInfoOptionBaseMode,
                           BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Information Option Reservations"

   def __init__( self, parent, session, subnetConfig, infoOpt ):
      self.vrf = parent.vrf
      param = ( subnetConfig, infoOpt, self.vrf )
      DhcpServerReservationsInfoOptionBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
