#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import Tracing
from InbandTelemetryTypes import tacPortProfileType
import CliMode
from CliMode.SamplePolicy import ( SamplePolicyConfigMode,
                                   SamplePolicyMatchRuleConfigMode,
                                   SamplePolicyActionRuleConfigMode )

__defaultTraceHandle__ = Tracing.Handle( 'InbandTelemetry' )
t0 = __defaultTraceHandle__.trace0

FEATURE = 'inband-sample-policy'
class InbandTelemetryModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return "monitor telemetry inband"

   def __init__( self ):
      self.modeKey = "mon-telemetry-inband"
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, None )

class InbandTelemetryConfigMode( InbandTelemetryModeBase, BasicCli.ConfigModeBase ):
   name = "mon-telemetry-inband"

   def __init__( self, parent, session, context ):
      self.telemetryContext = context
      InbandTelemetryModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.telemetryContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      t0( 'Commit telemetry inband' )
      context = self.telemetryContext
      if context:
         context.commit()
         self.telemetryContext = None

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()

class ProfileModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return f"profile {self.localProfileType} {self.profileName}"

   def __init__( self, param ):
      self.profileName = param[ 0 ]
      self.profileType = param[ 1 ]
      if self.profileType == tacPortProfileType.IntCore:
         self.localProfileType = 'core'
      elif self.profileType == tacPortProfileType.IntEdge:
         self.localProfileType = 'edge'
      else:
         self.localProfileType = 'vxlan'
      self.modeKey = "profile-%s" % self.localProfileType
      self.longModeKey = "profile-{}-{}".format( self.localProfileType,
                                                 self.profileName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=InbandTelemetryConfigMode )

class ProfileConfigMode( ProfileModeBase, BasicCli.ConfigModeBase ):
   name = "port profile configuration"

   def __init__( self, parent, session, context ):
      self.profileContext = context
      param = ( context.profileName_, context.profileType_ )
      ProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( f'Exit {self.profileType} profile {self.profileName}' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.profileContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      t0( 'Commit port profile' )
      context = self.profileContext
      if context and ( self.profileName != 'default' or
            self.profileType == tacPortProfileType.IntVxlan ):
         context.commit()
         self.profileContext = None

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()

# Configuration mode for sample-policy
class IntSamplePolicyConfigMode( SamplePolicyConfigMode ):
   name = "INT sample-policy configuration"

# Match rule configuration mode
class IntSamplePolicyMatchRuleConfigMode( SamplePolicyMatchRuleConfigMode ):
   name = 'INT sample-policy match rule configuration'

# Action configuration mode
class IntSamplePolicyActionRuleConfigMode( SamplePolicyActionRuleConfigMode ):
   name = 'INT sample-policy action rule configuration'
