# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliMode
import Tracing

th = Tracing.Handle( 'PseudowireCliMode' )
t0 = th.t0

class PatchPanelMode( CliMode.ConfigMode ):
   def __init__( self ):
      self.modeKey = 'patch'
      self.longModeKey = 'patch'
      CliMode.ConfigMode.__init__( self, None )

   def enterCmd( self ):
      return 'patch panel'

class PatchMode( CliMode.ConfigMode ):
   def __init__( self, patchName ):
      self.patchName = patchName
      self.modeKey = 'patch-%s' % patchName
      self.longModeKey = 'patch-%s' % patchName
      CliMode.ConfigMode.__init__( self, patchName, parentMode=PatchPanelMode )

   def enterCmd( self ):
      return 'patch ' + self.patchName

# The PatchQosMode is only applicable for remote connectors (bgp/ldp) where they
# will be used to enter the qos submode.
# However, PatchQosMode is overloaded and used for local connectors to ensure
# a consistent ordering between all connectors in CliSave, which is why local
# connector constructs (such as intfName, vlanTag, and noErrDisable) are
# part of the mode constructor
# Local connectors must also define a modeKey since this uniquely identifies the
# PatchQosMode instance in CliSavePlugin when creating a mode instance.
class PatchQosBaseMode( CliMode.ConfigMode ):
   def __init__( self, patchName, connName, pwName=None, vpwsName=None,
                 intfName=None, vlan=None, altPwName=None, noErrDisable=False,
                 staticMplsPw=False ):
      self.connName = connName
      # This represents vpwsPwName for bgp pw and pwName for ldp pw
      self.pwName = pwName
      # This represents vpwsEviName
      self.vpwsName = vpwsName
      self.intfName = intfName
      self.vlan = vlan
      self.altPwName = altPwName
      self.noErrDisable = noErrDisable
      # This is True if the connector is a static MPLS connector
      self.staticMplsPw = staticMplsPw
      if vpwsName:
         self.modeKey = f'patch-{patchName}-pw-{pwName}-vpws-{vpwsName}'
         self.connType = 'bgp'
      elif self.staticMplsPw:
         self.modeKey = f'patch-{patchName}-pw-{pwName}'
         self.connType = 'mplsStatic'
      elif self.pwName:
         if self.altPwName:
            self.modeKey = 'patch-{}-pw-{}-alternate-{}'.format( patchName, pwName,
                                                                 altPwName )
         else:
            self.modeKey = f'patch-{patchName}-pw-{pwName}'
         self.connType = 'ldp'
      elif self.vlan is not None:
         assert self.intfName is not None
         self.modeKey = 'patch-%s-intf-%s-vlan-%d' % ( patchName, intfName, vlan )
         self.connType = 'tagged'
      elif self.noErrDisable:
         # Updating the modeKey, currently there is no submode for local connectors
         # and the regular intf modekey should be sufficient. This is to ensure that
         # error disable has a unique modeKey which does not cause any issues with
         # future changes to local connectors
         assert self.intfName is not None
         self.modeKey = f'patch-{patchName}-intf-{intfName}-no-errdisable'
         self.connType = 'port'
      else:
         assert self.intfName is not None
         self.modeKey = f'patch-{patchName}-intf-{intfName}'
         self.connType = 'port'
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, self.longModeKey, parentMode=PatchMode )

   def enterCmd( self ):
      if self.connType == 'bgp':
         return 'connector {} pseudowire bgp vpws {} pseudowire {}'.format(
                 self.connName or "", self.vpwsName, self.pwName )
      elif self.connType == 'ldp':
         cmd = 'connector {} pseudowire ldp {}'.format( self.connName or "",
                                                        self.pwName )
         if self.altPwName:
            cmd += ' alternate %s' % self.altPwName
         return cmd
      elif self.connType == 'mplsStatic':
         cmd = 'connector {} pseudowire mpls static {}'.format(
               self.connName or "", self.pwName )
         return cmd
      elif self.connType == 'tagged':
         return 'connector %s interface %s dot1q vlan %d' % ( self.connName or "",
                 self.intfName, self.vlan )
      elif self.connType == 'port' and self.noErrDisable:
         connName = self.connName or ""
         return f'connector {connName} interface {self.intfName} no-errdisable'
      return 'connector {} interface {}'.format( self.connName or "",
                                                 self.intfName )

class FxcMode( CliMode.ConfigMode ):
   def __init__( self, patchName ):
      self.patchName = patchName
      self.modeKey = 'fxc-%s' % patchName
      self.longModeKey = 'flexible-cross-connect-%s' % patchName
      CliMode.ConfigMode.__init__( self, patchName, parentMode=PatchPanelMode )
      t0( "FxcMode init", repr( patchName ), "modeKey", repr( self.modeKey ),
            "longModeKey", repr( self.longModeKey ) )

   def enterCmd( self ):
      return 'flexible-cross-connect ' + self.patchName

class MplsLdpPseudowiresMode( CliMode.ConfigMode ):
   def __init__( self ):
      self.modeKey = 'mpls-ldp-pw'
      self.longModeKey = 'mpls-ldp-pw'
      CliMode.ConfigMode.__init__( self, None, parentMode=CliMode.LdpMode.LdpMode )

   def enterCmd( self ):
      return 'pseudowires'
   
class LdpPseudowireMode( CliMode.ConfigMode ):
   def __init__( self, ldpPwName ):
      self.ldpPwName = ldpPwName
      self.modeKey = "mpls-ldp-pw-" + ldpPwName
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, ldpPwName,
                                   parentMode=MplsLdpPseudowiresMode )

   def enterCmd( self ):
      return "pseudowire " + self.ldpPwName

class MplsPseudowiresMode( CliMode.ConfigMode ):
   def __init__( self ):
      self.modeKey = "mpls-pw"
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, None )

   def enterCmd( self ):
      return "mpls pseudowires"

class MplsStaticPseudowiresMode( CliMode.ConfigMode ):
   def __init__( self ):
      self.modeKey = 'mpls-pw-static'
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, None,
                                   parentMode=MplsPseudowiresMode )

   def enterCmd( self ):
      return 'static pseudowires'

class StaticPseudowireMode( CliMode.ConfigMode ):
   def __init__( self, staticPwName ):
      self.staticPwName = staticPwName
      self.modeKey = "mpls-pw-static-" + self.staticPwName
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, staticPwName,
                                   parentMode=MplsStaticPseudowiresMode )

   def enterCmd( self ):
      return "pseudowire " + self.staticPwName

# See http://aid/9180 (VPLS LDP AD)
class LdpPseudowireProfileMode( CliMode.ConfigMode ):
   def __init__( self, ldpPwProfileName ):
      self.ldpPwProfileName = ldpPwProfileName
      self.modeKey = "mpls-ldp-pw-pf-" + ldpPwProfileName
      self.longModeKey = self.modeKey
      CliMode.ConfigMode.__init__( self, ldpPwProfileName,
                                   parentMode=MplsLdpPseudowiresMode )

   def enterCmd( self ):
      return "profile " + self.ldpPwProfileName
